//=============================================================================
// NrNovelGamePlus.js
//=============================================================================

/*:
 * @plugindesc ログ追加プラグイン V1.0.0
 * @author NJ
 *
 * @param LogWindowBackground
 * @desc ログ全体の背景画像（img/pictures/ に配置）
 * @default
 *
 * @param NameColor
 * @desc 名前表示部分の文字色（例：#FF0000）
 * @default #000000
 *
 * @param NameFontSize
 * @desc 名前表示のフォントサイズ（例：28）
 * @default 28
 *
 * @param LogColor
 * @desc ログ本文の文字色（例：#333333）
 * @default #000000
 *
 * @param LogFontSize
 * @desc ログ本文のフォントサイズ（例：22）
 * @default 22
 *
 * @param PlayButtonOffsetX
 * @desc 再生ボタンのX座標オフセット
 * @default 0
 * 
 * @param PlayButtonOffsetY
 * @desc 再生ボタンのY座標オフセット（名前の下からの相対値）
 * @default 0
 * 
 * @param CloseButtonImage
 * @desc img/pictures/ に配置する閉じるボタン画像ファイル名（拡張子除く）
 * @default
 *
 * @param PlayButtonImage
 * @desc img/pictures/ に配置する再生ボタン画像ファイル名（拡張子除く）
 * @default
 *
 * @param NamedBlockBackground
 * @desc img/pictures/ に配置する名前付きログ用背景画像ファイル名（拡張子除く）
 * @default
 *
 * @param NormalBlockBackground
 * @desc img/pictures/ に配置する通常ログ用背景画像ファイル名（拡張子除く）
 * @default
 *
 * @param FaceIcons
 * @type struct<FaceIcon>[]
 * @desc \NrN[名前]で指定できる顔画像の登録（例：名前→画像ファイル）
 * @default []
 *
 * @param IconWidth
 * @desc 顔アイコンの横幅
 * @default 96
 *
 * @param IconHeight
 * @desc 顔アイコンの縦幅
 * @default 96
 *
 * @param IconOffsetX
 * @desc 顔アイコンのX軸
 * @default 30
 *
 * @param IconOffsetY
 * @desc 顔アイコンのY軸
 * @default 10
 *
 * @param BaseLineCount
 * @text 名前なしログの行数
 * @desc 名前なしログブロックで確保する行数（初期値: 4）
 * @default 4
 *
 * @param LineSpacing
 * @desc 行間スペース
 * @default 4
 *
 * @param BlockSpacing
 * @desc ログウィンドウ同士のスペース
 * @default 16
 *
 * @help
 * このプラグインはMV用に調整されたログ機能を提供します。
 *
 * プラグインコマンド:
 *   LogStart     # ログ記録開始
 *   LogStop      # ログ記録停止
 *   LogClear     # ログ削除
 *   ShowLog      # ログ画面を表示
 *   SetLogName 名前  # 名前を設定（例: SetLogName アリス）
 *
 * バージョン:
 * v1.0.0 初回
 *
 * 利用規約：
 *  プラグイン作者に無断で使用、改変、再配布は不可です。
 */

/*~struct~FaceIcon:
 * @param name
 * @text 登録名
 * @desc \NrN[登録名] で指定する名前
 *
 * @param file
 * @text 顔画像ファイル名
 * @desc img/pictures/ に置いた画像ファイル名（拡張子なし）
 */

(() => {
    const params = PluginManager.parameters("NrNovelGamePlus");
    const CloseButtonImage = params["CloseButtonImage"] || "";
    const PlayButtonImage = params["PlayButtonImage"] || "";
    const NamedBlockBackground = params["NamedBlockBackground"] || "";
    const NormalBlockBackground = params["NormalBlockBackground"] || "";
    const NameColor = params["NameColor"] || "#000000";
    const NameFontSize = Number(params["NameFontSize"] || 28);
    const LogColor = params["LogColor"] || "#000000";
    const LogFontSize = Number(params["LogFontSize"] || 22);
    const FaceIcons = JSON.parse(params["FaceIcons"] || "[]").map(JSON.parse);
    const faceIconMap = {};
    for (const icon of FaceIcons) {
        faceIconMap[icon.name] = icon.file;
    }
    const IconWidth = Number(params["IconWidth"] || 96);
    const IconHeight = Number(params["IconHeight"] || 96);
    const IconOffsetX = Number(params["IconOffsetX"] || 30);
    const IconOffsetY = Number(params["IconOffsetY"] || 10);
    const LogWindowBackground = params["LogWindowBackground"] || "";
    const LineSpacing = Number(params["LineSpacing"] || 4);
    const BlockSpacing = Number(params["BlockSpacing"] || 16);
    const PlayButtonOffsetX = Number(params["PlayButtonOffsetX"] || 0);
    const PlayButtonOffsetY = Number(params["PlayButtonOffsetY"] || 0);
    const BaseLineCount = Number(params["BaseLineCount"] || 4);

    let isLogging = false;
    let messageLogBuffer = [];
    let currentMessageBlock = { name: "", messages: [], voice: null, iconName: null };

    const _Game_Interpreter_pluginCommand = Game_Interpreter.prototype.pluginCommand;
    Game_Interpreter.prototype.pluginCommand = function(command, args) {
        _Game_Interpreter_pluginCommand.call(this, command, args);

        if (command === "LogStart") {
            isLogging = true;
            if (CloseButtonImage) ImageManager.loadPicture(CloseButtonImage);
            if (PlayButtonImage) ImageManager.loadPicture(PlayButtonImage);
            if (NamedBlockBackground) ImageManager.loadPicture(NamedBlockBackground);
            if (NormalBlockBackground) ImageManager.loadPicture(NormalBlockBackground);
        }

        if (command === "LogStop") {
            isLogging = false;
        }

        if (command === "LogClear") {
            $gameMessage.clearMessageLog();
        }

        if (command === "ShowLog") {
            SceneManager.push(Scene_MessageLog);
        }

        if (command === "SetLogName") {
            const nameArg = args[0];
            if (isLogging && nameArg) {
                currentMessageBlock.name = `【${nameArg}】`;
            }
        }
    };

    const _Window_Message_terminateMessage = Window_Message.prototype.terminateMessage;
    Window_Message.prototype.terminateMessage = function() {
        if (isLogging && (currentMessageBlock.messages.length > 0 || currentMessageBlock.name || currentMessageBlock.voice)) {
            messageLogBuffer.push(currentMessageBlock);
        }
        currentMessageBlock = { name: "", messages: [], voice: null };
        _Window_Message_terminateMessage.call(this);
    };

    const _Game_Message_add = Game_Message.prototype.add;
    Game_Message.prototype.add = function(text) {
        if (isLogging) {
            if (!currentMessageBlock.voice) {
                const voiceMatch = text.match(/\\SV\[(.+?)\]/);
                if (voiceMatch) {
                    let raw = voiceMatch[1];
                    raw = raw.replace(/\\v\[(\d+)\]/gi, (_, id) => $gameVariables.value(Number(id)));
                    raw = raw.replace(/\\n\[(\d+)\]/gi, (_, id) => {
                        const actor = $gameActors.actor(Number(id));
                        return actor ? actor.name() : "";
                    });
                    currentMessageBlock.voice = raw;
                }
            }

            const processedText = processControlCharacters(text);
            const autoBrokenText = applyAutoLineBreak(processedText);

            autoBrokenText.split('\n').forEach(line => {
                currentMessageBlock.messages.push(line);
            });

            if (currentMessageBlock.name === "" && processedText.includes("【")) {
                const nameMatch = processedText.match(/【.+?】/);
                if (nameMatch) {
                    currentMessageBlock.name = nameMatch[0];
                }
            }
        }
        _Game_Message_add.call(this, text);
    };

    Game_Message.prototype.getMessageLog = function() {
        return messageLogBuffer;
    };

    Game_Message.prototype.clearMessageLog = function() {
        messageLogBuffer = [];
    };

    function processControlCharacters(text) {
        text = text.replace(/\\SV\[((?:\\[vV]\[(\d+)\]|\\[nN]\[(\d+)\]|[^\\\]]+)+)\]/g, (_, content) => {
            let resolved = content;

            resolved = resolved.replace(/\\[vV]\[(\d+)\]/g, (_, varId) => {
                return String($gameVariables.value(Number(varId)));
            });

            resolved = resolved.replace(/\\[nN]\[(\d+)\]/g, (_, actorId) => {
                const actor = $gameActors.actor(Number(actorId));
                return actor ? actor.name() : "";
            });

            if (isLogging && resolved) {
                currentMessageBlock.voice = resolved;
            }
            return "";
        });

        text = text.replace(/\\n\[(\d+)\]/g, (_, actorId) => {
            const actor = $gameActors.actor(Number(actorId));
            return actor ? actor.name() : "";
        });

        text = text.replace(/\\N\[(\d+)\]/g, (_, actorId) => {
            const actor = $gameActors.actor(Number(actorId));
            return actor ? actor.name() : "";
        });

        text = text.replace(/\\SV\[(.+?)\]/g, (_, rawContent) => {
            const resolvedContent = processControlCharacters(rawContent);
            if (isLogging) currentMessageBlock.voice = resolvedContent;
            return "";
        });

        text = text.replace(/\\NW\[(.+?)\]/g, (_, content) => {
            if (isLogging) currentMessageBlock.name = `【${content.trim()}】`;
            return "";
        });

        text = text.replace(/\\SET\[\d+\]/g, () => {
            if (isLogging) {
                currentMessageBlock.customSet = "";
            }
            return "";
        });

        text = text.replace(
            /\\NrN\[\s*(?:\\v\[(\d+)\]|(\S+?))\s*\]/gi,
            (_, v1, str) => {
                const resolved = v1 ? String($gameVariables.value(Number(v1))) : str;
                if (isLogging && resolved) currentMessageBlock.iconName = resolved;
                return "";
            }
        );

        text = text.replace(/\\SN\[(\d+)\]/gi, (_, id) => {
            const skill = $dataSkills[Number(id)];
            return skill ? skill.name : '';
        });

        text = text.replace(/\\IN\[(\d+)\]/gi, (_, id) => {
            const item = $dataItems[Number(id)];
            return item ? item.name : '';
        });

        text = text.replace(/\\WN\[(\d+)\]/gi, (_, id) => {
            const weapon = $dataWeapons[Number(id)];
            return weapon ? weapon.name : '';
        });

        text = text.replace(/\\AN\[(\d+)\]/gi, (_, id) => {
            const armor = $dataArmors[Number(id)];
            return armor ? armor.name : '';
        });

        text = text.replace(/\\RB\[(.+?),(.+?)\]/gi, (_, base, ruby) => {
            return base;
        });

        const unusedPatterns = [
            /\\SP\[[^\]]*\]/gi,
            /\\AT\[[^\]]*\]/gi,
            /\\SET\[[^\]]*\]/gi,
            /\\CO\[[^\]]*\]/gi,
            /\\MX\[[^\]]*\]/gi,
            /\\MY\[[^\]]*\]/gi,
            /\\PX\[[^\]]*\]/gi,
            /\\PY\[[^\]]*\]/gi,
            /\\SW\[[^\]]*\]/gi,
            /\\FO\[[^\]]*\]/gi,
            /\\OC\[[^\]]*\]/gi,
            /\\OP\[[^\]]*\]/gi,
            /\\OW\[[^\]]*\]/gi,
            /\\RC\[[^\]]*\]/gi
        ];

        const unusedNoArg = [
            /\\FB/gi,
            /\\FI/gi,
            /\\DF/gi,
            /\\SV/gi,
            /\\LD/gi,
            /\\A/gi,
            /\\ES/gi,
            /\\WE/gi
        ];

        unusedPatterns.forEach(pattern => {
            text = text.replace(pattern, '');
        });

        unusedNoArg.forEach(pattern => {
            text = text.replace(pattern, '');
        });

        return text;
    }


    function Scene_MessageLog() {
        this.initialize(...arguments);
    }

    Scene_MessageLog.prototype = Object.create(Scene_MenuBase.prototype);
    Scene_MessageLog.prototype.constructor = Scene_MessageLog;

    Scene_MessageLog.prototype.initialize = function() {
        Scene_Base.prototype.initialize.call(this);
    };

    Scene_MessageLog.prototype.create = function() {
        Scene_Base.prototype.create.call(this);

        this.createBackground();
        this.createLogWindow();

        this._titleWindow = new Window_MessageLogTitle();
        if (LogWindowBackground) this._titleWindow.visible = false;

        this.addChild(this._logWindow);
        this.addChild(this._titleWindow);

        this.createReturnButton();
        this._logWindow.scrollToBottom();
    };

    Scene_MessageLog.prototype.createBackground = function() {
        if (LogWindowBackground) {
            const bitmap = ImageManager.loadPicture(LogWindowBackground);
            this._backgroundSprite = new Sprite(bitmap);
        } else {
            this._backgroundSprite = new Sprite();
            this._backgroundSprite.bitmap = SceneManager._backgroundBitmap || new Bitmap(Graphics.width, Graphics.height);
            this._backgroundSprite.opacity = 192;
        }
        this.addChild(this._backgroundSprite);
    };

    Scene_MessageLog.prototype.createLogWindow = function() {
        this._logWindow = new Window_MessageLog();
        this.addChild(this._logWindow);
    };

    Scene_MessageLog.prototype.createReturnButton = function() {
   if (CloseButtonImage) {
        const bitmap = ImageManager.loadPicture(CloseButtonImage);
        const btn = new Sprite(bitmap);
        btn.x = Graphics.boxWidth / 2 - bitmap.width / 2;
        btn.y = Graphics.boxHeight - 80;
        btn.update = function () {
            Sprite.prototype.update.call(this);
            if (TouchInput.isTriggered()) {
                const x = TouchInput.x, y = TouchInput.y;
                if (x >= this.x && x <= this.x + this.width && y >= this.y && y <= this.y + this.height) {
                    SceneManager.pop();
                }
            }
        };
        this._returnButton = btn;
        this.addChild(btn);
    } else {
        const btn = new Sprite(new Bitmap(200, 50));
        btn.bitmap.fillRect(0, 0, 200, 50, 'rgba(0, 0, 0, 0.6)');
        btn.bitmap.drawText('戻る', 0, 0, 200, 50, 'center');
        btn.x = Graphics.boxWidth / 2 - 100;
        btn.y = Graphics.boxHeight - 80;
        btn.update = function () {
            Sprite.prototype.update.call(this);
            if (TouchInput.isTriggered()) {
                const x = TouchInput.x, y = TouchInput.y;
                if (x >= this.x && x <= this.x + 200 && y >= this.y && y <= this.y + 50) {
                    SceneManager.pop();
                }
            }
        };
        this._returnButton = btn;
        this.addChild(btn);
    }
    };

    function Window_MessageLog() {
        this.initialize(...arguments);
    }

    Window_MessageLog.prototype = Object.create(Window_Base.prototype);
    Window_MessageLog.prototype.constructor = Window_MessageLog;

    Window_MessageLog.prototype.initialize = function() {
        const x = 25, y = 60, w = Graphics.boxWidth - 50, h = Graphics.boxHeight - y - 100;
        Window_Base.prototype.initialize.call(this, x, y, w, h);

        this._scrollY = 0;
        this._contentHeight = 0;
        this._lineSpacing = LineSpacing;
        this._visibleBlocks = [];
        this._renderedBlocks = {};
        this._blockHeightCache = null;

        this.createScrollBar();
        this.refresh();
    };

    Window_MessageLog.prototype.createScrollBar = function() {
        const barW = 8;
        const trackH = this.height;
        const offsetX = -10;

        this._scrollBarBack = new Sprite(new Bitmap(barW, trackH));
        this._scrollBarBack.bitmap.fillAll('rgba(0,0,0,0.2)');
        this._scrollBarBack.x = this.width - barW - offsetX;
        this._scrollBarBack.y = 0;
        this.addChild(this._scrollBarBack);

        const knobH = 40;
        this._scrollBar = new Sprite(new Bitmap(barW, knobH));
        this._scrollBar.bitmap.fillAll('rgba(255,255,255,0.9)');
        this._scrollBar.x = this._scrollBarBack.x;
        this._scrollBar.y = 0;
        this.addChild(this._scrollBar);
    };

    Window_MessageLog.prototype.drawLogBlockBackground = function(y, h, hasName) {
        const imgName = hasName ? NamedBlockBackground : NormalBlockBackground;
        if (imgName) {
            const bmp = ImageManager.loadPicture(imgName);
            if (bmp.isReady()) {
                const context = this.contents.context;
                const sw = bmp.width;
                const sh = bmp.height;
                const dw = this.contents.width;
                const dh = h - BlockSpacing;
                context.drawImage(bmp.canvas, 0, 0, sw, sh, 0, y, dw, dh);
            } else {
                bmp.addLoadListener(() => this.refresh());
            }
        } else {
            this.contents.fillRect(0, y, this.contents.width, h - BlockSpacing, 'rgba(255,255,255,0.9)');
        }
    };

    Window_MessageLog.prototype.refresh = function() {
        this._blockHeightCache = null;
        this._contentHeight = $gameMessage.getMessageLog().reduce((sum, block) => {
            return sum + this.estimateBlockHeight(block);
        }, 0);
        
        Object.values(this._renderedBlocks).forEach(bitmap => bitmap.clear());
        this._renderedBlocks = {};
        this._visibleBlocks = [];
        
        this.redrawVisibleArea();
        this.updateScrollBar();
    };

    Window_MessageLog.prototype.drawLogBlockToBitmap = function(block, index, y) {
        if (this._renderedBlocks[index]) return this._renderedBlocks[index];

        const lh = this.lineHeight();
        const bp = 10;
        const cw = this.width - this.padding * 2;
        const hblock = this.estimateBlockHeight(block);
        const bitmap = new Bitmap(cw, hblock);
        
        const mx = 250;
        const hasName = !!block.name;
        const iconFile = faceIconMap[block.iconName];

        const bgName = hasName ? NamedBlockBackground : NormalBlockBackground;
        if (bgName) {
            const bg = ImageManager.loadPicture(bgName);
            if (bg.isReady()) {
                bitmap.blt(bg, 0, 0, bg.width, bg.height, 0, 0, cw, hblock - BlockSpacing);
            } else {
                bg.addLoadListener(() => this.refresh());
            }
        } else {
            bitmap.fillRect(0, 0, cw, hblock - BlockSpacing, 'rgba(255,255,255,0.9)');
        }

        let iy = bp;

        if (iconFile) {
            const bmp = ImageManager.loadPicture(iconFile);
            if (bmp.isReady()) {
                const dx = IconOffsetX;
                const dy = iy + IconOffsetY;
                const dw = IconWidth;
                const dh = IconHeight;
                bitmap.blt(bmp, 0, 0, bmp.width, bmp.height, dx, dy, dw, dh);
            } else {
                bmp.addLoadListener(() => this.refresh());
            }
        }

        if (hasName) {
            bitmap.fontSize = NameFontSize;
            bitmap.textColor = NameColor;

            const nameX = mx;
            const nameY = iy;
            const nameTextWidth = bitmap.measureTextWidth(block.name);

            bitmap.drawText(block.name, nameX, nameY, cw - mx * 2, lh, "left");

            if (block.voice) {
                if (PlayButtonImage) {
                    const btnImg = ImageManager.loadPicture(PlayButtonImage);
                    const px = nameX + nameTextWidth + PlayButtonOffsetX;
                    const py = nameY + PlayButtonOffsetY;

                    if (btnImg.isReady()) {
                        bitmap.blt(btnImg, 0, 0, btnImg.width, btnImg.height, px, py, btnImg.width, btnImg.height);

                        const btnWidth = btnImg.width;
                        const screenX = this.absoluteX() + this.padding + px;
                        const screenYFunc = () => this.absoluteY() + this.padding + y + py - this._scrollY;

                        const btnRect = {
                            x: screenX,
                            y: screenYFunc,
                            width: btnWidth,
                            height: btnImg.height
                        };

                        this.addInteractiveButton(btnRect, () => {
                            AudioManager.playVoice({
                                name: block.voice,
                                volume: 100,
                                pitch: 100,
                                pan: 0
                            });
                        });
                    } else {
                        btnImg.addLoadListener(() => this.refresh());
                    }
                } else {
                    const playText = "▶ 再生";
                    const btnWidth = this.contents.measureTextWidth(playText);
                    const px = nameX + nameTextWidth + PlayButtonOffsetX;
                    const py = nameY + PlayButtonOffsetY;

                    bitmap.fontSize = NameFontSize;
                    bitmap.textColor = '#3399FF';
                    bitmap.drawText(playText, px, py, cw - px, lh, "left");

                    //当たり判定確認用。念のために残しておきます。
                    //bitmap.paintOpacity = 160;
                    //bitmap.fillRect(px, py, btnWidth, lh, 'rgba(255, 0, 0, 0.4)');
                    //bitmap.paintOpacity = 255;

                    const screenX = this.absoluteX() + this.padding + px;
                    const screenYFunc = () => this.absoluteY() + this.padding + y + py - this._scrollY;

                    this.addInteractiveButton({
                        x: screenX,
                        y: screenYFunc,
                        width: btnWidth,
                        height: lh
                    }, () => {
                        AudioManager.playVoice({
                            name: block.voice,
                            volume: 100,
                            pitch: 100,
                            pan: 0
                        });
                    });
                }
            }

            iy += lh * 2 + LineSpacing;
        } else {
            iy += lh / 2;
        }

        bitmap.fontSize = LogFontSize;
        bitmap.textColor = LogColor;

        const lines = block.messages.slice(0, BaseLineCount);
        const originalContents = this.contents;

        this.contents = bitmap;

        for (let j = 0; j < lines.length; j++) {
            const rawText = lines[j];
            if (typeof rawText !== 'string') continue;

            const text = Window_Base.prototype.convertEscapeCharacters.call(this, rawText);
            this.drawTextEx(text, mx, iy);
            iy += lh + LineSpacing;
        }

        this.contents = originalContents;
        this._renderedBlocks[index] = bitmap;
        return bitmap;
    };

    Window_MessageLog.prototype.redrawVisibleArea = function() {
        const messages = $gameMessage.getMessageLog();
        const padding = this.standardPadding();
        const visibleTop = this._scrollY;
        const visibleBottom = visibleTop + this.height - padding * 2;

        this._interactiveButtons = [];
        this._renderedBlocks = {};
        this.contents.clear();

        let y = 0;
        const newVisibleBlocks = [];

        for (let i = 0; i < messages.length; i++) {
            const block = messages[i];
            const h = this.estimateBlockHeight(block);

            if (y + h > visibleTop && y < visibleBottom) {
                const bitmap = this.drawLogBlockToBitmap(block, i, y);
                const dy = y - visibleTop;
                this.contents.blt(bitmap, 0, 0, bitmap.width, bitmap.height, 0, dy);
                newVisibleBlocks.push(i);
            }

            y += h;
        }

        this._visibleBlocks.forEach(index => {
            if (!newVisibleBlocks.includes(index)) {
                const bitmap = this._renderedBlocks[index];
                if (bitmap) bitmap.clear();
                delete this._renderedBlocks[index];
            }
        });

        this._visibleBlocks = newVisibleBlocks;
    };

    Window_MessageLog.prototype.estimateBlockHeight = function(block) {
        const lh = this.lineHeight();
        const hasName = !!block.name;
        const hasFace = !!faceIconMap[block.iconName];

        const totalLines = BaseLineCount + (hasName ? 2 : 1);
        const textHeight = totalLines * (lh + LineSpacing) + 20 + BlockSpacing;

        if (hasFace) {
            const faceHeight = IconHeight + IconOffsetY * 2;
            return Math.max(textHeight, faceHeight);
        } else {
            return textHeight;
        }
    };

    Window_MessageLog.prototype.updateScrollBar = function() {

        const ch = this._contentHeight;
        const vh = this.height;
        if (ch <= vh) {
            this._scrollBar.visible = false;
            return;
        }
        
        const ratio = vh / ch;
        const knobH = Math.max(32, vh * ratio);
        const maxScr = ch - vh;
        const barY = (this._scrollY / maxScr) * (vh - knobH);

        if (this._scrollBar.height !== knobH) {
            this._scrollBar.bitmap = new Bitmap(this._scrollBar.width, knobH);
            this._scrollBar.bitmap.fillAll('rgba(255,255,255,0.9)');
        }
        this._scrollBar.y = barY;
        this._scrollBar.visible = true;
    };

    Window_MessageLog.prototype.scrollToBottom = function() {
        const padding = this.standardPadding();
        const innerHeight = this.height - padding * 2;
        const extra = 8;
        const maxScroll = Math.max(0, this._contentHeight - innerHeight + extra);
        this._scrollY = maxScroll;
    this._needsRedraw = true;
        this.redrawVisibleArea();
    };

    Window_MessageLog.prototype.scrollDown = function(amount) {
        const max = Math.max(0, this._contentHeight - (this.height - this.padding * 2));
        const ny = Math.min(this._scrollY + amount, max);
        if (ny !== this._scrollY) {
            this._scrollY = ny;
            this.redrawVisibleArea();
            this.updateScrollBar();
        }
    };

    Window_MessageLog.prototype.scrollUp = function(amount) {
        const ny = Math.max(0, this._scrollY - amount);
        if (ny !== this._scrollY) {
            this._scrollY = ny;
            this.redrawVisibleArea();
            this.updateScrollBar();
        }
    };

    Scene_MessageLog.prototype.start = function() {
        Scene_Base.prototype.start.call(this);
        this._logWindow.scrollToBottom();
        this._logWindow.updateScrollBar();
    };

    Window_MessageLog.prototype.handleScrollbarDragging = function() {
        const localY = TouchInput.y - this.y;
        const bar = this._scrollBar;
        const knobY = bar.y;
        const knobH = bar.height;
        const trackH = this.height;

        if (TouchInput.isPressed()) {
            if (this._scrollDragging) {
                const barY = localY - this._dragOffsetY;
                const maxScroll = Math.max(0, this._contentHeight - trackH);
                const scrollRatio = barY / (trackH - knobH);
                this._scrollY = Math.min(maxScroll, Math.max(0, scrollRatio * maxScroll));
                this.redrawVisibleArea();
                this.updateScrollBar();
            } else {
                const touchX = TouchInput.x - this.x;
                const touchY = TouchInput.y - this.y;

                const knobRect = new Rectangle(
                    bar.x,
                    knobY,
                    bar.width,
                    knobH
                );

                if (touchX >= knobRect.x &&
                    touchX <= knobRect.x + knobRect.width &&
                    touchY >= knobRect.y &&
                    touchY <= knobRect.y + knobRect.height) {
                    this._scrollDragging = true;
                    this._dragOffsetY = localY - knobY;
                }
            }
        } else {
            this._scrollDragging = false;
        }
    };

    Window_MessageLog.prototype.update = function() {
        Window_Base.prototype.update.call(this);

        if (TouchInput.isTriggered()) {
            const tx = TouchInput.x;
            const ty = TouchInput.y;

            if (this._interactiveButtons && this._interactiveButtons.length > 0) {
                for (const btn of this._interactiveButtons) {
                    const x = btn.rect.x;
                    const y = typeof btn.rect.y === 'function' ? btn.rect.y() : btn.rect.y;
                    const { width, height } = btn.rect;

                    if (tx >= x && tx <= x + width && ty >= y && ty <= y + height) {
                        btn.callback();
                        break;
                    }
                }
            }
        }

        if (TouchInput.wheelY > 0) {
            this.scrollDown(40);
        } else if (TouchInput.wheelY < 0) {
            this.scrollUp(40);
        }

        this.handleScrollbarDragging();
    };


    Window_MessageLog.prototype.destroy = function() {
        Object.values(this._renderedBlocks).forEach(bitmap => bitmap.clear());
        this._renderedBlocks = {};
        Window_Base.prototype.destroy.call(this);
    };

    function Window_MessageLogTitle() {
        this.initialize(...arguments);
    }

    Window_MessageLogTitle.prototype = Object.create(Window_Base.prototype);
    Window_MessageLogTitle.prototype.constructor = Window_MessageLogTitle;
    Window_MessageLogTitle.prototype.initialize = function() {
        const w = 360, h = 64, x = (Graphics.width - w) / 2, y = 0;
        Window_Base.prototype.initialize.call(this, new Rectangle(x, y, w, h));

        this.opacity = 255;
        this.backOpacity = 192;
        this.contentsOpacity = 255;
        this.refresh();
    };

    Window_MessageLogTitle.prototype.refresh = function() {
        this.contents.clear();
        this.contents.fontSize = 30;
        this.changeTextColor(this.normalColor());

        const t = "バックログ", tw = this.textWidth(t + "　");
        const ox = (this.width - this.padding - tw) / 2;
        this.drawText(t, ox, 3, tw, 'left');
        this.resetTextColor();
    };

    Window_MessageLog.prototype.addInteractiveButton = function (rect, callback) {
        if (!this._interactiveButtons) this._interactiveButtons = [];
        this._interactiveButtons.push({ rect, callback });
    };

    window.NovelGameLog = {
        start: function() {
            isLogging = true;
            if (CloseButtonImage) ImageManager.loadPicture(CloseButtonImage);
            if (PlayButtonImage) ImageManager.loadPicture(PlayButtonImage);
            if (NamedBlockBackground) ImageManager.loadPicture(NamedBlockBackground);
            if (NormalBlockBackground) ImageManager.loadPicture(NormalBlockBackground);
        },
        stop: function() {
            isLogging = false;
        },
        clear: function() {
            $gameMessage.clearMessageLog();
        },
        show: function() {
            SceneManager.push(Scene_MessageLog);
        },
        setName: function(name) {
            if (isLogging) {
                currentMessageBlock.name = `【${name}】`;
            }
        }
    };

    const _Window_Message_convertEscapeCharacters = Window_Message.prototype.convertEscapeCharacters;
    Window_Message.prototype.convertEscapeCharacters = function(text) {
        text = text.replace(
            /\\NrN\[\s*(?:\\v\[(\d+)\]|([^\]]+))\s*\]/gi,
            (_, v1, str) => {
                const resolved = v1 ? String($gameVariables.value(Number(v1))) : str;
                if (isLogging && resolved) currentMessageBlock.iconName = resolved.trim();
                return "";
            }
        );

        return _Window_Message_convertEscapeCharacters.call(this, text);
    };

    Window_Base.prototype.createTextState = function(text, x, y, width) {
        const textState = {};
        textState.index = 0;
        textState.text = this.convertEscapeCharacters(text);
        textState.x = x;
        textState.y = y;
        textState.width = width;
        textState.height = this.calcTextHeight(textState, false);
        textState.drawing = true;
        textState.outputWidth = 0;
        textState.outputHeight = 0;
        return textState;
    };

    function applyAutoLineBreak(text) {
        const parameters = PluginManager.parameters('AutoLineBreak');
        const maxChars = Number(parameters['MaxCharsPerLine'] || 20);
        const insertAutoBreakSpace = String(parameters['InsertAutoBreakSpace']) === 'true';
        const forbiddenStartChars = ['）', '」', '』', '】', '、', '。', '…'];
        const nwActive = /\\[Nn][Ww]\[/.test(text);

        let count = 0;
        let result = '';
        for (let i = 0; i < text.length; i++) {
            const c = text[i];
            const next = text[i + 1];

            if (c === '\n') {
                count = 0;
                result += c;
                continue;
            }

            if (c === '\\') {
                const match = text.slice(i).match(/^\\[A-Za-z]+\[[^\]]*\]/);
                if (match) {
                    result += match[0];
                    i += match[0].length - 1;
                    continue;
                }
            }

            result += c;

            if (c !== '\r') count++;

            if (count >= maxChars) {
                if (!forbiddenStartChars.includes(next)) {
                    result += '\n';
                    if (insertAutoBreakSpace && nwActive) {
                        result += '　';
                    }
                    count = 0;
                }
            }
        }

        return result;
    }

    Window_MessageLog.prototype.processAllText = function(textState, targetBitmap) {
        const text = textState.text;
        let index = textState.index;
        targetBitmap.fontSize = this.contents.fontSize;
        targetBitmap.textColor = this.contents.textColor;

        while (index < text.length) {
            const c = text[index++];
            if (c === "\x1b") {
                const code = Window_Base.prototype.obtainEscapeCode.call(this, text, index);
                index = Window_Base.prototype.obtainEscapeParam.call(this, text, index);
                Window_Base.prototype.processEscapeCharacter.call(this, code, textState);
            } else {
                targetBitmap.drawText(c, textState.x, textState.y, textState.width, this.lineHeight(), "left");
                if (targetBitmap.measureTextWidth) {
                    textState.x += targetBitmap.measureTextWidth(c);
                } else {
                    textState.x += 10;
                }
            }
        }

        textState.index = index;
    };

    Window_Base.prototype.absoluteX = function() {
        return this.x + (this.parent ? this.parent.x : 0);
    };

    Window_Base.prototype.absoluteY = function() {
        return this.y + (this.parent ? this.parent.y : 0);
    };

})();