//=============================================================================
// NrTextFile.js
//=============================================================================

/*:
 * @plugindesc 外部テキスト逐次読み込みプラグイン Ver1.0.0
 *
 * @param NamePrefixes
 * @text 名前接頭辞
 * @type string[]
 * @default []
 *
 * @param Background
 * @text 初期背景
 * @type select
 * @option 通常
 * @value 0
 * @option 暗くする
 * @value 1
 * @option 透明
 * @value 2
 * @default 0
 *
 * @param Position
 * @text 初期ウィンドウ位置
 * @type select
 * @option 左
 * @value 0
 * @option 中
 * @value 1
 * @option 右
 * @value 2
 * @default 2
 *
 * @help
 * ■ コマンド使用方法 ■
 * 
 * 【@switch / @sw】
 * 例: @sw 1 true
 * - スイッチ番号1をONにします。falseや0でOFFにできます。
 * 
 * @choice
 * 選択肢の表示と分岐を行います。書式例：
 * @choice(
 *   (①背景)0,      // 0: 通常 1: 暗くする 2: 透明
 *   (②ウィンドウ位置)2,  // 0: 左 1: 中 2: 右
 *   (③デフォルト)0,      // -1: なし 0~5: 選択肢番号デフォルト選択
 *   (④キャンセル)-2)     // -2: 分岐あり -1: 禁止 0~5: 選択肢番号キャンセル時選択
 * @branch("はい", 0)      // 選択肢名、選択肢の番号-1
 *   (処理内容)
 * @branch("いいえ", 1)
 *   (処理内容)
 * @cancel                 // キャンセル時の処理
 *   (処理内容)
 * @endChoice
 * 
 * @if
 * 条件分岐コマンド。例：
 * @if(0, 1, 0)
 * パラメータ説明：
 * ①判定タイプ
 *  0: スイッチ判定
 *  1: 変数判定
 *  2: セルフスイッチ判定
 *  3: タイマー判定
 *  4: アクター関連判定
 *  5: 敵キャラ判定
 *  6: キャラクター方向判定
 *  7: 所持金判定
 *  8: アイテム所持判定
 *  9: 武器所持判定
 * 10: 防具所持判定
 * 11: ボタン押下判定
 * 12: スクリプト判定
 * 13: 乗り物判定
 * ②判定対象番号（スイッチ番号や変数番号など）
 * ③比較値（スイッチなら0=ON、1=OFF、変数なら0=定数、1=変数など）
 * ④比較対象（値または変数番号など）
 * ⑤比較式
 *    0: ==
 *    1: >=
 *    2: <=
 *    3: >
 *    4: <
 *    5: !=
 * @else
 * (条件に合わない場合の処理)
 * @endIf
 * 
 * @wait
 * ウェイト（フレーム数）を挿入。例：@wait 60（60フレーム待機）
 * 
 * @fadeout
 * 画面フェードアウト。例：@fadeout 90（90フレームかけてフェードアウト）
 * 
 * @var
 * 変数操作コマンド。例：
 * @var 1 0 0 100
 * パラメータ説明：
 * ①変数番号
 * ②操作種類
 *   0: 代入
 *   1: 加算
 *   2: 減算
 * ③値の種類
 *   0: 定数
 *   1: 変数
 *   2: ランダム範囲（例「1-3」）
 *   3: スクリプト（JavaScript式）
 * ④値またはスクリプト
 * 
 * @script / @sc
 * JavaScriptコードを実行。@endで終了。
 * 
 * @sp / @showpicture
 * ピクチャを表示します。例：
 * @sp 1 Face1
 * - ピクチャ番号1に "Face1" を表示します。
 *
 * @hp / @hidepicture
 * ピクチャを消去します。例：
 * @hp 1
 * - ピクチャ番号1を消去します。
 *
 * @mp / @movepicture
 * ピクチャを移動します。書式：
 * @mp ピクチャ番号 原点 X座標 Y座標 幅 高さ 不透明度 合成 モーション時間 ウェイト 予備1 予備2
 * - 例：@mp 1 0 200 300 100 100 255 0 60 30 0 0
 *
 * @pc / @plugincommand
 * プラグインコマンドを実行します。記述された行の残りをそのまま実行：
 *
 * @bgm
 * BGMを再生します。例：
 * @bgm 夜 [音量] [ピッチ] [パン]
 * - 音量, ピッチ, パンは省略可能（デフォルト：90, 100, 0）
 *
 * @bgs
 * BGSを再生します。例：
 * @bgs 風 80
 *
 * @me
 * MEを再生します。例：
 * @me 勝利
 *
 * @se
 * SEを再生します。例：
 * @se ピンポン 120 100 -10
 *
 * @stopbgm / @stopbgs / @stopme / @stopse
 * 各種音声を即時停止します。
 *
 * @fadebgm / @fadebgs
 * BGM/BGS をフェードアウトで停止します。
 * 例：@fadebgm 60 （60フレームかけてフェードアウト）
 *
 * ---  
 * これらのコマンドをテキストファイルに記述し、
 * NrTextFile.load("ファイル名.txt") で読み込んで実行してください。
 */

(function () {
    const pluginName = 'NrTextFile';
    const params = PluginManager.parameters(pluginName);
    const namePrefixes = JSON.parse(params['NamePrefixes'] || '[]');
    let currentBg = Number(params['Background'] || 0);
    let currentPos = Number(params['Position'] || 2);

    function loadTextFile(path, onSuccess, onError) {
        const xhr = new XMLHttpRequest();
        xhr.open('GET', 'data/' + path);
        xhr.overrideMimeType('text/plain');
        xhr.onload = () => (xhr.status < 400 ? onSuccess(xhr.responseText) : onError());
        xhr.onerror = onError;
        xhr.send();
    }

    async function sleep(ms) {
        return new Promise(resolve => setTimeout(resolve, ms));
    }

    async function processCommand(line) {
        const parts = line.trim().split(/\s+/);
        const cmd = parts[0].toLowerCase();

        switch (cmd) {
            case '@fadeout': {
                const dur = Number(parts[1] || 30);
                $gameScreen.startFadeOut(dur);
                await sleep(dur * (1000 / 60));
                break;
            }
            case '@fadein': {
                const dur = Number(parts[1] || 30);
                $gameScreen.startFadeIn(dur);
                await sleep(dur * (1000 / 60));
                break;
            }
            case '@wait': {
                const dur = Number(parts[1] || 60);
                await sleep(dur * (1000 / 60));
                break;
            }
            case '@var': {
                const varId = Number(parts[1]);
                const op = Number(parts[2]);
                const valType = parts[3];
                const valRaw = parts.slice(4).join(' ');
                let value = 0;
                switch (valType) {
                    case '0': value = Number(valRaw); break;
                    case '1': value = $gameVariables.value(Number(valRaw)); break;
                    case '2': {
                        const [min, max] = valRaw.split('-').map(Number);
                        value = Math.floor(Math.random() * (max - min + 1)) + min;
                        break;
                    }
                    case '3': value = eval(valRaw); break;
                }
                switch (op) {
                    case 0: $gameVariables.setValue(varId, value); break;
                    case 1: $gameVariables.setValue(varId, $gameVariables.value(varId) + value); break;
                    case 2: $gameVariables.setValue(varId, $gameVariables.value(varId) - value); break;
                }
                break;
            }
            case '@switch':
            case '@sw': {
                const switchId = Number(parts[1]);
                const valStr = parts[2] || 'true';
                const val = valStr.toLowerCase() === 'true' || valStr === '1';
                $gameSwitches.setValue(switchId, val);
                break;
            }
            case '@bg': {
                currentBg = Number(parts[1] || 0);
                break;
            }
            case '@pos': {
                currentPos = Number(parts[1] || 2);
                break;
            }
            case '@choice':
            case '@branch':
            case '@cancel':
            case '@endchoice':
            case '@if':
            case '@else':
            case '@endif': {
                break;
            }
            case '@script':
            case '@sc': {
                break;
            }
            case '@showpicture':
            case '@sp': {
                const pictureId = Number(parts[1]);
                const name = parts.slice(2).join(' ');
                const origin = 0;
                const x = 0;
                const y = 0;
                const scaleX = 100;
                const scaleY = 100;
                const opacity = 255;
                const blendMode = 0;
                $gameScreen.showPicture(pictureId, name, origin, x, y, scaleX, scaleY, opacity, blendMode);
                break;
            }
            case '@hidepicture':
            case '@hp': {
                const pictureId = Number(parts[1]);
                $gameScreen.erasePicture(pictureId);
                break;
            }
            case '@movepicture':
            case '@mp': {
                const args = parts.slice(1).map(s => s.trim());
                const pictureId = Number(args[0]);
                const origin = Number(args[1]);
                const x = Number(args[2]);
                const y = Number(args[3]);
                const scaleX = Number(args[4]);
                const scaleY = Number(args[5]);
                const opacity = Number(args[6]);
                const blendMode = Number(args[7]);
                const duration = Number(args[8]);
                const wait = args.length >= 10 ? Number(args[9]) : 0;

                $gameScreen.movePicture(pictureId, origin, x, y, scaleX, scaleY, opacity, blendMode, duration);
                if (wait > 0) await sleep(wait * (1000 / 60));
                break;
            }
            case '@plugincommand':
            case '@pc': {
                const commandLine = line.replace(/^\s*@(?:plugincommand|pc)\s*/i, '');
                const interpreter = new Game_Interpreter();
                interpreter.pluginCommand(commandLine, []);
                break;
            }
            case '@bgm':
            case '@bgs':
            case '@me':
            case '@se': {
                const name = parts[1];
                const volume = Number(parts[2] || 90);
                const pitch = Number(parts[3] || 100);
                const pan = Number(parts[4] || 0);
                const sound = { name, volume, pitch, pan };
                switch (cmd) {
                    case '@bgm': AudioManager.playBgm(sound); break;
                    case '@bgs': AudioManager.playBgs(sound); break;
                    case '@me':  AudioManager.playMe(sound); break;
                    case '@se':  AudioManager.playSe(sound); break;
                }
                break;
            }
            case '@stopbgm': {
                AudioManager.stopBgm();
                break;
            }
            case '@stopbgs': {
                AudioManager.stopBgs();
                break;
            }
            case '@stopme': {
                AudioManager.stopMe();
                break;
            }
            case '@stopse': {
                AudioManager.stopSe();
                break;
            }
            case '@fadebgm': {
                const duration = Number(parts[1] || 30);
                AudioManager.fadeOutBgm(duration);
                break;
            }
            case '@fadebgs': {
                const duration = Number(parts[1] || 30);
                AudioManager.fadeOutBgs(duration);
                break;
            }
            case '@shake': {
                const power = Number(parts[1] || 5);
                const speed = Number(parts[2] || 5);
                const duration = Number(parts[3] || 60);
                $gameScreen.startShake(power, speed, duration);
                break;
            }
            case '@flash': {
                const r = Number(parts[1] || 255);
                const g = Number(parts[2] || 255);
                const b = Number(parts[3] || 255);
                const a = Number(parts[4] || 128);
                const duration = Number(parts[5] || 30);
                $gameScreen.startFlash([r, g, b, a], duration);
                break;
            }
            case '@weather': {
                const type = parts[1] || 'none'; // none, rain, storm, snow
                const power = Number(parts[2] || 0);
                const duration = Number(parts[3] || 30);
                $gameScreen.changeWeather(type, power, duration);
                break;
            }
            case '@tint': {
                const r = Number(parts[1] || 0);
                const g = Number(parts[2] || 0);
                const b = Number(parts[3] || 0);
                const gray = Number(parts[4] || 0);
                const duration = Number(parts[5] || 60);
                $gameScreen.startTint([r, g, b, gray], duration);
                break;
            }

            default:
                break;
        }
    }

    async function processLines(lines, namePrefixes) {
        let scriptMode = false;
        let scriptBuffer = [];
        let messageBuffer = [];

        let ifStack = [];
        let execMode = true;

        let choiceMode = false;
        let choiceBuffer = [];
        let choiceBranches = {};
        let cancelBranch = null;
        let choiceParams = [0, 2, -1, -1];

        const labelMap = new Map();
        lines.forEach((line, i) => {
            const m = line.trim().match(/^@label\s+(.+)$/);
            if (m) labelMap.set(m[1], i);
        });

        async function flushMessage() {
            if (messageBuffer.length === 0) return;
            const text = messageBuffer.join('\n');
            messageBuffer = [];
            $gameMessage.setBackground(currentBg);
            $gameMessage.setPositionType(currentPos);
            $gameMessage.add(text);
            await new Promise(resolve => {
                const check = () => {
                    if (!$gameMessage.isBusy()) resolve();
                    else setTimeout(check, 100);
                };
                check();
            });
        }


        for (let i = 0; i < lines.length; i++) {
            let line = lines[i].trim();
            if (line === "") {
                await flushMessage();
                continue;
            }

            if (line.startsWith('@goto')) {
                const labelName = line.split(/\s+/)[1];
                const targetIndex = labelMap.get(labelName);
                if (targetIndex !== undefined) {
                    i = targetIndex - 1;
                }
                continue;
            }

            if (line.startsWith('@if')) {
                await flushMessage();
                const args = line.match(/\((.*?)\)/)[1].split(',').map(s => s.trim());
                const result = evaluateIfCondition(args);
                ifStack.push(result);
                execMode = ifStack.every(Boolean);
                continue;
            }
            if (line === '@else') {
                await flushMessage();
                const current = ifStack.pop();
                ifStack.push(!current);
                execMode = ifStack.every(Boolean);
                continue;
            }
            if (line === '@endif') {
                await flushMessage();
                ifStack.pop();
                execMode = ifStack.every(Boolean);
                continue;
            }
            if (!execMode) continue;

            if (scriptMode) {
                if (line === '@end') {
                    await eval(scriptBuffer.join('\n'));
                    scriptBuffer = [];
                    scriptMode = false;
                } else {
                    scriptBuffer.push(line);
                }
                continue;
            }
            if (line.startsWith('@script') || line.startsWith('@sc')) {
                await flushMessage();
                scriptMode = true;
                scriptBuffer = [];
                continue;
            }

            if (line.startsWith('@choice')) {
                await flushMessage();
                choiceMode = true;
                choiceBuffer = [];
                choiceBranches = {};
                cancelBranch = null;
                const args = line.match(/\((.*?)\)/)[1].split(',').map(s => s.trim()).map(Number);
                choiceParams = args;
                continue;
            }
            if (choiceMode && line.startsWith('@branch')) {
                const m = line.match(/@branch\("(.+?)",\s*(\d+)\)/);
                const text = m[1];
                const index = Number(m[2]);
                choiceBranches[index] = [];
                choiceBuffer.push(text);
                continue;
            }
            if (choiceMode && line === '@cancel') {
                cancelBranch = [];
                continue;
            }
            if (choiceMode && line === '@endChoice') {
                const selected = await new Promise(resolve => {
                    $gameMessage.setBackground(choiceParams[0]);
                    $gameMessage.setPositionType(choiceParams[1]);
                    $gameMessage.setChoices(choiceBuffer, choiceParams[2], choiceParams[3]);
                    $gameMessage.setChoiceCallback(n => resolve(n));
                });

                let target = choiceBranches[selected] || cancelBranch || [];
                await processLines(target, namePrefixes);
                choiceMode = false;
                continue;
            }
            if (choiceMode) {
                const keys = Object.keys(choiceBranches);
                if (keys.length > 0) {
                    const lastKey = Number(keys[keys.length - 1]);
                    choiceBranches[lastKey].push(line);
                } else if (cancelBranch !== null) {
                    cancelBranch.push(line);
                }
                continue;
            }

            if (line.startsWith('@')) {
                await flushMessage();
                await processCommand(line);
                continue;
            }

            const m = line.match(/^(\d+):(.*)$/);
            if (m) {
                const index = Number(m[1]) - 1;
                const prefix = namePrefixes[index] || '';
                let text = m[2].trim();
                text = text.replace(/\\n(?!\[\d+\])/g, '\n');
                messageBuffer.push(prefix + text);
            } else {
                let text = line;
                text = text.replace(/\\n(?!\[\d+\])/g, '\n');
                messageBuffer.push(text);
            }
        }
        await flushMessage();
    }

    function evaluateIfCondition(args) {
        const type = Number(args[0]);
        const id = Number(args[1]);
        const valType = Number(args[2]);
        const valRaw = args[3];
        const compare = Number(args[4] || 0);

        let left, right;

        switch (type) {
            case 0:
                left = $gameSwitches.value(id);
                right = valRaw === '0' || valRaw === 'false' ? false : true;
                break;
            case 1:
                left = $gameVariables.value(id);
                right = valType === 0 ? Number(valRaw) : $gameVariables.value(Number(valRaw));
                break;
            default:
                return false;
        }

        switch (compare) {
            case 0: return left == right;
            case 1: return left >= right;
            case 2: return left <= right;
            case 3: return left > right;
            case 4: return left < right;
            case 5: return left != right;
            default: return false;
        }
    }

    window.NrTextFile = {
        _queue: [],
        _busy: false,

        command: function (path, interpreter = null) {
            this._queue.push({ path, interpreter });
            this._processNext();
        },

        load: function (path) {
            this.command(path);
        },

        async _processNext() {
            if (this._busy || this._queue.length === 0) return;

            this._busy = true;
            const { path, interpreter } = this._queue.shift();

            const lines = await new Promise((resolve, reject) => {
                loadTextFile(path, text => resolve(text.split(/\r?\n/)), reject);
            });

            if (interpreter) interpreter.setWaitMode('nr_textfile');

            await processLines(lines, namePrefixes);

            if (interpreter) interpreter.setWaitMode('');
            this._busy = false;

            this._processNext();
        }
    };

})();
