//=============================================================================
// OnFUN.js
//=============================================================================

/*:
 * @plugindesc データ登録用プラグイン
 * @author Onmoremind
 *
 * @param Namespace
 * @text 名前空間
 * @desc 使用する名前空間を設定
 * @default OnFUN
 *
 * @param Conditions
 * @text 独自条件分岐
 * @desc 条件分岐専用コードを登録
 * @type struct<Condition>[]
 * @default []
 *
 * @param FunctionGroups
 * @text 関数グループ
 * @desc グループごとに独自関数を登録
 * @type struct<FunctionGroup>[]
 * @default []
 *
 * @param Objects
 * @text カスタムオブジェクトリスト
 * @desc 自由に名前を付けたオブジェクトを登録
 * @type struct<CustomObject>[]
 * @default []
 *
 * @param Arrays
 * @text 配列リスト
 * @desc 配列を登録
 * @type struct<Array>[]
 * @default []
 *
 * @param Arrays2D
 * @text 二次元配列リスト
 * @desc 二次元配列を登録
 * @type struct<Array2D>[]
 * @default []
 *
 * @param Data
 * @text 単一データ
 * @desc 単一のデータを登録
 * @type struct<Data>[]
 * @default []
 * 
 * @param EnableDebugLog
 * @text デバッグログを出力
 * @desc コンソールにデバッグログを出力するか設定します (true = 出力 / false = 非表示)
 * @type boolean
 * @default true
 *
 *
 * @help
 * ========================================================
 * 利用方法
 * ========================================================
 * ゲーム起動時にプラグインパラメーターで設定した項目を登録。
 * 
 * 利用規約：
 *  プラグイン作者に無断で使用、改変、再配布は不可。
 */


/*~struct~Condition:
 * @param memo
 * @text メモ
 * @desc 条件内容のメモ
 * @type note
 * @default 
 *
 * @param name
 * @text 条件名
 * @desc 名前空間.条件名() で呼び出し可能
 * @default 
 *
 * @param code
 * @text 条件コード
 * @desc 条件分岐で評価するJavaScriptを記述
 * @type note
 * @default 
 */

/*~struct~FunctionGroup:
 * @param groupName
 * @text グループ名
 * @desc このグループの名前を設定します（例: UI、表情など）
 * @default Default
 *
 * @param functions
 * @text 関数リスト
 * @desc このグループに属する関数を設定します。
 * @type struct<Function>[]
 * @default []
 */

/*~struct~Object:
 * @param memo
 * @text メモ
 * 
 * @param name
 * @text 名前
 * @desc オブジェクト名
 * @default 
 *
 * @param data
 * @text データ
 * @desc 
 * @type note
 * @default 
 */

/*~struct~Array:
 * @param memo
 * @text メモ
 * 
 * @param name
 * @text 名前
 * @desc 配列名
 * @default 
 *
 * @param data
 * @text データ
 * @desc 配列の内容を記述
 * @type note
 * @default 
 */

/*~struct~Array2D:
 * @param memo
 * @text メモ
 * 
 * @param name
 * @text 名前
 * @desc 二次元配列名
 * @default MyArray2D
 *
 * @param data
 * @text データ
 * @desc 二次元配列の内容を記述
 * @type note
 * @default 
 */

/*~struct~Data:
 * @param memo
 * @text メモ
 * 
 * @param name
 * @text 名前
 * @desc データの名前。名前空間.name でアクセス
 * @default 
 *
 * @param value
 * @text 値
 * @desc データの内容（数値、文字列、オブジェクト）
 * @type note
 * @default 
 */

/*~struct~Function:
 * @param memo
 * @text メモ
 * @desc 関数内容のメモ
 * @type note
 *
 * @param name
 * @text 関数名
 * @desc 関数の名前を設定します。例: 日常、イベント立ち絵
 * @default
 *
 * @param code
 * @text 関数コード
 * @desc 実行するJavaScriptコードを記述します。
 * @type note
 * @default
 */

/*~struct~CustomObject:
 * @param memo
 * @text メモ
 * 
 * @param name
 * @text 名前
 * @desc このオブジェクトの名前を指定
 * @default 
 *
 * @param properties
 * @text プロパティ
 * @desc このオブジェクトのkeyと値を設定
 * @type struct<Property>[]
 * @default 
 */

/*~struct~Property:
 * @param key
 * @text キー
 * @desc プロパティの名前（例: HP、職業など）
 * @default 
 *
 * @param value
 * @text 値
 * @desc プロパティの値（例: 100、"勇者" など）
 * @default 
 */


(() => {
    'use strict';

    const { TouchInput, SceneManager, AudioManager, $gameMap, $gameSystem } = window;
    const pluginName = 'OnFUN';

    const parameters = PluginManager.parameters(pluginName);
    const conditionsParam = JSON.parse(parameters['Conditions'] || '[]');
    const functionGroupsParam = JSON.parse(parameters['FunctionGroups'] || '[]');
    const objectsParam = JSON.parse(parameters['Objects'] || '[]');
    const arraysParam = JSON.parse(parameters['Arrays'] || '[]');
    const arrays2DParam = JSON.parse(parameters['Arrays2D'] || '[]');
    const dataParam = JSON.parse(parameters['Data'] || '[]');
    const enableDebugLog = parameters['EnableDebugLog'] === 'true';


    const namespaceName = parameters['Namespace'] || 'OnFUN';
    window[namespaceName] = window[namespaceName] || {};
    const namespace = window[namespaceName];

    if (enableDebugLog) {
    console.log(`DEBUG: 現在の名前空間は '${namespaceName}'`);
    console.log(`DEBUG: 名前空間の初期状態 →`, namespace);
    }
    /** ============================
      *  条件処理リストの登録
      *  ============================ */
    conditionsParam.forEach((condData, index) => {
        try {
            const parsed = JSON.parse(condData);
            const conditionName = parsed.name || `condition${index + 1}`;
            let conditionCode = parsed.code || '';

            if (conditionName && conditionCode) {
                conditionCode = conditionCode
                    .replace(/^"|"$/g, '') 
                    .replace(/\\n/g, '\n') 
                    .replace(/\\"/g, '"'); 

                namespace[conditionName] = function () {
                    try {
                        return !!eval(conditionCode);
                    } catch (e) {
                        return false;
                    }
                };

            }
        } catch (e) {
        }
    });

    /** ============================
     *  関数グループの登録処理
     *  ============================ */
    functionGroupsParam.forEach((groupData, groupIndex) => {
        try {
            const parsedGroup = JSON.parse(groupData);
            const groupName = parsedGroup.groupName || `Group${groupIndex + 1}`;
            const functions = JSON.parse(parsedGroup.functions || '[]');

            namespace[groupName] = namespace[groupName] || {};
            const groupNamespace = namespace[groupName];

            functions.forEach((funcData, funcIndex) => {
                try {
                    const parsedFunc = JSON.parse(funcData);
                    const functionName = parsedFunc.name || `function${funcIndex + 1}`;
                    let functionCode = parsedFunc.code || '';

                    functionCode = functionCode
                        .replace(/^"|"$/g, '')  
                        .replace(/\\n/g, '\n')  
                        .replace(/\\"/g, '"'); 

                    groupNamespace[functionName] = function (...args) {
                        const context = this;
                        return new Function('args', functionCode).call(context, args);
                    };
                    if (enableDebugLog) {
                    console.log(`DEBUG: 関数 '${groupName}.${functionName}' が登録 →`, groupNamespace[functionName]);
                    console.log(`DEBUG: '${groupName}.${functionName}' のコード →`, functionCode);
                    }

                } catch (syntaxError) {
                    console.warn(`${namespaceName}: 関数コードの解析中に構文エラーが発生(グループ: ${groupName}, 関数: ${functionName})`, syntaxError);
                    console.warn(`問題のコード: ${functionCode}`);
                }
            });
            if (enableDebugLog) {
                console.log(`DEBUG: 関数グループ '${groupName}' の内容 →`, groupNamespace);
            }

        } catch (e) {
            console.warn(`${namespaceName}: 関数グループの登録中にエラーが発生(グループインデックス: ${groupIndex})`, e);
        }
    });


    /** ============================
    *  オブジェクトリストの登録処理
    *  ============================ */
    objectsParam.forEach((objData, index) => {
        try {
            const parsed = JSON.parse(objData);
            const objectName = parsed.name || `Object${index + 1}`;
            let properties = parsed.properties || [];

            if (typeof properties === 'string') {
                properties = JSON.parse(properties);
            }

            const objectContent = {};
            properties.forEach(propData => {
                const property = JSON.parse(propData);
                objectContent[property.key] = property.value;
            });

            namespace[objectName] = objectContent;

        } catch (e) {
        }
    });


    /** ============================
     *  一次元配列リストの登録処理
     *  ============================ */
    arraysParam.forEach((arrayData, index) => {
        try {
            const parsed = JSON.parse(arrayData);
            const arrayName = parsed.name || `Array${index + 1}`;
            let arrayContent = parsed.data || '[]';  // `undefined` を防ぐ

            if (typeof arrayContent === "string") {
                try {
                    arrayContent = JSON.parse(arrayContent);

                    if (typeof arrayContent === "string") {
                        arrayContent = JSON.parse(arrayContent);
                    }
                } catch (e) {
                    console.warn(`${namespaceName}: '${arrayName}' の JSON パースに失敗。文字列のままで。`, e);
                }
            }

            if (!Array.isArray(arrayContent)) {
                console.warn(`${namespaceName}: '${arrayName}' は配列にあらず、ラップする`);
                arrayContent = [arrayContent];
            }

            const finalArray = arrayContent.map(item =>
                (typeof item === "string" && !isNaN(item)) ? Number(item) : item
            );

            namespace[arrayName] = finalArray;
            if (enableDebugLog) {
                console.log(`DEBUG: 一次元配列 '${arrayName}' が登録→`, finalArray);
            }
        } catch (e) {
            console.warn(`${namespaceName}: 一次元配列の登録中にエラーが発生`, e);
        }
    });

    /** ============================
     *  二次元配列リストの登録処理
     *  ============================ */
    arrays2DParam.forEach((arrayData, index) => {
        try {
            const parsed = JSON.parse(arrayData);
            const arrayName = parsed.name || `Array2D${index + 1}`;
            let arrayContent = parsed.data || '[]';  // `undefined` を防ぐ

            if (typeof arrayContent === "string") {
                try {
                    arrayContent = JSON.parse(arrayContent);

                    if (typeof arrayContent === "string") {
                        arrayContent = JSON.parse(arrayContent);
                    }
                } catch (e) {
                    console.warn(`${namespaceName}: '${arrayName}' の JSON パースに失敗。文字列として登録。`, e);
                }
            }

            if (!Array.isArray(arrayContent) || !arrayContent.every(row => Array.isArray(row))) {
                console.warn(`${namespaceName}: '${arrayName}' は二次元配列形式ではないので空二次元配列を設定。`);
                arrayContent = [[]]; 
            }

            const finalArray = arrayContent.map(row =>
                row.map(item => (typeof item === "string" && !isNaN(item)) ? Number(item) : item)
            );

            namespace[arrayName] = finalArray;
            if (enableDebugLog) {
                console.log(`DEBUG: 二次元配列 '${arrayName}' が登録 →`, finalArray);
            }
        } catch (e) {
            console.warn(`${namespaceName}: 二次元配列の登録中にエラーが発生`, e);
        }
    });




    /** ============================
     *  単一データリストの登録処理
     *  ============================ */
    dataParam.forEach((dataItem, index) => {
        try {
            const parsed = JSON.parse(dataItem);
            const dataName = parsed.name || `Data${index + 1}`;
            let dataValue = parsed.value || '';

            dataValue = JSON.parse(dataValue);
            namespace[dataName] = dataValue;
        } catch (e) {
        }
    });

})();

