//=============================================================================
// Onchat.js
//=============================================================================
/*:
 * @plugindesc スクロール式ログsystem
 * @author Onmoremind
 * 
 * @param SwitchID
 * @text 表示スイッチID
 * @type switch
 * @desc チャットログを表示・非表示にするスイッチのID
 * @default 1
 *
 * @param DisableChatSwitchID
 * @text チャット無効化スイッチID
 * @type switch
 * @desc 指定されたスイッチがONの間、チャット機能を無効にします
 * @default 2
 *
 * @param MaxLogEntries
 * @text 最大ログ件数
 * @type number
 * @desc 保存するチャットログの最大件数（古いログから順に削除されます）
 * @default 100
 *
 * @param CustomBackgroundImage
 * @text カスタム背景画像
 * @type file
 * @dir img/pictures
 * @desc 背景画像を指定します。空欄なら背景画像なし
 * @default
 *
 * @param PaddingTop
 * @text 上余白
 * @type number
 * @desc ウィンドウ内の上余白（px）
 * @default 8
 *
 * @param PaddingBottom
 * @text 下余白
 * @type number
 * @desc ウィンドウ内の下余白（px）
 * @default 8
 *
 * @param PaddingLeft
 * @text 左余白
 * @type number
 * @desc ウィンドウ内の左余白（px）
 * @default 12
 *
 * @param PaddingRight
 * @text 右余白
 * @type number
 * @desc ウィンドウ内の右余白（px）
 * @default 12
 *
 * @param LineSpacing
 * @text 行間
 * @type number
 * @desc テキスト行と行の間隔（px）
 * @default 4
 *
 * @param DefaultStampWidth
 * @text デフォルトスタンプ幅
 * @type number
 * @desc スタンプ画像のデフォルト幅（px）
 * @default 100
 *
 * @param DefaultStampHeight
 * @text デフォルトスタンプ高さ
 * @type number
 * @desc スタンプ画像のデフォルト高さ（px）
 * @default 100
 *
 * @param WindowX
 * @text ウィンドウのX座標
 * @type number
 * @desc ピクセル単位のX座標
 * @default 0
 *
 * @param WindowY
 * @text ウィンドウのY座標
 * @type number
 * @desc ピクセル単位のY座標
 * @default 0
 *
 * @param WindowWidth
 * @text ウィンドウ幅
 * @type number
 * @desc ピクセル単位の幅
 * @default 480
 *
 * @param WindowHeight
 * @text ウィンドウ高さ
 * @type number
 * @desc ピクセル単位の高さ
 * @default 216
 *
 * @param WindowOpacity
 * @text ウィンドウの透明度
 * @type number
 * @min 0
 * @max 255
 * @desc 0で完全透明、255で不透明
 * @default 255
 *
 * @param FontSize
 * @text フォントサイズ
 * @type number
 * @min 8
 * @desc チャットログの基本フォントサイズ
 * @default 28
 *
 * @param IconSize
 * @text アイコンサイズ
 * @type number
 * @desc アイコン表示サイズ（px）
 * @default 32
 *
 * @param ScrollAmount
 * @text 1回あたりのスクロール量
 * @type number
 * @desc マウスホイールやコマンドで1度に動かすピクセル数
 * @default 40
 *
 * @param VoiceCompleteSwitchID
 * @text ボイス再生完了スイッチID
 * @type switch
 * @desc \SC[n] のボイス再生が終わった時にONになるスイッチID（0で無効）
 * @default 0
 *
 * @param OnScrollSwitch
 * @text ホイール管理スイッチID
 * @type number
 * @desc このスイッチがONの時にホイールしない
 * @default 0
 * 
 * 
 *
 * @help
 *-----------------------------------------------------------------------------
 * 【追加できるコマンド】
 *   chatleft   <メッセージ>   // 左揃えのテキストログを追加
 *   chatright  <メッセージ>   // 右揃えのテキストログを追加
 *   chatsize   w h x y opacity // ウィンドウのサイズや位置、透明度を変更
 *   chatscroll <number>        // スクロール量を設定
 *   chatclear                  // チャットログをクリア
 *
 * @help
 *
 * chatleft	  左揃えでチャットログにテキストを追加します。	chatleft こんにちは、冒険者！
 * chatright  右揃えでチャットログにテキストを追加します。	chatright ようこそ、旅人よ。
 * chatsize	  チャットウィンドウのサイズ、位置、透明度を変更します。	chatsize 500 300 10 10 200
 * chatscroll チャットのスクロール速度を設定します。	chatscroll 3
 * chatclear  チャットログをすべてクリアします。	chatclear
 * \C[n]      文字色を変更します。	\C[2]赤色の文字
 * \I[n]      アイコンを表示します。	\I[64]
 * \V[n]      変数の値を表示します。	\V[1]
 * \N[n]      アクター名を表示します。	\N[1]
 * \ITEM[n]   アイテム名を表示します。	\ITEM[3]
 * \WEAPON[n] 武器名を表示します。	\WEAPON[5]
 * \ARMOR[n]  防具名を表示します。	\ARMOR[2]
 * \G         通貨単位を表示します。	\G
 * \SC[n]     変数に登録されたボイスを再生します。※SimpleVoiceControl.js必須。 \SC[n]もしくは\SC[\v[n]]
 * \n         改行します。　\n
 * 
 *  スクリプト呼び出し onchat("left", "メッセージ");
 *             onchat("right", { stamp:"画像名" });
 *             SceneManager._scene._chatLogWindow.clearLog(); //ログ削除
 *
 * 更新履歴:
 * 2025 / 4 / 15 ボイス再生終了時にパラメータで指定したスイッチが起動する機能追加。
 * 2025 / 4 / 14 改行コマンド追加。
 * 2025 / 4 / 10 SimpleVoiceControlと併用して使用できるコマンド追加。
 *
 * 利用規約：
 *  プラグイン作者に無断で使用、改変、再配布は不可。
 *-----------------------------------------------------------------------------
 */

(function () {
    "use strict";

    const parameters = PluginManager.parameters('Onchat');

    const disableChatSwitchId = Number(parameters['DisableChatSwitchID'] || 0);
    const switchId = Number(parameters['SwitchID'] || 1);
    const maxLogEntries = Number(parameters['MaxLogEntries'] || 100);

    const paddingTop = Number(parameters['PaddingTop'] || 8);
    const paddingBottom = Number(parameters['PaddingBottom'] || 8);
    const paddingLeft = Number(parameters['PaddingLeft'] || 12);
    const paddingRight = Number(parameters['PaddingRight'] || 12);

    const lineSpacing = Number(parameters['LineSpacing'] || 4);
    const fontSize = Number(parameters['FontSize'] || 28);
    const iconSize = Number(parameters['IconSize'] || 32);

    const defaultWindowX = Number(parameters['WindowX'] || 0);
    const defaultWindowY = Number(parameters['WindowY'] || 0);
    const defaultWindowWidth = Number(parameters['WindowWidth'] || 480);
    const defaultWindowHeight = Number(parameters['WindowHeight'] || 216);
    const defaultWindowOpacity = Number(parameters['WindowOpacity'] || 255);

    const defaultScrollAmount = Number(parameters['ScrollAmount'] || 40);

    const customBackgroundImage = parameters['CustomBackgroundImage'] || '';
	const voiceCompleteSwitchId = Number(parameters['VoiceCompleteSwitchID'] || 0);

    const OnchatSw = Number(parameters['OnScrollSwitch'] || 0);

    let chatEntries = [];

    let scrollY = 0;
    let totalLogHeight = 0;
    let maxScrollY = 0;

    let currentWindowX = defaultWindowX;
    let currentWindowY = defaultWindowY;
    let currentWindowWidth = defaultWindowWidth;
    let currentWindowHeight = defaultWindowHeight;
    let currentWindowOpacity = defaultWindowOpacity;

    let scrollAmount = defaultScrollAmount;

    let $chatLogData = {
        entries: [],
        scrollY: 0,
        wasVisible: false,
        windowX: currentWindowX,
        windowY: currentWindowY,
        windowWidth: currentWindowWidth,
        windowHeight: currentWindowHeight,
        windowOpacity: currentWindowOpacity
    };

    function Window_ChatLog() {
        this.initialize(...arguments);
    }

    Window_ChatLog.prototype = Object.create(Window_Base.prototype);
    Window_ChatLog.prototype.constructor = Window_ChatLog;

    Window_ChatLog.prototype.initialize = function () {
        Window_Base.prototype.initialize.call(
            this,
            currentWindowX,
            currentWindowY,
            currentWindowWidth,
            currentWindowHeight
        );

        this.opacity = currentWindowOpacity;
        this.contents.fontSize = fontSize;

        this.backOpacity = 192;
        this.contentsOpacity = 255;

        if (customBackgroundImage) {
            this._backgroundSprite = new Sprite();
            const bgBitmap = ImageManager.loadPicture(customBackgroundImage);
            this._backgroundSprite.bitmap = bgBitmap;
            this.addChildToBack(this._backgroundSprite);

            bgBitmap.addLoadListener(() => {
               const w = bgBitmap.width;
               const h = bgBitmap.height;
                this.setWindowSize(w, h, currentWindowX, currentWindowY, currentWindowOpacity);
            });
        }

        this.hide();

        this.updateVisibility();

        this.redrawLog();

    };

    Window_ChatLog.prototype.updateVisibility = function () {
        if ($gameSwitches.value(switchId)) {
            this.showWindow();
        } else {
            this.hideWindow();
        }
    };

    Window_ChatLog.prototype.textPadding = function () {
        return 0;
    };

    Window_ChatLog.prototype.lineHeight = function () {
        return this.contents.fontSize + 6;
    };

    Window_ChatLog.prototype.convertEscapeCharacters = function(text) {
        text = text.replace(/\\+n(?![\[\d])/g, '\n');
        text = text.replace(/\\/g, '\x1b');

        text = text.replace(/\x1bV\[(\d+)\]/gi, (_, n) => {
            return $gameVariables.value(Number(n));
        });

        text = text.replace(/\\SV\[([^\]]+)\]/gi, (_, arg) => {
            const expanded = this.convertEscapeCharacters(arg);
            this._lastVoiceName = expanded;
            return '';
        });

        text = text.replace(/\x1bSC\[(\d+)\]/gi, (_, n) => {
            const varValue = $gameVariables.value(Number(n));
            this._lastVoiceName = varValue;
            return '';
        });

        text = text.replace(/\x1bN\[(\d+)\]/gi, (_, n) => {
            const actor = $gameActors.actor(Number(n));
            return actor ? actor.name() : '';
        });
    
        text = text.replace(/\x1bITEM\[(\d+)\]/gi, (_, id) => {
            const item = $dataItems[parseInt(id)];
            return item ? item.name : "";
        });
    
        text = text.replace(/\x1bWEAPON\[(\d+)\]/gi, (_, id) => {
            const wep = $dataWeapons[parseInt(id)];
            return wep ? wep.name : "";
        });
    
        text = text.replace(/\x1bARMOR\[(\d+)\]/gi, (_, id) => {
            const arm = $dataArmors[parseInt(id)];
            return arm ? arm.name : "";
        });
    
        text = text.replace(/\x1bG/gi, TextManager.currencyUnit);

        return text;
    };


    Window_ChatLog.prototype.scrollYPlus = function () {
        scrollY = Math.min(scrollY + scrollAmount, maxScrollY);
        this.redrawLog();
    };


    Window_ChatLog.prototype.scrollYMinus = function () {
        scrollY = Math.max(scrollY - scrollAmount, 0);
        this.redrawLog();
    };

    Window_ChatLog.prototype.redrawLog = function () {
        this.contents.clear();
        this._playedVoices = [];
        this.updateTotalHeight();

        let currentY = this.contents.height - paddingBottom + scrollY;

        for (let i = chatEntries.length - 1; i >= 0; i--) {
            const entry = chatEntries[i];
            let entryHeight;
            if (entry.height) {
                entryHeight = entry.height;
            } else if (entry.text && entry.text.includes('\n')) {
                entryHeight = (fontSize + lineSpacing) * 2;
            } else {
                entryHeight = fontSize + lineSpacing;
            }

            if (currentY < -entryHeight) {
                break;
            }
    
            if (entry.text && entry.text.includes('\n')) {
                const lineCount = entry.text.split('\n').length;
                entryHeight = (fontSize + lineSpacing) * lineCount;
            }
    
            currentY -= entryHeight;

            if (currentY > this.contents.height) {
                continue;
            }

            if (entry.type === "text") {
                this.contents.fontSize = fontSize;

                const textX = (entry.align === "right")
                    ? this.contents.width - paddingRight
                    : paddingLeft;

                this.drawChatText(entry.text, textX, currentY, entry.align);
            }

            else if (entry.type === "stamp") {
                const bitmap = ImageManager.loadPicture(entry.stampName);
                if (bitmap && bitmap.isReady()) {
                    const stampW = entry.width;
                    const stampH = entry.height;
                    let stampX;

                    if (entry.align === "right") {
                        stampX = this.contents.width - stampW - paddingRight;
                    } else {
                        stampX = paddingLeft;
                    }

                    this.contents.blt(
                        bitmap,
                        0, 0,
                        bitmap.width,
                        bitmap.height,
                        stampX, currentY,
                        stampW, stampH
                    );
                } else {
                    bitmap.addLoadListener(() => {
                        this.redrawLog();
                    });
                }
            }
        }

        this.drawScrollBar();

        this.contents.fontSize = fontSize;
    };

    Window_ChatLog.prototype.drawScrollBar = function () {
        const barWidth = 8;
        const marginRight = 4;
        const visibleHeight = this.contents.height - paddingTop - paddingBottom;

        if (totalLogHeight <= visibleHeight) return;

        const trackX = this.contents.width - barWidth - marginRight;
        const trackY = paddingTop;
        const trackH = visibleHeight;

        this.contents.paintOpacity = 48;
        this.contents.fillRect(trackX, trackY, barWidth, trackH, "#000000");

        const barRatio = visibleHeight / totalLogHeight;
        const barHeight = Math.floor(visibleHeight * barRatio);
        const maxScroll = totalLogHeight - visibleHeight;
        const barY = paddingTop + Math.floor(((maxScroll - scrollY) / maxScroll) * (visibleHeight - barHeight));

        this.contents.paintOpacity = 128;
        this.contents.fillRect(trackX, barY, barWidth, barHeight, "#FFFFFF");

        this.contents.paintOpacity = 255;
    };

    Window_ChatLog.prototype.updateTotalHeight = function () {
            let total = 0;
            for (let i = chatEntries.length - 1; i >= 0; i--) {
                const entry = chatEntries[i];
                let entryHeight;
                if (entry.height) {
                    entryHeight = entry.height;
                } else if (entry.text && entry.text.includes('\n')) {
                    entryHeight = (fontSize + lineSpacing) * 2;
                } else {
                    entryHeight = fontSize + lineSpacing;
                }
                if (entry.text && entry.text.includes('\n')) {
                    const normalizedText = this.convertEscapeCharacters(entry.text);
                    const lineCount = normalizedText.split('\n').length;
                    entryHeight = (fontSize + lineSpacing) * lineCount;
                }
                total += entryHeight;
            }
            totalLogHeight = total;
    
            const visibleHeight = this.contents.height - paddingTop - paddingBottom;
            maxScrollY = Math.max(totalLogHeight - visibleHeight, 0);
    
            scrollY = Math.min(scrollY, maxScrollY);
        };

    Window_ChatLog.prototype.scrollToBottom = function () {
        this.updateTotalHeight();
        scrollY = 0;
        this.redrawLog();
    };

    Window_ChatLog.prototype.clearLog = function () {
        chatEntries = [];
        scrollY = 0;
        totalLogHeight = 0;
        maxScrollY = 0;
        this.contents.clear();
    };

    Window_ChatLog.prototype.hideWindow = function () {
        this.visible = false;
        this.hide();
    };

    Window_ChatLog.prototype.showWindow = function () {
        this.visible = true;
        this.show();
        this.redrawLog();
    };

    Window_ChatLog.prototype.setWindowSize = function (width, height, x, y, opacity) {
        if (width !== undefined) currentWindowWidth = width;
        if (height !== undefined) currentWindowHeight = height;
        if (x !== undefined) currentWindowX = x;
        if (y !== undefined) currentWindowY = y;
        if (opacity !== undefined) currentWindowOpacity = opacity;

        this.move(currentWindowX, currentWindowY, currentWindowWidth, currentWindowHeight);
        this.opacity = currentWindowOpacity;
        this.createContents();
        this.redrawLog();
    };

    Window_ChatLog.prototype.drawChatText = function(text, x, y, align) {
        const normalizedText = text.replace(/\\+n(?![\[\d])/g, '\n');
        const textState = {
            index: 0,
            x: x,
            y: y,
            left: x,
            text: this.convertEscapeCharacters(normalizedText),
            drawing: true
        };

        this.resetTextColor();
        this.contents.fontBold = false;
        this.contents.fontItalic = false;
        this.contents.outlineWidth = 4;

        while (textState.index < textState.text.length) {
            const c = textState.text.charAt(textState.index);
            if (c === '\x1b') {
                textState.index++;
                const code = this.obtainEscapeCode(textState);
                this.processChatEscapeCharacter(code, textState);
            } else if (c === '\n') {
                textState.x = textState.left;
                textState.y += this.lineHeight();
                textState.index++;
            } else {
                const w = this.textWidth(c);
                this.contents.drawText(c, textState.x, textState.y, w, this.lineHeight(), 'left');
                textState.x += w;
                textState.index++;
            }
        }
    };

    Window_ChatLog.prototype.processChatEscapeCharacter = function(code, textState) {
        switch (code) {
            case 'C': {
                const colorIndex = this.obtainEscapeParam(textState);
                this.changeTextColor(this.textColor(colorIndex));
                break;
            }
            case 'I': {
                const iconIndex = this.obtainEscapeParam(textState);
                this.drawIcon(iconIndex, textState.x, textState.y + 2);
                textState.x += this.iconWidth;
                break;
            }
            case 'FS': {
                const size = this.obtainEscapeParam(textState);
                this.contents.fontSize = size;
                break;
            }
            case 'V': {
                const variableId = this.obtainEscapeParam(textState);
                const value = $gameVariables.value(variableId).toString();
                for (let i = 0; i < value.length; i++) {
                    const char = value.charAt(i);
                    const w = this.textWidth(char);
                    this.contents.drawText(char, textState.x, textState.y, w, this.lineHeight(), 'left');
                    textState.x += w;
                }
                break;
            }
            case 'FB': {
                this.contents.fontBold = !this.contents.fontBold;
                break;
            }
            case 'FI': {
                this.contents.fontItalic = !this.contents.fontItalic;
                break;
            }
            case 'OW': {
                const width = this.obtainEscapeParam(textState);
                this.contents.outlineWidth = width;
                break;
            }
            case 'SV': {
                const match = textState.text.slice(textState.index).match(/^\[([^\]]+)\]/);
                if (match) {
                    let fileName = match[1];
                    textState.index += match[0].length;

                    fileName = this.convertEscapeCharacters(fileName);

                    if (this._playedVoices && this._playedVoices.includes(fileName)) {
                    } else {
                        this._lastVoiceName = fileName;
                    }
                }
                break;
            }
        }
    };

    Window_ChatLog.prototype.obtainEscapeCode = function(textState) {
        const regExp = /^([A-Z]{1,3})/i;
        const match = regExp.exec(textState.text.slice(textState.index));
        if (match) {
            textState.index += match[1].length;
            return match[1].toUpperCase();
        }
        return '';
    };

    Window_ChatLog.prototype.obtainEscapeParam = function(textState) {
        const regExp = /^\[(\d+)\]/;
        const match = regExp.exec(textState.text.slice(textState.index));
        if (match) {
            textState.index += match[0].length;
            return Number(match[1]);
        } else {
            return 0;
        }
    };

    Window_ChatLog.prototype.addChatText = function (text, align) {
        if ($gameSwitches.value(disableChatSwitchId)) return;

        const rawText = text;
        const displayText = this.convertEscapeCharacters(text);

        const hasVoiceTag = rawText.includes("\\SV[") || rawText.includes("\\SC[");

        if (hasVoiceTag && window.SimpleVoiceControlParams && window.SimpleVoiceControlParams.enableVoicePlayback) {
            if (!this._lastVoiceName) {
                const svMatch = /\\SV\[(.+?)\]/i.exec(rawText);
                if (svMatch && svMatch[1]) {
                    this._lastVoiceName = svMatch[1];
                } else {
                    return; // ボイス名が不明 → 処理を中断
                }
            }

            const sound = {
                name: this._lastVoiceName,
                volume: ConfigManager.voiceVolume || 100,
                pitch: 100,
                pan: 0
            };

            if (this._currentVoiceBuffer && this._currentVoiceBuffer.isPlaying()) {
                this._currentVoiceBuffer.stop();
            }

            const buffer = AudioManager.createBuffer('voice', sound.name);
            this._currentVoiceBuffer = buffer;

            buffer.addLoadListener(function () {
                buffer.play(false, 0);
                buffer.addStopListener(function () {
                    if (voiceCompleteSwitchId > 0) {
                        $gameSwitches.setValue(voiceCompleteSwitchId, true);
                    }
                });
            });

            this._lastVoiceName = null;
        }

        const entryHeight = fontSize + lineSpacing;
        const entry = {
            type: "text",
            text: displayText,
            align: align,
            fontSize: fontSize,
            height: entryHeight
        };

        chatEntries.push(entry);
        if (chatEntries.length > maxLogEntries) {
            chatEntries.shift();
        }

        this.scrollToBottom();
        //this.showWindow();
    };

    Window_ChatLog.prototype.addStamp = function (stampName, align, width, height) {
        if ($gameSwitches.value(disableChatSwitchId)) return;

        const defaultWidth = Number(parameters['DefaultStampWidth']) || 100;
        const defaultHeight = Number(parameters['DefaultStampHeight']) || 100;
        const bitmap = ImageManager.loadPicture(stampName);
        bitmap.addLoadListener(() => {
            let finalWidth = bitmap.width;
            let finalHeight = bitmap.height;

            if (finalWidth > defaultWidth || finalHeight > defaultHeight) {
                const widthRatio = defaultWidth / finalWidth;
                const heightRatio = defaultHeight / finalHeight;
                const scale = Math.min(widthRatio, heightRatio);

                finalWidth = Math.floor(finalWidth * scale);
                finalHeight = Math.floor(finalHeight * scale);
            }

            const entry = {
                type: "stamp",
                stampName: stampName,
                align: align,
                width: finalWidth,
                height: finalHeight
            };

            chatEntries.push(entry);
            if (chatEntries.length > maxLogEntries) {
                chatEntries.shift();
            }

            this.scrollToBottom();
            this.redrawLog();
            this.showWindow();
        });
    };


    const _Game_Interpreter_pluginCommand = Game_Interpreter.prototype.pluginCommand;
    Game_Interpreter.prototype.pluginCommand = function (command, args) {
        if (_Game_Interpreter_pluginCommand) {
            _Game_Interpreter_pluginCommand.call(this, command, args);
        }

        if (!SceneManager._scene || !SceneManager._scene._chatLogWindow) {
            return;
        }

        const chatWindow = SceneManager._scene._chatLogWindow;
        if (!args) args = [];

        switch (command) {
            case "chatleft":
            case "chatright": {
                const text = args.join(" ").replace(/_/g, " ");
                const align = (command === "chatleft") ? "left" : "right";
                chatWindow.addChatText(text, align);
                break;
            }
            case "chatsize": {
                const w = Number(args[0] || currentWindowWidth);
                const h = Number(args[1] || currentWindowHeight);
                const x = Number(args[2] || currentWindowX);
                const y = Number(args[3] || currentWindowY);
                const op = (args[4] !== undefined) ? Number(args[4]) : currentWindowOpacity;
                chatWindow.setWindowSize(w, h, x, y, op);
                break;
            }
            case "chatscroll": {
                const val = Number(args[0] || defaultScrollAmount);
                scrollAmount = val;
                break;
            }
            case "chatclear": {
                chatWindow.clearLog();
                break;
            }
            case "chatpos": {
                const rawX = args[0] || currentWindowX;
                const rawY = args[1] || currentWindowY;

                const x = Window_ChatLog.prototype.convertEscapeCharacters.call(chatWindow, rawX);
                const y = Window_ChatLog.prototype.convertEscapeCharacters.call(chatWindow, rawY);

                chatWindow.setWindowSize(
                    currentWindowWidth,
                    currentWindowHeight,
                    Number(x),
                    Number(y),
                    currentWindowOpacity
                );
                break;
            }
            case "chatstopvoice":
                if (chatWindow._currentVoiceBuffer && chatWindow._currentVoiceBuffer.isPlaying()) {
                    chatWindow._currentVoiceBuffer.stop();
                    chatWindow._currentVoiceBuffer = null;
                }
                break;
        }
    };

    function onchat(align, content) {
        if (!SceneManager._scene || !SceneManager._scene._chatLogWindow) {
            console.warn("ChatLog Window is not initialized");
            return;
        }
        const chatWindow = SceneManager._scene._chatLogWindow;

        if (align !== "left" && align !== "right") {
            console.warn("Invalid align: use 'left' or 'right'");
            return;
        }

        if (typeof content === "string") {
            const normalizedContent = content.replace(/\\+n(?![\[\d])/g, '\n');
            chatWindow.addChatText(normalizedContent, align);
        } else if (typeof content === "object" && content.stamp) {
            chatWindow.addStamp(content.stamp, align, content.width, content.height);
        } else {
            console.warn("Invalid content. Must be a string or {stamp:'img', ...}");
        }
    }
    window.onchat = onchat;

    const _Scene_Map_createAllWindows = Scene_Map.prototype.createAllWindows;
    Scene_Map.prototype.createAllWindows = function () {
        _Scene_Map_createAllWindows.call(this);

        this._chatLogWindow = new Window_ChatLog();
        this.addWindow(this._chatLogWindow);

        this._chatLogWindow.restoreFromSaveData();
    };

    const _Scene_Map_updateMain = Scene_Map.prototype.updateMain;
    Scene_Map.prototype.updateMain = function () {
        _Scene_Map_updateMain.call(this);

        if (this._chatLogWindow) {
            this._chatLogWindow.update();

            if ($gameScreen.brightness() < 255) {
                this._chatLogWindow.hide();
            } else if ($gameSwitches.value(switchId)) {
                this._chatLogWindow.show();
            } else {
                this._chatLogWindow.hide();
            }
        }
    };

    Window_ChatLog.prototype.restoreFromSaveData = function () {
        if ($chatLogData.entries) {
            chatEntries = JSON.parse(JSON.stringify($chatLogData.entries));
        }
        scrollY = $chatLogData.scrollY || 0;
        totalLogHeight = 0;

        currentWindowX = $chatLogData.windowX || currentWindowX;
        currentWindowY = $chatLogData.windowY || currentWindowY;
        currentWindowWidth = $chatLogData.windowWidth || currentWindowWidth;
        currentWindowHeight = $chatLogData.windowHeight || currentWindowHeight;
        currentWindowOpacity = $chatLogData.windowOpacity || currentWindowOpacity;

        this.move(currentWindowX, currentWindowY, currentWindowWidth, currentWindowHeight);
        this.opacity = currentWindowOpacity;
        this.createContents();

        if ($chatLogData.wasVisible) {
            this.showWindow();
        } else {
            this.hideWindow();
        }
        this.redrawLog();
    };

    const _Scene_Map_start = Scene_Map.prototype.start;
    Scene_Map.prototype.start = function () {
        _Scene_Map_start.call(this);

        if ($chatLogData.wasVisible && this._chatLogWindow) {
            this._chatLogWindow.showWindow();
        }
    };

    const _Scene_Map_terminate = Scene_Map.prototype.terminate;
    Scene_Map.prototype.terminate = function () {
        if (this._chatLogWindow) {
            this._chatLogWindow.saveChatLogData();
        }
        _Scene_Map_terminate.call(this);
    };

    Window_ChatLog.prototype.saveChatLogData = function () {
        $chatLogData.entries = JSON.parse(JSON.stringify(chatEntries));
        $chatLogData.scrollY = scrollY;
        $chatLogData.wasVisible = this.visible;
        $chatLogData.windowX = currentWindowX;
        $chatLogData.windowY = currentWindowY;
        $chatLogData.windowWidth = currentWindowWidth;
        $chatLogData.windowHeight = currentWindowHeight;
        $chatLogData.windowOpacity = currentWindowOpacity;
    };

    const _DataManager_makeSaveContents = DataManager.makeSaveContents;
    DataManager.makeSaveContents = function () {
        const contents = _DataManager_makeSaveContents.call(this);
        contents.chatLogData = JSON.parse(JSON.stringify($chatLogData));
        return contents;
    };

    const _DataManager_extractSaveContents = DataManager.extractSaveContents;
    DataManager.extractSaveContents = function (contents) {
        _DataManager_extractSaveContents.call(this, contents);
        if (contents.chatLogData) {
            $chatLogData = JSON.parse(JSON.stringify(contents.chatLogData));
        }
    };

    Window_ChatLog.prototype._onWheelHandler = function(event) {
        if (!this.visible) return;
        if ($gameSwitches.value(OnchatSw)) return;

        if (event.deltaY > 0) {
            this.scrollYMinus();
        }
        else if (event.deltaY < 0) {
            this.scrollYPlus();
        }

        event.preventDefault();
    };

    Window_ChatLog.prototype.attachScrollHandler = function() {
        if (!this._wheelBoundHandler) {
            this._wheelBoundHandler = this._onWheelHandler.bind(this);
            document.addEventListener('wheel', this._wheelBoundHandler, { passive: false });
        }
    };

    Window_ChatLog.prototype.detachScrollHandler = function() {
        if (this._wheelBoundHandler) {
            document.removeEventListener('wheel', this._wheelBoundHandler);
            this._wheelBoundHandler = null;
        }
    };

    Window_ChatLog.prototype.update = function() {
        Window_Base.prototype.update.call(this);

        if ($gameSwitches.value(OnchatSw)) {
            this.detachScrollHandler();
        } else {
            this.attachScrollHandler();
        }
    };
    
    window.stopChatVoice = function() {
        if (SceneManager._scene && SceneManager._scene._chatLogWindow) {
            const chatWindow = SceneManager._scene._chatLogWindow;
            if (chatWindow._currentVoiceBuffer && chatWindow._currentVoiceBuffer.isPlaying()) {
                chatWindow._currentVoiceBuffer.stop();
                chatWindow._currentVoiceBuffer = null;
            }
        }
    };

})();
