//=============================================================================
// Onpicturecheck.js
//=============================================================================

/*:
 * @plugindesc 使用中ピクチャ判定
 * @author Onmoremind
 *
 * @help
 * 画面表示中のpictureを判定(Dtext対応)
 * なしで使用している場合はSpineの可能性がアリの表示
 * テストプレイ時のみ有効
 * 
 * controlボタンでウィンドウ表示をON/OFF(デフォルト)
 *
 * @param toggleKey
 * @text ウィンドウ切り替えキー
 * @desc ウィンドウのON/OFFを切り替えるキーを指定
 * 使用出来る入力keyはツクールが対応しているものだけ
 * @default control
 */

(function () {
    if (!Utils.isOptionValid('test')) return;

    const parameters = PluginManager.parameters('Onpicturecheck');
    const toggleKey = parameters['toggleKey'] || 'control';

    let isWindowVisible = false; 

    function isDynamicPicture(pictureName) {
        return /^[0-9]+$/.test(pictureName);
    }

    function Window_PictureInfo() {
        this.initialize.apply(this, arguments);
    }

    Window_PictureInfo.prototype = Object.create(Window_Base.prototype);
    Window_PictureInfo.prototype.constructor = Window_PictureInfo;

    Window_PictureInfo.prototype.lineHeight = function () {
        return 20;
    };

    Window_PictureInfo.prototype.initialize = function () {
        const width = Graphics.boxWidth;
        const height = Graphics.boxHeight;
        Window_Base.prototype.initialize.call(this, 0, 0, width, height);
        this.opacity = 255;
        this.contents.fontSize = 14;
        this.refresh();
    };

    Window_PictureInfo.prototype.refresh = function () {
        this.contents.clear();
        if (!isWindowVisible) return; 

        const maxPictures = 100;
        const numColumns = 4;
        const columnWidth = this.contentsWidth() / numColumns;
        const numRows = Math.ceil(maxPictures / numColumns);

        for (let col = 0; col < numColumns; col++) {
            for (let row = 0; row < numRows; row++) {
                const pictureId = row + col * numRows + 1;
                if (pictureId > maxPictures) break;

                const picture = $gameScreen.picture(pictureId);
                let text = "";

                if (picture && picture.name() && picture.name().trim() !== "") {
                    const pictureName = picture.name();

                    if (isDynamicPicture(pictureName)) {
                        text = `ID: ${pictureId} (DText動的ピクチャ)`;
                    } else {
                        try {
                            const bitmap = ImageManager.loadPicture(pictureName);
                            if (bitmap.isReady() && bitmap.width > 0 && bitmap.height > 0) {
                                const width = bitmap.width;
                                const height = bitmap.height;
                                const origin = picture._origin === 1 ? "中央" : "左上";
                                text = `ID: ${pictureId}, 幅: ${width}px, 高さ: ${height}px, 原点: ${origin}`;
                            } else {
                                text = `ID: ${pictureId} の画像がロードされていません。`;
                            }
                        } catch (e) {
                            text = `ID: ${pictureId} の画像を読み込めませんでした。`;
                        }
                    }
                } else if (picture && !picture.name()) {
                    text = `ID: ${pictureId} 「Spine使用中の可能性アリ」`;
                }

                if (text) {
                    const x = col * columnWidth;
                    const y = row * this.lineHeight();
                    this.drawText(text, x, y, columnWidth, 'left');
                }
            }
        }
    };

    const _Scene_Map_createAllWindows = Scene_Map.prototype.createAllWindows;
    Scene_Map.prototype.createAllWindows = function () {
        _Scene_Map_createAllWindows.call(this);
        this._pictureInfoWindow = new Window_PictureInfo();
        this._pictureInfoWindow.zIndex = 1; 
        this.addWindow(this._pictureInfoWindow);

        if (this._messageWindow) {
            this._messageWindow.zIndex = 10;
            this._windowLayer.children.sort((a, b) => a.zIndex - b.zIndex);
        }
    };

    const _Scene_Map_update = Scene_Map.prototype.update;
    Scene_Map.prototype.update = function () {
        _Scene_Map_update.call(this);

        if (Input.isTriggered(toggleKey)) { 
            isWindowVisible = !isWindowVisible;
            this._pictureInfoWindow.refresh();
        }

        if (this._pictureInfoWindow) {
            this._pictureInfoWindow.visible = isWindowVisible;  
        }
    };

})();
