//=============================================================================
// SavePlayTime.js
//=============================================================================

/*:
 * @plugindesc セーブデータ毎の合計プレイ時間を記録できるプラグイン
 * @author NJ
 *
 * @help
 * セーブごとの合計プレイ時間を 00:00:00 形式で記録・表示します。
 *
 * 【プラグインコマンド】
 * SavePlayTime SetVariable 1
 *  - 合計プレイ時間を変数 ID 1 に格納します。
 *
 * 【スクリプトコマンド】
 * - $gameSystem.getSavePlayTime();  
 *   - 合計プレイ時間を 00:00:00 の形式で返します。
 *
 * 利用規約：
 *  プラグイン作者に無断で使用、改変、再配布は不可です。
 */

(() => {
    'use strict';

    const pluginName = 'SavePlayTime';

    function formatTime(totalSeconds) {
        const hours = Math.floor(totalSeconds / 3600);
        const minutes = Math.floor((totalSeconds % 3600) / 60);
        const seconds = Math.floor(totalSeconds % 60);
        return `${hours.toString().padStart(2, '0')}:${minutes
            .toString()
            .padStart(2, '0')}:${seconds.toString().padStart(2, '0')}`;
    }

    const _Game_System_initialize = Game_System.prototype.initialize;
    Game_System.prototype.initialize = function () {
        _Game_System_initialize.call(this);
        this._savePlayTime = 0;
        this._lastUpdateTime = Date.now();
    };

    const _Scene_Map_update = Scene_Map.prototype.update;
    Scene_Map.prototype.update = function () {
        _Scene_Map_update.call(this);
        this.updateSavePlayTime();
    };

    Scene_Map.prototype.updateSavePlayTime = function () {
        const currentTime = Date.now();
        const deltaTime = (currentTime - $gameSystem._lastUpdateTime) / 1000;
        $gameSystem._savePlayTime += deltaTime;
        $gameSystem._lastUpdateTime = currentTime;
    };

    const _Scene_Battle_update = Scene_Battle.prototype.update;
    Scene_Battle.prototype.update = function () {
        _Scene_Battle_update.call(this);
        this.updateSavePlayTime();
    };

    Scene_Battle.prototype.updateSavePlayTime = function () {
        const currentTime = Date.now();
        const deltaTime = (currentTime - $gameSystem._lastUpdateTime) / 1000;
        $gameSystem._savePlayTime += deltaTime;
        $gameSystem._lastUpdateTime = currentTime;
    };

    const _DataManager_makeSaveContents = DataManager.makeSaveContents;
    DataManager.makeSaveContents = function () {
        const contents = _DataManager_makeSaveContents.call(this);
        contents.savePlayTime = $gameSystem._savePlayTime;
        return contents;
    };

    const _DataManager_extractSaveContents =
        DataManager.extractSaveContents;
    DataManager.extractSaveContents = function (contents) {
        _DataManager_extractSaveContents.call(this, contents);
        $gameSystem._savePlayTime = contents.savePlayTime || 0;
        $gameSystem._lastUpdateTime = Date.now();
    };

    Game_System.prototype.getSavePlayTime = function () {
        const totalSeconds = Math.floor(this._savePlayTime || 0);
        return formatTime(totalSeconds);
    };

    const _Game_Interpreter_pluginCommand =
        Game_Interpreter.prototype.pluginCommand;
    Game_Interpreter.prototype.pluginCommand = function (
        command,
        args
    ) {
        _Game_Interpreter_pluginCommand.call(this, command, args);
        if (command === 'SavePlayTime') {
            if (args[0] === 'SetVariable' && args[1]) {
                const variableId = Number(args[1]);
                const totalSeconds = Math.floor($gameSystem._savePlayTime);
                const formattedTime = formatTime(totalSeconds);
                $gameVariables.setValue(variableId, formattedTime);
            }
        }
    };
})();
