//=============================================================================
// SimpleVoiceControl.js
//=============================================================================

/*:
 * @plugindesc SimpleVoiceControl V1.1.0
 * @author NJ
 * 
 * @param VoiceCutSwitch
 * @type switch
 * @desc ボイスカットを制御するスイッチID。このスイッチがONの場合、メッセージが進むと再生中のボイスを停止します。
 * @default 1
 * 
 * @param EnableVoicePlayback
 * @type boolean
 * @text ボイス再生の有効化
 * @desc ボイス再生を有効にするかどうかを設定します。[true: 再生する / false: 再生しない]
 * @default true
 * 
 * @help
 * このプラグインはSimpleVoiceプラグインを拡張し、以下の機能を提供します。
 * 
 * - 制御文字`\SV[ファイル名]`でボイスを再生
 * - 指定スイッチがONの場合、メッセージ終了時に再生中のボイスを停止
 * - 再生機能をプラグインパラメータで有効/無効を切り替え可能
 *
 * 使用例:
 * \SV[voice1] => ボイス「voice1」を再生
 * 
 * 注意:
 * - SimpleVoiceプラグインが必要です。
 * - プラグイン管理でSimpleVoiceの「下」に配置してください。
 *
 * バージョン
 * v1.0.0 初回
 * v1.0.1 ボイス再生の有効化機能を追加。
 * v1.0.2 MPP_MessageEXとの競合を修正。
 * v1.1.3 Onchatとの連携を可能に。
 *
 * 利用規約：
 *  プラグイン作者に無断で使用、改変、再配布は不可です。
 */

(function() {
    'use strict';

    const parameters = PluginManager.parameters('SimpleVoiceControl');
    const voiceCutSwitchId = Number(parameters['VoiceCutSwitch'] || 1);
    const enableVoicePlayback = parameters['EnableVoicePlayback'] === 'true';

    let lastPlayedVoiceName = null;

    const _Window_Message_processEscapeCharacter = Window_Message.prototype.processEscapeCharacter;
    Window_Message.prototype.processEscapeCharacter = function(code, textState) {
        if (code === 'SV') {
            const match = textState.text.slice(textState.index).match(/^\[([^\]]+)\]/);
            if (match) {
                const fileName = match[1];
                textState.index += match[0].length;

                if (enableVoicePlayback) {
                    lastPlayedVoiceName = fileName;
                    AudioManager.playVoice({
                        name: fileName,
                        volume: ConfigManager.voiceVolume || 100,
                        pitch: 100,
                        pan: 0
                    });
                }
            }
        } else {
            _Window_Message_processEscapeCharacter.call(this, code, textState);
        }
    };

    const _Window_Message_terminateMessage = Window_Message.prototype.terminateMessage;
    Window_Message.prototype.terminateMessage = function() {
        if ($gameSwitches.value(voiceCutSwitchId) && lastPlayedVoiceName) {
            AudioManager.stopVoice(lastPlayedVoiceName);
            lastPlayedVoiceName = null;
        }
        _Window_Message_terminateMessage.call(this);
    };

    window.SimpleVoiceControlParams = {
        enableVoicePlayback: enableVoicePlayback
    };

})();
