//=============================================================================
// SlotMachine.js
//=============================================================================

/*:
 * @plugindesc スロットゲームプラグイン V1.0.0
 * @author NJ
 *
 * @param windowX
 * @text スロットウィンドウX座標
 * @desc スロットウィンドウの左上X位置
 * @default 100
 *
 * @param windowY
 * @text スロットウィンドウY座標
 * @desc スロットウィンドウの左上Y位置
 * @default 100
 *
 * @param windowWidth
 * @text スロットウィンドウの横幅
 * @desc ウィンドウの幅(推奨300以上)
 * @default 350
 *
 * @param windowHeight
 * @text スロットウィンドウの高さ
 * @desc ウィンドウの高さ(推奨300以上)
 * @default 350
 *
 * @param stopType
 * @text 停止方法
 * @desc "自動停止" or "手動停止"
 * @default 手動停止
 *
 * @param reelCount
 * @text リールの数
 * @desc 通常は3 (3リール)
 * @default 3
 *
 * @param symbols
 * @text シンボルリスト
 * @type struct<SymbolData>[]
 * @desc 各シンボルの設定（画像と名前のセット）
 * @default []
 *
 * @param reels
 * @text 各リールのシンボル順
 * @type struct<ReelData>[]
 * @desc 各リールのシンボルの並び順
 * @default []
 *
 * @param paylines
 * @text 役リスト
 * @type struct<Payline>[]
 * @desc 役リストを設定（シンボル3つと役名）
 * @default []
 *
 * @param resultVariable
 * @text 結果保存用変数ID
 * @type variable
 * @desc 揃った役の名前を保存する変数のID
 * @default 10
 *
 * @param symbolCheckMode
 * @text シンボル判定モード
 * @type number
 * @min 1
 * @max 4
 * @desc 1:中央のみ / 2:中央+上段 / 3:上段+中央+下段 / 4:全て(斜め含む)
 * @default 1
 *
 * @param reelRowCount
 * @text リールの行数
 * @type number
 * @min 3
 * @max 4
 * @desc リールの縦の行数（3 or 4）
 * @default 4
 *
 * @help
 * 使い方:
 * ShowSlot                  - スロットウィンドウを表示
 *   $gameMap._interpreter.pluginCommand("ShowSlot", []);
 * 
 * CShowSlot                 - スロットウィンドウを表示 (中央のみ)
 *   $gameMap._interpreter.pluginCommand("CShowSlot", []);
 * 
 * StartSlot                 - スロット回転開始
 *   $gameMap._interpreter.pluginCommand("StartSlot", []);
 * 
 * StopSlot                  - 手動停止 (stopType が「手動停止」の場合のみ有効)
 *   $gameMap._interpreter.pluginCommand("StopSlot", []);
 * 
 * SlowStopSlot              - スロットをゆっくりと停止
 *   $gameMap._interpreter.pluginCommand("SlowStopSlot", []);
 * 
 * EndSlot                   - スロットウィンドウを閉じる
 *   $gameMap._interpreter.pluginCommand("EndSlot", []);
 * 
 * CheckSlotResult 1         - 現在の役をチェック (1:中央, 2:上+中, 3:上+中+下, 4:全体)
 *   $gameMap._interpreter.pluginCommand("CheckSlotResult", ["1"]);
 * 
 * UseSlotReels 1 2 3        - 使用するリールを設定 (例: 1 2 3)
 *   $gameMap._interpreter.pluginCommand("UseSlotReels", ["1", "2", "3"]);
 *
 * SetCentralWeight Test 1   - 中央ライン専用の役の重みを設定する (CShowSlot 専用)
 *   $gameMap._interpreter.pluginCommand("SetCentralWeight", ["役名", "重み"]);
 *
 * バージョン
 * v1.0.0 初回
 *
 * 利用規約：
 *  プラグイン作者に無断で使用、改変、再配布は不可です。
 */

/*~struct~SymbolData:
 * @param name
 * @text シンボル名
 * @desc スロットのシンボル名
 * @default
 *
 * @param image
 * @text シンボル画像
 * @desc シンボルの画像名（img/pictures/ にあるファイル名）
 * @default
 */

/*~struct~Payline:
 * @param symbol1
 * @text シンボル1
 * @desc 1つ目のリールのシンボル名
 * @default 
 *
 * @param symbol2
 * @text シンボル2
 * @desc 2つ目のリールのシンボル名
 * @default 
 *
 * @param symbol3
 * @text シンボル3
 * @desc 3つ目のリールのシンボル名
 * @default 
 *
 * @param name
 * @text 役名
 * @desc この組み合わせで揃ったときの役名
 * @default 
 *
 * @param weight
 * @text 出現率
 * @type number
 * @min 1
 * @max 100
 * @desc 役の出現しやすさ（1～100）
 * @default 10
 */

/*~struct~ReelData:
 * @param symbols
 * @text シンボル一覧
 * @type string[]
 * @desc このリールで使用するシンボルの並び順
 * @default ["リプレイ", "7", "肉", "ケーキ"]
 */

var Imported = Imported || {};
Imported.SlotMachine = true;

var SlotMachine = SlotMachine || {};

(function () {
    var parameters = PluginManager.parameters("SlotMachine");
    SlotMachine.windowX      = Number(parameters["windowX"] || 100);
    SlotMachine.windowY      = Number(parameters["windowY"] || 100);
    SlotMachine.windowWidth  = Number(parameters["windowWidth"] || 350);
    SlotMachine.windowHeight = Number(parameters["windowHeight"] || 350);
    SlotMachine.stopType     = String(parameters["stopType"] || "手動停止");
    SlotMachine.reelCount    = Number(parameters["reelCount"] || 3);
    SlotMachine.resultVariable = Number(parameters["resultVariable"] || 10);
    SlotMachine.symbolCheckMode = Number(parameters["symbolCheckMode"] || 1);
    SlotMachine.reelRowCount = Number(parameters["reelRowCount"] || 4);
	SlotMachine.centralOnlyWeights = {};

    try {
        SlotMachine.paylines = JSON.parse(parameters["paylines"] || "[]").map(p => {
            const data = JSON.parse(p);
            return {
                symbols: [data.symbol1, data.symbol2, data.symbol3],
                name: data.name,
                weight: Number(data.weight) || 10
            };
        });
    } catch (e) {
        console.error("SlotMachine.paylines パース失敗:", e);
        SlotMachine.paylines = [];
    }

    try {
        SlotMachine.symbols = JSON.parse(parameters["symbols"] || "[]").map(o => {
            return (typeof o === "string") ? JSON.parse(o) : o;
        });
    } catch (e) {
        console.error("SlotMachine.symbols パース失敗:", e);
        SlotMachine.symbols = [];
    }

    try {
        let reelData = JSON.parse(parameters["reels"] || "[]");

        SlotMachine.reels = reelData.map(reel => {
            if (typeof reel === "string") {
                try {
                    let parsedReel = JSON.parse(reel);
                    return parsedReel.symbols || [];
                } catch (e) {
                    console.error("❌ リールデータのJSONパースに失敗:", reel, e);
                    return [];
                }
            } else if (reel && reel.symbols) {
                return reel.symbols;
            } else {
                console.error("❌ リールデータの形式が不正:", reel);
                return [];
            }
        });

        if (!Array.isArray(SlotMachine.reels)) {
            console.error("❌ `SlotMachine.reels` が配列ではありません！", SlotMachine.reels);
            SlotMachine.reels = [];
        }

    } catch (e) {
        console.error("❌ `SlotMachine.reels` の取得に失敗:", e);
        SlotMachine.reels = [];
    }

    var _Game_Interpreter_pluginCommand = Game_Interpreter.prototype.pluginCommand || function() {};

    Game_Interpreter.prototype.pluginCommand = function (command, args) {
        _Game_Interpreter_pluginCommand.apply(this, arguments);

        switch (command) {
            case "ShowSlot":
                if (SceneManager._scene && SceneManager._scene.showSlot) {
                    SceneManager._scene.showSlot();
                }
                break;
            case "StartSlot":
                if (SceneManager._scene && SceneManager._scene.startSlot) {
                    SceneManager._scene.startSlot();
                }
                break;
            case "StopSlot":
                if (SceneManager._scene && SceneManager._scene.stopSlot) {
                    SceneManager._scene.stopSlot();
                }
                break;
            case "EndSlot":
                if (SceneManager._scene && SceneManager._scene.endSlot) {
                    SceneManager._scene.endSlot();
                }
                break;
            case "CheckSlotResult":
                let mode = args.length > 0 ? parseInt(parseVariableString(args[0]), 10) : SlotMachine.symbolCheckMode;
                if (isNaN(mode) || mode < 1 || mode > 4) {
                    console.error(`CheckSlotResult: 無効な判定モード (${args[0]})`);
                    return;
                }
                if (SceneManager._scene && SceneManager._scene.checkSlotResult) {
                    SceneManager._scene.checkSlotResult(mode);
                }
                break;
            case "CShowSlot":
                if (SceneManager._scene && SceneManager._scene.CShowSlot) {
                    SceneManager._scene.CShowSlot();
                }
                break;
            case "SetPaylineWeight":
                if (args.length < 2) {
                    console.error("SetPaylineWeight: 引数が不足しています");
                    return;
                }

                let paylineName = parseVariableString(args[0]);
                let weight = parseInt(parseVariableString(args[1]), 10);

                if (isNaN(weight) || weight < 1) {
                    console.error("SetPaylineWeight: 出現率の値が無効です:", args[1]);
                    return;
                }

                let payline = SlotMachine.paylines.find(p => p.name === paylineName);
                if (payline) {
                    payline.weight = weight;
                } else {
                    console.error(`SetPaylineWeight: 役 "${paylineName}" が見つかりません`);
                }
                break;
                case "UseSlotReels":
                    if (args.length > 0) {
                        const selectedReels = args.map(arg => Number(parseVariableString(arg)) - 1);
        
                        SlotMachine.currentReels = selectedReels.map(index => {
                            const reel = SlotMachine.reels[index];
                            if (Array.isArray(reel)) {
                                return reel;
                            } else if (typeof reel === "string") {
                                try {
                                    const parsedReel = JSON.parse(reel);
                                    if (Array.isArray(parsedReel)) {
                                        return parsedReel;
                                    } else {
                                        console.error(`❌ ${index + 1}列目のリールデータが配列ではありません。`, parsedReel);
                                        return [];
                                    }
                                } catch (e) {
                                    console.error(`❌ ${index + 1}列目のリールデータのJSONパースに失敗しました。`, reel, e);
                                    return [];
                                }
                            } else {
                                console.error(`❌ ${index + 1}列目のリールデータが無効です。`, reel);
                                return [];
                            }
                        });
                    } else {
                        SlotMachine.currentReels = null;
                    }
                    break;
                    case "SetCentralWeight":
                        if (args.length < 2) {
                            console.error("SetCentralWeight: 引数が不足しています");
                            return;
                        }

                        let centralPaylineName = parseVariableString(args[0]);
                        let centralWeight = parseInt(parseVariableString(args[1]), 10);

                        if (isNaN(centralWeight) || centralWeight < 0) {
                            console.error("SetCentralWeight: 無効な重みです:", args[1]);
                            return;
                        }

                        SlotMachine.centralOnlyWeights[centralPaylineName] = centralWeight;
                        break;
        }
    };

    function Window_Slot() {
        this.initialize.apply(this, arguments);
    }

    Window_Slot.prototype = Object.create(Window_Base.prototype);
    Window_Slot.prototype.constructor = Window_Slot;

    Window_Slot.prototype.initialize = function () {
        Window_Base.prototype.initialize.call(
            this,
            SlotMachine.windowX,
            SlotMachine.windowY,
            SlotMachine.windowWidth,
            SlotMachine.windowHeight
        );
        this.opacity = 0;
        this.backOpacity = 0;
        this.openness = 255;
        this._symbolWidth  = 100;
        this._symbolHeight = 100;
        this._symbolBitmaps = {};
        this._reelsData = [];
        this._reelOffsets = [];
        this._reelStates = [];
        this._spinning = false;
        this.loadSymbolImages();
        this.setupReels();
        this.refresh();
    };

    Window_Slot.prototype.loadSymbolImages = function () {
        let loadCount = 0;
        const totalImages = SlotMachine.symbols.length;

        SlotMachine.symbols.forEach(symbol => {
            if (symbol.image) {
                const bitmap = ImageManager.loadPicture(symbol.image);
                this._symbolBitmaps[symbol.image] = bitmap;                
                bitmap.addLoadListener(() => {
                    loadCount++;
                    if (loadCount >= totalImages) {
                        this.refresh();
                    }
                });
            }
        });
    };

    Window_Slot.prototype.setupReels = function () {
        const reels = SlotMachine.currentReels || SlotMachine.reels;

        if (!Array.isArray(reels)) {
            console.error("❌ setupReels: `SlotMachine.reels` が配列ではありません！", reels);
            return;
        }

        this._reelsData = [];
        this._reelOffsets = [];
        this._reelStates = [];

        const reelCount = SlotMachine.reelCount;

        for (let i = 0; i < reelCount; i++) {
            if (i >= reels.length) break;

            let reelSymbolNames = reels[i];

            if (typeof reelSymbolNames === "string") {
                try {
                    reelSymbolNames = JSON.parse(reelSymbolNames);
                } catch (e) {
                    console.error(`❌ ${i + 1}列目のリールデータのJSONパースに失敗！`, reelSymbolNames, e);
                    reelSymbolNames = [];
                }
            }

            if (!Array.isArray(reelSymbolNames)) {
                console.error(`❌ ${i + 1}列目のリールデータが配列ではありません！`, reelSymbolNames);
                reelSymbolNames = [];
            }

            let reelData = reelSymbolNames.map(name => {
                const found = SlotMachine.symbols.find(s => s.name === name);
                if (!found) {
                    console.warn(`⚠ シンボル "${name}" が見つかりません`, SlotMachine.symbols);
                }
                return found || { name: "Unknown", image: "" };
            });

            this._reelsData.push(reelData);
            this._reelOffsets.push(-this._symbolHeight * 1);
            this._reelStates.push(false);
        }
    };

    Window_Slot.prototype.refresh = function () {
        this.contents.clear();
        for (let reelIndex = 0; reelIndex < this._reelsData.length; reelIndex++) {
            const reel = this._reelsData[reelIndex];
            for (let j = 0; j < Math.min(4, reel.length); j++) {
                const symbol = reel[j];
                if (!symbol || !symbol.image) {
                    console.error(`❌ シンボルが未定義 (reelIndex=${reelIndex}, j=${j})`, symbol);
                    continue;
                }
                const bitmap = this._symbolBitmaps[symbol.image];
                if (!bitmap) {
                    console.error(`❌ 画像が読み込めていない: ${symbol.image}`);
                    continue;
                }
                const xPos = reelIndex * this._symbolWidth;
                const yPos = j * this._symbolHeight + this._reelOffsets[reelIndex];
                this.contents.blt(
                    bitmap,
                    0, 0,
                    bitmap.width, bitmap.height,
                    xPos, yPos,
                    this._symbolWidth, this._symbolHeight
                );
            }
        }
    };

    Window_Slot.prototype.adjustHeightForCenterReel = function () {
        let symbolHeight = this._symbolHeight;
        let newHeight = symbolHeight + this.padding * 2;

        const customY = SlotMachine.windowY;
        const centerY = (Graphics.height - newHeight) / 2;

        const newY = (typeof customY === "number" && customY >= 0) ? customY : centerY;

        this.move(this.x, newY, this.width, newHeight);
        this._reelOffsets = this._reelOffsets.map(() => -symbolHeight * 3);
        this.refresh();
    };

    Scene_Map.prototype.showSlot = function () {
        if (!this._slotWindow) {
            this._slotWindow = new Window_Slot();
            this.addWindow(this._slotWindow);
        }
        this._slotWindow.setupReels();
        this._slotWindow.refresh();
    };

    Scene_Map.prototype.CShowSlot = function () {
        if (!this._slotWindow) {
            this._slotWindow = new Window_Slot();
            this.addWindow(this._slotWindow);
        }
        this._slotWindow.setupReels();
        this._slotWindow.refresh();
        this._slotWindow.adjustHeightForCenterReel();
        this._slotWindow._isCentralView = true;
    };

    Scene_Map.prototype.startSlot = function () {
        if (this._slotWindow) {
            this._slotWindow.startSpin();
        }
    };

    Scene_Map.prototype.stopSlot = function () {
        if (this._slotWindow) {
            this._slotWindow.requestStop();
        }
    };

Scene_Map.prototype.slowStopSlot = function () {
    if (this._slotWindow) {
        this._slotWindow.requestSlowStop();
    }
};

    Scene_Map.prototype.endSlot = function () {
        if (this._slotWindow) {
            this._slotWindow.close();
            this.removeChild(this._slotWindow);
            this._slotWindow = null;
        }
    };

    function parseVariableString(str) {
        return str.replace(/\\v\[(\d+)\]/g, (_, varId) => $gameVariables.value(Number(varId)));
    }

    Window_Slot.prototype.startSpin = function () {
        this._spinning = true;
        for (let i = 0; i < this._reelsData.length; i++) {
            this._reelStates[i] = "spin";
            setTimeout(() => {
                this.updateSpin(i);
            }, i * 200);
        }

        if (SlotMachine.stopType === "自動停止") {
            setTimeout(() => {
                this.requestAllStop();
            }, 3000);
        }
    };

    Window_Slot.prototype.updateSpin = function (reelIndex) {
        if (!this._reelStates[reelIndex]) return;

        this._reelOffsets[reelIndex] += 40;
        if (this._reelOffsets[reelIndex] >= this._symbolHeight) {
            this._reelOffsets[reelIndex] = 0;
            this.shiftReelSymbols(reelIndex);
        }
        this.refresh();
        if (this._reelStates[reelIndex] === "stopping") {
            if (this._reelOffsets[reelIndex] === 0) {
                this._reelStates[reelIndex] = false;
                if (this._reelStates.every(state => state === false)) {
                    this._spinning = false;
                }
                return;
            }
        }

        setTimeout(() => {
            this.updateSpin(reelIndex);
        }, 30);
    };

    Window_Slot.prototype.shiftReelSymbols = function (reelIndex) {
        const reel = this._reelsData[reelIndex];
        const first = reel.shift();
        reel.push(first);
    };

    Scene_Map.prototype.checkSlotResult = function (mode) {
        if (this._slotWindow) {
            this._slotWindow.checkWinningCombination(mode);
        }
    };

    Window_Slot.prototype.requestStop = function () {
        if (SlotMachine.stopType === "手動停止") {
            const nextStoppingReel = this._reelStates.findIndex(state => state === "spin");
            if (nextStoppingReel !== -1) {
                this._reelStates[nextStoppingReel] = "stopping";
                this.applySlidingEffect(nextStoppingReel);
            }
        }
    };

    Window_Slot.prototype.applySlidingEffect = function (reelIndex) {
        let chosenPayline = this.choosePaylineByWeight();
        if (!chosenPayline) return;

        let reelSymbols = this._reelsData[reelIndex];
        let targetSymbol = chosenPayline.symbols[reelIndex];
        let targetIndex = reelSymbols.findIndex(symbol => symbol.name === targetSymbol);

        if (targetIndex !== -1) {
            while (this._reelsData[reelIndex][1].name !== targetSymbol) {
                this.shiftReelSymbols(reelIndex);
            }
        } else {
            console.warn(`⚠️ リール ${reelIndex + 1}: シンボル "${targetSymbol}" が見つかりません`);
        }
    };

    Window_Slot.prototype.requestAllStop = function () {
        for (let i = 0; i < this._reelStates.length; i++) {
            if (this._reelStates[i] === "spin") {
                this._reelStates[i] = "stopping";
            }
        }

        const checkInterval = setInterval(() => {
            if (this._reelStates.every(state => state === false)) {
                clearInterval(checkInterval);
                this.checkWinningCombination();
            }
        }, 100);
    };

    Window_Slot.prototype.choosePaylineByWeight = function () {
        let paylines = SlotMachine.paylines;
        let totalWeight = 0;

        if (this._isCentralView) {
            paylines = paylines.map(payline => {
                const centralWeight = SlotMachine.centralOnlyWeights[payline.name] || 0;
                totalWeight += centralWeight;
                return { ...payline, weight: centralWeight };
            }).filter(payline => payline.weight > 0);
        } else {
            totalWeight = paylines.reduce((sum, payline) => sum + payline.weight, 0);
        }

        if (totalWeight === 0) {
            console.error("❌ 役の総重みが 0 です。設定を確認してください。");
            return null;
        }

        const guaranteedPayline = paylines.find(payline => payline.weight >= 100);
        if (guaranteedPayline) {
            return guaranteedPayline;
        }

        let randomValue = Math.random() * totalWeight;
        let accumulatedWeight = 0;

        for (let payline of paylines) {
            accumulatedWeight += payline.weight;
            if (randomValue <= accumulatedWeight) {
                return payline;
            }
        }
        return null;
    };

    Window_Slot.prototype.checkWinningCombination = function (mode) {
        if (!SlotMachine.paylines.length) return;    
        let results = this.getSymbolPatterns(mode);
    
        let winningHand = "ハズレ";
        for (const payline of SlotMachine.paylines) {
            let match = results.some(pattern => {
                return pattern.every((symbol, index) => {
                    return payline.symbols[index] === "All" || payline.symbols[index] === symbol;
                });
            });

            if (match) {
                winningHand = payline.name;
                break;
            }
        }

        $gameVariables.setValue(SlotMachine.resultVariable, winningHand);
    };

    Window_Slot.prototype.checkWinningCombination = function (mode) {
        if (!SlotMachine.paylines.length) return;    
        let results = this.getSymbolPatterns(mode);

        let winningHand = "ハズレ";
        for (const payline of SlotMachine.paylines) {
            let match = results.some(pattern => {
                return pattern.every((symbol, index) => {
                    return payline.symbols[index] === "All" || payline.symbols[index] === symbol;
                });
            });

            if (match) {
                winningHand = payline.name;
                break;
            }
        }

        $gameVariables.setValue(SlotMachine.resultVariable, winningHand);
    };

    Window_Slot.prototype.getSymbolPatterns = function (mode) {
        const patterns = [];
        const reels = this._reelsData;
        const rowCount = SlotMachine.reelRowCount;
        const targetReels = SlotMachine.currentReels ? this._reelsData.slice(0, SlotMachine.currentReels.length) : this._reelsData;

        if (rowCount < 3) return patterns;

        const centerIndex = (this._isCentralView)
            ? ((rowCount === 3) ? 0 : 2)
            : 1;                         

        if (mode >= 1) {
            const pattern = targetReels.map(reel => (reel[centerIndex] && reel[centerIndex].name) || "Unknown");
            patterns.push(pattern);
        }

        if (mode >= 2) {
            const upperIndex = this._isCentralView ? centerIndex - 1 : 0;
            const pattern = targetReels.map(reel => (reel[upperIndex] && reel[upperIndex].name) || "Unknown");
            patterns.push(pattern);
        }

        if (mode >= 3) {
            const lowerIndex = this._isCentralView ? centerIndex + 1 : 2;
            const pattern = targetReels.map(reel => (reel[lowerIndex] && reel[lowerIndex].name) || "Unknown");
            patterns.push(pattern);
        }

        if (mode >= 4) {
            if (rowCount === 3) {
                const diag1 = [], diag2 = [];
                for (let i = 0; i < targetReels.length; i++) {
                    diag1.push(targetReels[i][i] ? targetReels[i][i].name : "Unknown");
                    diag2.push(targetReels[i][rowCount - 1 - i] ? targetReels[i][rowCount - 1 - i].name : "Unknown");
                }
                patterns.push(diag1, diag2);
            } else if (rowCount === 4) {
                const diag1 = [], diag2 = [];
                for (let i = 0; i < targetReels.length; i++) {
                    diag1.push(targetReels[i][i] ? targetReels[i][i].name : "Unknown");
                    diag2.push(targetReels[i][rowCount - 1 - i] ? targetReels[i][rowCount - 1 - i].name : "Unknown");
                }
                patterns.push(diag1, diag2);
            }
        }

        return patterns;
    };
})();