//=============================================================================
// SpineHitBox.js
//=============================================================================

/*:
 * @plugindesc 境界ボックス名を取得するプラグイン V1.1.0
 * @author NJ
 * 
 * @help
 * このプラグインは指定座標または範囲内にあるSpineの境界ボックスの名前を取得できます。
 * 
 * 【使用例】
 * 1. 指定座標で判定（変数1, 2番の座標を使用）
 *    $gameVariables.setValue(3, $gameScreen.spine(10).hitTestWithName("\V[1]", "\V[2]"));
 * 
 * 2. 画面クリック時のマウス座標で判定
 *    let hitBox = $gameScreen.spine(10).hitTestWithName("mouseX", "mouseY");
 *
 * 3. 範囲内で判定（X:10～50, Y:20～-30 の範囲）
 *    let hitBox = $gameScreen.spine(10).hitTestWithRange(10, 50, 20, -30);
 *
 * 【注意】
 * - PictureSpineプラグインが必要です。
 * - プラグイン管理でPictureSpineの「下」に配置してください。
 *
 * 【バージョン履歴】
 * v1.0.0 初回リリース
 * v1.1.0 範囲検索機能追加 (hitTestWithRange)
 */

function convertVariables(value) {
    if (typeof value === "string") {
        let match = value.match(/^\\V\[(\d+)\]$/);
        if (match) {
            let varId = Number(match[1]);
            let varValue = Number($gameVariables.value(varId));
            return isNaN(varValue) ? 0 : varValue;
        }
        if (value.toLowerCase() === "mousex") return TouchInput.x || 0;
        if (value.toLowerCase() === "mousey") return TouchInput.y || 0;
    }
    return isNaN(Number(value)) ? 0 : Number(value);
}

(function() {
    'use strict';

    const _Game_Screen_spine = Game_Screen.prototype.spine;
    Game_Screen.prototype.spine = function(pictureId) {
        const spineObj = _Game_Screen_spine.call(this, pictureId);
        if (spineObj && spineObj instanceof Object) {
            ensureSpineExtension(spineObj);
        }
        return spineObj;
    };

    function ensureSpineExtension(spineObj) {
        if (spineObj.hitTestWithName && spineObj.hitTestWithRange) return;

        spineObj.hitTestWithName = function(x, y, screen = true) {
            x = convertVariables(x);
            y = convertVariables(y);
            screen = (typeof screen === "string") ? screen === "true" : screen;

            if (isNaN(x) || isNaN(y)) return "";

            let hitBoxes = this.hitTest(x, y, screen);
            if (!hitBoxes || hitBoxes.length === 0) return "";

            hitBoxes.sort((a, b) => (b.zIndex || 0) - (a.zIndex || 0));

            return hitBoxes[0].attachmentName || "";
        };

spineObj.hitTestWithRange = function(x1, x2, y1, y2, screen = true) {
    x1 = convertVariables(x1);
    x2 = convertVariables(x2);
    y1 = convertVariables(y1);
    y2 = convertVariables(y2);
    screen = (typeof screen === "string") ? screen === "true" : screen;

    console.log(`[DEBUG] hitTestWithRange 実行: 範囲 (${x1}, ${x2}, ${y1}, ${y2})`);

    // 範囲の大きさに応じて step を動的に調整
    const width = Math.abs(x2 - x1);
    const height = Math.abs(y2 - y1);
    const step = Math.max(5, Math.min(width, height) / 10); // 範囲が大きいほど step を大きくする

    let hitBoxes = this.hitTestRect(x1, x2, y1, y2, screen, step);
    console.log(`[DEBUG] hitTestRect の結果: ${hitBoxes.length} 個のヒットボックス`);

    if (!hitBoxes || hitBoxes.length === 0) {
        console.warn("[DEBUG] 範囲内のヒットボックスなし");
        return "";
    }

    hitBoxes.sort((a, b) => (b.zIndex || 0) - (a.zIndex || 0));

    return hitBoxes[0].attachmentName || "";
};

spineObj.hitTestRect = function(minX, maxX, minY, maxY, screen = true, step = 5) {
    let result = [];
    let centerX = (minX + maxX) / 2;
    let centerY = (minY + maxY) / 2;

    // **中心点でヒットテスト**
    let centerHit = this.hitTest(centerX, centerY, screen) || [];
    if (centerHit.length > 0) {
        return centerHit;
    }

    // **周辺点でヒットテスト**
    const offsets = [
        { x: 0, y: 0 },          // 中心
        { x: -step, y: 0 },      // 左
        { x: step, y: 0 },       // 右
        { x: 0, y: -step },      // 上
        { x: 0, y: step },       // 下
        { x: -step, y: -step },  // 左上
        { x: step, y: -step },   // 右上
        { x: -step, y: step },   // 左下
        { x: step, y: step }     // 右下
    ];

    for (let offset of offsets) {
        let x = centerX + offset.x;
        let y = centerY + offset.y;
        let hit = this.hitTest(x, y, screen) || [];
        if (hit.length > 0) {
            result = result.concat(hit);
        }
    }

    // **周辺点でヒットボックスが見つからなかった場合、範囲全体をスキャン**
    if (result.length === 0) {
        for (let x = minX; x <= maxX; x += step) {
            for (let y = minY; y <= maxY; y += step) {
                let hit = this.hitTest(x, y, screen) || [];
                if (hit.length > 0) {
                    result = result.concat(hit);
                }
            }
        }
    }

    return result;
};
    }

})();