//=============================================================================
// SpineHitBoxChecker.js
//=============================================================================

/*:
 * @plugindesc SpineHitBoxの強化版 V1.0.1
 * @author NJ
 * 
 * @param conditionGroups
 * @text 条件グループ
 * @desc シーンごとに異なる条件を設定
 * @type struct<ConditionGroup>[]
 * @default []
 *
 * @param currentHitBoxVariableId
 * @text 現在の境界ボックス名変数ID
 * @desc 取得した境界ボックス名を代入する変数ID
 * @type variable
 * @default 0
 * 
 * @help
 * 注意:
 * - PictureSpineプラグインが必要です。
 * - プラグイン管理でPictureSpineの「下」に配置してください。
 *
 * CommonEvent:1  - コモンイベント1を起動。
 *
 * バージョン
 * v1.0.0 初回
 * v1.0.1 一度だけ hitTest するように修正し、パフォーマンスを改善
 *
 * 利用規約：
 *  プラグイン作者に無断で使用、改変、再配布は不可です。
 */

/*~struct~ConditionGroup:
 * @param groupName
 * @text グループ名
 * @desc 条件グループの名前
 * @type string
 * @default scene
 * 
 * @param startSwitchId
 * @text 起動スイッチID
 * @desc このスイッチがONのとき、条件が有効化される
 * @type switch
 * @default 0
 * 
 * @param startVariableId
 * @text 起動変数ID
 * @desc この変数の値が一致する場合に、条件が有効化される
 * @type variable
 * @default 0
 * 
 * @param startVariableValue
 * @text 起動変数値
 * @desc 起動変数の期待値。値が一致した場合に条件が有効化される
 * @type number
 * @default 0
 * 
 * @param stopSwitchIds
 * @text 停止スイッチリスト
 * @desc いずれかのスイッチがONの場合、この条件グループは無効化される
 * @type switch[]
 * @default []
 *
 * @param conditions
 * @text 境界ボックス条件リスト
 * @desc 境界ボックス判定の条件を設定
 * @type struct<Condition>[]
 * @default []
 */

/*~struct~Condition:
 * @param memo
 * @text メモ
 * @desc 何を対象にするかを記述（任意）
 * @type note
 * @default
 *
 * @param pictureId
 * @text 判定するSpineのピクチャID
 * @desc 境界ボックスを取得するSpineピクチャのID
 * @type number
 * @default 1
 *
 * @param targetHitBoxName
 * @text 対象の境界ボックス名(カンマ区切り可)
 * @desc 例: boxA,boxB
 *       いずれか一致すれば実行。何もないところを指定する場合は"noname"
 * @type string
 * @default
 *
 * @param resultVariableId
 * @text 代入変数ID
 * @desc 取得した境界ボックス名を代入する変数ID
 * @type variable
 * @default 1
 *
 * @param resultValue
 * @text 変数代入値
 * @desc 境界ボックスを認識した際、代入変数に挿入する値
 * @type number
 * @default 1
 *
 * @param stopSwitchIds
 * @text 禁止スイッチリスト
 * @desc いずれかのスイッチがONのとき、この条件の判定を無効化する
 * @type switch[]
 * @default []
 *
 * @param executeScript
 * @text 実行スクリプト
 * @desc 対象の境界ボックスが一致した際に一度だけ実行。
 * @type note
 * @default
 *
 * @param triggerType
 * @text トリガータイプ
 * @desc トリガー実行スクリプトを起動するためのキー設定
 * @type select
 * @option none
 * @option leftClick
 * @option TriggerleftClick
 * @option rightClick
 * @option keyboard
 * @option leftClickReleased
 * @default none
 *
 * @param triggerKey
 * @text トリガーキーボード
 * @desc トリガータイプがkeyboardのときに押されるキー
 *       ※対応するキーはRPGツクールMVの標準。
 * @type string
 * @default
 *
 * @param triggerExecuteScript
 * @text トリガー実行スクリプト
 * @desc 対象ボックスを認識中、トリガーが入力されたとき一度だけ実行
 * @type note
 * @default
 */

(() => {
    'use strict';

    const pluginName = 'SpineHitBoxChecker';
    const parameters = PluginManager.parameters(pluginName);

    const conditionGroups = JSON.parse(parameters['conditionGroups'] || '[]').map(groupStr => {
        const g = JSON.parse(groupStr);
        g.startSwitchId = Number(g.startSwitchId || 0);
        g.startVariableId = Number(g.startVariableId || 0);
        g.startVariableValue = Number(g.startVariableValue || 0);
        g.stopSwitchIds = JSON.parse(g.stopSwitchIds || '[]').map(id => Number(id)); // 停止スイッチリスト
        g.conditions = JSON.parse(g.conditions || '[]').map(cStr => JSON.parse(cStr));
        return g;
    });

    const currentHitBoxVariableId = Number(parameters['currentHitBoxVariableId'] || 0);

    function isConditionGroupEnabled(group) {
        const switchOk = (group.startSwitchId === 0) || $gameSwitches.value(group.startSwitchId);
        const varOk = (group.startVariableId === 0) ||
            ($gameVariables.value(group.startVariableId) === group.startVariableValue);
        const stopSwitchOk = !group.stopSwitchIds.some(id => $gameSwitches.value(id));
        return switchOk && varOk && stopSwitchOk;
    }

    function checkTriggerInput(condition) {
        const type = condition.triggerType;
        if (!type || type === 'none') return false;
        switch (type) {
            case 'leftClick': return TouchInput.isPressed();
            case 'TriggerleftClick': return TouchInput.isTriggered();
            case 'rightClick': return TouchInput.isCancelled();
            case 'keyboard': return condition.triggerKey ? Input.isTriggered(condition.triggerKey) : false;
            case 'leftClickReleased': return TouchInput.isReleased();
        }
        return false;
    }

    function runScriptSafely(script) {
        if (script && script.trim()) {
            try {
                let decoded = script;
                while (decoded.startsWith("\"") && decoded.endsWith("\"")) {
                    decoded = JSON.parse(decoded);
                }

                decoded = decoded.replace(/CommonEvent\s*:\s*(\d+)/g, 'CommonEvent($1)');

                const wrapped = new Function(`
                    const CommonEvent = (id) => {
                        if ($dataCommonEvents[id]) {
                            $gameTemp.reserveCommonEvent(id);
                        } else {
                            console.error('[ERROR] 無効なコモンイベントID: ' + id);
                        }
                    };
                    ${decoded}
                `);

                wrapped();

            } catch (e) {
                console.error("スクリプト実行エラー:", e);
            }
        }
    }

    function setVariableIfChanged(variableId, value) {
        if (variableId > 0 && $gameVariables.value(variableId) !== value) {
            $gameVariables.setValue(variableId, value);
        }
    }

    let mousePosX = 0;
    let mousePosY = 0;
    document.addEventListener("mousemove", event => {
        if (!Graphics._canvas) return;
        const rect = Graphics._canvas.getBoundingClientRect();
        mousePosX = (event.clientX - rect.left) * (Graphics.width / rect.width);
        mousePosY = (event.clientY - rect.top) * (Graphics.height / rect.height);
    });

    function isConditionEnabled(condition) {
        let stopSwitchIds = condition.stopSwitchIds || "[]";
        if (typeof stopSwitchIds === "string") {
            try {
                stopSwitchIds = JSON.parse(stopSwitchIds).map(id => Number(id));
            } catch (e) {
                console.error("[ERROR] stopSwitchIds の解析に失敗しました:", e);
                stopSwitchIds = [];
            }
        }
        return !stopSwitchIds.some(id => $gameSwitches.value(id));
    }

    const triggerExecuted = {};

    let prevMouseX = 0;
    let prevMouseY = 0;

    const hitBoxMemory = {};

    function updateSpineHitBoxCache(pictureId) {
        const picture = $gameScreen.picture(pictureId);
        if (!picture) return;

        const spine = $gameScreen.spine(pictureId);
        if (!spine || !spine._skeleton) return;

        const mouseX = mousePosX;
        const mouseY = mousePosY;
        const hitBoxes = spine.hitTest(mouseX, mouseY, true) || [];
        const currentName = (hitBoxes.length > 0) ? (hitBoxes[0].slotName || "noname") : "noname";

        const mem = hitBoxMemory[pictureId] || { last: "", count: 0, stable: "noname" };

        if (currentName === mem.last) {
            mem.count++;
        } else {
            mem.last = currentName;
            mem.count = 1;
        }

        if (mem.count >= 2 && mem.stable !== currentName) {
            mem.stable = currentName;
        }

        hitBoxMemory[pictureId] = mem;

        const frontHitBox = mem.stable;
        const oldHitBox = spine._lastHitBoxName || "noname";
        const changed = (frontHitBox !== oldHitBox);
        spine._lastHitBoxName = frontHitBox;

        if (typeof spineHitBoxCache !== "object") window.spineHitBoxCache = {};
        spineHitBoxCache[pictureId] = {
            frontHitBox: frontHitBox,
            changed: changed
        };

        if ($gameVariables.value(currentHitBoxVariableId) !== frontHitBox) {
            setVariableIfChanged(currentHitBoxVariableId, frontHitBox);
        }
    }

    function processSpineConditions(group) {
        if (!group) return;

        if (typeof window.spineHitBoxCache !== "object") {
            window.spineHitBoxCache = {};
        }
        const spineHitBoxCache = window.spineHitBoxCache;

        const pictureIdSet = new Set();
        for (const condition of group.conditions) {
            if (!isConditionEnabled(condition)) continue;
            const pid = Number(condition.pictureId || 1);
            pictureIdSet.add(pid);
        }

        pictureIdSet.forEach(pid => {
            updateSpineHitBoxCache(pid);
        });

        let index = 0;
        for (const condition of group.conditions) {
            index++;
            if (!isConditionEnabled(condition)) {
                continue;
            }

            const pictureId = Number(condition.pictureId || 1);
            const targetHitBoxNames = condition.targetHitBoxName
                ? condition.targetHitBoxName.split(",").map(n => n.trim()).filter(n => n)
                : [];
            const resultVariableId = Number(condition.resultVariableId || 0);
            const resultValue = Number(condition.resultValue || 1);
            const executeScript = condition.executeScript || "";
            const triggerExecuteScript = condition.triggerExecuteScript || "";

            const cacheData = spineHitBoxCache[pictureId] || { frontHitBox: "noname", changed: false };
            const frontHitBox = cacheData.frontHitBox;
            const changed = cacheData.changed;

            if (!triggerExecuted[pictureId]) {
                triggerExecuted[pictureId] = {};
            }
            if (triggerExecuted[pictureId][index] == null) {
                triggerExecuted[pictureId][index] = false;
            }

            if (changed && targetHitBoxNames.includes(frontHitBox)) {
                setVariableIfChanged(resultVariableId, resultValue);
                runScriptSafely(executeScript);
            }

            if (targetHitBoxNames.includes(frontHitBox)) {
                if (checkTriggerInput(condition)) {
                    runScriptSafely(triggerExecuteScript);
                }
            } else {
                triggerExecuted[pictureId][index] = false;
            }
        }
    }

    const _Scene_Map_update = Scene_Map.prototype.update;
    Scene_Map.prototype.update = function () {
        _Scene_Map_update.apply(this, arguments);
        for (const group of conditionGroups) {
            if (isConditionGroupEnabled(group)) {
                processSpineConditions(group);
            }
        }
    };

    window.SetHitBoxNameToVariable = function(pictureId, variableId) {
        const spine = $gameScreen.spine(pictureId);
        if (!spine || !spine._skeleton) return;

        const hitBoxes = spine.hitTest(mousePosX, mousePosY, true) || [];
        const hitBoxName = (hitBoxes.length > 0) ? (hitBoxes[0].slotName || "noname") : "noname";

        if (variableId > 0) {
            $gameVariables.setValue(variableId, hitBoxName);
        }
    };

    window.resetSpineHitBoxCache = function(pictureId) {
        if (window.spineHitBoxCache && window.spineHitBoxCache[pictureId]) {
            window.spineHitBoxCache[pictureId].frontHitBox = "noname";
            window.spineHitBoxCache[pictureId].changed = false;
        }

        if (hitBoxMemory && hitBoxMemory[pictureId]) {
            hitBoxMemory[pictureId].last   = "noname";
            hitBoxMemory[pictureId].count  = 0;
            hitBoxMemory[pictureId].stable = "noname";
        }

        const spine = $gameScreen.spine(pictureId);
        if (spine) {
            spine._lastHitBoxName = "noname";
        }

        if (currentHitBoxVariableId > 0) {
            $gameVariables.setValue(currentHitBoxVariableId, "noname");
        }
    };

    window.resetAllSpineHitBoxes = function() {
        if (window.spineHitBoxCache) {
            for (const pid in window.spineHitBoxCache) {
                window.spineHitBoxCache[pid].frontHitBox = "noname";
                window.spineHitBoxCache[pid].changed = false;
            }
        }
        if (hitBoxMemory) {
            for (const pid in hitBoxMemory) {
                hitBoxMemory[pid].last   = "noname";
                hitBoxMemory[pid].count  = 0;
                hitBoxMemory[pid].stable = "noname";
            }
        }
        if ($gameScreen._pictures) {
            for (const pic of $gameScreen._pictures) {
                if (pic && pic._spine) {
                    pic._spine._lastHitBoxName = "noname";
                }
            }
        }
        if (currentHitBoxVariableId > 0) {
            $gameVariables.setValue(currentHitBoxVariableId, "noname");
        }
    };
})();
