//=============================================================================
// WaitPlus.js
//=============================================================================

/*:
 * @plugindesc 入力または経過時間による待機 + 無操作時にスクリプト実行 V1.0.0（統合版）@author NJ
 *
 * @param IdleTime
 * @type number
 * @desc 待機とみなすまでの時間（秒）
 * @default 60
 *
 * @param OnIdleScript
 * @type note
 * @desc 待機状態時に実行されるJavaScriptスクリプト
 * @default ""
 *
 * @param ResetAfterExecute
 * @type boolean
 * @desc スクリプト実行後、タイマーをリセットするか
 * @default true
 *
 * @param ResetDelay
 * @type number
 * @desc スクリプト実行後にリセットするまでの待機時間（秒）
 * @default 0
 *
 * @param DetectMouseMove
 * @type boolean
 * @default true
 *
 * @param DetectMouseClick
 * @type boolean
 * @default true
 *
 * @param DetectArrowKey
 * @type boolean
 * @default true
 *
 * @param DetectOtherKey
 * @type boolean
 * @default true
 *
 * @param EnableSwitchId
 * @type switch
 * @default 0
 *
 * @param DisableSwitchIds
 * @type string
 * @default
 *
 * @param ConditionVariableId
 * @type variable
 * @default 0
 *
 * @param ConditionCompareType
 * @type select
 * @option equal
 * @option greaterEqual
 * @option lessEqual
 * @default equal
 *
 * @param ConditionValue
 * @type number
 * @default 0
 *
 * @help
 * 【統合機能】
 * - WaitForKeyOrTime、WaitForKeysOrTime、WaitForAllKeys
 * - WaitForOkOrClickOrTime（クリック or OKキー or 時間）
 * - 一定時間無操作時に任意スクリプト実行（条件付き）
 *
 * プラグインコマンド:
 *   WaitForKeyOrTime keyName waitFrames
 *    keyName: 監視するキー名 (例: ok, cancel, left, right, up, down, shift, control, tab, pageup, pagedown)
 *    waitFrames: 待機するフレーム数 (1/60秒単位)。0なら無限待機（キー入力待ちのみ）
 *
 *   WaitForOkOrClickOrTime waitFrames
 *    waitFrames: 待機するフレーム数 (1/60秒単位)。0なら無限待機（入力待ちのみ）
 *
 * 例:
 *   WaitForKeyOrTime ok 120            - OKキーが押されるか2秒経過で進行
 *   WaitForKeysOrTime ok,cancel 120    - OKキーかキャンセルキーが押されるか2秒経過で進行 
 *   WaitForAllKeys ok,shift 120        - OKキーとシフトキーを同時押しされるか2秒経過で進行
 *   WaitForOkOrClickOrTime 180         - OKキーかクリック、3秒経過で進行
 *
 * スクリプトコマンドからも使用可能:
 *   this.waitForKeyOrTime('ok', 120);
 *   this.waitForKeysOrTime(['ok', 'cancel'], 120);
 *   this.waitForAllKeys(['ok', 'shift'], 120);
 *   this.waitForOkOrClickOrTime(180);
 *
 * 使用できるキーはツクールに標準で搭載されているものだけです。他のキーを使用する場合は他プラグインなどで拡張をしてください。
 *
 * バージョン
 * v1.0.0 初回
 *
 * 利用規約：
 *  プラグイン作者に無断で使用、改変、再配布は不可です。
 */

(function() {
    const params = PluginManager.parameters('WaitPlus');

    const idleTime = Number(params['IdleTime'] || 60);
    const onIdleScript = String(params['OnIdleScript'] || '');
    const resetAfterExecute = params['ResetAfterExecute'] === 'true';
    const resetDelay = Number(params['ResetDelay'] || 0);

    const detectMouseMove = params['DetectMouseMove'] === 'true';
    const detectMouseClick = params['DetectMouseClick'] === 'true';
    const detectArrowKey = params['DetectArrowKey'] === 'true';
    const detectOtherKey = params['DetectOtherKey'] === 'true';

    const enableSwitchId = Number(params['EnableSwitchId'] || 0);
    const disableSwitchIds = (params['DisableSwitchIds'] || '').split(',').map(s => Number(s.trim())).filter(id => id > 0);

    const condVarId = Number(params['ConditionVariableId'] || 0);
    const condCompare = params['ConditionCompareType'] || 'equal';
    const condValue = Number(params['ConditionValue'] || 0);

    let lastActivityTime = Date.now();
    let lastMouseX = null;
    let lastMouseY = null;
    let idleExecuted = false;

    const convertEscapeCharacters = function(text) {
        return (text + '').replace(/\\V\[(\d+)\]/gi, (_, vId) => $gameVariables.value(Number(vId))).replace(/\\/g, '\x1b');
    };

    const _Game_Interpreter_pluginCommand = Game_Interpreter.prototype.pluginCommand;
    Game_Interpreter.prototype.pluginCommand = function(command, args) {
        _Game_Interpreter_pluginCommand.call(this, command, args);
        if (Array.isArray(args)) {
            args = args.map(arg => convertEscapeCharacters(arg));
        } else {
            args = [];
        }

        switch (command) {
            case 'WaitForKeyOrTime':
                this.waitForKeysOrTime([args[0]], Number(args[1] || 0));
                break;
            case 'WaitForKeysOrTime':
                this.waitForKeysOrTime(args[0].split(',').map(s => s.trim()), Number(args[1] || 0));
                break;
            case 'WaitForAllKeys':
                this.waitForAllKeys(args[0].split(',').map(s => s.trim()), Number(args[1] || 0));
                break;
            case 'WaitForOkOrClickOrTime':
                this.waitForOkOrClickOrTime(Number(args[0] || 0));
                break;
        }
    };

    Game_Interpreter.prototype.waitForKeysOrTime = function(keys, waitFrames) {
        this._waitKeys = keys;
        this._waitCount = waitFrames;
        this._waitUnlimited = waitFrames === 0;
        this._waitForKeysOrTime = true;
    };

    Game_Interpreter.prototype.waitForAllKeys = function(keys, waitFrames) {
        this._waitAllKeys = keys;
        this._waitCount = waitFrames;
        this._waitUnlimitedAll = waitFrames === 0;
        this._waitForAllKeys = true;
    };

    Game_Interpreter.prototype.waitForOkOrClickOrTime = function(waitFrames) {
        this._waitCount = waitFrames;
        this._waitForOkOrClickOrTime = true;
    };

    const _Game_Interpreter_updateWait = Game_Interpreter.prototype.updateWait;
    Game_Interpreter.prototype.updateWait = function() {
        if (this._waitForKeysOrTime) {
            if (!this._waitUnlimited && this._waitCount > 0) this._waitCount--;
            if (this._waitKeys.some(k => Input.isTriggered(k)) || (!this._waitUnlimited && this._waitCount === 0)) {
                this._waitForKeysOrTime = false;
                return false;
            }
            return true;
        } else if (this._waitForAllKeys) {
            if (!this._waitUnlimitedAll && this._waitCount > 0) this._waitCount--;
            if (this._waitAllKeys.every(k => Input.isPressed(k)) || (!this._waitUnlimitedAll && this._waitCount === 0)) {
                this._waitForAllKeys = false;
                return false;
            }
            return true;
        } else if (this._waitForOkOrClickOrTime) {
            if (this._waitCount > 0) {
                this._waitCount--;
            }

            const triggered = Input.isTriggered('ok') || TouchInput.isTriggered();
            const timeout = this._waitCount <= 0;

            if (triggered || timeout) {
                this._waitForOkOrClickOrTime = false;
                return false;
            }

            return true;
        }
        return _Game_Interpreter_updateWait.call(this);
    };

    function isIdleEnabled() {
        if (!$gameSwitches || !$gameVariables) return false;
        if (enableSwitchId > 0 && !$gameSwitches.value(enableSwitchId)) return false;
        if (disableSwitchIds.some(id => $gameSwitches.value(id))) return false;
        if (condVarId > 0) {
            const val = $gameVariables.value(condVarId);
            switch (condCompare) {
                case 'equal':        if (val !== condValue) return false; break;
                case 'greaterEqual': if (val < condValue) return false; break;
                case 'lessEqual':    if (val > condValue) return false; break;
            }
        }
        return true;
    }

    function resetIdleTimer() {
        lastActivityTime = Date.now();
        idleExecuted = false;
    }

    if (detectMouseMove) {
        const _TouchInput_onMouseMove = TouchInput._onMouseMove;
        TouchInput._onMouseMove = function(event) {
            if (lastMouseX !== event.pageX || lastMouseY !== event.pageY) {
                resetIdleTimer();
                lastMouseX = event.pageX;
                lastMouseY = event.pageY;
            }
            if (_TouchInput_onMouseMove) _TouchInput_onMouseMove.call(this, event);
        };
    }

    if (detectMouseClick) {
        const _TouchInput_onMouseDown = TouchInput._onMouseDown;
        TouchInput._onMouseDown = function(event) {
            resetIdleTimer();
            _TouchInput_onMouseDown.call(this, event);
        };
    }

    const _Input_onKeyDown = Input._onKeyDown;
    Input._onKeyDown = function(event) {
        const code = event.code;
        const isArrow = ['ArrowUp', 'ArrowDown', 'ArrowLeft', 'ArrowRight'].includes(code);
        if ((detectArrowKey && isArrow) || (detectOtherKey && !isArrow)) {
            resetIdleTimer();
        }
        _Input_onKeyDown.call(this, event);
    };

    const _SceneManager_updateMain = SceneManager.updateMain;
    SceneManager.updateMain = function() {
        _SceneManager_updateMain.call(this);

        if (!isIdleEnabled()) {
            resetIdleTimer();
            return;
        }

        const elapsed = (Date.now() - lastActivityTime) / 1000;
        if (!idleExecuted && elapsed >= idleTime) {
            try {
                const script = JSON.parse(onIdleScript);
                eval(script);
            } catch (e) {
                console.error('[WaitPlus] スクリプト実行エラー:', e);
            }

            idleExecuted = true;
            if (resetAfterExecute) {
                setTimeout(() => resetIdleTimer(), resetDelay * 1000);
            }
        }
    };
})();
