//=============================================================================
// PictureAnimationState.js
//=============================================================================

/*:
 * @target MZ
 * @plugindesc ピクチャにループアニメーション状態を付与するプラグイン v1.1
 * @author ラーメン好き好き亀有
 * @url https://your-url.com
 *
 * @help PictureAnimationState.js
 *
 * ピクチャに繰り返しアニメーションの「状態」を付与できるプラグインです。
 * 動画編集ソフトでループアニメーションを画像に適用するような感覚で使えます。
 *
 * 【主な機能】
 * ・プラグインパラメータで複数のアニメーション状態を定義
 * ・プラグインコマンドでピクチャに状態を付与/解除（選択式）
 * ・拡大縮小、回転、移動、透明度変化などを組み合わせ可能
 * ・イージング関数対応で滑らかなアニメーション
 *
 * 【使い方】
 * 1. プラグインパラメータで「アニメーション状態リスト」を設定
 * 2. プラグインコマンド「状態を付与」でピクチャにアニメーションを適用
 * 3. 解除したい場合は「状態を解除」コマンドを使用
 *
 * ※注意: 新しい状態を追加した場合は、GeneratePictureAnimationPlugin.jsで
 *   プラグインを再生成してください。
 *
 * 【プラグインコマンド】
 * ・状態を付与 - ピクチャにアニメーション状態を適用
 * ・状態を解除 - ピクチャからアニメーション状態を削除
 * ・全ての状態を解除 - すべてのピクチャの状態をクリア
 *
 * @command applyState
 * @text 状態を付与
 * @desc 指定したピクチャにアニメーション状態を付与します
 *
 * @arg pictureId
 * @text ピクチャ番号
 * @desc アニメーションを適用するピクチャの番号
 * @type number
 * @min 1
 * @max 100
 * @default 1
 *
 * @arg stateIndex
 * @text 状態
 * @desc 適用する状態を選択してください
 * @type select
 * @option 脈打つ
 * @value 0
 * @option 縦に伸び縮み
 * @value 1
 * @option ゆっくり回転
 * @value 2
 * @option 左右に揺れる
 * @value 3
 * @option ふわふわ浮遊
 * @value 4
 * @option 点滅
 * @value 5
 * @default 0
 *
 * @command removeState
 * @text 状態を解除
 * @desc 指定したピクチャからアニメーション状態を解除します
 *
 * @arg pictureId
 * @text ピクチャ番号
 * @desc 状態を解除するピクチャの番号
 * @type number
 * @min 1
 * @max 100
 * @default 1
 *
 * @command clearAllStates
 * @text 全ての状態を解除
 * @desc すべてのピクチャから状態を解除します
 *
 * @param animationStates
 * @text アニメーション状態リスト
 * @desc 利用可能なアニメーション状態の定義リスト
 * @type struct<AnimationState>[]
 * @default ["{\"name\":\"脈打つ\",\"duration\":40,\"loop\":\"pingpong\",\"easing\":\"easeInOut\",\"scaleX\":10,\"scaleY\":10,\"rotation\":0,\"offsetX\":0,\"offsetY\":0,\"opacity\":0}","{\"name\":\"縦に伸び縮み\",\"duration\":30,\"loop\":\"pingpong\",\"easing\":\"easeOut\",\"scaleX\":0,\"scaleY\":15,\"rotation\":0,\"offsetX\":0,\"offsetY\":0,\"opacity\":0}","{\"name\":\"ゆっくり回転\",\"duration\":180,\"loop\":\"repeat\",\"easing\":\"linear\",\"scaleX\":0,\"scaleY\":0,\"rotation\":360,\"offsetX\":0,\"offsetY\":0,\"opacity\":0}","{\"name\":\"左右に揺れる\",\"duration\":60,\"loop\":\"pingpong\",\"easing\":\"easeInOut\",\"scaleX\":0,\"scaleY\":0,\"rotation\":15,\"offsetX\":0,\"offsetY\":0,\"opacity\":0}","{\"name\":\"ふわふわ浮遊\",\"duration\":90,\"loop\":\"pingpong\",\"easing\":\"easeInOut\",\"scaleX\":0,\"scaleY\":0,\"rotation\":0,\"offsetX\":0,\"offsetY\":-20,\"opacity\":0}","{\"name\":\"点滅\",\"duration\":30,\"loop\":\"pingpong\",\"easing\":\"linear\",\"scaleX\":0,\"scaleY\":0,\"rotation\":0,\"offsetX\":0,\"offsetY\":0,\"opacity\":-200}"]
 */

/*~struct~AnimationState:
 * @param name
 * @text 状態名
 * @desc この状態を識別する名前
 * @type string
 * @default 新しい状態
 *
 * @param duration
 * @text アニメーション時間
 * @desc 1サイクルの時間（フレーム数）
 * @type number
 * @min 1
 * @default 60
 *
 * @param loop
 * @text ループ方式
 * @desc アニメーションのループ方式
 * @type select
 * @option 繰り返し（0→100→0→100...）
 * @value repeat
 * @option 往復（0→100→0...）
 * @value pingpong
 * @default repeat
 *
 * @param easing
 * @text イージング
 * @desc アニメーションの緩急
 * @type select
 * @option リニア（等速）
 * @value linear
 * @option イーズイン（加速）
 * @value easeIn
 * @option イーズアウト（減速）
 * @value easeOut
 * @option イーズインアウト（加速→減速）
 * @value easeInOut
 * @default easeInOut
 *
 * @param scaleX
 * @text X方向拡大率変化
 * @desc X方向の拡大率変化（%）。0で変化なし
 * @type number
 * @min -100
 * @max 100
 * @default 0
 *
 * @param scaleY
 * @text Y方向拡大率変化
 * @desc Y方向の拡大率変化（%）。0で変化なし
 * @type number
 * @min -100
 * @max 100
 * @default 10
 *
 * @param rotation
 * @text 回転角度変化
 * @desc 回転角度変化（度）。0で変化なし
 * @type number
 * @min -360
 * @max 360
 * @default 0
 *
 * @param offsetX
 * @text X座標移動量
 * @desc X座標の移動量（ピクセル）。0で変化なし
 * @type number
 * @min -999
 * @max 999
 * @default 0
 *
 * @param offsetY
 * @text Y座標移動量
 * @desc Y座標の移動量（ピクセル）。0で変化なし
 * @type number
 * @min -999
 * @max 999
 * @default 0
 *
 * @param opacity
 * @text 不透明度変化
 * @desc 不透明度の変化量（0-255）。0で変化なし
 * @type number
 * @min -255
 * @max 255
 * @default 0
 */

(() => {
    'use strict';

    const pluginName = 'PictureAnimationState';
    const parameters = PluginManager.parameters(pluginName);
    
    // パラメータをパース
    const animationStates = JSON.parse(parameters['animationStates'] || '[]').map(state => {
        const parsed = JSON.parse(state);
        return {
            name: parsed.name || '新しい状態',
            duration: Number(parsed.duration) || 60,
            loop: parsed.loop || 'repeat',
            easing: parsed.easing || 'easeInOut',
            scaleX: Number(parsed.scaleX) || 0,
            scaleY: Number(parsed.scaleY) || 0,
            rotation: Number(parsed.rotation) || 0,
            offsetX: Number(parsed.offsetX) || 0,
            offsetY: Number(parsed.offsetY) || 0,
            opacity: Number(parsed.opacity) || 0
        };
    });

    // プラグインコマンド登録
    PluginManager.registerCommand(pluginName, 'applyState', args => {
        const pictureId = Number(args.pictureId);
        const stateIndex = Number(args.stateIndex);
        $gameScreen.applyPictureAnimationStateByIndex(pictureId, stateIndex);
    });

    PluginManager.registerCommand(pluginName, 'removeState', args => {
        const pictureId = Number(args.pictureId);
        $gameScreen.removePictureAnimationState(pictureId);
    });

    PluginManager.registerCommand(pluginName, 'clearAllStates', args => {
        $gameScreen.clearAllPictureAnimationStates();
    });

    //-----------------------------------------------------------------------------
    // Game_Screen
    //-----------------------------------------------------------------------------

    const _Game_Screen_initialize = Game_Screen.prototype.initialize;
    Game_Screen.prototype.initialize = function() {
        _Game_Screen_initialize.call(this);
        this._pictureAnimationStates = {};
    };

    const _Game_Screen_clear = Game_Screen.prototype.clear;
    Game_Screen.prototype.clear = function() {
        _Game_Screen_clear.call(this);
        this._pictureAnimationStates = {};
    };

    Game_Screen.prototype.applyPictureAnimationState = function(pictureId, stateName) {
        const stateData = animationStates.find(s => s.name === stateName);
        if (!stateData) {
            console.warn(`Animation state not found: ${stateName}`);
            return;
        }

        const picture = this.picture(pictureId);
        if (!picture) {
            console.warn(`Picture not found: ${pictureId}`);
            return;
        }

        this._pictureAnimationStates[pictureId] = {
            state: stateData,
            frame: 0,
            baseScaleX: picture._scaleX,
            baseScaleY: picture._scaleY,
            baseRotation: picture._angle,
            baseX: picture._x,
            baseY: picture._y,
            baseOpacity: picture._opacity
        };
    };

    Game_Screen.prototype.applyPictureAnimationStateByIndex = function(pictureId, stateIndex) {
        const stateData = animationStates[stateIndex];
        if (!stateData) {
            console.warn(`Animation state index out of range: ${stateIndex}`);
            return;
        }

        const picture = this.picture(pictureId);
        if (!picture) {
            console.warn(`Picture not found: ${pictureId}`);
            return;
        }

        this._pictureAnimationStates[pictureId] = {
            state: stateData,
            frame: 0,
            baseScaleX: picture._scaleX,
            baseScaleY: picture._scaleY,
            baseRotation: picture._angle,
            baseX: picture._x,
            baseY: picture._y,
            baseOpacity: picture._opacity
        };
    };

    Game_Screen.prototype.removePictureAnimationState = function(pictureId) {
        delete this._pictureAnimationStates[pictureId];
    };

    Game_Screen.prototype.clearAllPictureAnimationStates = function() {
        this._pictureAnimationStates = {};
    };

    Game_Screen.prototype.getPictureAnimationState = function(pictureId) {
        return this._pictureAnimationStates[pictureId] || null;
    };

    const _Game_Screen_updatePictures = Game_Screen.prototype.updatePictures;
    Game_Screen.prototype.updatePictures = function() {
        _Game_Screen_updatePictures.call(this);
        this.updatePictureAnimationStates();
    };

    Game_Screen.prototype.updatePictureAnimationStates = function() {
        for (const pictureId in this._pictureAnimationStates) {
            const animState = this._pictureAnimationStates[pictureId];
            const picture = this.picture(Number(pictureId));
            
            if (!picture) {
                delete this._pictureAnimationStates[pictureId];
                continue;
            }

            this.updatePictureAnimationFrame(picture, animState);
        }
    };

    Game_Screen.prototype.updatePictureAnimationFrame = function(picture, animState) {
        const state = animState.state;
        
        // フレームを進める
        animState.frame++;
        if (animState.frame >= state.duration) {
            if (state.loop === 'repeat') {
                animState.frame = 0;
            } else if (state.loop === 'pingpong') {
                // 往復の場合は duration * 2 でリセット
                if (animState.frame >= state.duration * 2) {
                    animState.frame = 0;
                }
            }
        }

        // 進行度を計算（0.0 〜 1.0）
        let progress = animState.frame / state.duration;
        
        // 往復の場合は前半と後半で処理を変える
        if (state.loop === 'pingpong' && animState.frame >= state.duration) {
            progress = 2.0 - progress; // 1.0 → 0.0 に反転
        }

        // イージングを適用
        const easedProgress = this.applyEasing(progress, state.easing);

        // 各パラメータを更新
        if (state.scaleX !== 0) {
            const scaleChange = (state.scaleX / 100) * easedProgress;
            picture._scaleX = animState.baseScaleX + scaleChange * 100;
        }

        if (state.scaleY !== 0) {
            const scaleChange = (state.scaleY / 100) * easedProgress;
            picture._scaleY = animState.baseScaleY + scaleChange * 100;
        }

        if (state.rotation !== 0) {
            const rotationChange = state.rotation * easedProgress;
            picture._angle = animState.baseRotation + rotationChange;
        }

        if (state.offsetX !== 0) {
            const xChange = state.offsetX * easedProgress;
            picture._x = animState.baseX + xChange;
        }

        if (state.offsetY !== 0) {
            const yChange = state.offsetY * easedProgress;
            picture._y = animState.baseY + yChange;
        }

        if (state.opacity !== 0) {
            const opacityChange = state.opacity * easedProgress;
            picture._opacity = Math.max(0, Math.min(255, animState.baseOpacity + opacityChange));
        }
    };

    Game_Screen.prototype.applyEasing = function(t, easingType) {
        switch (easingType) {
            case 'linear':
                return t;
            case 'easeIn':
                return t * t;
            case 'easeOut':
                return t * (2 - t);
            case 'easeInOut':
                return t < 0.5 ? 2 * t * t : -1 + (4 - 2 * t) * t;
            default:
                return t;
        }
    };

    //-----------------------------------------------------------------------------
    // Game_Picture
    //-----------------------------------------------------------------------------

    const _Game_Picture_initialize = Game_Picture.prototype.initialize;
    Game_Picture.prototype.initialize = function() {
        _Game_Picture_initialize.call(this);
    };

    // ピクチャが表示される際に基準値を更新
    const _Game_Picture_show = Game_Picture.prototype.show;
    Game_Picture.prototype.show = function(name, origin, x, y, scaleX, scaleY, opacity, blendMode) {
        _Game_Picture_show.call(this, name, origin, x, y, scaleX, scaleY, opacity, blendMode);
        
        // アニメーション状態が既に存在する場合、基準値を更新
        const pictureId = $gameScreen._pictures.indexOf(this) + 1;
        const animState = $gameScreen.getPictureAnimationState(pictureId);
        if (animState) {
            animState.baseScaleX = this._scaleX;
            animState.baseScaleY = this._scaleY;
            animState.baseRotation = this._angle;
            animState.baseX = this._x;
            animState.baseY = this._y;
            animState.baseOpacity = this._opacity;
        }
    };

})();
