//=============================================================================
// PictureAnimationState.js
//=============================================================================

/*:
 * @target MZ
 * @plugindesc ピクチャにループアニメーション状態を付与するプラグイン
 * @author ラーメン好き好き亀有
 * @url https://your-url.com
 *
 * @help PictureAnimationState.js
 *
 * ピクチャに繰り返しアニメーションの「状態」を付与できるプラグインです。
 * 動画編集ソフトでループアニメーションを画像に適用するような感覚で使えます。
 *
 * 【主な機能】
 * ・プラグインパラメータで複数のアニメーション状態を定義
 * ・プラグインコマンドでピクチャに状態を付与/解除
 * ・拡大縮小、回転、移動、透明度変化などを組み合わせ可能
 * ・イージング関数対応で滑らかなアニメーション
 * ・効果音を複数設定して抽選で再生可能
 * ・効果音の再生タイミングを自由に設定可能
 * ・往復ループの折り返し時にも効果音を再生可能
 *
 * 【使い方】
 * 1. プラグインパラメータで「アニメーション状態リスト」を設定
 * 2. プラグインコマンド「状態を付与」でピクチャにアニメーションを適用
 * 3. 解除したい場合は「状態を解除」コマンドを使用
 *
 * 【効果音の設定】
 * ・「効果音設定リスト」に複数の効果音を登録可能
 * ・複数設定した場合、再生時に抽選される
 * ・再生された効果音は次の抽選では除外され、その次で対象に戻る
 * ・例: 2つ設定すると交互に再生、3つ設定すると3回に1回は必ず別の音
 * ・「効果音のタイミング」でフレーム数を指定（0から開始）
 * ・往復ループの場合、「折り返し時も再生」で折り返しでも効果音再生
 *
 * 【プラグインコマンド】
 * ・状態を付与 - ピクチャにアニメーション状態を適用
 * ・状態を解除 - ピクチャからアニメーション状態を削除
 * ・全ての状態を解除 - すべてのピクチャの状態をクリア
 *
 * @command applyState
 * @text 状態を付与
 * @desc 指定したピクチャにアニメーション状態を付与します
 *
 * @arg pictureId
 * @text ピクチャ番号
 * @desc アニメーションを適用するピクチャの番号
 * @type number
 * @min 1
 * @max 100
 * @default 1
 *
 * @arg stateName
 * @text 状態名
 * @desc 適用する状態の名前（プラグインパラメータで定義したもの）
 * @type string
 * @default
 *
 * @command removeState
 * @text 状態を解除
 * @desc 指定したピクチャからアニメーション状態を解除します
 *
 * @arg pictureId
 * @text ピクチャ番号
 * @desc 状態を解除するピクチャの番号
 * @type number
 * @min 1
 * @max 100
 * @default 1
 *
 * @command clearAllStates
 * @text 全ての状態を解除
 * @desc すべてのピクチャから状態を解除します
 *
 * @param animationStates
 * @text アニメーション状態リスト
 * @desc 利用可能なアニメーション状態の定義リスト
 * @type struct<AnimationState>[]
 * @default ["{\"name\":\"脈打つ\",\"duration\":\"40\",\"loop\":\"pingpong\",\"easing\":\"easeInOut\",\"scaleX\":\"10\",\"scaleY\":\"10\",\"rotation\":\"0\",\"offsetX\":\"0\",\"offsetY\":\"0\",\"opacity\":\"0\"}","{\"name\":\"縦に伸び縮み\",\"duration\":\"30\",\"loop\":\"pingpong\",\"easing\":\"easeOut\",\"scaleX\":\"0\",\"scaleY\":\"15\",\"rotation\":\"0\",\"offsetX\":\"0\",\"offsetY\":\"0\",\"opacity\":\"0\"}","{\"name\":\"横に伸び縮み\",\"duration\":\"30\",\"loop\":\"pingpong\",\"easing\":\"easeOut\",\"scaleX\":\"15\",\"scaleY\":\"0\",\"rotation\":\"0\",\"offsetX\":\"0\",\"offsetY\":\"0\",\"opacity\":\"0\"}","{\"name\":\"ゆっくり回転\",\"duration\":\"180\",\"loop\":\"repeat\",\"easing\":\"linear\",\"scaleX\":\"0\",\"scaleY\":\"0\",\"rotation\":\"360\",\"offsetX\":\"0\",\"offsetY\":\"0\",\"opacity\":\"0\"}","{\"name\":\"左右に揺れる\",\"duration\":\"60\",\"loop\":\"pingpong\",\"easing\":\"easeInOut\",\"scaleX\":\"0\",\"scaleY\":\"0\",\"rotation\":\"15\",\"offsetX\":\"0\",\"offsetY\":\"0\",\"opacity\":\"0\"}","{\"name\":\"小刻みに震える\",\"duration\":\"4\",\"loop\":\"repeat\",\"easing\":\"linear\",\"scaleX\":\"0\",\"scaleY\":\"0\",\"rotation\":\"0\",\"offsetX\":\"2\",\"offsetY\":\"2\",\"opacity\":\"0\"}","{\"name\":\"ふわふわ浮遊\",\"duration\":\"90\",\"loop\":\"pingpong\",\"easing\":\"easeInOut\",\"scaleX\":\"0\",\"scaleY\":\"0\",\"rotation\":\"0\",\"offsetX\":\"0\",\"offsetY\":\"-20\",\"opacity\":\"0\"}","{\"name\":\"上下に大きく揺れる\",\"duration\":\"120\",\"loop\":\"pingpong\",\"easing\":\"easeInOut\",\"scaleX\":\"0\",\"scaleY\":\"0\",\"rotation\":\"0\",\"offsetX\":\"0\",\"offsetY\":\"-40\",\"opacity\":\"0\"}","{\"name\":\"横にスライド\",\"duration\":\"80\",\"loop\":\"pingpong\",\"easing\":\"easeInOut\",\"scaleX\":\"0\",\"scaleY\":\"0\",\"rotation\":\"0\",\"offsetX\":\"30\",\"offsetY\":\"0\",\"opacity\":\"0\"}","{\"name\":\"点滅\",\"duration\":\"30\",\"loop\":\"pingpong\",\"easing\":\"linear\",\"scaleX\":\"0\",\"scaleY\":\"0\",\"rotation\":\"0\",\"offsetX\":\"0\",\"offsetY\":\"0\",\"opacity\":\"-200\"}","{\"name\":\"ゆっくり点滅\",\"duration\":\"90\",\"loop\":\"pingpong\",\"easing\":\"easeInOut\",\"scaleX\":\"0\",\"scaleY\":\"0\",\"rotation\":\"0\",\"offsetX\":\"0\",\"offsetY\":\"0\",\"opacity\":\"-150\"}","{\"name\":\"チカチカ\",\"duration\":\"8\",\"loop\":\"repeat\",\"easing\":\"linear\",\"scaleX\":\"0\",\"scaleY\":\"0\",\"rotation\":\"0\",\"offsetX\":\"0\",\"offsetY\":\"0\",\"opacity\":\"-100\"}","{\"name\":\"回転しながら浮遊\",\"duration\":\"90\",\"loop\":\"pingpong\",\"easing\":\"easeInOut\",\"scaleX\":\"0\",\"scaleY\":\"0\",\"rotation\":\"360\",\"offsetX\":\"0\",\"offsetY\":\"-15\",\"opacity\":\"0\"}","{\"name\":\"脈打ちながら回転\",\"duration\":\"60\",\"loop\":\"repeat\",\"easing\":\"easeInOut\",\"scaleX\":\"20\",\"scaleY\":\"20\",\"rotation\":\"360\",\"offsetX\":\"0\",\"offsetY\":\"0\",\"opacity\":\"0\"}","{\"name\":\"揺れながら光る\",\"duration\":\"70\",\"loop\":\"pingpong\",\"easing\":\"easeInOut\",\"scaleX\":\"5\",\"scaleY\":\"5\",\"rotation\":\"10\",\"offsetX\":\"0\",\"offsetY\":\"0\",\"opacity\":\"-80\"}","{\"name\":\"ポップアップ\",\"duration\":\"20\",\"loop\":\"repeat\",\"easing\":\"easeOut\",\"scaleX\":\"15\",\"scaleY\":\"15\",\"rotation\":\"0\",\"offsetX\":\"0\",\"offsetY\":\"0\",\"opacity\":\"0\"}","{\"name\":\"強調表示\",\"duration\":\"50\",\"loop\":\"pingpong\",\"easing\":\"easeInOut\",\"scaleX\":\"8\",\"scaleY\":\"8\",\"rotation\":\"0\",\"offsetX\":\"0\",\"offsetY\":\"0\",\"opacity\":\"-50\"}","{\"name\":\"待機中\",\"duration\":\"120\",\"loop\":\"pingpong\",\"easing\":\"easeInOut\",\"scaleX\":\"2\",\"scaleY\":\"2\",\"rotation\":\"0\",\"offsetX\":\"0\",\"offsetY\":\"-5\",\"opacity\":\"0\"}","{\"name\":\"エラー表示\",\"duration\":\"6\",\"loop\":\"repeat\",\"easing\":\"linear\",\"scaleX\":\"0\",\"scaleY\":\"0\",\"rotation\":\"5\",\"offsetX\":\"4\",\"offsetY\":\"0\",\"opacity\":\"0\"}","{\"name\":\"成功演出\",\"duration\":\"40\",\"loop\":\"pingpong\",\"easing\":\"easeOut\",\"scaleX\":\"12\",\"scaleY\":\"12\",\"rotation\":\"0\",\"offsetX\":\"0\",\"offsetY\":\"-10\",\"opacity\":\"-30\"}"]
 */

/*~struct~AnimationState:
 * @param name
 * @text 状態名
 * @desc この状態を識別する名前
 * @type string
 * @default 新しい状態
 *
 * @param duration
 * @text アニメーション時間
 * @desc 1サイクルの時間（フレーム数）
 * @type number
 * @min 1
 * @default 60
 *
 * @param loop
 * @text ループ方式
 * @desc アニメーションのループ方式
 * @type select
 * @option 繰り返し（0→100→0→100...）
 * @value repeat
 * @option 往復（0→100→0...）
 * @value pingpong
 * @default repeat
 *
 * @param easing
 * @text イージング
 * @desc アニメーションの緩急
 * @type select
 * @option リニア（等速）
 * @value linear
 * @option イーズイン（加速）
 * @value easeIn
 * @option イーズアウト（減速）
 * @value easeOut
 * @option イーズインアウト（加速→減速）
 * @value easeInOut
 * @default easeInOut
 *
 * @param scaleX
 * @text X方向拡大率変化
 * @desc X方向の拡大率変化（%）。0で変化なし
 * @type number
 * @min -100
 * @max 100
 * @default 0
 *
 * @param scaleY
 * @text Y方向拡大率変化
 * @desc Y方向の拡大率変化（%）。0で変化なし
 * @type number
 * @min -100
 * @max 100
 * @default 10
 *
 * @param rotation
 * @text 回転角度変化
 * @desc 回転角度変化（度）。0で変化なし
 * @type number
 * @min -360
 * @max 360
 * @default 0
 *
 * @param offsetX
 * @text X座標移動量
 * @desc X座標の移動量（ピクセル）。0で変化なし
 * @type number
 * @min -999
 * @max 999
 * @default 0
 *
 * @param offsetY
 * @text Y座標移動量
 * @desc Y座標の移動量（ピクセル）。0で変化なし
 * @type number
 * @min -999
 * @max 999
 * @default 0
 *
 * @param opacity
 * @text 不透明度変化
 * @desc 不透明度の変化量（0-255）。0で変化なし
 * @type number
 * @min -255
 * @max 255
 * @default 0
 *
 * @param soundSettings
 * @text 効果音設定リスト
 * @desc 再生する効果音のリスト（複数設定した場合は抽選される）
 * @type struct<SoundEffect>[]
 * @default []
 *
 * @param soundTiming
 * @text 効果音のタイミング
 * @desc 効果音を再生するフレーム（0から指定）
 * @type number
 * @min 0
 * @default 0
 *
 * @param soundOnReturn
 * @text 折り返し時も再生
 * @desc 往復ループの折り返し時にも効果音を再生するか
 * @type boolean
 * @default false
 */

/*~struct~SoundEffect:
 * @param soundName
 * @text 効果音ファイル名
 * @desc 再生する効果音（audio/seフォルダ内）
 * @type file
 * @dir audio/se
 * @default
 *
 * @param soundVolume
 * @text 効果音の音量
 * @desc 効果音の音量（0-100）
 * @type number
 * @min 0
 * @max 100
 * @default 90
 *
 * @param soundPitch
 * @text 効果音のピッチ
 * @desc 効果音のピッチ（50-150）
 * @type number
 * @min 50
 * @max 150
 * @default 100
 *
 * @param soundPan
 * @text 効果音の位相
 * @desc 効果音の位相（-100～100）
 * @type number
 * @min -100
 * @max 100
 * @default 0
 */

(() => {
    'use strict';

    const pluginName = 'PictureAnimationState';
    const parameters = PluginManager.parameters(pluginName);
    
    // パラメータをパース
    const animationStates = JSON.parse(parameters['animationStates'] || '[]').map(state => {
        const parsed = JSON.parse(state);
        
        // 効果音設定リストをパース
        const soundSettings = JSON.parse(parsed.soundSettings || '[]').map(sound => {
            const soundParsed = JSON.parse(sound);
            return {
                soundName: soundParsed.soundName || '',
                soundVolume: Number(soundParsed.soundVolume) || 90,
                soundPitch: Number(soundParsed.soundPitch) || 100,
                soundPan: Number(soundParsed.soundPan) || 0
            };
        }).filter(sound => sound.soundName); // 効果音名が空のものは除外
        
        return {
            name: parsed.name || '新しい状態',
            duration: Number(parsed.duration) || 60,
            loop: parsed.loop || 'repeat',
            easing: parsed.easing || 'easeInOut',
            scaleX: Number(parsed.scaleX) || 0,
            scaleY: Number(parsed.scaleY) || 0,
            rotation: Number(parsed.rotation) || 0,
            offsetX: Number(parsed.offsetX) || 0,
            offsetY: Number(parsed.offsetY) || 0,
            opacity: Number(parsed.opacity) || 0,
            soundSettings: soundSettings,
            soundTiming: Number(parsed.soundTiming) || 0,
            soundOnReturn: parsed.soundOnReturn === 'true'
        };
    });

    // プラグインコマンド登録
    PluginManager.registerCommand(pluginName, 'applyState', args => {
        const pictureId = Number(args.pictureId);
        const stateName = args.stateName;
        $gameScreen.applyPictureAnimationState(pictureId, stateName);
    });

    PluginManager.registerCommand(pluginName, 'removeState', args => {
        const pictureId = Number(args.pictureId);
        $gameScreen.removePictureAnimationState(pictureId);
    });

    PluginManager.registerCommand(pluginName, 'clearAllStates', args => {
        $gameScreen.clearAllPictureAnimationStates();
    });

    //-----------------------------------------------------------------------------
    // Game_Screen
    //-----------------------------------------------------------------------------

    const _Game_Screen_initialize = Game_Screen.prototype.initialize;
    Game_Screen.prototype.initialize = function() {
        _Game_Screen_initialize.call(this);
        this._pictureAnimationStates = {};
    };

    const _Game_Screen_clear = Game_Screen.prototype.clear;
    Game_Screen.prototype.clear = function() {
        _Game_Screen_clear.call(this);
        this._pictureAnimationStates = {};
    };

    Game_Screen.prototype.applyPictureAnimationState = function(pictureId, stateName) {
        const stateData = animationStates.find(s => s.name === stateName);
        if (!stateData) {
            console.warn(`Animation state not found: ${stateName}`);
            return;
        }

        const picture = this.picture(pictureId);
        if (!picture) {
            console.warn(`Picture not found: ${pictureId}`);
            return;
        }

        this._pictureAnimationStates[pictureId] = {
            state: stateData,
            frame: 0,
            baseScaleX: picture._scaleX,
            baseScaleY: picture._scaleY,
            baseRotation: picture._angle,
            baseX: picture._x,
            baseY: picture._y,
            baseOpacity: picture._opacity,
            soundPlayedForward: false,
            soundPlayedReturn: false,
            lastPlayedSoundIndex: -1,
            availableSoundIndices: []
        };
    };

    Game_Screen.prototype.removePictureAnimationState = function(pictureId) {
        delete this._pictureAnimationStates[pictureId];
    };

    Game_Screen.prototype.clearAllPictureAnimationStates = function() {
        this._pictureAnimationStates = {};
    };

    Game_Screen.prototype.getPictureAnimationState = function(pictureId) {
        return this._pictureAnimationStates[pictureId] || null;
    };

    const _Game_Screen_updatePictures = Game_Screen.prototype.updatePictures;
    Game_Screen.prototype.updatePictures = function() {
        _Game_Screen_updatePictures.call(this);
        this.updatePictureAnimationStates();
    };

    Game_Screen.prototype.updatePictureAnimationStates = function() {
        for (const pictureId in this._pictureAnimationStates) {
            const animState = this._pictureAnimationStates[pictureId];
            const picture = this.picture(Number(pictureId));
            
            if (!picture) {
                delete this._pictureAnimationStates[pictureId];
                continue;
            }

            this.updatePictureAnimationFrame(picture, animState);
        }
    };

    Game_Screen.prototype.updatePictureAnimationFrame = function(picture, animState) {
        const state = animState.state;
        
        // フレームを進める
        animState.frame++;
        if (animState.frame >= state.duration) {
            if (state.loop === 'repeat') {
                animState.frame = 0;
                animState.soundPlayedForward = false;
                animState.soundPlayedReturn = false;
            } else if (state.loop === 'pingpong') {
                // 往復の場合は duration * 2 でリセット
                if (animState.frame >= state.duration * 2) {
                    animState.frame = 0;
                    animState.soundPlayedForward = false;
                    animState.soundPlayedReturn = false;
                }
            }
        }

        // 効果音の再生判定
        if (state.soundSettings && state.soundSettings.length > 0) {
            // 前半（0→duration）での効果音再生
            if (animState.frame === state.soundTiming && !animState.soundPlayedForward) {
                this.playRandomAnimationSound(animState);
                animState.soundPlayedForward = true;
            }
            
            // 往復の後半（duration→duration*2）での効果音再生
            if (state.loop === 'pingpong' && state.soundOnReturn) {
                const returnTiming = state.duration + state.soundTiming;
                if (animState.frame === returnTiming && !animState.soundPlayedReturn) {
                    this.playRandomAnimationSound(animState);
                    animState.soundPlayedReturn = true;
                }
            }
            
            // フレームがリセットされたら再生フラグもリセット
            if (animState.frame < state.soundTiming) {
                animState.soundPlayedForward = false;
            }
            if (state.loop === 'pingpong' && animState.frame < state.duration + state.soundTiming) {
                animState.soundPlayedReturn = false;
            }
        }

        // 進行度を計算（0.0 〜 1.0）
        let progress = animState.frame / state.duration;
        
        // 往復の場合は前半と後半で処理を変える
        if (state.loop === 'pingpong' && animState.frame >= state.duration) {
            progress = 2.0 - progress; // 1.0 → 0.0 に反転
        }

        // イージングを適用
        const easedProgress = this.applyEasing(progress, state.easing);

        // 各パラメータを更新
        if (state.scaleX !== 0) {
            const scaleChange = (state.scaleX / 100) * easedProgress;
            picture._scaleX = animState.baseScaleX + scaleChange * 100;
        }

        if (state.scaleY !== 0) {
            const scaleChange = (state.scaleY / 100) * easedProgress;
            picture._scaleY = animState.baseScaleY + scaleChange * 100;
        }

        if (state.rotation !== 0) {
            const rotationChange = state.rotation * easedProgress;
            picture._angle = animState.baseRotation + rotationChange;
        }

        if (state.offsetX !== 0) {
            const xChange = state.offsetX * easedProgress;
            picture._x = animState.baseX + xChange;
        }

        if (state.offsetY !== 0) {
            const yChange = state.offsetY * easedProgress;
            picture._y = animState.baseY + yChange;
        }

        if (state.opacity !== 0) {
            const opacityChange = state.opacity * easedProgress;
            picture._opacity = Math.max(0, Math.min(255, animState.baseOpacity + opacityChange));
        }
    };

    Game_Screen.prototype.playRandomAnimationSound = function(animState) {
        const state = animState.state;
        const soundSettings = state.soundSettings;
        
        if (!soundSettings || soundSettings.length === 0) return;
        
        // 効果音が1つしかない場合はそれを再生
        if (soundSettings.length === 1) {
            this.playAnimationSound(soundSettings[0]);
            return;
        }
        
        // 利用可能な効果音のプールが空の場合、または初回の場合は再構築
        if (animState.availableSoundIndices.length === 0) {
            // 前回再生した効果音以外の全インデックスを追加
            for (let i = 0; i < soundSettings.length; i++) {
                if (i !== animState.lastPlayedSoundIndex) {
                    animState.availableSoundIndices.push(i);
                }
            }
            
            // それでも空の場合（初回など）は全インデックスを追加
            if (animState.availableSoundIndices.length === 0) {
                for (let i = 0; i < soundSettings.length; i++) {
                    animState.availableSoundIndices.push(i);
                }
            }
        }
        
        // プールからランダムに選択
        const randomIndex = Math.floor(Math.random() * animState.availableSoundIndices.length);
        const selectedSoundIndex = animState.availableSoundIndices[randomIndex];
        
        // 選択した効果音を再生
        this.playAnimationSound(soundSettings[selectedSoundIndex]);
        
        // 使用した効果音をプールから削除
        animState.availableSoundIndices.splice(randomIndex, 1);
        
        // 次回の抽選対象外にするため記録
        animState.lastPlayedSoundIndex = selectedSoundIndex;
    };

    Game_Screen.prototype.playAnimationSound = function(soundSetting) {
        const se = {
            name: soundSetting.soundName,
            volume: soundSetting.soundVolume,
            pitch: soundSetting.soundPitch,
            pan: soundSetting.soundPan
        };
        AudioManager.playSe(se);
    };

    Game_Screen.prototype.applyEasing = function(t, easingType) {
        switch (easingType) {
            case 'linear':
                return t;
            case 'easeIn':
                return t * t;
            case 'easeOut':
                return t * (2 - t);
            case 'easeInOut':
                return t < 0.5 ? 2 * t * t : -1 + (4 - 2 * t) * t;
            default:
                return t;
        }
    };

    //-----------------------------------------------------------------------------
    // Game_Picture
    //-----------------------------------------------------------------------------

    const _Game_Picture_initialize = Game_Picture.prototype.initialize;
    Game_Picture.prototype.initialize = function() {
        _Game_Picture_initialize.call(this);
    };

    // ピクチャが表示される際に基準値を更新
    const _Game_Picture_show = Game_Picture.prototype.show;
    Game_Picture.prototype.show = function(name, origin, x, y, scaleX, scaleY, opacity, blendMode) {
        _Game_Picture_show.call(this, name, origin, x, y, scaleX, scaleY, opacity, blendMode);
        
        // アニメーション状態が既に存在する場合、基準値を更新
        const pictureId = $gameScreen._pictures.indexOf(this) + 1;
        const animState = $gameScreen.getPictureAnimationState(pictureId);
        if (animState) {
            animState.baseScaleX = this._scaleX;
            animState.baseScaleY = this._scaleY;
            animState.baseRotation = this._angle;
            animState.baseX = this._x;
            animState.baseY = this._y;
            animState.baseOpacity = this._opacity;
            animState.soundPlayedForward = false;
            animState.soundPlayedReturn = false;
            animState.lastPlayedSoundIndex = -1;
            animState.availableSoundIndices = [];
        }
    };

})();
