//=============================================================================
// CharacterImageManager.js
//=============================================================================

/*:
 * @help CharacterImageManager.js
 * @plugindesc 歩行グラフィックを構成する
 * 
 * @param layers
 * @desc 表示順を防具タイプ名で指定。
 * @type String[]
 * 
 * @param Directory Name
 * @desc 装備イメージを格納しているフォルダ名
 * @type String
 * @default appearance
 * 
 * 
 */

(function () {
    'use strict';

    var parameters = PluginManager.parameters('CharacterImageManager');
    const layers = JSON.parse(parameters['layers']);
    const imageDirectory = parameters['Directory Name']
    const bodyFileName = 'body';
    const hairFileName = 'hair';

    Game_Player.prototype.actor = function () {
        return $gameParty.leader();
    }

    const _Sprite_Character_Initialize = Sprite_Character.prototype.initialize;
    Sprite_Character.prototype.initialize = function (character) {
        _Sprite_Character_Initialize.call(this, character);
        this.clearLayers();
    };

    Sprite_Character.prototype.initLayers = function () {
        this._layers = [];
    };

    Sprite_Character.prototype.clearLayers = function () {
        this._layers = null;
    };

    Sprite_Character.prototype.hasLayers = function () {
        return !!this._layers;
    }

    const _Sprite_Character_SetCharacterBitmap = Sprite_Character.prototype.setCharacterBitmap;
    Sprite_Character.prototype.setCharacterBitmap = function () {
        _Sprite_Character_SetCharacterBitmap.call(this);
        if (this._character instanceof Game_Player)
            this.initLayers();
        else
            this.clearLayers();

        if (this.hasLayers()) {
            this.bitmap.addLoadListener(this.drawLayers.bind(this))
        }
    };

    Sprite_Character.prototype.drawLayers = function () {
        const promises = [];

        for (const part of layers) {
            let fileName = this.getLayerImage(part);
            this._layers[part] = fileName;
            if (!fileName) continue;
            promises.unshift(this.makePromiseLoadBitmap(fileName));
        }

        this.hide();
        Promise.all(promises).then((bitmaps) => {

            const newBitmap = new Bitmap(this.bitmap.width, this.bitmap.height);
            newBitmap.blt(this.bitmap, 0, 0, this.bitmap.width, this.bitmap.height, 0, 0);

            for (const b of bitmaps) {
                newBitmap.blt(b, 0, 0, this.bitmap.width, this.bitmap.height, 0, 0);
            }

            this.bitmap = newBitmap;
            this.show();
            this.bitmap.baseTexture.update();
            this.updateCharacterFrame();
            this._refresh();
        }).catch((e) => {
            console.log('failure to load equipment image ', e)
        });
    }

    Sprite_Character.prototype.makePromiseLoadBitmap = function (filePath, order) {
        return new Promise((resolve, reject) => {
            var bitmap = ImageManager.loadCharacter(filePath);
            if (!bitmap) {
                reject();
                return null;
            }
            bitmap.addLoadListener(() => {
                resolve(bitmap);
            });
        });
    }

    const _Sprite_Character_isImageChanged = Sprite_Character.prototype.isImageChanged;
    Sprite_Character.prototype.isImageChanged = function () {
        return _Sprite_Character_isImageChanged.apply(this, arguments) ||
            this.hasLayers() && this.isLayerContentChanged();
    };

    Sprite_Character.prototype.isLayerContentChanged = function () {
        return !layers.every(part => this._layers[part] === this.getLayerImage(part))
    };

    Sprite_Character.prototype.getLayerImage = function (part) {
        let fileName = '';
        switch (part) {
            case '素体': fileName = bodyFileName;
                break;
            case '髪': fileName = hairFileName;
                break;
            case '頭': fileName = hairbandFileName;
                break;
            default: fileName = $gamePlayer.getEquipmentWithEtypeName(part);
                break;
        }
        if (!fileName) return '';
        let dir = imageDirectory + '/';
        if (this._characterName.slice(-'_damage'.length) === '_damage')
            dir = imageDirectory + '/damage/'
        return dir + fileName;
    }

    Game_Player.prototype.getEquipmentWithEtypeName = function (part) {
        const actor = $gameParty.members().find(a => a.actorId() === 1)
        const armor = actor.armors().find(a => $dataSystem.equipTypes.indexOf(part) === a.etypeId);
        return armor?.meta.appearance
    }

})();