/*:ja
 * @target MZ
 * @author fude
 *
 * @help EventSystem.js
 * 本プラグインは、制作者が明示的に許可した場合を除き、いかなる形であれ使用、複製、改変、再配布することを禁じます。
 * 本プラグインを無断で使用したこと、またはその利用によって生じた不具合・損害について、制作者は一切の責任を負いません。
 *
 * @command SendMessasge
 * @text メッセージを送信
 * @arg message
 * @type text
 * 
 * @command sendMessageAroundTheCharacter
 * @text 周囲のイベントにメッセージを送信

 * @arg message
 * @type text
 * @text メッセージ
 * @desc 送信するメッセージ
 * 
 * @arg sourceId
 * @type number
 * @default 0
 * @text 発信元ID
 * @desc イベントIDを指定。0でこのイベント、-1でプレイヤー
 * @min -1
 * 
 * @arg isForward
 * @type boolean
 * @default false
 * @text 前方に限定
 * @desc 発信元からの方向を前方に限定
 * 
 * @arg range
 * @type number
 * @default 5
 * @text 範囲
 * @desc 発信元からの範囲
 * 
 * @command Subscribe
 * @text メッセージ監視を登録
 * 
 * @arg listenerId
 * @type text
 * @text リスナーID
 * @desc メッセージ監視登録を行うリスナーのID
 * 
 * @arg message
 * @type text
 * @text メッセージ
 * @desc 監視するメッセージ
 * 
 * @arg cmnId
 * @type text
 * @text コモンイベントID
 * @desc メッセージ受信時実行するコモンイベント（IDまたは名前）
 * @default 0
 * 
 * @command Unsubscribe
 * @text メッセージ監視を解除
 * 
 * @arg listenerId
 * @type text
 * @text リスナーID
 * @desc メッセージ監視解除を行うリスナーのID
 * 
 * @arg message
 * @type text
 * @desc 監視解除するメッセージ
 * 
 * @command RegisterListener
 * @text リスナーを登録
 * @arg listenerId
 * @type text
 * @text リスナーID
 * 
 * @command DeleteListener
 * @text リスナーを削除
 * @arg listenerId
 * @type text
 * @text リスナーID
 * 
 * @command RegisterTemporaryEventListener
 * @text イベント中限定のリスナーを作成
 * @desc ページの注釈から監視メッセージを登録する このコマンドを呼び出したページ処理が終了した時点でリスナを削除
 * 
 * 
 * */

(function () {
    const pluginName = 'EventSystem';

    var _Game_System_initialize = Game_System.prototype.initialize;
    Game_System.prototype.initialize = function () {
        _Game_System_initialize.apply(this, arguments);
        this._eventSystem = new EventSystem();
    };

    Game_System.prototype.eventSystem = function () {
        return this._eventSystem;
    };

    var _Game_System_onLoad = Game_System.prototype.onLoad;
    Game_System.prototype.onLoad = function () {
        if (_Game_System_onLoad)
            _Game_System_onLoad.apply(this, arguments);
        if (!this.eventSystem())
            this._eventSystem = new EventSystem();
    };

    class EventListener {
        constructor(id) {
            this._id = id
            this._actions = [];
        }

        receiveMessage(message) {
            const action = this._actions[message]
            if (action)
                action();
        }

        subscribe(message, action) {
            this._actions[message] = action;
        }

        unsubscribe(message) {
            delete this._actions[message];
        }

        listenerId() {
            return this._id;
        }

        hasSubscription(message) {
            return !!this._actions[message]
        }
    }

    class EventSystem {
        constructor() {
            this._listeners = [];
            this._eventQue = [];
        }

        interpreter() {
            return $gameMap._interpreter;
        }

        sendMessage(message) {
            this._listeners.forEach(listener => {
                listener.receiveMessage(message);
            })
        }

        sendMessageAroundTheCharacter(message, character, range, isforward) {
            const px = Math.floor(character.x);
            const py = Math.floor(character.y);
            const pd = isforward ? character.direction() : 0;

            const eventsByDir = $gameMap.events().filter(e => {
                const ex = Math.floor(e.x);
                const ey = Math.floor(e.y);
                switch (pd) {
                    case 8:
                        return ex === px && ey <= py;
                    case 2:
                        return ex === px && ey >= py;
                    case 4:
                        return ex <= px && ey === py;
                    case 6:
                        return ex >= px && ey === py;
                    default: return true;
                }
            });
            const nearEvents = eventsByDir.filter(e => $gameMap.distance(px, py, e.x, e.y) <= range);
            const listeners = nearEvents.filter(e => e.page() && e.getStartComment(e.page()).match(/EventListener/));
            listeners.forEach(e => {
                e.receiveMessage(message);
            });
        }

        subscribe(listenerId, message, action) {
            if (this.checkErrorSubscribe(listenerId, message, action))
                return; //エラーがある

            const listener = this.getListener(listenerId);
            listener.subscribe(message, action);
        }

        unsubscribe(listenerId, message) {
            if (this.checkErrorUnsubscribe(listenerId, message))
                return; //エラーがある
            const listener = this.getListener(listenerId);
            listener.unsubscribe(message);
        }

        registerListener(listenerId) {
            if (this.checkErrorRegisterListener(listenerId))
                return; //エラーがある
            this._listeners.push(new EventListener(listenerId));
        }

        deleteListener(listenerId) {
            if (this.checkErrorDeleteListener(listenerId))
                return; //エラーがある
            this._listeners = this._listeners.filter(l => l.listenerId() !== listenerId);
        }

        getListener(listenerId) {
            return this._listeners.find(l => l.listenerId() === listenerId);
        }

        checkErrorSubscribe(listenerId, message, action) {
            let error = false;
            if (!this.getListener(listenerId)) {
                console.log('リスナーID『' + listenerId + '』は登録されていないため、メッセージの登録ができません。')
                error = true;
            }

            if (!action) {
                console.log('アクションが未指定のため、メッセージの登録ができません。')
                error = true;
            }

            if (!message) {
                console.log('メッセージが指定されていないため、メッセージの登録ができません。')
                error = true;
            }
            return error;
        }

        checkErrorUnsubscribe(listenerId, message) {
            let error = false;
            const listener = this.getListener(listenerId)
            if (!listener) {
                console.log('リスナーID『' + listenerId + '』は登録されていないため、メッセージの登録解除できません。')
                error = true;
            }
            else if (!listener.hasSubscription(message)) {
                console.log('リスナーID『' + listenerId + '』にメッセージ『' + message + '』は登録されていないため、メッセージの登録解除できません。')
                error = true;
            }
            return error;
        }

        checkErrorRegisterListener(listenerId) {
            if (this.getListener(listenerId)) {
                console.log('リスナーID『' + listenerId + '』は登録済のため、登録できません。')
                return true;
            }
            return false;
        }

        checkErrorDeleteListener(listenerId) {
            if (!this.getListener(listenerId)) {
                console.log('リスナーID『' + listenerId + '』は登録されていないため、リスナを削除できません。')
                return true;
            }
            return false;
        }

        getActionQueEvent(teId) {
            const cmnev = $dataCommonEvents.find(te => !!te && (te.id === teId || te.name === teId));
            if (!cmnev) {
                console.log('コモンイベント' + teId + 'が見つかりません。')
                return
            }

            const eventCommands = cmnev.list

            return this.queEvent.bind(this, eventCommands, 0);
        }

        invokeEvent(que) {
            if (!que) return;
            if (!this.interpreter().isRunning())
                this.interpreter().setup(que.eventCommands, que.eventId);
            else
                $gameMap.getLastInterpreter(this.interpreter()).setupChild(que.eventCommands, que.eventId);
        }

        update() {
            if (this._eventQue.length <= 0)
                return
            this.invokeEvent(this._eventQue.shift());
        }

        queEvent(eventCommands, eventId) {
            this._eventQue.push({ eventCommands, eventId })
        }

        isEventRunning() {
            return this.interpreter().isRunning()
        }

    }

    Game_Map.prototype.getLastInterpreter = function (intprt) {
        if (!intprt._childInterpreter) {
            return intprt;
        }
        return this.getLastInterpreter(intprt._childInterpreter);
    }


    const _Game_Mapupdate_Interpreter = Game_Map.prototype.updateInterpreter;
    Game_Map.prototype.updateInterpreter = function () {
        $gameSystem.eventSystem().update()
        _Game_Mapupdate_Interpreter.apply(this, arguments);
    };

    const _Game_Map_isEventRunning = Game_Map.prototype.isEventRunning;
    Game_Map.prototype.isEventRunning = function () {
        return _Game_Map_isEventRunning.apply(this, arguments) || $gameSystem.eventSystem().isEventRunning();
    };

    Game_Event.prototype.receiveMessage = function (message) {
        const pages = this.event().pages;
        if (!pages) return;

        const page = pages.find(p => p.conditions.switch1Id === $dataSystem.switches.indexOf(message))
        if (!page) return;
        $gameSystem.eventSystem().queEvent(page.list, this.eventId());
    }

    function extractEventCommand(eventCommands, message) {
        const result = [];
        const reg = new RegExp(`\\Listen{${message}}`);

        let match = false;
        for (let i = 0; i < eventCommands.length; i++) {
            const currentCommand = eventCommands[i];

            if (match) {
                if (currentCommand.code === 409)
                    break;
                result.push(currentCommand)
                continue
            }

            const nextCommand = eventCommands[i + 1];
            if (!nextCommand) break;

            const isCurrentSkip = currentCommand.code === 109;
            const isNextComment = (nextCommand.code === 108 || nextCommand.code === 408);
            match = isCurrentSkip && isNextComment && nextCommand.parameters[0].match(reg);
        }

        return result;
    }

    PluginManager.registerCommand(pluginName, 'SendMessasge', function (args) {
        $gameSystem.eventSystem().sendMessage(args.message);
    });

    PluginManager.registerCommand(pluginName, 'Subscribe', function (args) {
        const action = $gameSystem.eventSystem().getActionQueEvent(args.cmnId);
        $gameSystem.eventSystem().subscribe(args.listenerId, args.message, action);
    });

    PluginManager.registerCommand(pluginName, 'Unsubscribe', function (args) {
        $gameSystem.eventSystem().unsubscribe(args.listenerId, args.message);
    });

    PluginManager.registerCommand(pluginName, 'RegisterListener', function (args) {
        $gameSystem.eventSystem().registerListener(args.listenerId);
    });

    PluginManager.registerCommand(pluginName, 'DeleteListener', function (args) {
        $gameSystem.eventSystem().deleteListener(args.listenerId);
    });

    PluginManager.registerCommand(pluginName, 'RegisterTemporaryEventListener', function (args) {
        const page = { list: this._list }
        const comment = Game_Event.prototype.getStartComment.call(this, page);
        const messages = [];
        comment.replace(/\\Listen{(.+?)}/gi, function () {
            messages.push(arguments[1]);
        }.bind(this));

        if (messages.length <= 0) return;

        this._eventListenerId = Symbol(); // シンボルはセーブデータに含めることはできないので注意　ここではイベント中限定のライフサイクルなので使用する
        $gameSystem.eventSystem().registerListener(this._eventListenerId);
        for (const message of messages) {
            const eventCommands = extractEventCommand(this._list, message);
            if (eventCommands.length <= 0) continue;
            const action = $gameSystem.eventSystem().queEvent.bind($gameSystem.eventSystem(), eventCommands, this.eventId());
            $gameSystem.eventSystem().subscribe(this._eventListenerId, message, action);
        }
    });

    PluginManager.registerCommand(pluginName, 'sendMessageAroundTheCharacter', function (args) {
        const id = +args.sourceId;
        const range = +args.range;
        const message = args.message;
        const isForward = JSON.parse(args.isForward ? args.isForward : false);
        const character = id === 0 ? $gameMap.event(this.eventId()) : id === -1 ? $gamePlayer : $gameMap.event(id);
        if (character)
            $gameSystem.eventSystem().sendMessageAroundTheCharacter(message, character, range, isForward);
    });

    const _Game_Interpreter_terminate = Game_Interpreter.prototype.terminate;
    Game_Interpreter.prototype.terminate = function () {
        _Game_Interpreter_terminate.apply(this, arguments);
        if (this._eventListenerId) {
            console.log('deletelistener')
            $gameSystem.eventSystem().deleteListener(this._eventListenerId);
            this._eventListenerId = null;
        }
    }

    window[EventSystem.name] = EventSystem;
    window[EventListener.name] = EventListener;

}());