/*=============================================================================
 ExtraGauge.js
----------------------------------------------------------------------------

/*:
 * @plugindesc ゲージ追加プラグイン
 * @target MZ
 * @base PluginCommonBase
 * @author トリアコンタン
 * 
 *
 * @param GaugeList
 * @text ゲージリスト
 * @desc 各画面に追加するゲージのリストです。
 * @default []
 * @type struct<Gauge>[]
 * 
 * @param alignGaugeBaseX
 * @text 整列ゲージ基準X
 * @type number
 * 
 * @param alignGaugeBaseY
 * @text 整列ゲージ基準Y
 * @type number

 *
 * @help ExtraGauge.js
 *
 * このプラグインはトリアコンタン様の提供するExtraGauge.jsを改変し作成したものです。
 * このプラグインを無断利用する上での責任は一切追いかねます
 * 
 * このプラグインの利用にはベースプラグイン『PluginCommonBase.js』が必要です。
 * 『PluginCommonBase.js』は、RPGツクールMZのインストールフォルダ配下の
 * 以下のフォルダに格納されています。
 * dlc/BasicResources/plugins/official
 *
 */

/*~struct~Gauge:
 *
 * @param SceneName
 * @text 対象シーン
 * @desc 追加対象のシーンです。オリジナルのシーンを対象にする場合はシーンクラス名を直接記入します。
 * @type select
 * @default Scene_Title
 * @option タイトル
 * @value Scene_Title
 * @option マップ
 * @value Scene_Map
 * @option ゲームオーバー
 * @value Scene_Gameover
 * @option バトル
 * @value Scene_Battle
 * @option メインメニュー
 * @value Scene_Menu
 * @option アイテム
 * @value Scene_Item
 * @option スキル
 * @value Scene_Skill
 * @option 装備
 * @value Scene_Equip
 * @option ステータス
 * @value Scene_Status
 * @option オプション
 * @value Scene_Options
 * @option セーブ
 * @value Scene_Save
 * @option ロード
 * @value Scene_Load
 * @option ゲーム終了
 * @value Scene_End
 * @option ショップ
 * @value Scene_Shop
 * @option 名前入力
 * @value Scene_Name
 * @option デバッグ
 * @value Scene_Debug
 *
 * @param Id
 * @text 識別子
 * @desc ゲージの識別子です。特に使用されませんが、分かりやすい名称を設定すると管理がしやすくなります。
 * @default gauge01
 *
 * @param SwitchId
 * @text 表示スイッチID
 * @desc 指定したスイッチがONの場合のみ画面に表示されます。0を指定すると常に表示されます。
 * @default 0
 * @type switch
 *
 * @param OpacityVariable
 * @text 不透明度変数ID
 * @desc 不透明度を取得する変数番号です。0を指定すると常に不透明度255で表示されます。
 * @default 0
 * @type variable
 *
 * @param Layout
 * @text レイアウト
 * @desc ゲージの表示座標と幅、高さです。スクリプトを使用する場合、変数witch, heightでUIエリアの幅と高さを取得できます。
 * @type struct<Layout>
 * @default {"x":"width / 2","y":"30","width":"width * 0.8","height":"36","GaugeX":"0","GaugeHeight":"0","Vertical":"false"}
 *
 * @param CurrentMethod
 * @text 現在値取得方法
 * @desc ゲージの現在値を取得する方法です。変数、スクリプトのいずれかを設定します。
 * @default {"VariableId":"1","Script":"","FixedValue":""}
 * @type struct<Method>
 *
 * @param MaxMethod
 * @text 最大値1取得方法
 * @desc ゲージの最大値1を取得する方法です。変数、スクリプト、固定値のいずれかを設定します。
 * @default {"VariableId":"0","Script":"","FixedValue":"100"}
 * @type struct<Method>
 *
 * 
 * @param isBalance
 * @text バランスゲージ有効化
 * @desc ゲージをバランスゲージにするか
 * @type boolean
 * @on する
 * @off しない
 * @default false

 * @param isSperms
 * @text 精液量ゲージ有効化
 * @desc ゲージを精液量ゲージにするか
 * @type boolean
 * @on する
 * @off しない
 * @default false
 * 
 * @param MaxMethod2
 * @parent isBalance
 * @text 最大値2取得方法
 * @desc ゲージの最大値2を取得する方法です。変数、スクリプト、固定値のいずれかを設定します。
 * @type struct<Method>
 * @default {"VariableId":"0","Script":"","FixedValue":"100"}
 * 
 * 
 * @param Detail
 * @text 詳細設定
 * @desc ゲージの配置や色などの細かい設定です。
 * @type struct<Detail>
 * @default
 *
 * @param LowerPicture
 * @text 下ピクチャ
 * @desc ゲージの下に表示されるピクチャです。ゲージの中心と画像の中心が合わせて表示されます。
 * @default
 * @type struct<Picture>
 *
 * @param UpperPicture
 * @text 上ピクチャ
 * @desc ゲージの上に表示されるピクチャです。ゲージの中心と画像の中心が合わせて表示されます。
 * @default
 * @type struct<Picture>
 *
 * @param Battler
 * @text バトラー情報
 * @desc ゲージの主体となるバトラー情報の参照方法を指定します。現在値、最大値をスクリプトで決める場合のみ使用します。
 * @default
 * @type struct<Battler>
 */

/*~struct~Layout:
 *
 * @param x
 * @text X座標
 * @desc X座標です。原点は中央です。数値以外を指定した場合はスクリプトとして評価します。
 * @default width / 2
 *
 * @param y
 * @text Y座標
 * @desc Y座標です。原点は中央です。数値以外を指定した場合はスクリプトとして評価します。
 * @default 30
 *
 * @param realTime
 * @text リアルタイム座標反映
 * @desc ゲージを表示後、X座標、Y座標が変更されたときにゲージの位置を再評価します。
 * @default false
 * @type boolean
 *
 * @param width
 * @text 横幅
 * @desc 横幅です。数値以外を指定した場合はスクリプトとして評価します。
 * @default width * 0.8
 *
 * @param height
 * @text 高さ
 * @desc 高さです。数値以外を指定した場合はスクリプトとして評価します。
 * @default 36
 *
 * @param GaugeX
 * @text ゲージX座標
 * @desc ゲージのX座標です。ラベルが長い文字の場合は変更してください。
 * @default 0
 *
 * @param GaugeHeight
 * @text ゲージ高さ
 * @desc ゲージの高さです。0を指定すると全体の高さに合わせられます。
 * @default 0
 *
 * @param Vertical
 * @text 縦ゲージ
 * @desc 有効にすると縦方向ゲージになります。ラベルなども縦方向になるので注意してください。
 * @default false
 * @type boolean
 * 
 * @param AlignCol
 * @text 整列準列番号
 * @desc メニューシーン表示用
 * @default 0
 * @type number
 * 
 * @param AlignRow
 * @text 整列準行番号
 * @desc メニューシーン表示用
 * @default 0
 * @type number

 */

/*~struct~Picture:
 *
 * @param FileName
 * @text ファイル名
 * @desc ピクチャのファイル名です。
 * @default
 * @type file
 * @dir img/pictures
 *
 * @param OffsetX
 * @text X座標補正値
 * @desc X座標の補正値です。
 * @default 0
 * @type number
 * @min -9999
 *
 * @param OffsetY
 * @text Y座標補正値
 * @desc Y座標の補正値です。
 * @default 0
 * @type number
 * @min -9999
 */

/*~struct~Method:
 *
 * @param VariableId
 * @text 取得変数ID
 * @desc ゲージの値を取得する変数番号です。スクリプトより優先して参照されます。
 * @default 0
 * @type variable
 *
 * @param Script
 * @text 取得スクリプト
 * @desc ゲージの値を取得するスクリプトです。固定値より優先して参照されます。
 * @default
 * @type combo
 * @option battler.hp; // HP
 * @option battler.mhp; // 最大HP
 * @option battler.mp; // MP
 * @option battler.mmp; // 最大MP
 * @option battler.tp; // TP
 * @option battler.maxTp(); // 最大MP
 * @option meta.value; // メモ欄[value]の値
 *
 * @param FixedValue
 * @text 固定値
 * @desc ゲージの値を固定値で設定します。現在値に固定値を指定することは推奨しません。
 * @default
 * @type number
 * @min 1
 */

/*~struct~Detail:
 *
 * @param RisingSmoothness
 * @text 上昇中のなめらかさ
 * @desc 大きい数を指定するとゲージがゆっくりと上昇します。
 * @default 1
 * @type number
 * @min 1
 *
 * @param FallingSmoothness
 * @text 下降中のなめらかさ
 * @desc 大きい数を指定するとゲージがゆっくりと下降します。
 * @default 1
 * @type number
 * @min 1
 *
 * @param GaugeColorPreset
 * @text ゲージ色のプリセット
 * @desc ゲージ色をプリセットから簡易指定します。詳細指定があればそちらが優先されます。
 * @default hp
 * @type select
 * @option
 * @option hp
 * @option mp
 * @option tp
 * @option time
 *
 * @param GaugeColorLeftLeft
 * @text ゲージ色(左)(左左)
 * @desc 左側領域の左側ゲージ色です。テキストカラー番号かCSS色指定(rgba(0, 0, 0, 0))を指定します。
 * @default 0
 *
 * @param GaugeColorLeftRight
 * @text ゲージ色(右)(左右)
 * @desc 左側領域の右側のゲージ色です。テキストカラー番号かCSS色指定(rgba(0, 0, 0, 0))を指定します。
 * @default 0
 * 
 * @param GaugeColorRightLeft
 * @text ゲージ色(右左)
 * @desc 右側領域の左側のゲージ色です。テキストカラー番号かCSS色指定(rgba(0, 0, 0, 0))を指定します。
 * @default 0
 *
 * @param GaugeColorRightRight
 * @text ゲージ色(右右)
 * @desc 右側領域の右側のゲージ色です。テキストカラー番号かCSS色指定(rgba(0, 0, 0, 0))を指定します。
 * @default 0
 *
 * @param BackColor
 * @text ゲージ背景色
 * @desc ゲージ背景色です。テキストカラー番号かCSS色指定(rgba(0, 0, 0, 0))を指定します。
 * @default 0
 *
 * @param Label
 * @text ラベル
 * @desc ゲージの左に表示されるラベル文字列です。
 * @default
 *
 * @param LabelFont
 * @text ラベルフォント
 * @desc ラベルを表示するときのフォント情報です。未指定の場合はゲージのデフォルト値が使用されます。
 * @default
 * @type struct<Font>
 *
 * @param DrawValue
 * @text 現在値を描画する
 * @desc ゲージの右側に現在値を描画します。
 * @default true
 * @type boolean
 * 
 * @param DrawMaxValue
 * @text 現在地と最大値を描画する
 * @desc ゲージの右側に現在地と最大値を描画します。
 * @default true
 * @type boolean
 * 
 * @param DrawValueAsText
 * @text 現在値をテキストとして表示する
 * @default false
 * @type boolean
 * 
 * @param ValueAsTextSettings
 * @parent DrawValueAsText
 * @text 現在値をテキストとして表示する設定
 * @type struct<ValueAsTextSetting>[]
 * 
 * @param ValueFont
 * @text 現在値フォント
 * @desc 現在値を表示するときのフォント情報です。未指定の場合はゲージのデフォルト値が使用されます。
 * @default
 * @type struct<Font>
 *
 * @param FlashIfFull
 * @text 満タン時にフラッシュ
 * @desc ゲージの現在値が最大値以上になるとゲージをフラッシュさせます。
 * @default false
 * @type boolean
 * 
 * @param temporaryDisp
 * @text 一時表示有効化
 * @desc ゲージ値変動時、一時的にゲージを表示する
 * @type boolean
 * @on する
 * @off しない
 * @default false
 * 
 * @param temporaryDispDuration
 * @text 一時表示時間
 * @type number
 * @default 300

 * 
 * @param fadeFrame
 * @text ゆっくり消失
 * @type number
 * @default 60
 * 
 * @param icon
 * @text ゲージアイコン
 * @type number
 * @default 0
 */

/*~struct~ValueAsTextSetting:
 * @param range
 * @text 範囲
 * @desc 例 50～80 → 50,80
 * @type text
 * 
 * @param valueText
 * @text 表示するテキスト
 * @type text
*/

/*~struct~Font:
 *
 * @param Face
 * @text フォント名
 * @desc フォント名です。別途フォントロードプラグインが必要です。
 * @default
 * @dir fonts
 *
 * @param Size
 * @text フォントサイズ
 * @desc フォントサイズです。
 * @default 0
 * @type number
 *
 * @param Color
 * @text テキストカラー
 * @desc テキストカラーです。テキストカラー番号かCSS色指定(rgba(0, 0, 0, 0))を指定します。
 * @default 0
 * @type number
 *
 * @param OutlineColor
 * @text アウトラインカラー
 * @desc アウトラインカラーです。テキストカラー番号かCSS色指定(rgba(0, 0, 0, 0))を指定します。
 * @default 0
 * @type number
 *
 * @param OutlineWidth
 * @text アウトライン横幅
 * @desc アウトラインの横幅です。
 * @default 0
 * @type number
 */

/*~struct~Battler:
 *
 * @param Type
 * @text バトラー種別
 * @desc ゲージの主体となるバトラーの取得方法です。
 * @default
 * @type select
 * @option アクターID
 * @value ActorId
 * @option パーティの並び順
 * @value PartyIndex
 * @option 敵キャラID
 * @value EnemyId
 * @option 敵グループの並び順(戦闘画面で有効)
 * @value TroopIndex
 * @option メニュー画面で選択したアクター(メニュー詳細画面で有効)
 * @value MenuActor
 *
 * @param ActorId
 * @text アクターID
 * @desc 種別選択で『アクターID』を選択したときのアクターIDです。
 * @default 0
 * @type actor
 *
 * @param EnemyId
 * @text 敵キャラID
 * @desc 種別選択で『敵キャラID』を選択したときの敵キャラIDです。
 * @default 0
 * @type enemy
 *
 * @param Index
 * @text 並び順
 * @desc 種別選択で『パーティの並び順』『敵グループの並び順』を選択したときの並び順です。先頭は[0]です。
 * @default 0
 * @type number
 */

(() => {
    'use strict';
    const script = document.currentScript;
    const param = PluginManagerEx.createParameter(script);
    if (!param.GaugeList) {
        param.GaugeList = [];
    }

    const _Scene_Base_create = Scene_Base.prototype.create;
    Scene_Base.prototype.create = function () {
        _Scene_Base_create.apply(this, arguments);
        if (!(this instanceof Scene_Map)) {
            this.createExtraGauges();
        }
    };

    const _Scene_Map_create = Scene_Map.prototype.create;
    Scene_Map.prototype.create = function () {
        _Scene_Map_create.apply(this, arguments);
        this.createExtraGauges();
    };

    Scene_Base.prototype.createExtraGauges = function () {
        this._extraGauges = this.findExtraGaugeList().map(data => {
            return new Sprite_ExtraGaugeContainer(data, data.Detail || {}, data.Layout || {});
        });
    };

    const _Scene_Base_createWindowLayer = Scene_Base.prototype.createWindowLayer;
    Scene_Base.prototype.createWindowLayer = function () {
        if (this instanceof Scene_Message) {
            this.addExtraGauge();
        }
        _Scene_Base_createWindowLayer.apply(this, arguments);
    };

    const _Scene_Base_start = Scene_Base.prototype.start;
    Scene_Base.prototype.start = function () {
        _Scene_Base_start.apply(this, arguments);
        this.addExtraGauge();
    };

    Scene_Base.prototype.addExtraGauge = function () {
        if (this._extraGaugesAdd) {
            return;
        }
        this._extraGauges.forEach(extraGauge => {
            this.addChild(extraGauge);
        });
        this._extraGaugesAdd = true;
    };

    Scene_Base.prototype.findExtraGaugeList = function () {
        const currentSceneName = PluginManagerEx.findClassName(this);
        return (param.GaugeList || []).filter(function (data) {
            return data.SceneName === currentSceneName;
        }, this);
    };

    const _Sprite_Gauge_initialize = Sprite_Gauge.prototype.initialize;
    Sprite_Gauge.prototype.initialize = function (data, detail, layout) {
        if (data) {
            this._data = data;
            this._detail = detail;
            this._layout = layout;
        }
        this._statusType = 'hp';
        _Sprite_Gauge_initialize.apply(this, arguments);
    };


    /**
     * Sprite_ExtraGaugeContainer
     * 追加ゲージとピクチャを含むコンテナです。
     */
    class Sprite_ExtraGaugeContainer extends Sprite {
        constructor(data, detail, layout) {
            super();
            this._data = data;
            this._detail = detail;
            this._layout = layout;
            this._openness = 0;
            this._tempDispDuration = 0;
            this.create();
        }

        create() {
            if (this._data.isBalance) {
                this._gauge = new Sprite_ExtraBalanceGauge(this._data, this._detail, this._layout);
            } else if (this._data.isSperms) {
                this._gauge = new Sprite_SpermsGauge(this._data, this._detail, this._layout);
            }
            else {
                this._gauge = new Sprite_ExtraGauge(this._data, this._detail, this._layout);
            }
            this._lower = this.createPicture(this._data.LowerPicture);
            this.addChild(this._gauge);
            this._upper = this.createPicture(this._data.UpperPicture);
            this._iconSprite = this.createIcon(this._detail.icon, this.x, this.y);
            if (this._iconSprite)
                this._gauge.move(ImageManager.iconWidth, 4)
            this.setupPosition();

            this.update();
        }

        createIcon(iconIndex, x, y) {
            if (!iconIndex) return null;
            const bitmap = ImageManager.loadSystem("IconSet");
            const pw = ImageManager.iconWidth;
            const ph = ImageManager.iconHeight;
            const sx = (iconIndex % 16) * pw;
            const sy = Math.floor(iconIndex / 16) * ph;
            const iconBitmap = new Bitmap(pw, ph);
            iconBitmap.blt(bitmap, sx, sy, pw, ph, x, y);
            const iconSprite = new Sprite(iconBitmap);
            this.addChild(iconSprite);
            return iconSprite;
        };

        setupPosition() {
            if (this._layout.AlignCol && this._layout.AlignRow) {
                this.x = +param.alignGaugeBaseX + (294 * (this._layout.AlignCol - 1));
                this.y = +param.alignGaugeBaseY + (42 * (this._layout.AlignRow - 1));
                return;
            }
            this.x = this._gauge.findLayoutValue(this._layout.x);
            this.y = this._gauge.findLayoutValue(this._layout.y);
        }

        update() {
            super.update();
            if (this._layout.realTime) {
                this.setupPosition();
            }
            // this.updateVisibly();
            this.updateOpacity();
        }

        // updateVisibly() {
        //     this.visible = this.isVisible();
        // }

        updateOpacity() {
            this._openness = (this._openness + this.calcDeltaOpenness()).clamp(0, 1);
            this.opacity = 255 * this._openness;
        }

        calcDeltaOpenness() {
            const openness = 1 / (this._detail.fadeFrame || 1);
            return this.isVisible() ? openness : -openness;
        }

        isVisible() {
            if (this._detail.temporaryDisp) {
                return Graphics.frameCount <= this._tempDispDuration;
            }
            return !this._data.SwitchId || $gameSwitches.value(this._data.SwitchId);
        }

        createPicture(pictureData) {
            if (!pictureData || !pictureData.FileName) {
                return null;
            }
            const sprite = new Sprite();
            // sprite.anchor.x = 0.5;
            // sprite.anchor.y = 0.5;
            sprite.bitmap = ImageManager.loadPicture(pictureData.FileName);
            sprite.x = pictureData.OffsetX || 0;
            sprite.y = pictureData.OffsetY || 0;
            this.addChild(sprite);
            return sprite;
        }
    }

    /**
     * Sprite_ExtraGauge
     * 追加ゲージを扱うクラスです。
     */
    class Sprite_ExtraGauge extends Sprite_Gauge {
        constructor(data, detail, layout) {
            super(data, detail, layout);
            this.setup(this.findBattler(), this._detail.GaugeColorPreset);
            this.setupPosition();
        }

        findBattler() {
            const battlerData = this._data.Battler;
            if (!battlerData) {
                return $gameParty.menuActor();
            }
            const methodName = `findBattler${battlerData.Type}`;
            if (this[methodName]) {
                return this[methodName](battlerData);
            } else {
                return $gameParty.menuActor();
            }
        }

        findBattlerActorId(battlerData) {
            return $gameActors.actor(battlerData.ActorId);
        }

        findBattlerPartyIndex(battlerData) {
            return $gameParty.members()[battlerData.Index];
        }

        findBattlerEnemyId(battlerData) {
            return new Game_Enemy(battlerData.EnemyId, 0, 0);
        }

        findBattlerTroopIndex(battlerData) {
            return $gameTroop.members()[battlerData.Index];
        }

        updateBitmap() {
            const value = this.currentValue();
            const maxValue = this.currentMaxValue();
            if (value !== this._targetValue || maxValue !== this._targetMaxValue) {
                this.updateTargetValue(value, maxValue);
                if (this.parent)
                    this.parent._tempDispDuration = Graphics.frameCount + this._detail.temporaryDispDuration;
            }

            const visible = this.parent ? this.parent.isVisible() : false;

            if (!visible) {
                this._prevVisible = visible;
                return;
            }
            else if (!this._detail.temporaryDisp && !this._prevVisible) {
                this._value = this._targetValue;
                this._maxValue = this._targetMaxValue;
            }
            this.updateGaugeAnimation();
            this.updateFlashing();
            this._prevVisible = visible;
        }

        updateFlashing() {
            if (!this._detail.FlashIfFull) {
                return;
            }
            if (this.isFull()) {
                this._flashingCount++;
                if (this._flashingCount % 20 < 10) {
                    this.setBlendColor(this.flashingColor1());
                } else {
                    this.setBlendColor(this.flashingColor2());
                }
            } else {
                this.setBlendColor([0, 0, 0, 0]);
            }
        }

        flashingColor1() {
            return [255, 0, 100, 96];
        }

        flashingColor2() {
            return [255, 0, 100, 64];
        }

        isFull() {
            return this._value >= this._maxValue;
        }

        setupPosition() {
            // this.anchor.x = 0.5;
            // this.anchor.y = 0.5;
            if (this._layout.Vertical) {
                this.rotation = (270 * Math.PI) / 180;
            }
        }

        bitmapWidth() {
            return this.findLayoutValue(this._layout.width) || super.bitmapWidth();
        }

        bitmapHeight() {
            return this.findLayoutValue(this._layout.height) || super.bitmapHeight();
        }

        textHeight() {
            return this.bitmapHeight();
        }

        gaugeHeight() {
            return this.findLayoutValue(this._layout.GaugeHeight) || this.bitmapHeight();
        }

        gaugeX() {
            return this.findLayoutValue(this._layout.GaugeX) || 0;
        }

        findLayoutValue(value) {
            if (isNaN(value)) {
                try {
                    const width = $dataSystem.advanced.uiAreaWidth;
                    const height = $dataSystem.advanced.uiAreaHeight;
                    return eval(value);
                } catch (e) {
                    console.error(e);
                    return 0;
                }
            } else {
                return value;
            }
        }

        currentValue() {
            return this.findValue(this._data.CurrentMethod);
        }

        currentMaxValue() {
            return Math.max(this.findValue(this._data.MaxMethod), 1)
        }

        findValue(method) {
            if (!method) {
                return 0;
            } else if (method.VariableId) {
                return $gameVariables.value(method.VariableId)
            } else if (method.Script) {
                const battler = this._battler;
                const meta = battler.isActor() ? battler.actor().meta : battler.enemy().meta;
                try {
                    return eval(method.Script);
                } catch (e) {
                    console.error(e);
                    return 0;
                }
            } else {
                return method.FixedValue;
            }
        }

        label() {
            return this._detail.Label || '';
        }

        labelColor() {
            return this.findColor(this.findLabelFont().Color, super.labelColor());
        }

        labelOutlineColor() {
            return this.findColor(this.findLabelFont().OutlineColor, super.labelOutlineColor());
        }

        labelOutlineWidth() {
            return this.findLabelFont().OutlineWidth || super.labelOutlineWidth();
        }

        labelFontFace() {
            return this.findLabelFont().Face || super.labelFontFace();
        }

        labelFontSize() {
            return this.findLabelFont().Size || super.labelFontSize();
        }

        findLabelFont() {
            return this._detail.LabelFont || {};
        }

        valueColor() {
            return this.findColor(this.findValueFont().Color, super.valueColor());
        }

        valueOutlineColor() {
            return this.findColor(this.findValueFont().OutlineColor, super.valueOutlineColor());
        }

        valueOutlineWidth() {
            return this.findValueFont().OutlineWidth || super.valueOutlineWidth();
        }

        valueFontFace() {
            return this.findValueFont().Face || super.valueFontFace();
        }

        valueFontSize() {
            return this.findValueFont().Size || super.valueFontSize();
        }

        findValueFont() {
            return this._detail.ValueFont || {};
        }

        gaugeBackColor() {
            return this.findColor(this._detail.BackColor, super.gaugeBackColor());
        }

        gaugeColor1() {
            return this.findColor(this._detail.GaugeColorLeftLeft, super.gaugeColor1());
        }

        gaugeColor2() {
            return this.findColor(this._detail.GaugeColorLeftRight, super.gaugeColor2());
        }

        isValid() {
            return true;
        }

        smoothness() {
            if (this._value <= this._targetValue) {
                return this._detail.RisingSmoothness || 1;
            } else {
                return this._detail.FallingSmoothness || 1;
            }
        }

        drawValue() {
            if (this._detail.DrawMaxValue) {
                this.drawValueWithMax();
            }
            else if (this._detail.DrawValue) {
                super.drawValue();
            }
        }

        drawValueWithMax() {
            const currentValue = this.currentValue();
            const maxValue = this.currentMaxValue();
            const width = this.bitmapWidth();
            const height = this.textHeight() - 4;
            const marginX = -12;
            this.setupValueFont();
            // this.bitmap.fontSize = $gameSystem.mainFontSize() - 8;
            const maxValueWidth = Math.floor(this.bitmap.measureTextWidth(maxValue));
            this.bitmap.drawText(currentValue + "/", maxValueWidth * -1 + marginX, +4, width, height, "right");
            this.bitmap.drawText(maxValue, marginX, +4, width, height, "right");
            this.bitmap.fontSize = this.valueFontSize();
        };


        drawGaugeRect(x, y, width, height) {
            const rate = this.gaugeRate();
            const fillW = Math.floor((width - 2) * rate);
            const fillH = height - 2;
            const color0 = this.gaugeBackColor();
            const color1 = this.gaugeColor1();
            const color2 = this.gaugeColor2();
            // this.bitmap.fillRect(x, y, width, height, color0);
            // this.bitmap.gradientFillRect(x + 1, y + 1, fillW, fillH, color1, color2);

            // solid
            this.bitmap.fillTrapEx(x, y, width, height, color0, color0, '|', ')');
            // this.bitmap.gradientFillRect(x + 1, y + 1, fillW, fillH, color1, color2, );
            this.bitmap.fillTrapEx(x + 1, y + 1, fillW, fillH, color1, color2, '|', '|', "atop");
        };

        findColor(code, defaultColor = null) {
            if (!code) {
                return defaultColor ? defaultColor : ColorManager.normalColor();
            } else if (isNaN(code)) {
                return code;
            } else {
                return ColorManager.textColor(code);
            }
        }
    }


    /**
     * Sprite_SpermsGauge
     */
    class Sprite_SpermsGauge extends Sprite_ExtraGauge {
        constructor(data, detail, layout) {
            super(data, detail, layout);
            this.frmCnt = 0;
            this.prev = 0;
        }

        gaugeRate = function () {
            if (this.isValid()) {
                const value = this._value;
                const maxValue = this._maxValue;
                const rate = value > maxValue ? maxValue : easeOutSine(value / maxValue)
                return maxValue > 0 ? rate : 0;
            } else {
                return 0;
            }
        };

        currentValue() {
            return this._battler.sex.uterus.spermSum;
        }

        currentMaxValue() {
            return this._battler.sex.uterus._capacity
        }

        actualMaxVale() {
            return this._battler.sex.uterus._aCapacity
        }

        findValue(method) {
            if (!method) {
                return 0;
            } else if (method.VariableId) {
                return $gameVariables.value(method.VariableId)
            } else if (method.Script) {
                const battler = this._battler;
                const meta = battler.isActor() ? battler.actor().meta : battler.enemy().meta;
                try {
                    return eval(method.Script);
                } catch (e) {
                    console.error(e);
                    return 0;
                }
            } else {
                return method.FixedValue;
            }
        }

        isValid() {
            return true;
        }

        smoothness() {
            if (this._battler.sex.uterus.hasPenisIn()) return 1;
            if (this._value <= this._targetValue) {
                return this._detail.RisingSmoothness || 1;
            } else {
                return this._detail.FallingSmoothness || 1;
            }
        }

        gaugeOutlineColor() {
            return 'rgba(210,30,60,255)';
        };

        drawGaugeRect(x, y, width, height) {
            width /= 2, height /= 2;
            x += width / 2, y += height / 2;
            const rate = this.gaugeRate();
            var fillW = Math.floor(width * rate);
            const color0 = this.gaugeBackColor();
            const color1 = this.gaugeColor1();
            const color2 = this.gaugeColor2();
            const color3 = this.gaugeOutlineColor();
            const infrateRate = this.currentValue() / this.currentMaxValue()
            let c = 1 - this._battler.sex.uterus.penisDepth;
            if (c <= 0) {
                x -= (100 * c) / 2;
                y -= (100 * c) / 2;
                width += (100 * c);
                height += (100 * c);
            }

            if (infrateRate > 1) {
                x -= (width * infrateRate - width) / 2;
                y -= (height * infrateRate - height) / 2
                width *= infrateRate;
                height *= infrateRate;
            }


            this.bitmap.fillTrap(x + 2, y + 2, width - 4, height - 4, 'rgb(130,30,50)', color0);
            this.bitmap.fillTrapInternal(x + 2, y + 2, fillW - 4, height - 4, color1, color2, "atop", this.frmCnt);
            this.bitmap.outlineTrap(x + 2, y + 2, width - 4, height - 4, color3, color3, "desover");
            // this.bitmap.context.shadowColor = 'rgb(150,30,60)';
            // this.bitmap.context.shadowOffsetX = 0; //影のずらし量(X)
            // this.bitmap.context.shadowOffsetY = 0; //影のずらし量(Y)
            // this.bitmap.context.shadowBlur = 100; //影のぼけ量
            this.bitmap.fillTrap(x, y, width, height, "#00000000", "#00000000");
            this.frmCnt++
        };

        updateGaugeAnimation() {
            if (this._duration > 0) {
                const d = this._duration;
                this._value = (this._value * (d - 1) + this._targetValue) / d;
                this._maxValue = (this._maxValue * (d - 1) + this._targetMaxValue) / d;
                this._duration--;
                this.redraw();
            } else {
                this._value = this._targetValue;
                this._maxValue = this._targetMaxValue;
                this.redraw();
            }
        };

        isDisplay() {
            return $gameSwitches.value(this._data.SwitchId);
        }

        isVisible() {
            return this.opacity > 0;
        }

    }

    class Sprite_ExtraBalanceGauge extends Sprite_ExtraGauge {
        constructor(data, detail, layout) {
            super(data, detail, layout);
        }

        drawValue() {
            if (this._detail.DrawValueAsText) {
                this.drawValueAsText();
                return
            }
            if (this._detail.DrawValue) {
                this.drawValueOrg();
            }
        };

        drawValueAsText() {
            const currentValue = this.currentValue();
            const width = this.bitmapWidth();
            const height = this.bitmapHeight();
            const settings = this._detail.ValueAsTextSettings
            const setting = settings.find(s => {
                const range = s.range.split(",").map(elm => +elm)
                return this.isBetween(range[0], range[1], currentValue)
            })
            if (!setting) return;
            this.setupValueFont();
            this.bitmap.drawText(setting.valueText, 0, 0, width, height, "center");
        }

        isBetween(lower, upper, subject) {
            return subject >= lower && subject <= upper;
        }

        drawValueOrg() {
            const currentValue = this.currentValue();
            const currentmax = this.currentMaxValue();
            const currentmax2 = this.currentMaxValue2();
            const width = this.bitmapWidth();
            const height = this.bitmapHeight();
            this.setupValueFont();
            this.bitmap.drawText(currentValue, 0, 0, width, height, "center");
            if (this._detail.DrawMaxValue) {
                this.bitmap.drawText(currentmax, 0, 0, width, height, "left");
                this.bitmap.drawText(currentmax2, 0, 0, width, height, "right");
            }
        }

        drawGaugeRect = function (x, y, width, height) {
            // const rate = this.gaugeRate();
            const fillH = height - 2;
            const color0 = this.gaugeBackColor();
            const half = width / 2;
            let width_RtoL = -half;
            let width_LtoR = half;

            width_LtoR += (half * (this._value / this.currentMaxValue()));
            width_RtoL -= (half * (this._value / this.currentMaxValue2()));

            const leftColor1 = this.GaugeColorLeftLeft();
            const leftColor2 = this.GaugeColorLeftRight();
            const rightColor1 = this.GaugeColorRightLeft();
            const rightColor2 = this.GaugeColorRightRight();

            this.bitmap.fillRect(x, y, width, height, color0);
            this.bitmap.gradientFillRect(width - 1, y + 1, width_RtoL, fillH, rightColor1, rightColor2);
            this.bitmap.gradientFillRect(x + 1, y + 1, width_LtoR, fillH, leftColor1, leftColor2);
        };

        flashingColor1() {
            return [255, 0, 100, 96];
        }

        flashingColor2() {
            return [255, 0, 100, 64];
        }

        isFull() {
            return this._value - this.currentMaxValue() >= 0 || this._value + this.currentMaxValue2() <= 0

            // return this._value - $gameVariables.value(3) >= 0 || this._value + $gameVariables.value(4) <= 0
        }

        currentValue() {
            return this.findValue(this._data.CurrentMethod);
        }

        currentMaxValue() {
            return Math.max(this.findValue(this._data.MaxMethod), 1)
        }

        currentMaxValue2() {
            return this.findValue(this._data.MaxMethod2)
        }

        GaugeColorLeftLeft() {
            return this.findColor(this._detail.GaugeColorLeftLeft, super.gaugeColor1());
        }

        GaugeColorLeftRight() {
            return this.findColor(this._detail.GaugeColorLeftRight, super.gaugeColor2());
        }

        GaugeColorRightLeft() {
            return this.findColor(this._detail.GaugeColorRightLeft, super.gaugeColor1());
        }

        GaugeColorRightRight() {
            return this.findColor(this._detail.GaugeColorRightRight, super.gaugeColor2());
        }
    }

    //Bitmap
    Bitmap.prototype.fillTrap = function (x, y, width, height, color1, color2, atop) {
        const context = this._context;
        // const grad = context.createLinearGradient(x, y, x + width, y);
        const grad = context.createRadialGradient(width / 4 * 3, height, 50, width / 4 * 3, height, width / 4 * 3);
        if (atop) {
            context.globalCompositeOperation = 'source-atop';
        } else {
            context.globalCompositeOperation = 'source-over';
        }
        let startCoords = [];

        grad.addColorStop(0, color1);
        grad.addColorStop(1, color2);

        context.save();
        context.beginPath();

        context.arc(x + width / 2, y + height / 2, width / 2, 0, 2 * Math.PI);

        // startCoords = [x + height, y + height]
        // context.moveTo(x + height, y + height);
        // context.bezierCurveTo(x, y + height, x, y, x + height, y);
        // context.lineTo(x + width - height, y);
        // context.bezierCurveTo(x + width, y, x + width, y + height, x + width - height, y + height);

        // context.lineTo(startCoords[0], startCoords[1])
        context.fillStyle = grad;
        context.fill();
        context.restore();
        this._baseTexture.update();
    };

    Bitmap.prototype.fillTrapInternal = function (x, y, width, height, color1, color2, atop, frmCnt) {
        const context = this._context;
        const grad = context.createLinearGradient(x, y, x + width, y);
        if (atop) {
            context.globalCompositeOperation = 'source-atop';
        } else {
            context.globalCompositeOperation = 'source-over';
        }
        let startCoords = [];

        grad.addColorStop(0, color1);
        grad.addColorStop(1, color2);


        context.save();
        context.beginPath();

        startCoords = [x, y + height]
        context.moveTo(x, y + height)
        context.lineTo(x, y)
        context.lineTo(x + width, y)

        // ----
        let div = 100;
        for (var i = 1; i <= div; i++) {
            var noise = (PerlinNoise.noise(frmCnt * 0.011, i * 0.011, 0) - 0.5) * 4;
            // var noise = (PerlinNoise.noise(frmCnt * 0.01, i * 0.01, 0) - 0.5) * 5;
            context.lineTo((x + width) + 5 * noise, (y + height) / div * i);
        }
        // context.lineTo(x + width, y + height)

        // ----

        context.lineTo(startCoords[0], startCoords[1])
        context.fillStyle = grad;
        context.fill();
        context.restore();
        this._baseTexture.update();
    };

    Bitmap.prototype.outlineTrap = function (x, y, width, height, color1, color2, desover) {
        const context = this._context;
        const grad = context.createRadialGradient(0, 0, 100, 0, 0, width / 2);
        if (desover) {
            context.globalCompositeOperation = 'destination-over';
        } else {
            context.globalCompositeOperation = 'source-over';
        }
        let startCoords = [];

        grad.addColorStop(0, color1);
        grad.addColorStop(1, color2);

        context.save();
        context.beginPath();

        context.arc(x + width / 2, y + height / 2, width / 2, Math.PI, Math.PI - 0.01);

        context.strokeStyle = grad;
        context.lineWidth = 20.0;
        context.stroke();
        context.restore();
        this._baseTexture.update();
    };


})();
