/*:ja
 * @target MZ
 * @author fude
 * @help
 * 本プラグインは、制作者が明示的に許可した場合を除き、いかなる形であれ使用、複製、改変、再配布することを禁じます。
 * 本プラグインを無断で使用したこと、またはその利用によって生じた不具合・損害について、制作者は一切の責任を負いません。
 * @plugindesc 汎用処理
 * 
 * @command setItemIdMode
 * @text アイテムID格納
 * @desc 変数の操作：アイテム数の格納をアイテムIDの格納をするモードに設定する
 * 
 * @arg mode
 * @text モード
 * @type boolean
 * @default true
 * 
 */

class PluginParamsParser {
    static parse(params, typeData, predictEnable = true) {
        return new PluginParamsParser(predictEnable).parse(params, typeData);
    }

    constructor(predictEnable = true) {
        this._predictEnable = predictEnable;
    }

    parse(params, typeData, loopCount = 0) {
        if (++loopCount > 255) throw new Error("endless loop error");
        const result = {};
        for (const name in typeData) {
            result[name] = this.convertParam(params[name], typeData[name], loopCount);
        }
        if (!this._predictEnable) return result;
        if (typeof params === "object" && !(params instanceof Array)) {
            for (const name in params) {
                if (result[name]) continue;
                const param = params[name];
                const type = this.predict(param);
                result[name] = this.convertParam(param, type, loopCount);
            }
        }
        return result;
    }

    convertParam(param, type, loopCount) {
        if (typeof type === "string") {
            return this.cast(param, type);
        } else if (typeof type === "object" && type instanceof Array) {
            const aryParam = JSON.parse(param);
            if (typeof type[0] === "string") {
                return aryParam.map(strParam => this.cast(strParam, type[0]));
            } else {
                return aryParam.map(strParam => this.parse(JSON.parse(strParam), type[0]), loopCount);
            }
        } else if (typeof type === "object") {
            return this.parse(JSON.parse(param), type, loopCount);
        } else {
            throw new Error(`${type} is not string or object`);
        }
    }

    cast(param, type) {
        switch (type) {
            case "any":
                if (!this._predictEnable) throw new Error("Predict mode is disable");
                return this.cast(param, this.predict(param));
            case "string":
                return param;
            case "number":
                if (param.match(/\d+\.\d+/)) return parseFloat(param);
                return parseInt(param);
            case "boolean":
                return param === "true";
            default:
                throw new Error(`Unknow type: ${type}`);
        }
    }

    predict(param) {
        if (param.match(/^\d+$/) || param.match(/^\d+\.\d+$/)) {
            return "number";
        } else if (param === "true" || param === "false") {
            return "boolean";
        } else {
            return "string";
        }
    }
}


(function () {
    'use strict';
    const pluginName = 'GeneralPurpose';

    //=============================================================================
    // アイテムID格納モード
    //=============================================================================
    PluginManager.registerCommand(pluginName, 'setItemIdMode', function (args) {
        const mode = JSON.parse(args.mode);
        this._setItemIdMode = !!mode;
    });

    const game_interpreter_clear = Game_Interpreter.prototype.clear;
    Game_Interpreter.prototype.clear = function () {
        game_interpreter_clear.apply(this, arguments);
        this._setItemIdMode = false;
    };

    const gameInterpreter_gameDataOperand = Game_Interpreter.prototype.gameDataOperand;
    Game_Interpreter.prototype.gameDataOperand = function (type, param1, param2) {
        if (!this._setItemIdMode)
            return gameInterpreter_gameDataOperand.apply(this, arguments);
        if (type === 0)
            return $dataItems[param1].id;
    }

    //=============================================================================
    // マップ右端にスペースを作る
    //=============================================================================
    // override
    Game_Map.prototype.setDisplayPos = function (x, y) {
        if (this.isLoopHorizontal()) {
            this._displayX = x.mod(this.width());
            this._parallaxX = x;
        } else {
            const endX = $dataMap.meta.makeSpaceRightDisable ?
                this.width() - this.screenTileX() :
                this.width() - this.screenTileX() + 7
            this._displayX = endX < 0 ? endX / 2 : x.clamp(0, endX);
            this._parallaxX = this._displayX;
        }
        if (this.isLoopVertical()) {
            this._displayY = y.mod(this.height());
            this._parallaxY = y;
        } else {
            const endY = this.height() - this.screenTileY();
            this._displayY = endY < 0 ? endY / 2 : y.clamp(0, endY);
            this._parallaxY = this._displayY;
        }
    };

    // override
    Game_Map.prototype.scrollLeft = function (distance) {

        const needMakeSpace = !$dataMap.meta.makeSpaceRightDisable;

        if (this.isLoopHorizontal()) {
            this._displayX += $dataMap.width - distance;
            this._displayX %= $dataMap.width;
            if (this._parallaxLoopX) {
                this._parallaxX -= distance;
            }
        } else if (this.width() >= this.screenTileX() ||
            (needMakeSpace && this.width() + 7 >= this.screenTileX())) {
            const lastX = this._displayX;
            this._displayX = Math.max(this._displayX - distance, 0);
            this._parallaxX += this._displayX - lastX;
        }
    };

    // override
    Game_Map.prototype.scrollRight = function (distance) {

        const needMakeSpace = !$dataMap.meta.makeSpaceRightDisable;

        if (this.isLoopHorizontal()) {
            this._displayX += distance;
            this._displayX %= $dataMap.width;
            if (this._parallaxLoopX) {
                this._parallaxX += distance;
            }
        } else if (this.width() >= this.screenTileX() ||
            (needMakeSpace && this.width() + 7 >= this.screenTileX())) {
            const lastX = this._displayX;
            this._displayX = Math.min(
                this._displayX + distance,
                needMakeSpace ?
                    this.width() - this.screenTileX() + 7 :
                    this.width() - this.screenTileX()
            );

            this._parallaxX += this._displayX - lastX;
        }
    };


    //=============================================================================
    // 色調をリフレッシュ
    //=============================================================================
    // Game_Map
    const _Game_Map_setup = Game_Map.prototype.setup;
    Game_Map.prototype.setup = function (mapId) {
        _Game_Map_setup.apply(this, arguments);
        if ($dataMap.meta?.noTint)
            $gameScreen.clearTone();
        else
            $gameSystem.chronus().refreshTint();
    };


    //=============================================================================
    // 制作用マーカーを非表示
    //=============================================================================
    const _Sprite_Character_SetCharacterBitmap = Sprite_Character.prototype.setCharacterBitmap;
    Sprite_Character.prototype.setCharacterBitmap = function () {
        if (this._characterName.contains('$marker')) {
            this.bitmap = new Bitmap();
            this._isBigCharacter = false;
        } else {
            _Sprite_Character_SetCharacterBitmap.call(this);
        }
    };


    //=============================================================================
    // メニュー呼び出し時のSE変更
    //=============================================================================
    Scene_Map.prototype.callMenu = function () {
        AudioManager.playSe({ name: '/beemyu/yumekawa/myuu_077_YumeSE_BattleMagic01', volume: 90, pitch: 200 });
        SceneManager.push(Scene_Menu);
        Window_MenuCommand.initCommandPosition();
        $gameTemp.clearDestination();
        this._mapNameWindow.hide();
        this._waitCount = 2;
    };

    //=============================================================================
    // ニューゲーム押下時SE変更
    //=============================================================================
    Scene_Title.prototype.commandNewGame = function () {
        AudioManager.playSe({ name: '/beemyu/yumekawa/myuu_094_YumeSE_MagicFlash02', volume: 90, pitch: 100 });
        DataManager.setupNewGame();
        this._commandWindow.close();
        this.fadeOutAll();
        SceneManager.goto(Scene_Map);
    };

    //=============================================================================
    // メニューにロードコマンドを追加
    //=============================================================================
    var _Scene_Menu_createCommandWindow = Scene_Menu.prototype.createCommandWindow;
    Scene_Menu.prototype.createCommandWindow = function () {
        _Scene_Menu_createCommandWindow.call(this);
        this._commandWindow.setHandler('load', this.commandLoad.bind(this));
    };

    Scene_Menu.prototype.commandLoad = function () { //新規
        SceneManager.push(Scene_Load);
    };

    var _Window_MenuCommand_addSaveCommand = Window_MenuCommand.prototype.addSaveCommand;
    Window_MenuCommand.prototype.addSaveCommand = function () {
        _Window_MenuCommand_addSaveCommand.call(this);
        this.addCommand('ロード', 'load', true);
    }

    //=============================================================================
    // イベント斜め移動時歩行グラ表示不具合の対応
    //=============================================================================
    Sprite_Character.prototype.characterPatternY = function () {
        const direction = this.character().direction();
        const deg = DotMoveSystem.Degree.fromDirection(direction);
        const direction4 = deg.toDirection4(direction);
        return (direction4 - 2) / 2;
    };

    Sprite_Character.prototype.character = function () {
        return this._character
    };

    //=============================================================================
    // バルーン表示時SEを自動演奏
    //=============================================================================
    Game_Temp.prototype.requestBalloon = function (target, balloonId, seMute) {
        const request = { target: target, balloonId: balloonId, seMute };
        this._balloonQueue.push(request);
        if (target.startBalloon) {
            target.startBalloon();
        }
    };

    const _Spriteset_Map_createBalloon = Spriteset_Map.prototype.createBalloon;
    Spriteset_Map.prototype.createBalloon = function (request) {
        _Spriteset_Map_createBalloon.apply(this, arguments);
        if (!!request.seMute) return;
        let se;
        const folder = 'balloon/';
        switch (request.balloonId) {
            case 1: // びっくり
                se = { name: folder + 'Bikkuri', volume: 50, pitch: 150 }
                break;
            case 2: // はてな
                se = { name: folder + 'Hatena', volume: 50, pitch: 100 }
                break;
            case 3: // 音符
                break;
            case 4: // ハート
                se = { name: folder + 'Heart', volume: 50, pitch: 100 }
                break;
            case 5: // 怒り
                se = { name: 'Monster1', volume: 30, pitch: 140 }
                break;
            case 6: // 汗
                se = { name: folder + 'Ase', volume: 50, pitch: 150 }
                break;
            case 7: // くしゃくしゃ
                se = { name: folder + 'Kusya', volume: 50, pitch: 100 }
                break;
            case 8: // 沈黙
                se = { name: folder + 'Silence', volume: 50, pitch: 150 }
                break;
            case 9: // 電球
                se = { name: folder + 'Denkyu', volume: 50, pitch: 100 }
                break;
            case 10: // ZZZ
                break;
            case 11: // 照れ
                se = { name: folder + 'Tere', volume: 35, pitch: 120 }
                break;
            case 14: // わいわい
                se = { name: folder + 'Waiwai', volume: 35, pitch: 120 }
                break;
            case 15: // あせあせ
                se = { name: folder + 'AseAse', volume: 35, pitch: 110 }
                break;

        }
        if (se)
            AudioManager.playSe(se);
    };


    //=============================================================================
    // アイテム選択リストの位置を調整
    //=============================================================================
    Window_EventItem.prototype.updatePlacement = function () {
        if (this._messageWindow.y >= Graphics.boxHeight / 2) {
            this.y = Graphics.boxHeight / 4;
        } else {
            this.y = Graphics.boxHeight - this.height;
        }
    };
    Scene_Message.prototype.eventItemWindowRect = function () {
        const wx = this._messageWindow.x;
        const wy = Graphics.boxHeight / 4;;
        const ww = Graphics.boxWidth / 2;
        const wh = this.calcWindowHeight(6, true);
        return new Rectangle(wx, wy, ww, wh);
    };

    //=============================================================================
    // セルフスイッチを移動ルートなどから操作する
    //=============================================================================
    Game_Event.prototype.setSelfSwitch = function (type, value) {
        key = [$gameMap.mapId(), this.eventId(), type];
        $gameSelfSwitches.setValue(key, value);
    }

    //=============================================================================
    // マップ移動時コモン呼び出し
    //=============================================================================
    const _Scene_Map_onTransfer = Scene_Map.prototype.onTransfer;
    Scene_Map.prototype.onTransfer = function () {
        _Scene_Map_onTransfer.apply(this, arguments);
        const CMNID_ON_TRANSFER = 7;
        $gameTemp.reserveCommonEvent(CMNID_ON_TRANSFER)
    };

    //=============================================================================
    // ゲージの形状
    //=============================================================================
    Bitmap.prototype.fillTrapEx = function (x, y, width, height, color1, color2, typeLeft, typeRight, atop) {
        const context = this._context;
        const grad = context.createLinearGradient(x, y, x + width, y);
        if (atop) {
            context.globalCompositeOperation = 'source-atop';
        } else {
            context.globalCompositeOperation = 'source-over';
        }
        let startCoords = [];

        grad.addColorStop(0, color1);
        grad.addColorStop(1, color2);

        context.save();
        context.beginPath();

        switch (typeLeft) {
            case "|":
                startCoords = [x, y + height]
                context.moveTo(x, y + height)
                context.lineTo(x, y)
                break;
            case "/":
                startCoords = [x, y + height]
                context.moveTo(x, y + height)
                context.lineTo(x + height, y)
                break;
            case "<":
                startCoords = [x + height / 2, y + height]
                context.moveTo(x + height / 2, y + height)
                context.lineTo(x, y + height / 2)
                context.lineTo(x + height / 2, y)
                break;
            case "(":
                startCoords = [x + height, y + height]
                context.moveTo(x + height, y + height);
                context.bezierCurveTo(x, y + height, x, y, x + height, y);
                break;
            case ".":
                startCoords = [x, y + height]
                context.moveTo(x, y + height)
                break;
            case "\\":
                startCoords = [x + height, y + height]
                context.moveTo(x + height, y + height)
                context.lineTo(x, y)
                break;
        };

        switch (typeRight) {
            case "|":
                context.lineTo(x + width, y)
                context.lineTo(x + width, y + height)
                break;
            case "/":
                context.lineTo(x + width, y)
                context.lineTo(x + width - height, y + height)
                break;
            case ">":
                context.lineTo(x + width - height / 2, y)
                context.lineTo(x + width, y + height / 2)
                context.lineTo(x + width - height / 2, y + height)
                break;
            case ")":
                context.lineTo(x + width - height, y);
                context.bezierCurveTo(x + width, y, x + width, y + height, x + width - height, y + height);
                break;
            case ".":
                context.lineTo(x + width, y + height)
                break;
            case "\\":
                context.lineTo(x + width - height, y)
                context.lineTo(x + width, y + height)
                break;
        };

        context.lineTo(startCoords[0], startCoords[1])
        context.fillStyle = grad;
        context.fill();
        context.restore();
        this._baseTexture.update();
    };

    Sprite_Gauge.prototype.bitmapWidth = function () {
        return 224;
    };


    //=============================================================================
    // 場所移動時の自動演奏制御
    //=============================================================================
    // Game_Map.prototype.autoplay = function () {

    //     if ($dataMap.autoplayBgm && !$gameSwitches.value(18)) {
    //         if ($gamePlayer.isInVehicle()) {
    //             $gameSystem.saveWalkingBgm2();
    //         } else {
    //             AudioManager.playBgm($dataMap.bgm);
    //         }
    //     }
    //     if ($dataMap.autoplayBgs) {
    //         AudioManager.playBgs($dataMap.bgs);
    //     }
    // };


    //=============================================================================
    // カメラ位置をずらす
    //=============================================================================
    Game_Map.prototype.scrollOffsetX = function () {
        return -1;
    }

    Game_Map.prototype.scrollOffsetY = function () {
        return 1;
    }

    Game_Player.prototype.centerX = function () {
        return ($gameMap.screenTileX() - 1) / 2.0 + $gameMap.scrollOffsetX();
    };

    Game_Player.prototype.centerY = function () {
        return ($gameMap.screenTileY() - 1) / 2.0 + $gameMap.scrollOffsetY();
    };

    Game_Character.prototype.centerX = function () {
        return ($gameMap.screenTileX() - 1) / 2.0 + $gameMap.scrollOffsetX();
    };

    Game_Character.prototype.centerY = function () {
        return ($gameMap.screenTileY() - 1) / 2.0 + $gameMap.scrollOffsetY();
    };


    //=============================================================================
    // マップ名の位置を調整
    //=============================================================================
    Scene_Map.prototype.mapNameWindowRect = function () {
        const wx = 0;
        const ww = Graphics.boxWidth;
        const wh = this.calcWindowHeight(1, false);
        const wy = Math.floor(Graphics.boxHeight / 2 - wh / 2);
        return new Rectangle(wx, wy, ww, wh);
    };


    Window_MapName.prototype.refresh = function () {
        this.contents.clear();
        const text = $gameMap.displayName();
        if (text) {
            const margin = 256;
            const height = this.lineHeight();
            const textWidth = this.textWidth(text);
            const textX = Graphics.boxWidth / 2 - textWidth / 2;
            this.drawBackground(textX - margin / 2, 0, textWidth + margin, height);
            this.drawText(text, textX, 0, textWidth);
        }
    };

    //=============================================================================
    // データベースの名前を参照可能にする
    //=============================================================================
    const _Window_Base_convertEscapeCharacters = Window_Base.prototype.convertEscapeCharacters;
    Window_Base.prototype.convertEscapeCharacters = function (text) {

        // Original Processing
        text = _Window_Base_convertEscapeCharacters.call(this, text);

        // Convert Data Name and Icon
        text = text.replace(/\x1b(\*)?(class|skill|item|itemGpCost|weapon|armor|enemy|troop|state)\[(\d+)\]/gi, function () {
            var text = '';
            var icon = arguments[1];
            var type = arguments[2].toLowerCase();
            var id = parseInt(arguments[3])
            // get object
            var obj = null;
            if (id >= 1) {
                switch (type) {
                    case 'class':
                        obj = $dataClasses[id];
                        break;
                    case 'skill':
                        obj = $dataSkills[id];
                        break;
                    case 'item':
                        obj = $dataItems[id];
                        break;
                    case 'itemgpcost':
                        const gpCost = $dataItems[id]?.meta.gpCost;
                        obj = { name: gpCost | '' };
                        break;
                    case 'weapon':
                        obj = $dataWeapons[id];
                        break;
                    case 'armor':
                        obj = $dataArmors[id];
                        break;
                    case 'enemy':
                        obj = $dataEnemies[id];
                        break;
                    case 'troop':
                        obj = $dataTroops[id];
                        break;
                    case 'state':
                        obj = $dataStates[id];
                        break;
                }
            }
            if (obj) {
                // get object name
                text = obj.name;
                // get object icon
                if (icon === '*') {
                    switch (type) {
                        case 'skill':
                        case 'item':
                        case 'weapon':
                        case 'armor':
                        case 'state':
                            text = "\x1bI[" + obj.iconIndex + "]" + text;
                    }
                }
            }
            return text;
        }.bind(this));

        return text;

    };


    //=============================================================================
    // 他マップからイベントを参照可能にする対応
    //=============================================================================
    Game_Interpreter.prototype.isOnCurrentMap = function () {
        return true;
    };


    //=============================================================================
    // ゲームオーバー
    //=============================================================================
    // Scene_Map.prototype.checkGameover = function () {
    // //     if (!$gameParty.isAllDead() || $gameMap.isEventRunning())
    // //         return
    // //     // SceneManager.goto(Scene_Gameover);
    // //     const CMNID_HP_ZERO = 60;
    // //     $gameTemp.reserveCommonEvent(CMNID_HP_ZERO);
    // //     $gameMap._interpreter.setupReservedCommonEvent();
    // // };

    //=============================================================================
    // レベルアップ表示
    //=============================================================================
    // override
    Game_Actor.prototype.displayLevelUp = function (newSkills) {
        const text = TextManager.levelUp.format(
            this._name,
            TextManager.level,
            this._level
        );
        addNoticeWindow(text);

    };

    // ハートをアイコン表示する
    const _PluginManagerEx_convertEscapeCharactersEx = PluginManagerEx.convertEscapeCharactersEx;
    PluginManagerEx.convertEscapeCharactersEx = function (text) {
        text = _PluginManagerEx_convertEscapeCharactersEx.call(this, text);
        const key = this._selfSwitchKey;
        if (!key) {
            return text;
        }
        text = text.replace(/♡/gi, (_, p1) => {
            return '\\i[15]';
        });
        return text;
    };


    //=============================================================================
    // WASD移動に対応
    //=============================================================================
    Input.keyMapper[65] = 'left'
    Input.keyMapper[68] = 'right'
    Input.keyMapper[83] = 'down'
    Input.keyMapper[87] = 'up'
    Input.keyMapper[69] = 'pagedown'

    // 前方のプレイヤーを検出
    Game_Event.prototype.seekPlayerForward = function () {
        const px = $gamePlayer.x;
        const py = $gamePlayer.y;
        const d = this.direction();
        let result = false;

        for (let i = 0; i < 6; i++) {
            let sx = this.x;
            let sy = this.y;
            switch (d) {
                case 2: sy += i;
                    break;
                case 4: sx -= i;
                    break;
                case 6: sx += i;
                    break;
                case 8: sy -= i;
                    break;
            }
            if (px === sx && py === sy) {
                result = true;
                break;
            }
            else if (!this.canPass(sx, sy))
                break;

        }
        this.setExSelfSwitchValue(14, result);
    }


    // Game_Player.prototype.realMoveSpeed = function () {
    //     return this._moveSpeed + (this.isDashing() ? 0.9 : 0);
    // };


    //=============================================================================
    // メニューのステータスを非表示
    //=============================================================================
    Scene_Menu.prototype.create = function () {
        Scene_MenuBase.prototype.create.call(this);
        this.createCommandWindow();
        this.createGoldWindow();
        // this.createStatusWindow();
    };


    Scene_Menu.prototype.start = function () {
        Scene_MenuBase.prototype.start.call(this);
        // this._statusWindow.refresh();
    };

    //=============================================================================
    // メニューにエロステータスを追加
    //=============================================================================
    const _Window_MenuCommand_addMainCommands = Window_MenuCommand.prototype.addMainCommands;
    Window_MenuCommand.prototype.addMainCommands = function () {
        _Window_MenuCommand_addMainCommands.apply(this, arguments);
        this.addCommand('性経験', "ero_status", true);
    };

    const _Scene_Menu_createCommandWindow_forErostatus = Scene_Menu.prototype.createCommandWindow;
    Scene_Menu.prototype.createCommandWindow = function () {
        _Scene_Menu_createCommandWindow_forErostatus.apply(this, arguments);
        this._commandWindow.setHandler("ero_status", (() => { SceneManager.callCustomMenu('Scene_EroStatus'); }).bind(this));

    };

    //=============================================================================
    // アクターセレクトの表示を調整
    //=============================================================================
    Window_StatusBase.prototype.placeBasicGauges = function (actor, x, y) {
        this.placeGauge(actor, "hp", x, y);
    };
    Window_StatusBase.prototype.drawActorLevel = function (actor, x, y) {
        // do nothing
    };

    Sprite_Gauge.prototype.currentValue = function () {
        if (this._battler) {
            switch (this._statusType) {
                case "hp":
                    return $gameVariables.value(27);
                case "mp":
                    return this._battler.mp;
                case "tp":
                    return this._battler.tp;
                case "time":
                    return this._battler.tpbChargeTime();
            }
        }
        return NaN;
    };

    Sprite_Gauge.prototype.currentMaxValue = function () {
        if (this._battler) {
            switch (this._statusType) {
                case "hp":
                    return $gameVariables.value(28);
                case "mp":
                    return this._battler.mmp;
                case "tp":
                    return this._battler.maxTp();
                case "time":
                    return 1;
            }
        }
        return NaN;
    };


})();