/*:
 * @plugindesc タイミングを合わせてボタン入力するタイミングバーを実行します。
 * 
 * @command HzTimingBar
 * @text タイミングバー起動
 * @desc　タイミングバー起動
 * 
 * @arg hit_area
 * @text ヒットエリア
 * @desc ヒット範囲の最小値・最大値を0〜100の間で設定。min-max 例）70-90
 * @type text
 * 
 * @arg critical_area
 * @text ヒットクリティカルエリア
 * @desc クリティカル範囲の最小値・最大値を0〜100の間で設定。min-max
 * @type text
 * 
 * @arg require_area
 * @text 必須エリア
 * @desc 入力必須範囲（複数可）の最小値・最大値を0〜100の間で設定。min-max,min-max,...
 * @type text
 * 
 * @arg barWidth
 * @text バーの幅
 * @type number
 * @default 500
 *
 * @arg duration
 * @text 実行秒数
 * @type number
 * @default 3
 * 
 * @arg require_easy
 * @text 必須エリアのミス許容数
 * @type number
 * @default 0
 * 
 * @arg missMax
 * @text ミス許容最大数
 * @type number
 * @default 0
 * 
 * @param var_no
 * @text 変数番号
 * @desc 結果を返す変数番号。ミスの場合は0・ヒットの場合は1・クリティカルの場合は2が返される。
 * @type variable
 * 
 * @param bar width
 * @desc バーの幅
 * @default 500
 *
 * @param required SE
 * @desc 必須エリアヒット時のSE
 * @type file
 * @dir audio/se
 * @default Bow3
 *
 * @param hit SE
 * @desc ヒットエリアヒット時のSE
 * @type file
 * @dir audio/se
 * @default Attack2
 * 
 * @param critical SE
 * @desc クリティカルエリアヒット時のSE
 * @type file
 * @dir audio/se
 * @default Attack3
 *
 * @param miss SE
 * @desc 入力時（失敗）のSE
 * @type file
 * @dir audio/se
 * @default Buzzer1
 * 
 * @help
 * タイミングを合わせてボタン入力するタイミングバーを実行します。
 * 
 *  プラグインコマンド:
 *   HzTimingBar [var_no] [hit_area] [critical_area] [require_area] [x] [y]  # コマンド入力起動
 *   
 *   [var_no]
 *    【必須】結果を返す変数番号。ミスの場合は0・ヒットの場合は1・クリティカルの場合は2が返される。
 *   [hit_area] 
 *    【必須】ヒット範囲の最小値・最大値を0〜100の間で設定。min-max
 *    　　例）70-90
 *        
 *   [critical_area]
 *    【任意】クリティカル範囲の最小値・最大値を0〜100の間で設定。min-max
 *    　　例）90-95
 *   [require_area]
 *    【任意】入力必須範囲（複数可）の最小値・最大値を0〜100の間で設定。min-max,min-max,...
 *           入力必須範囲で全てボタン入力しないとヒット範囲・クリティカル範囲でボタン入力してもミスになります。
 *    　　　　※ 必ずヒット範囲・クリティカル範囲より前になるように設定して下さい。
 *      例） [10-20]         # 必須エリアは10〜20の範囲
 *          [20-30,50-60]   # 必須エリアは20〜30・50〜60の範囲
 *          []              # 必須エリア無し
 *   [x]
 *     【任意】コマンドの表示位置を指定します。（デフォルトでは画面中央）
 *   [y]
 *     【任意】コマンドの表示位置を指定します。（デフォルトでは画面中央）
 *   
 *  コマンド例）
 *    HzTimingBar 1 70-90 90-95     # ヒット範囲は70-90、クリティカル範囲は90-95。結果は変数番号１にセットされる。
 *    HzTimingBar 1 70-90 90-95 [10-30,40-60]    
 *                                # ヒット範囲は70-90、クリティカル範囲は90-95。結果は変数番号１にセットされる。
 *                                # 10-30・40-60の両方の範囲内でボタン入力しないとミス。
 *    HzTimingBar 1 80-90 90-95 [10-20] 413 20
 *                                # ヒット範囲は80-90、クリティカル範囲は90-95。結果は変数番号１にセットされる。
 *                                # 10-20の範囲内でボタン入力しないとミス。
 *                                # コマンドの表示位置は画面中央上端。
 */

// TODO:状態異常の効果追加（エリア非表示・高速/低速）

(function () {
    const pluginName = 'HzTimingBar';
    var parameters = PluginManager.parameters('HzTimingBar');
    var requiredSe = parameters['required SE'];
    var hitSe = parameters['hit SE'];
    var criticalSe = parameters['critical SE'];
    var missSe = parameters['miss SE'];
    const VARID_TIMINGBAR_RESULT = +parameters.var_no

    PluginManager.registerCommand(pluginName, 'HzTimingBar', function (args) {
        const barWidth = Number(args.barWidth);
        const hitAreaParm = String(args.hit_area);
        const criticalAreaParm = String(args.critical_area);
        const requiredAreaParm = String(args.require_area);
        const duration = Number(args.duration);
        const requireEasy = Number(args.require_easy);
        const missMax = Number(args.missMax);

        const x = !!args.x ? Number(args.x) : Graphics.boxWidth / 2;
        const y = !!args.y ? Number(args.y) : Graphics.boxHeight / 2;

        const hitArea = hitAreaParm.split("-").map(elm => +elm);
        const criticalArea = criticalAreaParm.split("-").map(elm => +elm);
        const requiredAreas = requiredAreaParm ? requiredAreaParm.split(",").map(ra => {
            return ra.split("-").map(elm => +elm);
        }) : [];

        const a = { x, y, barWidth, hitArea, criticalArea, requiredAreas, duration, requireEasy, missMax }
        this._timingBar = new HzTimingBar(a);
        AudioManager.playSe({ name: 'erotic/puls_3', volume: 90, pitch: 100, pan: 0 });
        this.setWaitMode("hzTimingBar");
    });

    // 待機状態の更新用関数に機能追加
    var _Game_Interpreter_updateWaitMode = Game_Interpreter.prototype.updateWaitMode;
    Game_Interpreter.prototype.updateWaitMode = function () {
        var waiting = null;
        switch (this._waitMode) {
            case 'hzTimingBar':
                // 待機状態の更新
                // ※ waitingには必ずtrueかfalseをセットすること！
                waiting = this._timingBar.update();
                if (!waiting) {
                    // 終了処理
                    this._timingBar.terminate();
                    this._timingBar = null;
                    //決定キーのトリガーイベントが残っているので待つ
                    this.wait(5);
                }
                break;
        }
        if (waiting !== null) {
            if (!waiting) {
                this._waitMode = '';
            }
            return waiting;
        }
        return _Game_Interpreter_updateWaitMode.call(this);
    };

    // コマンド入力実行用クラス
    function HzTimingBar() {
        this.initialize.apply(this, arguments);
    }

    HzTimingBar.DELAY = 15;
    HzTimingBar.HEIGHT = 12;
    HzTimingBar.RADIUS = 4;
    HzTimingBar.SPEED = 1;
    HzTimingBar.RESULT_FAILURE = 0;
    HzTimingBar.RESULT_HIT = 1;
    HzTimingBar.RESULT_CRITICALHIT = 2;

    HzTimingBar.prototype.initialize = function (args) {
        const x = args.x;
        const y = args.y;
        const hitArea = args.hitArea;

        this.barWidth = args.barWidth;
        this.maxFrame = args.duration * 60
        this._frame = -HzTimingBar.DELAY;
        this.requireEasyNum = args.requireEasy;
        this.missCnt = 0;
        this.missCntMax = args.missMax;
        this._hitCntRequired = 0;

        const criticalArea = args.criticalArea ? args.criticalArea : [-1, 0];
        this._requiredAreas = args.requiredAreas ? args.requiredAreas : [];
        this.reqs = [];
        this.result = 0;
        this._requiredNeedNum = Math.max(this._requiredAreas.length - this.requireEasyNum, 0);

        // 描画コンテナ
        this._container = new Sprite();
        this._container.position.set(x - this.barWidth / 2, y - HzTimingBar.HEIGHT / 2);

        // 枠
        var barFrameBmp = new Bitmap(this.barWidth + 4, HzTimingBar.HEIGHT + 4);
        roundedRectangle(barFrameBmp.context, 2, 2, this.barWidth, HzTimingBar.HEIGHT, HzTimingBar.RADIUS);
        barFrameBmp.context.fillStyle = "#FFFFFF";
        barFrameBmp.context.lineWidth = 2;
        barFrameBmp.context.strokeStyle = "#000000";
        barFrameBmp.context.fill();
        barFrameBmp.context.stroke();
        var barFrame = new Sprite(barFrameBmp);
        barFrame.position.set(-2, -2);
        this._container.addChild(barFrame);

        // 必須エリア
        for (const requreSetting of this._requiredAreas) {
            var reqWidth = (requreSetting[1] - requreSetting[0]) / 100 * this.barWidth;
            var reqBmp = new Bitmap(reqWidth, HzTimingBar.HEIGHT - 1);
            reqBmp.context.fillStyle = "#43D197";
            reqBmp.context.rect(0, 1, reqWidth, HzTimingBar.HEIGHT - 1);
            reqBmp.context.fill();
            var req = new Sprite(reqBmp);
            req.position.set(requreSetting[0] / 100 * this.barWidth, 0);
            this._container.addChild(req);
            this.reqs.push(req);
        }

        // ヒットエリア
        var hitWidth = (hitArea[1] - hitArea[0]) / 100 * this.barWidth;
        var hitBmp = new Bitmap(hitWidth, HzTimingBar.HEIGHT - 1);
        hitBmp.context.fillStyle = "#EBCE41";
        hitBmp.context.rect(0, 1, hitWidth, HzTimingBar.HEIGHT - 1);
        hitBmp.context.fill();
        this.hit = new Sprite(hitBmp);
        this.hit.position.set(hitArea[0] / 100 * this.barWidth, 0);
        this._container.addChild(this.hit);

        // クリティカルエリア
        var criticalWidth = (criticalArea[1] - criticalArea[0]) / 100 * this.barWidth;
        var criticalBmp = new Bitmap(criticalWidth, HzTimingBar.HEIGHT - 1);
        criticalBmp.context.fillStyle = "#E47237";
        criticalBmp.context.rect(0, 1, criticalWidth, HzTimingBar.HEIGHT - 1);
        criticalBmp.context.fill();
        this.critical = new Sprite(criticalBmp);
        this.critical.position.set(criticalArea[0] / 100 * this.barWidth, 0);
        this._container.addChild(this.critical);

        // カーソル
        var cursorBmp = new Bitmap(18, 32 + HzTimingBar.HEIGHT + 2);
        cursorBmp.context.fillStyle = "#000000";
        cursorBmp.context.strokeStyle = "#FFFFFF";
        cursorBmp.context.lineWidth = 1;
        cursorBmp.context.moveTo(1, 1);
        cursorBmp.context.lineTo(17, 1);
        cursorBmp.context.lineTo(9, 17);
        cursorBmp.context.lineTo(1, 1);
        cursorBmp.context.moveTo(9, HzTimingBar.HEIGHT + 17);
        cursorBmp.context.lineTo(17, HzTimingBar.HEIGHT + 33);
        cursorBmp.context.lineTo(1, HzTimingBar.HEIGHT + 33);
        cursorBmp.context.lineTo(9, HzTimingBar.HEIGHT + 17);
        cursorBmp.context.fill();
        cursorBmp.context.stroke();
        this._cursor = new Sprite(cursorBmp);
        this._cursor.opacity = 0;
        this._cursor.position.set(-9, -17);
        this._container.addChild(this._cursor);

        SceneManager._scene._spriteset.addChild(this._container);
    };

    HzTimingBar.prototype.isRequireHit = function (value) {
        return this.reqs.some(req => this.isBetween(req.x, req.x + req.width, value))
    }

    HzTimingBar.prototype.isHit = function (value) {
        return this.isBetween(this.hit.x, this.hit.x + this.hit.width, value)
    }

    HzTimingBar.prototype.isCritical = function (value) {
        return this.isBetween(this.critical.x, this.critical.x + this.critical.width, value)
    }

    HzTimingBar.prototype.isBetween = function (min, max, value) {
        return min <= value && value < max
    }

    // 更新処理（終了時はfalseを返す）
    HzTimingBar.prototype.update = function () {

        this._frame += HzTimingBar.SPEED;

        // カーソルアニメーション
        if (this._frame >= 0) {
            this._cursor.x += this.barWidth / this.maxFrame;
            this._cursor.x = this._cursor.x.clamp(0, this.barWidth);
            this._cursor.opacity = 255;

        } else {
            this._cursor.x = 0;
            this._cursor.opacity = (HzTimingBar.DELAY + this._frame) / HzTimingBar.DELAY * 255;
        }

        let result = null;
        if (Input.isTriggered('ok') && this._frame >= 0) {
            if (this.isCritical(this._cursor.x)) {
                $gameScreen.startShake(2, 10, 10)
                result = HzTimingBar.RESULT_CRITICALHIT;
            } else if (this.isHit(this._cursor.x)) {
                result = HzTimingBar.RESULT_HIT;
            }
            else if (this.isRequireHit(this._cursor.x)) {
                if (requiredSe)
                    AudioManager.playSe({ name: requiredSe, volume: 90, pitch: 100, pan: 0 });
                this._hitCntRequired++;
            } else {
                if (missSe)
                    AudioManager.playSe({ name: missSe, volume: 90, pitch: 100, pan: 0 });
                this.missCnt++;
            }
        }

        if (this._frame > this.maxFrame || this.missCnt > this.missCntMax) {
            AudioManager.playSe({ name: missSe, volume: 90, pitch: 100, pan: 0 });
            result = HzTimingBar.RESULT_FAILURE;
        }

        if (Number.isInteger(result)) {
            if (this._requiredNeedNum && this._hitCntRequired < this._requiredNeedNum)
                result = HzTimingBar.RESULT_FAILURE;
            if (result === 2 && criticalSe)
                AudioManager.playSe({ name: criticalSe, volume: 90, pitch: 100, pan: 0 });
            else if (result === 1 && hitSe)
                AudioManager.playSe({ name: hitSe, volume: 90, pitch: 100, pan: 0 });
            else if (result === 0 && missSe)
                AudioManager.playSe({ name: missSe, volume: 90, pitch: 100, pan: 0 });
            $gameVariables.setValue(VARID_TIMINGBAR_RESULT, result);
            return false;
        }
        return true;
    };

    // 終了処理
    HzTimingBar.prototype.terminate = function () {
        SceneManager._scene._spriteset.removeChild(this._container);
        this._container = null;
    };

    // 角丸長方形を描画
    function roundedRectangle(ctx, top, left, width, height, borderRadius) {
        ctx.beginPath();
        ctx.moveTo(left + borderRadius, top);
        ctx.lineTo(left + width - borderRadius, top);
        ctx.quadraticCurveTo(left + width, top, left + width, top + borderRadius);
        ctx.lineTo(left + width, top + height - borderRadius);
        ctx.quadraticCurveTo(left + width, top + height, left + width - borderRadius, top + height);
        ctx.lineTo(left + borderRadius, top + height);
        ctx.quadraticCurveTo(left, top + height, left, top + height - borderRadius);
        ctx.lineTo(left, top + borderRadius);
        ctx.quadraticCurveTo(left, top, left + borderRadius, top);
        ctx.closePath();
    }

})();