//=============================================================================
// Plugin for RPG Maker MZ
// SimpleLogMessage.js
//=============================================================================

/*:ja
 * @target MZ
 * @author fude
 * @help
 * 本プラグインは、制作者が明示的に許可した場合を除き、いかなる形であれ使用、複製、改変、再配布することを禁じます。
 * 本プラグインを無断で使用したこと、またはその利用によって生じた不具合・損害について、制作者は一切の責任を負いません。
 * @plugindesc シンプルな会話ウィンドウ
 * @base PluginCommonBase
 * 
 * @command showLog
 * @text 通知表示
 * @desc 指定のテキストで通知ウィンドウを表示します。
 *
 * @arg logs
 * @text ログ
 * @desc 表示するログ
 * @default
 * @type struct<LogParam>[]
 * 
 * @arg type
 * @text 通知タイプ
 * @desc 通知の種類
 * @type select
 * @option Talk
 * @option Check
 * @option Info
 * @option Get
 * @option News
 * @option SmartPhone
 * @default Talk
 * 
 * @arg name
 * @text 人物名
 * @desc 人物名
 * @type text
 * 
 */

/*~struct~LogParam:
 * @param text
 * @text テキスト
 * @desc 表示するテキスト
 * @type text
 * @default 
 * 
 * @param color
 * @text テキストカラー
 * @type select
 * @option グレー
 * @value \C[8]
 * @option 赤
 * @value \C[18]
 * @option 青
 * @value \C[1]
 * @option 黄
 * @value \C[14]
 * @option 水色
 * @value \C[23]
 * @option 紫
 * @value \C[31]
 * @option ピンク
 * @value \C[27]
 * @default 
 * 
 * @param duration
 * @text 表示時間
 * @desc 表示する時間 (1/60秒)
 * @type number
 * @default 70
 * 
 *
 * @param  se
 * @text 通知SEファイル名
 * @desc 通知を表示した時に演奏されるSEのファイル名。
 * @default
 * @require 1
 * @dir audio/se/
 * @type file
 * 
 * @param script
 * @text スクリプト
 * @desc ログ表示と同時に実行するスクリプト
 * @default
 * @type text
 */


(() => {
  const pluginName = 'SimpleLogMessage';

  const containerWidth = 400;
  const containerHeight = 200;
  const positionX = 0;
  const positionY = 164;
  const dispDuration = 180;
  const animDuration = 30;
  const fontSize = 18;
  const logTextHeight = fontSize + 6;

  const guideWidth = 1000;
  const guideFontSize = 17;
  const guideHeight = guideFontSize + 6;
  const iconSize = 15;

  const guideWidth2 = guideWidth * 0.23;

  const profWidth = 500;
  const profHeight = 40;

  const leftMergin = 100;
  const contentsX = leftMergin + 140;

  const maxDisplay = 5;

  PluginManager.registerCommand(pluginName, 'showLog', function (args) {
    const logs = JSON.parse(args.logs).map(log => JSON.parse(log));
    SimpleLogManager.show(logs, args.name, args.type);
  });

  PluginManager.registerCommand(pluginName, 'sumahoNotice', function (args) {
    const logs = [{ text, duration: 50 }];
    SimpleLogManager.createLogContainer();
    SceneManager._scene._logContainer.show(logs, who, '\\I[188] スマホ通知');
  });

  window.SimpleLogManager = function () { };

  SimpleLogManager.show = function (logs, name, type) {
    SimpleLogManager.createLogContainer();
    SceneManager._scene._logContainer.show(logs, name, type);
  }

  SimpleLogManager.showSimpleTalk = function (text, name) {
    const logs = [{ text, duration: 50 }];
    SimpleLogManager.createLogContainer();
    SceneManager._scene._logContainer.show(logs, name, 'Talk');
  }

  SimpleLogManager.tweet = function (text) {
    const logs = [{ text, duration: 50 }];
    SimpleLogManager.createLogContainer();
    SceneManager._scene._logContainer.show(logs, '\\P[1]', 'Talk');
  }

  SimpleLogManager.createLogContainer = function () {
    if (!SceneManager._scene._logContainer) {
      let logContainer = new SimpleLogContainer();
      SceneManager._scene._logContainer = logContainer;
      SceneManager._scene.addChild(logContainer);
    }
  }

  SimpleLogManager.isPlaying = function () {
    SimpleLogManager.createLogContainer();

  }

  class SimpleLogContainer extends PIXI.Container {
    constructor() {
      super();
      this.x = positionX;
      this.y = positionY;
      this._width = containerWidth;
      this._height = containerHeight;
      this.simpleLog = new SimpleLog();
      this.simpleLogGuide = new SimpleLogGuide();
      this.addChild(this.simpleLogGuide);
      this.addChild(this.simpleLog);
      this.state = 'next';
      this.frameCount = 0;
      this.lastType = '';
      this.lastName = '';
      this.que = [];
      this.idle = true;
      this.currentLog = {};
      this.seMute = false;
    }

    show(logs, name, type) {
      const shouldNotSkip = type.contains(...['Get', 'Lose', 'Info']);

      if (!shouldNotSkip) {
        if (this.isShowing(logs)) {
          return;
        }
        const stateTo = this.isSameParson(name, type) ? 'disp' : 'nothing';
        if (stateTo === 'nothing' || !this.idle)
          this.updateToStateWithoutSe(stateTo);
      }

      this.que.push(...logs.map(log => { return { log, name, type, key: JSON.stringify(logs) } }));

      // for (var i = 0; i < this.que.length; i++) {
      //   if (i >= maxDisplay) {
      //     this.que.shift()
      //   }
      // }
      this.updatePosition()
    }

    isPlaying() {
      return this.state !== 'nothing';
    }

    playSeWithType(type, isMoney) {
      switch (true) {
        case !!type.match(/Talk/):
        case !!type.match(/Check/):
          AudioManager.playSe({ name: 'beemyu/yumekawa/myuu_003_YumeSE_SystemCursor03', volume: 60, pitch: 130 })
          break;
        case !!type.match(/Info/):
          break;
        case !!type.match(/Phone/):
          if (this.lastType !== type)
            AudioManager.playSe({ name: 'beemyu/yumekawa/myuu_056_YumeSE_MapDoor01', volume: 90, pitch: 130 })
          break;
        case !!type.match(/News/):
          if (this.lastType !== type)
            AudioManager.playSe({ name: 'moon_clown/Vol31_Corsor_05', volume: 100, pitch: 200 })
          break;
        case !!type.match(/Get/):
          if (!isMoney)
            AudioManager.playSe({ name: 'moon_clown/Vol31_Button_01', volume: 100, pitch: 150 })
          else
            AudioManager.playSe({ name: 'Coin', volume: 100, pitch: 100 })
        default:
          break;
      }
    }

    procLog(log, name, type, key) {

      if (log.script) eval(log.script);
      if (!log.text) return false;

      if (this.guideInit || this.lastType !== type || this.lastName !== name) {
        this.simpleLogGuide.reset();
        this.simpleLogGuide.makeContents(type, name);
        this.frameCount = 0;
      }
      if (log.se && !this.seMute)
        AudioManager.playSe({ name: log.se, volume: 100, pitch: 100 });
      else
        this.playSeWithType(type, name === 'お金');
      this.lastType = type;
      this.lastName = name;
      this.guideInit = false;
      const text = log.color ? log.color + log.text : log.text;
      this.simpleLog.makeLog(text, Number(log.duration), key);
      return true;
    }

    selfSwitch(id, type, bool) {
      $gameSelfSwitches.setValue([$gameMap.mapId(), +id, type], JSON.parse(bool))
    }

    isSameParson(name, type) {
      return this.lastType === type && this.lastName === name;
    }

    updateToStateWithoutSe(stateTo) {
      this.seMute = true;
      while (this.state !== stateTo) this.update();
      this.seMute = false;
    }

    updatePosition() {
      const needUppper = $gameMessage.isBusy() && $gameMessage.positionType() === 1;
      this.y = needUppper ? positionY - 120 : positionY;
    }

    update() {
      if (this.idle && this.que.length > 0)
        this.state = 'next';

      switch (this.state) {

        case 'next':
          let next = this.que.shift()
          if (next) {
            if (!this.procLog(next.log, next.name, next.type, next.key))
              return;
            this.state = 'init';
            this.idle = false;
          } else {
            this.state = 'disp';
            this.idle = true;
          }
          break;

        case 'init':
          this.simpleLogGuide.updateInit(this.frameCount);
          this.simpleLog.updateInit(this.frameCount);
          this.frameCount++;
          if (this.simpleLogGuide.isInitDone() && this.simpleLog.isInitDone()) {
            this.state = 'next';
            this.frameCount = 0;
          }
          break;

        case 'disp':
          this.frameCount++;
          if (this.frameCount > dispDuration) {
            this.guideInit = true;
            this.state = 'exit';
            this.frameCount = 0;
          }
          break;

        case 'exit':
          for (var child of this.children) {
            child.updateExit(this.frameCount);
          }
          this.frameCount++;
          if (this.simpleLogGuide.isExitDone()) {
            this.state = 'end';
          }
          break;

        case 'end':
          this.frameCount = 0;
          this.lastName = null;
          this.lastType = null;
          this.simpleLog.dispose();
          this.state = 'nothing';
          break;
        case 'nothing':
          break;
      }
    }

    isShowing(logs) {
      const at = [...this.simpleLog.children.map(c => c.key), ...this.que.map(q => q.key)];
      const key = JSON.stringify(logs)
      return at.some(l => l === key);
    }

    gainGold(amount) {
      $gameMap._interpreter.command125([0, amount > 0 ? 0 : 1, amount])
    }
  }

  class SimpleLogGuide extends PIXI.Container {
    constructor() {
      super();
      this.y = containerHeight;

      this.guideBack = new GuideContentsBack(new Bitmap(guideWidth, guideHeight), guideWidth, animDuration);
      this.guideBack2 = new GuideContentsBack(new Bitmap(guideWidth2, guideHeight), guideWidth2, animDuration)
      this.guideText = new GuideContents(new Rectangle(0, 0, guideWidth, guideHeight), -17, 0, animDuration)

      this.guideProfBack = new GuideContentsBack(new Bitmap(profWidth, profHeight), profWidth, animDuration);
      this.guideProfBack.move(0, -profHeight);
      this.guideProfText = new GuideContents(new Rectangle(0, -profHeight, contentsX + 22, profHeight), 20, 0, animDuration + 10)

      this.reset();
      this.setDelay();
      this.addChild(this.guideBack)
      this.addChild(this.guideBack2)
      this.addChild(this.guideText)
      this.addChild(this.guideProfBack)
      this.addChild(this.guideProfText)

      this.updateInitWork = this.children;
      this.updateExitWork = this.children;
      for (var child of this.children) {
        child.finish = this.finish.bind(this);
      }
    }

    setDelay() {
      this.guideBack.setDelay(0, 30)
      this.guideBack2.setDelay(20, 38)
      this.guideText.setDelay(27, 45)
      this.guideProfBack.setDelay(35, 0)
      this.guideProfText.setDelay(45, 20)
    }

    reset() {
      this.guideBack.reset();
      this.guideBack2.reset();
      this.guideText.reset();
      this.guideProfBack.reset();
      this.guideProfText.reset();
      this.updateInitWork = this.children;
      this.updateExitWork = this.children;
    }

    makeContents(type, name) {
      const color = '#000000';
      this.guideBack._bitmap.paintOpacity = 140;
      this.guideBack._bitmap.fillTrapEx(0, 0, guideWidth * 0.7, guideHeight, color, 'rgba(0,0,0,0)', '|', '/');
      this.guideBack._bitmap.paintOpacity = 220;
      this.guideBack._bitmap.fillRect(0, 0, guideWidth * 0.7, 1, color);
      this.guideBack2._bitmap.paintOpacity = 200;
      this.guideBack2._bitmap.fillTrapEx(0, 0, guideWidth2, guideHeight, color, color, '|', '/');
      this.guideText.drawTextEx(this.putTypeIcon(type), leftMergin, guideHeight / 2 - guideFontSize / 2 - 2);

      this.guideProfBack._bitmap.paintOpacity = 255;
      this.guideProfBack._bitmap.fillTrapEx(0, 0, profWidth * 0.7, profHeight, color, 'rgba(0,0,0,0)', '|', '|');
      const nameWidth = this.guideProfText.textWidth(name)
      const a = (1 - nameWidth / this.guideProfText.width) * this.guideProfText.width / 2;
      this.guideProfText.drawTextEx(name, a, profHeight / 2 - guideFontSize / 2 - 2);
    }

    putTypeIcon(type) {
      switch (type) {
        case 'Talk':
        case 'News':
          type = '\\I[12]' + type;
          break;
        case 'SmartPhone':
          break;
      }
      return type
    }

    updateInit(frameCount) {
      for (var child of this.updateInitWork) {
        child.updateInit(frameCount);
      }
    }

    updateExit(frameCount) {
      for (var child of this.updateExitWork) {
        child.updateExit(frameCount);
      }
    }

    finish(obj, type) {
      if (type === 'init')
        this.updateInitWork = this.updateInitWork.filter(child => child != obj);
      else if (type === 'exit')
        this.updateExitWork = this.updateExitWork.filter(child => child != obj);
    }

    isInitDone() {
      return this.updateInitWork.length === 0;
    }

    isExitDone() {
      return this.updateExitWork.length === 0;
    }

  }

  class GuideContentsBack extends Sprite {
    constructor(bitmap, targetWidth, animDuration) {
      super(bitmap);
      this.targetWidth = targetWidth;
      this.animDuration = animDuration;
      this.delay = 0;
    }

    reset() {
      this._bitmap.clear();
      this.opacity = 0;
      this.width = 0;
    }

    updateInit(frameCount) {
      frameCount -= this.initDelay;
      if (frameCount >= this.animDuration) this.finish(this, 'init');

      var rate = easeOutSine(frameCount.clamp(0, this.animDuration) / this.animDuration);
      this.opacity = rate * 255;
      this.width = rate * this.targetWidth;
    }

    updateExit(frameCount) {
      frameCount -= this.exitDelay;
      if (frameCount > this.animDuration) this.finish(this, 'exit');
      var rate = easeInSine(frameCount.clamp(0, this.animDuration) / this.animDuration);
      this.opacity = (1 - rate) * 255;
      this.width = (1 - rate) * this.targetWidth;
    }

    setDelay(initDelay, exitDelay) {
      this.initDelay = initDelay;
      this.exitDelay = exitDelay;
    }
  }

  class GuideContents extends Window_Base {
    constructor(rect, slideFromX, slideToX, animDuration) {
      super(rect)
      this.slideFromX = slideFromX;
      this.slideToX = slideToX;
      this.animDuration = animDuration;
      this.initDelay = 0;
      this.exitDelay = 0;
    }

    reset() {
      this.contents.clear();
      this.opacity = 0;
      this.contentsOpacity = 0;
    }

    updateInit(frameCount) {
      frameCount -= this.initDelay;
      if (frameCount >= this.animDuration) this.finish(this, 'init');
      var rate = easeOutSine(frameCount.clamp(0, this.animDuration) / this.animDuration);
      this.contentsOpacity = rate * 255;
      this.x = this.slideToX + (1 - rate) * this.slideFromX;
    }

    updateExit(frameCount) {
      frameCount -= this.exitDelay;
      if (frameCount > this.animDuration) this.finish(this, 'exit');
      var rate = easeInSine(frameCount.clamp(0, this.animDuration) / this.animDuration);
      this.contentsOpacity = (1 - rate) * 255;
      this.x = this.slideToX + rate * -this.slideFromX;
    }

    updatePadding() {
      this.padding = 0;
    };

    setDelay(initDelay, exitDelay) {
      this.initDelay = initDelay;
      this.exitDelay = exitDelay;
    }

    resetFontSettings() {
      this.contents.fontFace = $gameSystem.mainFontFace();
      this.contents.fontSize = guideFontSize;
    };

    processDrawIcon = function (iconIndex, textState) {
      var size = this.getIconSize();
      var y = (logTextHeight - size) / 2;
      this.drawIcon(iconIndex, textState.x + 2, y);
      textState.x += size + 4;
    };

    drawIcon = function (iconIndex, x, y) {
      var bitmap = ImageManager.loadSystem('IconSet');
      var pw = ImageManager.iconWidth;
      var ph = ImageManager.iconHeight;
      var sx = iconIndex % 16 * pw;
      var sy = Math.floor(iconIndex / 16) * ph;
      var size = this.getIconSize();
      this.contents.blt(bitmap, sx, sy, pw, ph, x, y, size, size);
    };

    getIconSize = function () {
      return iconSize;
    };
  }

  class SimpleLog extends PIXI.Container {
    constructor() {
      super();
      this._width = Graphics.width;
      this._height = containerHeight;
      this.animDuration = 5;
      this.updateInitWork = [];
    }

    makeLog(text, duration, key) {
      let log = new Window_Log(text, new Rectangle(contentsX, containerHeight - logTextHeight, this._width, logTextHeight), duration, key);
      log.finish = this.finish.bind(this);
      this.addChild(log);
      this.updateInitWork.push(log);
      for (var i = 0; i < this.children.length; i++) {
        this.children[this.children.length - 1 - i].ajustPosition(i);
        this.children[this.children.length - 1 - i].ajustOpacity(i);
        if (i >= maxDisplay) {
          this.removeChildAt(0)
        }
      }
      for (var i = 0; i < this.updateInitWork.length; i++) {
        if (i >= maxDisplay) {
          this.updateInitWork.shift()
        }
      }
    }

    updateInit(frameCount) {
      if (this.alpha < 1) {
        var rate = easeInSine(frameCount / this.animDuration);
        this.alpha = rate;
        return;
      }

      if (this.children.filter(child => child.targetY && child.targetY != child.y).length > 0) {
        for (var child of this.children) {
          child.updatePosition();
        }
        return;
      }

      this.updateInitWork[0]?.updateInit();
    }

    updateExit(frameCount) {
      if (frameCount > this.animDuration) return;
      var rate = easeInSine(frameCount / this.animDuration);
      this.alpha = (1 - rate);
    }

    finish(obj) {
      this.updateInitWork = this.updateInitWork.filter(child => child != obj);
    }

    isInitDone() {
      return this.updateInitWork.length === 0;
    }

    dispose() {
      const length = this.children.length;
      for (let i = 0; i < length; i++) {
        this.removeChild(this.children[0]);
      }
    }


  }

  class Window_Log extends Window_Base {
    constructor(text, rect, duration, key) {
      super(rect);
      this.text = text;
      this.duration = duration;
      this.opacity = 0;
      this.contentsOpacity = 0;
      this.tWidth = this.textWidth(text);
      this.drawTextEx(text, 0, this.height / 2 - fontSize / 2);
      this.width = 0;
      this.animDuration = 20;
      this.frameCount = 0;
      this.key = key;
    }

    updateInit() {
      if (this.frameCount >= this.animDuration) {
        if (this.frameCount >= this.animDuration + this.duration)
          this.finish(this);
        this.frameCount++;
        return;
      }
      var rate = easeOutSine(this.frameCount / this.animDuration);
      this.width = rate * this.tWidth;
      this.contentsOpacity = rate * 255;
      this.frameCount++;
    }

    updatePosition() {
      if (this.targetY && this.y != this.targetY) {
        this.y -= 1;
        if (this.y <= this.targetY) {
          this.y = this.targetY;
        }
      }
    }

    lineHeight() {
      return fontSize + 11;
    };

    updatePadding() {
      this.padding = 0;
    };

    ajustPosition(index) {
      this.targetY = containerHeight - this.height * (index + 1);
    };

    ajustOpacity(index) {
      this.contentsOpacity = 255 * (1 - (index / maxDisplay));
    }

    resetFontSettings() {
      this.contents.fontFace = $gameSystem.mainFontFace();
      this.contents.fontSize = fontSize;
    };

    processDrawIcon = function (iconIndex, textState) {
      var size = this.getIconSize();
      var y = (logTextHeight - size) / 2;
      this.drawIcon(iconIndex, textState.x + 2, y);
      textState.x += size + 4;
    };

    drawIcon = function (iconIndex, x, y) {
      var bitmap = ImageManager.loadSystem('IconSet');
      var pw = ImageManager.iconWidth;
      var ph = ImageManager.iconHeight;
      var sx = iconIndex % 16 * pw;
      var sy = Math.floor(iconIndex / 16) * ph;
      var size = this.getIconSize();
      this.contents.blt(bitmap, sx, sy, pw, ph, x, y, size, size);
    };

    getIconSize = function () {
      return iconSize;
    };
  }


  // ウィンドウ位置が中のとき表示位置が被らないようにする
  const _Window_Message_startMessage = Window_Message.prototype.startMessage;
  Window_Message.prototype.startMessage = function () {
    _Window_Message_startMessage.apply(this, arguments);
    const container = SceneManager._scene._logContainer;
    if (container)
      container.updatePosition();
  };

  // Show Text
  // Game_Interpreter.prototype.command101 = function (params) {

  //   if ($gameMessage.isBusy()) {
  //     return false;
  //   }

  //   $gameMessage.setFaceImage(params[0], params[1]);
  //   $gameMessage.setBackground(params[2]);
  //   $gameMessage.setPositionType(params[3]);
  //   $gameMessage.setSpeakerName(params[4]);
  //   while (this.nextEventCode() === 401) {
  //     // Text data
  //     this._index++;
  //     var text = this.currentCommand().parameters[0];
  //     if (text.match(/\\s/)) {
  //       SimpleLogManager.showSimpleTalk(text, params[4], $gameMap.event(this.eventId()), 0);
  //     }
  //     else if (text.match(/\\S/)) {
  //       SimpleLogManager.showSimpleTalk(text, params[4], $gameMap.event(this.eventId()), 0);
  //     }
  //     else {
  //       $gameMessage.add(text);
  //     }
  //     text.replace(/\\spk\[(\d+)\]/, function () {
  //       console.log(arguments);
  //       $gameMessage.setSpeakerName(`\\N[${arguments[1]}]`);
  //     })
  //   }
  // }



  //   switch (this.nextEventCode()) {
  //     case 102: // Show Choices
  //       this._index++;
  //       this.setupChoices(this.currentCommand().parameters);
  //       break;
  //     case 103: // Input Number
  //       this._index++;
  //       this.setupNumInput(this.currentCommand().parameters);
  //       break;
  //     case 104: // Select Item
  //       this._index++;
  //       this.setupItemChoice(this.currentCommand().parameters);
  //       break;
  //   }
  //   this.setWaitMode("message");
  //   return true;
  // };

  // // Change Items
  // const command126 = Game_Interpreter.prototype.command126;
  // Game_Interpreter.prototype.command126 = function (params) {
  //   command126.call(this, params);

  //   let value = this.operateValue(params[1], params[2], params[3]);
  //   let item = $dataItems[params[0]];

  //   const name = item.name;
  //   const num = Math.abs(value);
  //   const iconID = item.iconIndex;
  //   const description = GetItemDescription(item);
  //   const isGain = value >= 0;
  //   const guideType = isGain ? 'Get' : 'Lose';
  //   const text = `\\I[${iconID}] ${name} ${isGain ? 'x' : '-'} ${num} ${isGain ? description : ''} `;
  //   const duration = 100;
  //   const log = { text, duration }

  //   SimpleLogManager.show([log], `アイテム`, `   ${guideType}`);
  //   return true;
  // };


  // // Change Weapons
  // const command127 = Game_Interpreter.prototype.command127;
  // Game_Interpreter.prototype.command127 = function (params) {
  //   command127.call(this, params);
  //   let value = this.operateValue(params[1], params[2], params[3]);
  //   let item = $dataWeapons[params[0]];

  //   const name = item.name;
  //   const num = Math.abs(value);
  //   const iconID = item.iconIndex;
  //   const description = GetItemDescription(item);
  //   const isGain = value >= 0;
  //   const guideType = isGain ? 'Get' : 'Lose';
  //   const text = `${isGain ? '' : '-'}\\I[${iconID}] ${name} x ${num} ${isGain ? description : ''} `;
  //   const duration = 100;
  //   const log = { text, duration }

  //   SimpleLogManager.show([log], `アイテム`, `   ${guideType}`);
  //   return true;
  // };

  // // Change Armors
  // const command128 = Game_Interpreter.prototype.command128;
  // Game_Interpreter.prototype.command128 = function (params) {
  //   command128.call(this, params);
  //   let value = this.operateValue(params[1], params[2], params[3]);
  //   let item = $dataArmors[params[0]];

  //   const name = item.name;
  //   const num = Math.abs(value);
  //   const iconID = item.iconIndex;
  //   const description = GetItemDescription(item);
  //   const isGain = value >= 0;
  //   const guideType = isGain ? 'Get' : 'Lose';
  //   const text = `${isGain ? '' : '-'}\\I[${iconID}] ${name} x ${num} ${isGain ? description : ''} `;
  //   const duration = 100;
  //   const log = { text, duration }

  //   SimpleLogManager.show([log], `アイテム`, `   ${guideType}`);
  //   return true;
  // };

  // // Change Gold
  // const command125 = Game_Interpreter.prototype.command125;
  // Game_Interpreter.prototype.command125 = function (params) {
  //   command125.call(this, params);
  //   const value = this.operateValue(params[0], params[1], params[2]);
  //   const isGain = value >= 0;
  //   const guideType = isGain ? 'Get' : 'Lose';
  //   const text = `\\I[208] ${Math.abs(value)} \\G`;
  //   const duration = 100;
  //   const log = { text, duration }

  //   SimpleLogManager.show([log], `お金`, `   ${guideType}`);
  //   return true;
  // };

  // 装備変更時
  // Game_Actor.prototype.changeEquipById = function (etypeId, itemId) {
  //   const slotId = etypeId - 1;
  //   if (this.equipSlots()[slotId] === 1) {
  //     this.changeEquip(slotId, $dataWeapons[itemId]);
  //   } else {
  //     this.changeEquip(slotId, $dataArmors[itemId]);
  //   }
  // };

  function GetItemDescription(item) {
    if (item.meta.ticker) {
      return item.meta.ticker;
    }
    return item.description.replace(/[\r\n]+.*/m, '');
  }

})();
