//=============================================================================
// Plugin for RPG Maker MZ
// TinyGetInfoWndMZ.js
//=============================================================================
// [Release Note]
// This is the MZ version of TinyGetInfoWnd.js the RMMV plugin.

/*:ja
 * @target MZ
 * @plugindesc アイテムの入手/消失を1行ウィンドウで表示します
 * @author 神無月サスケ (原案：ももまる)
 *
 * @param Event Command Switch
 * @text イベントコマンド用スイッチID
 * @desc このスイッチがONの時、イベントコマンドでアイテム類や
 * 所持金の増減を行った時ウィンドウ表示
 * @type switch
 * @default 10
 * 
 * @param Y position type
 * @text 表示Y座標タイプ
 * @desc 複数のウィンドウを並べる位置(0:上部 1:下部)
 * @type select
 * @option 上部
 * @value 0
 * @option 下部
 * @value 1
 * @default 0
 * 
 * @param Window Duration
 * @text 表示フレーム時間
 * @desc ウィンドウを表示するフレーム数
 * ※1秒＝60フレーム (既定値：200)
 * @type number
 * @min 0
 * @default 200
 * 
 * @param displayMax
 * @text 最大表示数
 * @desc 最大表示数を超えると最も古い通知が消える
 * @type number
 * @min 1
 * @max 100
 * @default 10
 * 
 * @param textGainItem
 * @text 入手時表示タイトル
 * @desc アイテムを入手した時に表示するタイトル部分のテキスト。
 * %1がアイテム種別(アイテム/武器/防具/お金)に置き換わる
 * @type string
 * @default %1入手！
 * 
 * @param Display Loss
 * @text 消失時も表示？
 * @desc 消失時にもウィンドウを表示するか
 * @type boolean
 * @on する
 * @off しない
 * @default true
 * 
 * @param textLoseItem
 * @parent Display Loss
 * @text 消失時表示タイトル
 * @desc アイテムを消失した時に表示するタイトル部分のテキスト。
 * %1がアイテム種別(アイテム/武器/防具/お金)に置き換わる
 * @type string
 * @default %1消失……。
 *
 * @param Display at Battle
 * @text 戦闘中も表示？
 * @desc 戦闘中にもウィンドウを表示するか
 * @type boolean
 * @on する
 * @off しない
 * @default true
 * 
 * @param wordMoney
 * @text 「お金」用テキスト
 * @desc アイテム種別の「お金」を表すテキスト
 * @type string
 * @default お金
 *
 * @param iconMoney
 * @text 「お金」用アイコンID
 * @desc お金入手/消失時に表示するアイコンID
 * @type number
 * @min 0
 * @default 313
 *
 * @param Item SE filename
 * @text アイテムSEファイル名
 * @desc アイテムを入手した時に演奏されるSEのファイル名。
 * 注意：消失した時は演奏されません。
 * @default Evasion1
 * @require 1
 * @dir audio/se/
 * @type file
 * 
 * @param Item SE volume
 * @text アイテムSEボリューム
 * @parent Item SE filename
 * @desc アイテムを入手した時に演奏されるSEのボリューム
 * @type number
 * @min 0
 * @default 90
 * 
 * @param Item SE pitch
 * @text アイテムSEピッチ
 * @parent Item SE filename
 * @desc アイテムを入手した時に演奏されるSEのピッチ
 * @type number
 * @max 100000
 * @min 10
 * @default 100
 * 
 * @param Weapon SE filename
 * @text 武器SEファイル名
 * @desc 武器を入手した時に演奏されるSEのファイル名です。
 * 注意：消失した時は演奏されません。
 * @default Equip1
 * @require 1
 * @dir audio/se/
 * @type file
 * 
 * @param Weapon SE volume
 * @text 武器SEボリューム
 * @parent Weapon SE filename
 * @desc 武器を入手した時に演奏されるSEのボリューム
 * @type number
 * @min 0
 * @default 90
 * 
 * @param Weapon SE pitch
 * @text 武器SEピッチ
 * @parent Weapon SE filename
 * @desc 武器を入手した時に演奏されるSEのピッチ
 * @type number
 * @max 100000
 * @min 10
 * @default 100
 * 
 * @param Armor SE filename
 * @text 防具SEファイル名
 * @desc 防具を入手した時に演奏されるSEのファイル名。
 * 注意：消失した時は演奏されません。
 * @default Evasion2
 * @require 1
 * @dir audio/se/
 * @type file
 * 
 * @param Armor SE volume
 * @text 防具SEボリューム
 * @parent Armor SE filename
 * @desc 防具を入手した時に演奏されるSEのボリューム
 * @type number
 * @min 0
 * @default 90
 * 
 * @param Armor SE pitch
 * @text 防具SEピッチ
 * @parent Armor SE filename
 * @desc 防具を入手した時に演奏されるSEのピッチ
 * @type number
 * @max 100000
 * @min 10
 * @default 100
 * 
 * @param Money SE filename
 * @text お金SEファイル名
 * @desc お金を入手した時に演奏されるSEのファイル名。
 * 注意：消失した時は演奏されません。
 * @default Coin
 * @require 1
 * @dir audio/se/
 * @type file
 * 
 * @param Money SE volume
 * @text お金SEボリューム
 * @parent Money SE filename
 * @desc お金を入手した時に演奏されるSEのボリューム
 * @type number
 * @min 0
 * @default 90
 * 
 * @param Money SE pitch
 * @text お金SEピッチ
 * @parent Money SE filename
 * @desc お金を入手した時に演奏されるSEのピッチ
 * @type number
 * @max 100000
 * @min 10
 * @default 100
 *
 * @command noticeInfo
 * @text 通知表示
 * @desc 指定のテキストで通知ウィンドウを表示します。
 *
 * @arg noticeText
 * @text テキスト
 * @desc 表示するテキスト
 * @type text
 * @default 
 *
 * @arg noticeIcon
 * @desc テキストの左に表示したいアイコン(ID指定)
 * @default 
 * @type number
 * 
 * @arg  seFilename
 * @text 通知SEファイル名
 * @desc 通知を表示した時に演奏されるSEのファイル名。
 * @default Coin
 * @require 1
 * @dir audio/se/
 * @type file
 * 
 * @arg seVolume
 * @text 通知SEボリューム
 * @parent Money SE filename
 * @desc 通知をした時に演奏されるSEのボリューム
 * @type number
 * @min 0
 * @default 90
 * 
 * @arg sePitch
 * @text 通知SEピッチ
 * @parent Money SE filename
 * @desc 通知をした時に演奏されるSEのピッチ
 * @type number
 * @max 100000
 * @min 10
 * @default 100
 * 
 * @command changeYPosition
 * @text Y座標変更
 * @desc コンテナのY座標を変更
 * 0指定で初期座標に設定
 *
 * @arg yPosition
 * @text y座標
 * @type number
 * 
 * 
 * 
 * @help
 * このプラグインは、RPGツクールMZに対応しています。
 *
 * このプラグインは、アイテムや装備、所持金の増減の際に、
 * 画面に1行ウィンドウを表示することを可能にします。
 *
 * ■概要
 * 画面に1行ウィンドウを表示させるには、以下のいずれかを行います。
 * ◆ プラグインコマンドを呼び出す：
 *  オプションで指定したアイテムや装備、お金の増減とともに、
 *  ウィンドウが表示されます。
 * ◆ オプションのスイッチ「イベントコマンド用スイッチID」をONにする：
 *  このスイッチがONの間、イベントコマンドで増減を行うと、
 *  同時にウィンドウが表示されるようになります。
 *
 * ■メモの書式
 * <info:the_explanation> : the_explanation の文章が、アイテムの説明として
 *  入手/消失時に表示されます。省略した場合は、アイテムの説明の1行目が
 *  表示されます。
 *
 * ■注意
 * アイテムの消失は、所持している数に関連します。
 * 例えば5個消失コマンドを実行して、3個しか持っていない場合「3個消失」と
 * 表示されます。また、該当アイテムをひとつも持っていない場合は、
 * 消失ウィンドウは表示されません。
 *
 * ■謝辞
 * このプラグインは、ももまる様のRGSS2素材の仕様をベースに作られました。
 * ももまる様に謝意を示します。
 *
 * ■ライセンス表記
 * このプラグインは MIT ライセンスで配布されます。
 * ご自由にお使いください。
 * http://opensource.org/licenses/mit-license.php
 *
 */

(() => {
  const pluginName = 'TinyGetInfoWndMZ';
  //
  // process parameters
  //
  const parameters = PluginManager.parameters(pluginName);
  const dispSwitchID = Number(parameters['Event Command Switch'] || 10);
  const yPosType = Number(parameters['Y position type'] || 0);
  let wndDuration = Number(parameters['Window Duration'] || 200);
  const isDisplayLoss = eval(parameters['Display Loss'] || 'true');
  const isDisplayBattle = eval(parameters['Display at Battle'] || 'true');
  const textGainItem = parameters['textGainItem'] || '';
  const textLoseItem = parameters['textLoseItem'] || '';
  const wordMoney = parameters['wordMoney'] || 'money';
  const iconMoney = Number(parameters['iconMoney'] || 360);
  const seFilename = parameters['Item SE filename'] || 'Evasion1';
  const seVolume = Number(parameters['Item SE volume'] || 90);
  const sePitch = Number(parameters['Item SE pitch'] || 100);
  const seWeaponFilename = parameters['Weapon SE filename'] || 'Equip1';
  const seWeaponVolume = Number(parameters['Weapon SE volume'] || 90);
  const seWeaponPitch = Number(parameters['Weapon SE pitch'] || 100);
  const seArmorFilename = parameters['Armor SE filename'] || 'Evasion2';
  const seArmorVolume = Number(parameters['Armor SE volume'] || 90);
  const seArmorPitch = Number(parameters['Armor SE pitch'] || 100);
  const seMoneyFilename = parameters['Money SE filename'] || 'Coin';
  const seMoneyVolume = Number(parameters['Money SE volume'] || 90);
  const seMoneyPitch = Number(parameters['Money SE pitch'] || 100);
  const displayMax = Number(parameters['displayMax'] || 5);


  // 定数
  const runupTime = 30; // インフォウィンドウの出現フレーム
  const AlignSpeed = 2; // インフォウィンドウの整列速度
  const infoWndH = 30;
  const fontSize = 18;


  //
  // utility functions
  //
  const tinyInfoContainer = () => {
    return SceneManager._scene._tinyInfoContainer;
  };

  window.addGetItemWindow = (type, id, value) => {
    const container = tinyInfoContainer();
    if (!container) return;

    switch (type) {
      case 'weapon':
        var item = $dataWeapons[id];
        var audio = { name: seWeaponFilename, volume: seWeaponVolume, pitch: seWeaponPitch };
        break;
      case 'armor':
        var item = $dataArmors[id];
        var audio = { name: seArmorFilename, volume: seArmorVolume, pitch: seArmorPitch };
        break;
      case 'item':
        var item = $dataItems[id];
        var audio = { name: seFilename, volume: seVolume, pitch: sePitch };
        break;
    }

    item.type = type;
    const window = new Window_GetInfo(item, value, audio)
    container.addTinyInfoWindow(window);
  };

  const addGetGoldWindow = value => {
    const container = tinyInfoContainer();
    if (!container) return;

    var audio = { name: seMoneyFilename, volume: seMoneyVolume, pitch: seMoneyPitch };
    const window = new Window_GetGoldInfo(value, audio)

    container.addTinyInfoWindow(window);
  };

  window.addNoticeWindow = (text, iconId, seFilename, seVolume, sePitch) => {
    const container = tinyInfoContainer();
    if (!container) return;

    var audio = { name: seFilename, volume: seVolume, pitch: sePitch };

    const window = new Window_NoticeInfo(text, iconId, audio);
    container.addTinyInfoWindow(window);
  }

  const noteDescription = data => {
    if (data.meta.info) {
      return data.meta.info;
    }
    return data.description.replace(/[\r\n]+.*/m, "");
  };


  //
  // process plugin commands
  //
  PluginManager.registerCommand(pluginName, 'noticeInfo', args => {
    addNoticeWindow(args.noticeText, args.noticeIcon, args.seFilename, args.seVolume, args.sePitch);
  });

  //
  // process plugin commands
  //
  PluginManager.registerCommand(pluginName, 'changeYPosition', args => {
    const yPosition = +args.yPosition;
    if (yPosition > 0)
      $gameSystem.setTinyInfoYPosition(yPosition)
    else
      $gameSystem.resetTinyInfoYPosition();
    const container = tinyInfoContainer();
    container.updateYPosition();
  });

  var _Game_System_initialize = Game_System.prototype.initialize;
  Game_System.prototype.initialize = function () {
    _Game_System_initialize.apply(this, arguments);
    this.resetTinyInfoYPosition();
  };

  Game_System.prototype.resetTinyInfoYPosition = function () {
    this._tinyInfoYposition = yPosType === 0 ? 170 : 270;
  }

  Game_System.prototype.setTinyInfoYPosition = function (y) {
    this._tinyInfoYposition = y;
  }

  Game_System.prototype.tinyInfoYPosition = function () {
    return this._tinyInfoYposition;
  }

  const _Scene_Map_createDisplayObjects = Scene_Map.prototype.createDisplayObjects;
  Scene_Map.prototype.createDisplayObjects = function () {
    _Scene_Map_createDisplayObjects.call(this);
    this._tinyInfoContainer = new TinyInfoContainer();
    this.addChildAt(this._tinyInfoContainer, 1);
  };

  const _Scene_Battle_createDisplayObjects = Scene_Battle.prototype.createDisplayObjects;
  Scene_Battle.prototype.createDisplayObjects = function () {
    _Scene_Battle_createDisplayObjects.call(this);
    this._tinyInfoContainer = new TinyInfoContainer();
    this.addChildAt(this._tinyInfoContainer, 1);
  };

  // _Scene_Map_updateMain = Scene_Map.prototype.updateMain;
  // Scene_Map.prototype.updateMain = function () {
  //   _Scene_Map_updateMain.apply(this, arguments);
  //   this._tinyInfoContainer.update();
  // }

  class TinyInfoContainer extends PIXI.Container {
    constructor() {
      super();
      this.x = 0;
      this.y = $gameSystem.tinyInfoYPosition();
      this._width = Graphics.boxWidth;
      this._height = infoWndH * displayMax;
      this._openness = 0;
      this._fadeFrame = 180;
      this.tinyInfoWnds = [];
      this._dispDuration = 0;
    }

    updateYPosition() {
      this.y = $gameSystem.tinyInfoYPosition();
    }

    updateOpacity() {
      this.alpha = (this.alpha + this.calcDeltaOpenness()).clamp(0, 1);
    }

    calcDeltaOpenness() {
      const openness = 1 / this._fadeFrame;
      return this.isVisible() ? 1 : -openness;
    }

    isVisible() {
      if ($gameParty.inBattle())
        this._dispDuration = 300;
      else
        this._dispDuration--;

      if (this._dispDuration < 0) this._dispDuration = 0;
      return this._dispDuration > 0;
    }

    addTinyInfoWindow(window) {

      this._dispDuration = 300;

      if (!(window instanceof Window_TinyInfo_Base)) return;

      if (yPosType === 0) {
        window.setPosition(this.tinyInfoWnds.length);
        this.tinyInfoWnds.push(window);
      } else {
        window.setPosition(0);
        this.tinyInfoWnds.unshift(window)
      }

      this.addChild(window);

      this.tinyInfoWnds.forEach((window, index) => {
        window.resetPosition(index);
      })

      let w = this.tinyInfoWnds.filter(window => window.state !== "exit")
      if (yPosType === 1) w = w.reverse();
      if (w.length > displayMax) {
        for (var i = 0; i < w.length; i++) {
          w[i].state = 'exit';
          w[i].count = 0;
          return;
        }
      }
    }

    update() {

      this.updateOpacity();

      let disposed = false;
      this.tinyInfoWnds = this.tinyInfoWnds.filter(window => {
        if (window.needsDispose()) {
          this.removeGetInfoWindow(window);
          disposed = true;
          return false;
        }
        return true;
      });

      if (disposed) {
        this.tinyInfoWnds.forEach((window, index) => {
          window.resetPosition(index);
        })
      }

      for (const window of this.tinyInfoWnds)
        window.update()
    }

    removeGetInfoWindow(window) {
      this.removeChild(window);
    }

    removeAllGetInfoWindows() {
      for (const window of this.tinyInfoWnds) {
        this.removeGetInfoWindow(window);
      }
      this.tinyInfoWnds = [];
    }

  }

  // -------------------------------------------------------------------------
  // Window_TinyInfo_Base
  // 

  function Window_TinyInfo_Base() {
    this.initialize.apply(this, arguments);
  }

  Window_TinyInfo_Base.prototype = Object.create(Window_Base.prototype);
  Window_TinyInfo_Base.prototype.constructor = Window_TinyInfo_Base;

  Window_TinyInfo_Base.prototype.initialize = function () {
    Window_Base.prototype.initialize.call(this, tinyInfoRect());
    this.reset();
    // fill background
    this.contents.paintOpacity = 170;
    this.contents.gradientFillRect(0, 0, Graphics.boxWidth * 0.4, this.height, '#000000', 'rgba(0,0,0,0)');
  };

  const tinyInfoRect = () => {
    const wx = 0;
    const wy = 0;
    const ww = Graphics.boxWidth;
    const wh = infoWndH;
    return new Rectangle(wx, wy, ww, wh);
  };

  Window_TinyInfo_Base.prototype.reset = function () {
    this.disposed = false;
    this.opacity = 0;
    this.backOpacity = 0;
    this.contentsOpacity = 0;
    this.state = 'enter';
    this.count = 0;
    this.contents.clear();
    if ($gameSystem.tinyWndDuration != null) {
      wndDuration = $gameSystem.tinyWndDuration;
    }
    return this;
  };

  Window_TinyInfo_Base.prototype.setPosition = function (index) {
    this.y = this.height * index;
    // this.y = yPosType == 0 ? this.height * index : -this.height * (index + 1);
  };

  Window_TinyInfo_Base.prototype.resetPosition = function (index) {
    this.toY = this.height * index;
    // this.toY = yPosType == 0 ? this.height * index : -this.height * (index + 1);
  };

  Window_TinyInfo_Base.prototype.drawItemName = function (item, x, y, width) {
    if (item) {
      var size = this.getIconSize();
      var iconY = (this.height - size) / 2;
      const textMargin = size + 4;
      const itemWidth = Math.max(0, width - textMargin);
      this.resetTextColor();
      this.drawIcon(item.iconIndex, x, iconY);
      this.drawTextEx(item.name, x + textMargin, y, itemWidth);
    }
  };

  Window_TinyInfo_Base.prototype.processDrawIcon = function (iconIndex, textState) {
    var size = this.getIconSize();
    var y = (this.height - size) / 2;
    this.drawIcon(iconIndex, textState.x + 2, y);
    textState.x += size + 4;
  };

  Window_TinyInfo_Base.prototype.drawIcon = function (iconIndex, x, y) {
    var bitmap = ImageManager.loadSystem('IconSet');
    var pw = ImageManager.iconWidth;
    var ph = ImageManager.iconHeight;
    var sx = iconIndex % 16 * pw;
    var sy = Math.floor(iconIndex / 16) * ph;
    var size = this.getIconSize();
    this.contents.blt(bitmap, sx, sy, pw, ph, x, y, size, size);
  };

  Window_TinyInfo_Base.prototype.getIconSize = function () {
    return 25;
  };

  Window_TinyInfo_Base.prototype.lineHeight = function () {
    return fontSize + 8;
  };

  Window_TinyInfo_Base.prototype.updatePadding = function () {
    this.padding = 0;
  };

  Window_TinyInfo_Base.prototype.update = function () {
    Window_Base.prototype.update.call(this);

    if (this.toY !== null) {
      if (this.y > this.toY) {
        this.y -= AlignSpeed;
        if (this.y < this.toY)
          this.y = this.toY;
      } else if (this.y < this.toY) {
        this.y += AlignSpeed;
        if (this.y > this.toY)
          this.y = this.toY;
      }
      if (this.y === this.toY) {
        this.toY = null;
      }
    }

    switch (this.state) {
      case 'enter':
        this.count++;
        var rate = easeOutSine(this.count / runupTime);
        this.x = 0 + (1 - rate) * 20;
        this.contentsOpacity = rate * 255;
        if (this.count > runupTime) {
          this.state = 'display';
          this.count = 0;
        }
        break;
      case 'display':
        this.count++;
        if (this.count >= wndDuration) {
          this.state = 'exit';
          this.count = 0;
        }
        break;
      case 'exit':
        if (yPosType === 0) {
          this.count++;
          this.contentsOpacity = (1 - this.count / runupTime) * 255;
          this.y -= 1;
        } else {
          // if (this.toY !== null) break;
          this.count++;
          this.contentsOpacity = (1 - this.count / runupTime) * 255;
          this.y += 1;
        }
        break;
    }
  };

  Window_TinyInfo_Base.prototype.needsDispose = function () {
    return this.contentsOpacity === 0;
  };

  Window_TinyInfo_Base.prototype.resetFontSettings = function () {
    Window_Base.prototype.resetFontSettings.apply(this, arguments);
    this.contents.fontSize = this.fontSize();
  };

  Window_TinyInfo_Base.prototype.fontSize = function () {
    return fontSize;
  };


  function Window_NoticeInfo() {
    this.initialize.apply(this, arguments);
  }

  Window_NoticeInfo.prototype = Object.create(Window_TinyInfo_Base.prototype);
  Window_NoticeInfo.prototype.constructor = Window_NoticeInfo;

  Window_NoticeInfo.prototype.initialize = function (text, iconId, audio) {
    Window_TinyInfo_Base.prototype.initialize.call(this);
    this.setup(text, iconId, audio);
  };

  Window_NoticeInfo.prototype.setup = function (text, iconId, audio) {
    this.drawContents(text, iconId);
    AudioManager.playSe(audio);
    this.update();
    return this;
  };

  Window_NoticeInfo.prototype.processDrawIcon = function (iconIndex, textState) {
    var size = this.getIconSize();
    var y = (textState.y) / 2 + 2;
    this.drawIcon(iconIndex, textState.x + 2, y);
    textState.x += size + 2;
  };

  Window_NoticeInfo.prototype.drawIcon = function (iconIndex, x, y) {
    var bitmap = ImageManager.loadSystem('IconSet');
    var pw = ImageManager.iconWidth;
    var ph = ImageManager.iconHeight;
    var sx = iconIndex % 16 * pw;
    var sy = Math.floor(iconIndex / 16) * ph;
    var size = this.getIconSize();
    this.contents.blt(bitmap, sx, sy, pw, ph, x, y, size, size);
  };

  Window_NoticeInfo.prototype.getIconSize = function () {
    return 20;
  };


  Window_NoticeInfo.prototype.drawContents = function (text, iconId) {

    var x = 6;
    var y = this.height / 2 - this.contents.fontSize / 2;
    this.contents.paintOpacity = 255;

    this.resetTextColor();
    if (iconId && iconId != 0) {
      this.drawIcon(iconId, x, y);
      x += this.getIconSize() + 4;
    }
    this.drawTextEx(text, x, y, this.textWidth(text) + 12, 'left');
  }


  // -------------------------------------------------------------------------
  // Window_GetInfo
  // 
  // The tiny window to display item gain/lose situation on map.

  function Window_GetInfo() {
    this.initialize.apply(this, arguments);
  }

  Window_GetInfo.prototype = Object.create(Window_TinyInfo_Base.prototype);
  Window_GetInfo.prototype.constructor = Window_GetInfo;

  Window_GetInfo.prototype.initialize = function (item, value, audio) {
    Window_TinyInfo_Base.prototype.initialize.call(this);
    this.setup(item, value, audio);
  };

  Window_GetInfo.prototype.setup = function (item, value, audio) {
    this.drawContents(item, value);
    if (value > 0)
      AudioManager.playSe(audio);
    this.update();
    return this;
  };

  Window_GetInfo.prototype.processDrawIcon = function (iconIndex, textState) {
    var size = this.getIconSize();
    var y = (this.height - size) / 2;
    this.drawIcon(iconIndex, textState.x + 2, y);
    textState.x += size + 4;
  };

  Window_GetInfo.prototype.drawIcon = function (iconIndex, x, y) {
    var bitmap = ImageManager.loadSystem('IconSet');
    var pw = ImageManager.iconWidth;
    var ph = ImageManager.iconHeight;
    var sx = iconIndex % 16 * pw;
    var sy = Math.floor(iconIndex / 16) * ph;
    var size = this.getIconSize();
    this.contents.blt(bitmap, sx, sy, pw, ph, x, y - 4, size, size);
  };

  Window_GetInfo.prototype.drawContents = function (item, value) {

    var x = 6;
    var y = this.height / 2 - this.contents.fontSize / 2;

    // draw item name, number, description
    this.contents.paintOpacity = 255;
    this.resetTextColor();
    if (value < 0) {
      this.contents.paintOpacity = 160;
    }

    const getText = value > 0 ? textGainItem.format(TextManager.item) : textLoseItem.format(TextManager.item);
    this.drawTextEx(getText, x, y, this.textWidth(getText) + 12, 'left');
    x += this.textWidth(getText);

    this.drawItemName(item, x, y, 300);
    x += this.getIconSize() + 4 + this.textWidth(item.name) + 5;

    if (item.type == 'item') {
      this.drawTextEx('\xd7', x, y, 24, 'center');
      x += 24
      this.drawTextEx(String(Math.abs(value)), x, y, 32, 'right');
      x += 24
    }
    this.resetTextColor();

    let desc = noteDescription(item)
    if (desc)
      this.drawTextEx(desc, x, y, Graphics.boxWidth - 384, 'left');
  };


  function Window_GetGoldInfo() {
    this.initialize.apply(this, arguments);
  }

  Window_GetGoldInfo.prototype = Object.create(Window_TinyInfo_Base.prototype);
  Window_GetGoldInfo.prototype.constructor = Window_GetGoldInfo;

  Window_GetGoldInfo.prototype.initialize = function (value, audio) {
    Window_TinyInfo_Base.prototype.initialize.call(this);
    this.setup(value, audio);
  };

  Window_GetGoldInfo.prototype.setup = function (value, audio) {
    this.drawContents(value);
    AudioManager.playSe(audio);
    this.update();
    return this;
  };


  Window_GetGoldInfo.prototype.drawContents = function (value) {

    var x = 6;
    var y = this.height / 2 - this.contents.fontSize / 2;
    this.contents.paintOpacity = 255;

    if (value < 0) {
      this.contents.paintOpacity = 160;
      if (-value > $gameParty.gold()) {
        value = -$gameParty.gold();
      }
    }

    const getText = value > 0 ? textGainItem.format(wordMoney) : textLoseItem.format(wordMoney);

    this.drawTextEx(getText, x, y, this.textWidth(getText) + 12, 'left');
    x += this.textWidth(getText);

    this.resetTextColor();
    this.drawIcon(iconMoney, x, y);
    x += this.getIconSize() + 4;
    const mainText = String(Math.abs(value)) + $dataSystem.currencyUnit;
    this.drawTextEx(mainText, x, y, 180, 'left');
  }


  const _windowBackLogInitialize = Window_BattleLog.prototype.initialize;
  Window_BattleLog.prototype.initialize = function (rect) {
    _windowBackLogInitialize.call(this, rect);
    this.visible = false;
  };

  Window_BattleLog.prototype.drawLineText = function (index) {
    addNoticeWindow(this._lines[index]);
    this._lines.pop();
  };

  Scene_Battle.prototype.updateLogWindowVisibility = function () {
    this._logWindow.visible = false;
  };

  // Scene_Battle.prototype.updateLogWindowVisibility = function () {
  //   this._logWindow.visible = !this._helpWindow.visible;
  // };

  // バトルリザルトメッセージ
  // BattleManager.displayVictoryMessage = function () {
  //   addNoticeWindow(TextManager.victory.format($gameParty.name()));
  // };

  // BattleManager.displayDefeatMessage = function () {
  //   addNoticeWindow(TextManager.defeat.format($gameParty.name()));
  // };

  // BattleManager.displayEscapeSuccessMessage = function () {
  //   addNoticeWindow(TextManager.escapeStart.format($gameParty.name()));
  // };

  // BattleManager.displayEscapeFailureMessage = function () {
  //   addNoticeWindow(TextManager.escapeStart.format($gameParty.name()));
  //   addNoticeWindow("\\." + TextManager.escapeFailure);
  // };

  // BattleManager.displayExp = function () {
  //   const exp = this._rewards.exp;
  //   if (exp > 0) {
  //     const text = TextManager.obtainExp.format(exp, TextManager.exp);
  //     addNoticeWindow("\\." + text);
  //   }
  // };

  // BattleManager.displayGold = function () {
  //   const gold = this._rewards.gold;
  //   if (gold > 0) {
  //     addNoticeWindow("\\." + TextManager.obtainGold.format(gold));
  //   }
  // };

  // BattleManager.displayDropItems = function () {
  //   const items = this._rewards.items;
  //   if (items.length > 0) {
  //     $gameMessage.newPage();
  //     for (const item of items) {
  //       addNoticeWindow(TextManager.obtainItem.format(item.name));
  //     }
  //   }
  // };

  const gameBattler_addstate = Game_Battler.prototype.addState;
  Game_Battler.prototype.addState = function (stateId) {
    gameBattler_addstate.apply(this, arguments);
    if (stateId <= 1) return;
    const state = $dataStates[stateId];
    if (!state) return;
    const message1 = state.message1;
    if (message1)
      addNoticeWindow(message1.format($gameParty.name()))
  };

  //
  // interpreter commands
  //

  // Change Gold
  const _Game_Interpreter_command125 = Game_Interpreter.prototype.command125;
  Game_Interpreter.prototype.command125 = function (params) {
    _Game_Interpreter_command125.call(this, params);

    const value = this.operateValue(params[0], params[1], params[2]);
    addGetGoldWindow(value);
    return true;
  };

  // Change Items
  const _Game_Interpreter_command126 = Game_Interpreter.prototype.command126;
  Game_Interpreter.prototype.command126 = function (params) {
    _Game_Interpreter_command126.call(this, params);

    const value = this.operateValue(params[1], params[2], params[3]);
    addGetItemWindow('item', params[0], value);
    return true;
  };

  // Change Weapons
  const _Game_Interpreter_command127 = Game_Interpreter.prototype.command127;
  Game_Interpreter.prototype.command127 = function (params) {
    _Game_Interpreter_command127.call(this, params);

    const value = this.operateValue(params[1], params[2], params[3]);
    addGetItemWindow('weapon', params[0], value);
    return true;
  };

  // Change Armors
  const _Game_Interpreter_command128 = Game_Interpreter.prototype.command128;
  Game_Interpreter.prototype.command128 = function (params) {
    _Game_Interpreter_command128.call(this, params);

    const value = this.operateValue(params[1], params[2], params[3]);
    addGetItemWindow('armor', params[0], value);
    return true;
  };

  Game_Party.prototype.gainItemWithLog = function (item, amount, includeEquip) {
    if (!item) return;
    addGetItemWindow('item', item.id, amount);
    $gameParty.gainItem(item, amount, includeEquip);
  };



})();
