//=============================================================================
// MPP_ActorEventImage.js
//=============================================================================
// Copyright (c) 2015 - 2024 Mokusei Penguin
// Released under the MIT license
// http://opensource.org/licenses/mit-license.php
//=============================================================================

/*:
 * @target MV MZ
 * @plugindesc The event image will be the same as that of the party members.
 * @author Mokusei Penguin
 * @url
 *
 * @help [version 2.0.0]
 *  - This plugin is for RPG Maker MV and MZ.
 * 
 * ▼ Plugin command
 *  - In MV, the variable N is referred to by writing 'v[N]' in the item for
 *    inputting a numerical value.
 *  - In MZ, in the item to enter a numerical value, select the text and
 *    write 'v[N]' to refer to the variable N.
 *  
 *  〇 MV / MZ
 *  
 *  〇 SetMemberImage eventId index  / setMemberImage
 *      eventId : event ID
 *      index   : Party Member Number (0 is the leader)
 *   - Set the event 'eventId' to the 'index'th party member image.
 * 
 *  〇 SetActorImage eventId actorId  / setActorImage
 *      eventId : event ID
 *      actorId : actor ID
 *   - Image event 'eventId' for actor 'actorId'
 * 
 * ================================
 * Mail : wood_penguin＠yahoo.co.jp (＠ is half-width)
 * Blog : http://woodpenguin.blog.fc2.com/
 * License : MIT license
 * 
 *  @command setMemberImage
 *      @desc 
 *      @arg eventId
 *          @desc 0:This event, 1..:EventID
 *          @type number
 *              @min 0
 *              @max 999
 *          @default 0
 *      @arg index
 *          @desc 0 is the leader
 *          @type number
 *              @min 0
 *              @max 99
 *          @default 0
 * 
 *  @command setActorImage
 *      @desc 
 *      @arg eventId
 *          @desc 0:This event, 1..:EventID
 *          @type number
 *              @min 0
 *              @max 999
 *          @default 0
 *      @arg actorId
 *          @desc 
 *          @type actor
 *          @default 1
 * 
 */

/*:ja
 * @target MV MZ
 * @plugindesc イベントの画像をパーティメンバーと同じにします。
 * @author 木星ペンギン
 * @url
 *
 * @help [version 2.0.0]
 *  - このプラグインはRPGツクールMVおよびMZ用です。
 * 
 * ▼ プラグインコマンド
 *  - MVでは数値を入力する項目で v[N] と記述することで変数N番を参照します。
 *  - MZでは数値を入力する項目で、テキストを選択して v[N] と記述することで
 *    変数N番を参照します。
 *  
 *  〇 MV / MZ
 *  
 *  〇 SetMemberImage eventId index  / メンバー画像設定
 *      eventId : イベントID
 *      index   : パーティメンバー番号(0が先頭)
 *   - イベント eventId 番を index 番目のパーティメンバーの画像にする
 * 
 *  〇 SetActorImage eventId actorId  / アクター画像設定
 *      eventId : イベントID
 *      actorId : アクターID
 *   - イベント eventId 番をアクター actorId 番の画像にする
 * 
 * ================================
 * Mail : wood_penguin＠yahoo.co.jp (＠は半角)
 * Blog : http://woodpenguin.blog.fc2.com/
 * License : MIT license
 * 
 *  @command setMemberImage
 *      @text メンバー画像設定
 *      @desc 
 *      @arg eventId
 *          @text イベントID
 *          @desc 0:このイベント, 1..:イベントID
 *          @type number
 *              @min 0
 *              @max 999
 *          @default 0
 *      @arg index
 *          @text パーティメンバー
 *          @desc 0が先頭
 *          @type number
 *              @min 0
 *              @max 99
 *          @default 0
 * 
 *  @command setActorImage
 *      @text アクター画像設定
 *      @desc 
 *      @arg eventId
 *          @text イベントID
 *          @desc 0:このイベント, 1..:イベントID
 *          @type number
 *              @min 0
 *              @max 999
 *          @default 0
 *      @arg actorId
 *          @text アクターID
 *          @desc 
 *          @type actor
 *          @default 1
 * 
 */

(() => {
    'use strict';

    const pluginName = 'MPP_ActorEventImage';
    
    //-------------------------------------------------------------------------
    // Game_Interpreter

    const _mzCommands = {
        SetMemberImage: { name: 'setMemberImage', keys: ['eventId', 'index'] },
        SetActorImage: { name: 'setActorImage', keys: ['eventId', 'actorId'] }
    };
    Object.assign(_mzCommands, {
        'メンバー画像設定': _mzCommands.SetMemberImage,
        'アクター画像設定': _mzCommands.SetActorImage,
    });

    const _Game_Interpreter_pluginCommand = Game_Interpreter.prototype.pluginCommand;
    Game_Interpreter.prototype.pluginCommand = function(command, args) {
        _Game_Interpreter_pluginCommand.apply(this, arguments);
        const mzCommand = _mzCommands[command];
        if (mzCommand) {
            const args2 = Object.assign(
                {}, ...mzCommand.keys.map((k, i) => ({ [k]: args[i] }))
            );
            PluginManager.callCommandMV(this, pluginName, mzCommand.name, args2);
        }
    };

    //-------------------------------------------------------------------------
    // PluginManager
    
    if (!PluginManager.registerCommand && !PluginManager._commandsMV) {
        PluginManager._commandsMV = {};

        PluginManager.registerCommandMV = function(pluginName, commandName, func) {
            const key = pluginName + ':' + commandName;
            this._commandsMV[key] = func;
        };
        
        PluginManager.callCommandMV = function(self, pluginName, commandName, args) {
            const key = pluginName + ':' + commandName;
            const func = this._commandsMV[key];
            if (typeof func === 'function') {
                func.bind(self)(args);
            }
        };
    }

    const _registerCommandName = PluginManager.registerCommand
        ? 'registerCommand'
        : 'registerCommandMV';
    
    PluginManager[_registerCommandName](pluginName, 'setMemberImage', function(args) {
        const mppValue = PluginManager.mppValue;
        const event = this.character(mppValue(args.eventId));
        const actor = $gameParty.members()[mppValue(args.index)];
        if (event && actor) {
            event.setImage(actor.characterName(), actor.characterIndex());
        }
    });

    PluginManager[_registerCommandName](pluginName, 'setActorImage', function(args) {
        const mppValue = PluginManager.mppValue;
        const event = this.character(mppValue(args.eventId));
        const actor = $gameActors.actor(mppValue(args.actorId));
        if (event && actor) {
            event.setImage(actor.characterName(), actor.characterIndex());
        }
    });

    PluginManager.mppValue = function(value) {
        const match = /^V\[(\d+)\]$/i.exec(value);
        return match ? $gameVariables.value(+match[1]) : +value;
    };
    
})();
