//=============================================================================
// Yanfly Engine Plugins - Skill Cost Extension - Cooldowns
// YEP_X_SkillCooldowns.js
//=============================================================================

var Imported = Imported || {};
Imported.YEP_X_SkillCooldowns = true;

var Yanfly = Yanfly || {};
Yanfly.SCD = Yanfly.SCD || {};
Yanfly.SCD.version = 1.12;

//=============================================================================
/*:ja
 * @plugindesc v1.12 (要YEP_SkillCore.js)スキルが連続で使われるのを防ぐクールダウンのシステムを導入できます。
 * @author Yanfly Engine Plugins
 *
 * @param ---クールダウン---
 * @default
 *
 * @param Cooldown Format
 * @text 表示形式
 * @parent ---クールダウン---
 * @desc クールダウンの表示形式
 * 残りターン数:%1
 * @default %1CD
 *
 * @param Cooldown Font Size
 * @text フォントサイズ
 * @parent ---クールダウン---
 * @type number
 * @min 0
 * @desc クールダウン表示のフォントサイズ
 * デフォルト:28
 * @default 20
 *
 * @param Cooldown Text Color
 * @text フォント色
 * @parent ---クールダウン---
 * @type number
 * @min 0
 * @max 31
 * @desc クールダウン表示のフォント色
 * @default 6
 *
 * @param Cooldown Icon
 * @text アイコン
 * @parent ---クールダウン---
 * @type number
 * @min 0
 * @desc クールダウン表示に使われるアイコン。アイコン無しは0
 * @default 75
 *
 * @param Cooldown After Battle
 * @text 戦闘後処理
 * @parent ---クールダウン---
 * @type number
 * @desc 戦闘後のクールダウンへの処理
 * @default -10
 *
 * @param Cooldown Steps
 * @text マップ歩数
 * @parent ---クールダウン---
 * @type number
 * @min 0
 * @desc マップ上でクールダウンを 1 するために必要な歩数
 * @default 5
 *
 * @param Cooldown Bypass
 * @text 影響を受けないスキル
 * @parent ---クールダウン---
 * @desc クールダウンの影響を受けないスキル(例えば攻撃/防御など)
 * @default 1 2 3 4 5 6 7
 *
 * @param Cooldown Bypass List
 * @text 影響を受けないスキル一覧
 * @parent ---クールダウン---
 * @type skill[]
 * @desc クールダウンの影響を受けないスキル(例えば攻撃/防御など)のリスト。RPGツクールMV 1.5.0以降が必要です。
 * @default []
 *
 * @param ---ウォームアップ---
 * @default
 *
 * @param Warmup Format
 * @text 表示形式
 * @parent ---ウォームアップ---
 * @desc ウォームアップの表示テキスト
 * 残りターン数:%1
 * @default %1WU
 *
 * @param Warmup Font Size
 * @text フォントサイズ
 * @parent ---ウォームアップ---
 * @type number
 * @min 1
 * @desc ウォームアップ表示のフォントサイズ
 * デフォルト:28
 * @default 20
 *
 * @param Warmup Text Color
 * @text フォント色
 * @parent ---ウォームアップ---
 * @type number
 * @min 0
 * @max 31
 * @desc ウォームアップ表示のフォント色
 * @default 4
 *
 * @param Warmup Icon
 * @text アイコン
 * @parent ---ウォームアップ---
 * @type number
 * @min 0
 * @desc ウォームアップの表示アイコン
 * 0を入力するとアイコン無し
 * @default 75
 *
 * @param ---Battle Core---
 * @text ---バトルコア---
 * @default
 *
 * @param Time Based
 * @text クールダウン基準
 * @parent ---Battle Core---
 * @type boolean
 * @on 時間基準
 * @off ターン基準
 * @desc 戦闘システムがティック基準の場合、クールダウン基準をティックか時間か設定。ターン:false / 時間:true
 * @default false
 *
 * @param Turn Time
 * @text ティック数
 * @parent ---Battle Core---
 * @type number
 * @min 1
 * @desc 1クールダウン・ターンに相当するのに必要なティック数
 * @default 100
 *
 * @help
 * 翻訳:ムノクラ
 * https://fungamemake.com/
 * https://twitter.com/munokura/
 *
 * ===========================================================================
 * 導入
 * ===========================================================================
 *
 * このプラグインを利用するには、YEP_SkillCoreが必要です。
 * リスト内ではYEP_SkillCoreの下にこのプラグインが来るようにしてください。
 *
 * このプラグインで、スキルに「クールダウン」を導入できます。
 * これにより、対象スキルの連続使用を制限することができます。
 *
 * ===========================================================================
 * クールダウンタイプ
 * ===========================================================================
 *
 * Cooldown (Standard)
 * この機能が働くと、特定のターンの間そのスキルを使うことができません。
 * また、クールダウンをさせるための方法はいくつかあります。
 * まずは、ただ単純に待つという方法。
 * 戦闘中、各ターンごとにクールダウンが進行します。
 * スキルなどで、これを早めることもできます。
 * 2つ目の方法は、戦闘を終了することです。
 * 戦闘終了によって、全てのクールダウンを特定の値だけ進行することができます。
 * (値はパラメータで指定可能)
 * 3つ目の方法は、フィールドマップを歩き回ることです。
 * 特定の歩数を歩くことでスキルのクールダウンを進行することができます。
 *
 * Warmups
 * ウォームアップは、クールダウンと同様の働きを持ちます。
 * 一定の時間が経つまでスキルを使わせないようにします。
 * しかしクールダウンと異なる点として、
 * 戦闘の最初に一度だけ起こる、という点が挙げられます。
 * ウォームアップタイマーにより、
 * 戦闘でそのスキルが使われるタイミングを制限することができ、
 * またそれは戦闘終了後は一時的に消去されます。
 * ウォームアップは、既存のクールダウン上に重ねられることは有りません。
 * スキルがウォームアップ段階の時にクールダウンが既に起こっている場合、
 * 両方が同時にアップデートされます。
 *
 * Linked Cooldowns
 * 使用したスキルが、ライブラリにクールダウンを持った他のスキルを
 * 発動させた時に起こります。
 * このクールダウンの他の特性は、通常のクールダウン(前述)と同様のものです。
 * このクールダウンタイプは、
 * スキルタイプクールダウン(後述)やグローバルクールダウンよりも優先されます。
 *
 * Skill Type Cooldowns
 * スキルタイプクールダウンが起こると、
 * バトラーのスキルライブラリ内で該当する全てのスキルが、
 * クールダウンの対象となります。
 * このクールダウンの他の特性は、通常のクールダウン(前述)と同様のものです。
 * このクールダウンタイプは、グローバルクールダウンよりも優先されます。
 *
 * 既にクールダウンを持つスキルに、さらに別のクールダウンが適用される場合、
 * 常に最大値が採用されます。
 * つまり、もしスキルが3ターンのクールダウンを持ち、
 * スキルタイプクールダウンは1に設定されていた時、3ターンの方が採用されます。
 * 同様に、もしスキルが3ターン、スキルタイプクールダウンが5ターンだった場合、
 * 5ターンの方が採用されます。
 *
 * ===========================================================================
 * メモタグ
 * ===========================================================================
 *
 * スキルのクールダウンを変更するには下記のメモタグを用いてください。
 *
 * スキルのメモタグ:
 *   <Cooldown: x>
 *   スキルに対するクールダウンを x ターンに設定します。
 *   このクールダウンは、このスキルにのみ単独で有効です。
 *   これはスキルタイプクールダウンやグローバルクールダウンより優先されます。
 *
 *   <Warmup: x>
 *   スキルに対するウォームアップを x ターンに設定します。
 *   新たな戦闘に入る時、スキルはウォームアップ中になり、
 *   ウォームアップが完了するまで使用することはできません。
 *
 *   <After Battle Cooldown: +x>
 *   <After Battle Cooldown: -x>
 *   戦闘終了後(勝利/敗北/逃走)、
 *    +x / -x だけ、スキルのクールダウン数を増減させます。
 *
 *   <Cooldown Steps: x>
 *   戦闘外で、毎 x 歩ごとにスキルのクールダウンが 1 進行します。
 *
 *   <Skill x Cooldown: y>
 *   <Skill name Cooldown: y>
 *   このスキルを使う時、スキルコストを消費した後で、
 *   x に該当するタイプを持つスキルは、ターン数 y のクールダウンを持ちます。
 *   これはグローバルクールダウンより優先されます。
 *
 *   <SType x Cooldown: y>
 *   このスキルを使う時、スキルコストを消費した後で、
 *   x に該当するタイプを持つスキルは、ターン数 y のクールダウンを持ちます。
 *   これはグローバルクールダウンより優先されます。
 *
 *   <Global Cooldown: x>
 *   このスキルを使う時、バトラーのライブラリにある全てのスキルには、
 *   ターン数 x のクールダウンが課せられます。
 *   これは個々のクールダウンやスキルタイプクールダウンよりも
 *   低い優先度として扱われます。
 *
 *   <Bypass Cooldown>
 *   その種類に関わらずスキルにクールダウンを無視させます。
 *   攻撃、防御、逃走等、クールダウンを適用しないスキルに使ってください。
 *
 * スキル、アイテムのメモタグ:
 *   <Skill x Cooldown: +y>
 *   <Skill x Cooldown: -y>
 *   <Skill name Cooldown: +y>
 *   <Skill name Cooldown: -y>
 *   このスキルにヒットした対象のスキル x が y の値で調整されます。
 *   これは使用者には無効で、対象のみに有効です。
 *
 *   <SType x Cooldown: +y>
 *   <SType x Cooldown: -y>
 *   このスキルにヒットした対象について、
 *   スキルライブラリ内の x タイプのクールダウンが y の値で調整されます。
 *   使用者には無効で、対象のみに有効です。
 *
 *   <Global Cooldown: +x>
 *   <Global Cooldown: -x>
 *   このスキルにヒットした対象について、
 *   スキルライブラリ内の全てのスキルのクールダウンが y の値で調整されます。
 *   使用者には無効で、対象のみに有効です。
 *
 * アクター、職業、敵、武器、防具、ステータスのメモタグ:
 *
 *   <Skill x Cooldown Duration: y%>
 *   <Skill name Cooldown Duration: y%>
 *   クールダウンのコストが適用された時、
 *   スキル x のクールダウン継続時間を、y% に変更します。
 *   これはスキル x にのみ作用します。
 *
 *   <SType x Cooldown Duration: y%>
 *   クールダウンのコストが適用された時、
 *   タイプ x のスキルのクールダウン継続時間を y% に変更します。
 *   これはスキル x にのみ作用します。
 *
 *   <Global Cooldown Duration: x%>
 *   クールダウンのコストが適用された時、全てのスキルのクールダウン継続時間を
 *   x% に変更します。
 *
 *   <Skill x Cooldown Rate: y%>
 *   <Skill name Cooldown Rate: y%>
 *   クールダウンカウンターが下がった時、
 *   スキル x に対するクールダウンレートを y% に変更します。
 *   これはスキル x にのみ作用します。
 *
 *   <SType x Cooldown Rate: y%>
 *   クールダウンカウンターが下がった時、
 *   タイプ x のスキルのクールダウンレートを y% に変更します。
 *   これはスキルタイプ x にのみ作用します。
 *
 *   <Global Cooldown Rate: x%>
 *   クールダウンカウンターが下がった時、
 *   全てのスキルのクールダウンレートを x% に変更します。
 *
 *   <Skill x Cooldown: +y>
 *   <Skill x Cooldown: -y>
 *   <Skill name Cooldown: +y>
 *   <Skill name Cooldown: -y>
 *   使用者が x というスキルを使った場合、
 *   そのスキルのクールダウンの値は増加/減少します。
 *   これは使用者がアクター、職業、クラス、敵である、
 *   またはその武器や防具を身に着けている、
 *   もしくはこのメモタグを持ったステートに影響を受けている時に発動します。
 *   これらの修正はレートと継続時間の計算が完了した時に適用されます。
 *
 *   <SType x Cooldown: +y>
 *   <SType x Cooldown: -y>
 *   使用者が x タイプのスキルを使った場合、
 *   そのスキルのクールダウンの値は増加/減少します。
 *   これは使用者がアクター、職業、クラス、敵である、
 *   またはその武器や防具を身に着けている、
 *   もしくはこのメモタグを持ったステートに影響を受けている時に発動します。
 *   これらの修正はレートと継続時間の計算が完了した時に適用されます。
 *
 *   <Global Cooldown: +x>
 *   <Global Cooldown: -x>
 *   使用者がスキルを使った場合(種類問わず)、
 *   そのスキルのクールダウンの値は増加/減少します。
 *   これは使用者がアクター、職業、クラス、敵である、
 *   またはその武器や防具を身に着けている、
 *   もしくはこのメモタグを持ったステートに影響を受けている時に発動します。
 *   これらの修正はレートと継続時間の計算が完了した時に適用されます。
 *
 *   <Skill x Warmup: +y>
 *   <Skill x Warmup: -y>
 *   <Skill name Warmup: +y>
 *   <Skill name Warmup: -y>
 *   戦闘開始時、スキル x のウォームアップの値は増加/減少します。
 *   これは使用者がアクター、職業、クラス、敵である、
 *   またはその武器や防具を身に着けている、
 *   もしくはこのメモタグを持ったステートに影響を受けている時に発動します。
 *   これらの修正はレートと継続時間の計算が完了した時に適用されます。
 *
 *   <SType x Warmup: +y>
 *   <SType x Warmup: -y>
 *   戦闘開始時、x タイプの全てのスキルのウォームアップの値は増加/減少します。
 *   これは使用者がアクター、職業、クラス、敵である、
 *   またはその武器や防具を身に着けている、
 *   もしくはこのメモタグを持ったステートに影響を受けている時に発動します。
 *   これらの修正はレートと継続時間の計算が完了した時に適用されます。
 *
 *   <Global Warmup: +x>
 *   <Global Warmup: -x>
 *   戦闘開始時、全てのスキルのウォームアップの値は増加/減少します。
 *   これは使用者がアクター、職業、クラス、敵である、
 *   またはその武器や防具を身に着けている、
 *   もしくはこのメモタグを持ったステートに影響を受けている時に発動します。
 *   これらの修正はレートと継続時間の計算が完了した時に適用されます。
 *
 * ===========================================================================
 * ルナティックモード - 特殊なクールダウン
 * ===========================================================================
 *
 * 特殊なコードをスキルのメモ欄に用いて、
 * スキルが使われる時のクールダウンの値を指定することができます。
 *
 * スキルのメモタグ:
 *   <Cooldown Eval>
 *   cooldown = x;
 *   cooldown += x;
 *   </Cooldown Eval>
 *   スキルに上記のタグを挿入し、クールダウンに必要なターン数を決定
 *
 *   <Warmup Eval>
 *   warmup = x;
 *   warmup += x;
 *   </Warmup Eval>
 *   スキルの上記のタグを挿入し、ウォームアップに必要なターン数を決定
 *
 * ===========================================================================
 * Yanfly Engine Plugins - Battle Engine Extension - Action Sequence Commands
 * ===========================================================================
 *
 * プラグインマネージャーに、このプラグインと共にYEP_BattleEngineCore.jsを、
 * 入れていれば、下記のクールダウン関連のアクションシーケンスも利用できます。
 *
 *===========================================================================
 * GLOBAL COOLDOWN: targets, +X
 * GLOBAL COOLDOWN: targets, -X
 * GLOBAL COOLDOWN: targets, X
 * - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
 * x の値で、クールダウンを全ての対象に設定できます。
 * クールダウンが有効な全てのスキルに対して適用されます。
 * - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
 * 例: global cooldown: target, +5
 *     global cooldown: user, -3
 *     global cooldown: enemies, 10
 *===========================================================================
 *
 *===========================================================================
 * SKILL X COOLDOWN: targets, +Y
 * SKILL X COOLDOWN: targets, -Y
 * SKILL X COOLDOWN: targets, Y
 * - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
 * 対象に対して、X というスキルに Y の値でクールダウンを付与します。
 * 特定のスキル X にのみ適用されます。
 * - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
 * 例: skill 10 cooldown: target, +5
 *     skill 12 cooldown: user, -3
 *     skill 15 cooldown: enemies, 10
 *===========================================================================
 *
 *===========================================================================
 * SKILL TYPE X COOLDOWN: targets, +Y
 * SKILL TYPE X COOLDOWN: targets, -Y
 * SKILL TYPE X COOLDOWN: targets, Y
 * - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
 * 対象に対して、X というスキルタイプに Y の値でクールダウンを付与します。
 * 特定のスキル X にのみ適用されます。
 * - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - - -
 * 例: skill type 1 cooldown: target, +5
 *     skill type 2 cooldown: user, -3
 *     skill type 5 cooldown: enemies, 10
 *===========================================================================
 *
 * ===========================================================================
 * Changelog
 * ===========================================================================
 *
 * Version 1.12:
 * - Updated for RPG Maker MV version 1.5.0.
 * - Added Parameter: Cooldown Bypass List
 *
 * Version 1.11:
 * - Lunatic Mode fail safes added.
 *
 * Version 1.10:
 * - Compatibility update with Equip Battle Skills.
 * - Documentation update. Added help information for <warmup: x>.
 *
 * Version 1.09:
 * - Fixed a bug with the <Skill x Cooldown Rate: y%>,
 * <SType x Cooldown Rate: y%>, and <Global Cooldown Rate: x%> notetags not
 * working as intended.
 *
 * Version 1.08:
 * - Updated for RPG Maker MV version 1.1.0.
 *
 * Version 1.07:
 * - Named versions of these notetags have been added:
 * <Skill x Cooldown: y>, <Skill x Cooldown: +/-y>,
 * <Skill x Cooldown Duration: y%>, <Skill x Cooldown: +/-y>,
 * <Skill x Warmup: +/-y>
 *
 * Version 1.06a:
 * - Fixed a bug with cooldown duration modifiers not modifying by the correct
 * value indicated.
 * - Added a fail safe for when there are no targets.
 *
 * Version 1.05:
 * - Fixed a bug that prevented <Cooldown Eval> from running properly.
 *
 * Version 1.04:
 * - Fixed a bug that didn't alter cooldowns correctly.
 *
 * Version 1.03:
 * - Optimized for Battle Engine Core v1.08.
 *
 * Version 1.02a:
 * - Added return for drawSkillCost to assist others scripters when making
 * compatibility notes.
 *
 * Version 1.01:
 * - Cooldowns can now be applied to skills that aren't learned by the actor.
 *
 * Version 1.00:
 * - Finished plugin!
 */
//=============================================================================


if (Imported.YEP_SkillCore) {

//=============================================================================
// Parameter Variables
//=============================================================================

Yanfly.Parameters = PluginManager.parameters('YEP_X_SkillCooldowns');
Yanfly.Param = Yanfly.Param || {};
Yanfly.Icon = Yanfly.Icon || {};

Yanfly.Param.CDFmt = String(Yanfly.Parameters['Cooldown Format']);
Yanfly.Param.CDFontSize = Number(Yanfly.Parameters['Cooldown Font Size']);
Yanfly.Param.CDTextColor = Number(Yanfly.Parameters['Cooldown Text Color']);
Yanfly.Icon.Cooldown = Number(Yanfly.Parameters['Cooldown Icon']);
Yanfly.Param.CDAfterBattle = Number(Yanfly.Parameters['Cooldown After Battle']);
Yanfly.Param.CDSteps = Number(Yanfly.Parameters['Cooldown Steps']);

Yanfly.Param.WUFmt = String(Yanfly.Parameters['Warmup Format']);
Yanfly.Param.WUFontSize = Number(Yanfly.Parameters['Warmup Font Size']);
Yanfly.Param.WUTextColor = Number(Yanfly.Parameters['Warmup Text Color']);
Yanfly.Param.CDTimeBased = String(Yanfly.Parameters['Time Based']);
Yanfly.Param.CDTurnTime = Number(Yanfly.Parameters['Turn Time']);
Yanfly.Icon.Warmup = Number(Yanfly.Parameters['Warmup Icon']);

Yanfly.SetupParameters = function() {
  Yanfly.Param.CDBypass = String(Yanfly.Parameters['Cooldown Bypass']);
  Yanfly.Param.CDBypass = Yanfly.Param.CDBypass.split(' ');
  for (var i = 0; i < Yanfly.Param.CDBypass.length; ++i) {
    Yanfly.Param.CDBypass[i] = parseInt(Yanfly.Param.CDBypass[i]);
  }
  var data = JSON.parse(Yanfly.Parameters['Cooldown Bypass List'] || '[]');
  for (var i = 0; i < data.length; ++i) {
    var id = parseInt(data[i]);
    if (id <= 0) continue;
    if (Yanfly.Param.CDBypass.contains(id)) continue;
    Yanfly.Param.CDBypass.push(id);
  }
};
Yanfly.SetupParameters()

//=============================================================================
// DataManager
//=============================================================================

Yanfly.SCD.DataManager_isDatabaseLoaded = DataManager.isDatabaseLoaded;
DataManager.isDatabaseLoaded = function() {
  if (!Yanfly.SCD.DataManager_isDatabaseLoaded.call(this)) return false;
  if (!Yanfly._loaded_YEP_X_SkillCooldowns) {
    this.processSCDNotetagsS($dataSkills);
    this.processSCDNotetags1($dataSkills);
    this.processSCDNotetags2($dataSkills);
    this.processSCDNotetags2($dataItems);
    this.processSCDNotetags2($dataActors);
    this.processSCDNotetags2($dataClasses);
    this.processSCDNotetags2($dataEnemies);
    this.processSCDNotetags2($dataWeapons);
    this.processSCDNotetags2($dataArmors);
    this.processSCDNotetags2($dataStates);
    this.processSCDNotetags3($dataActors);
    this.processSCDNotetags3($dataClasses);
    this.processSCDNotetags3($dataEnemies);
    this.processSCDNotetags3($dataWeapons);
    this.processSCDNotetags3($dataArmors);
    this.processSCDNotetags3($dataStates);
    Yanfly._loaded_YEP_X_SkillCooldowns = true;
  }
  return true;
};

DataManager.processSCDNotetagsS = function(group) {
  if (Yanfly.SkillIdRef) return;
  Yanfly.SkillIdRef = {};
  for (var n = 1; n < group.length; n++) {
    var obj = group[n];
    if (obj.name.length <= 0) continue;
    Yanfly.SkillIdRef[obj.name.toUpperCase()] = n;
  }
};

DataManager.processSCDNotetags1 = function(group) {
  for (var n = 1; n < group.length; n++) {
    var obj = group[n];
    var notedata = obj.note.split(/[\r\n]+/);

    obj.cooldown = {};
    obj.stypeCooldown = {}
    obj.globalCooldown = 0;
    obj.afterBattleCooldown = Yanfly.Param.CDAfterBattle;
    obj.cooldownSteps = Math.max(1, parseInt(Yanfly.Param.CDSteps));
    obj.warmup = 0;
    obj.bypassCooldown = Yanfly.Param.CDBypass.contains(obj.id);
    obj.cooldownEval = '';
    obj.warmupEval = '';
    var evalMode = 'none';

    for (var i = 0; i < notedata.length; i++) {
      var line = notedata[i];
      if (line.match(/<(?:COOLDOWN):[ ](\d+)>/i)) {
        obj.cooldown[obj.id] = parseInt(RegExp.$1);
      } else if (line.match(/<(?:AFTER BATTLE COOLDOWN):[ ]([\+\-]\d+)>/i)) {
        obj.afterBattleCooldown = parseInt(RegExp.$1);
      } else if (line.match(/<(?:COOLDOWN STEPS):[ ](\d+)>/i)) {
        obj.cooldownSteps = parseInt(RegExp.$1);
      } else if (line.match(/<(?:WARMUP):[ ](\d+)>/i)) {
        obj.warmup = parseInt(RegExp.$1);
      } else if (line.match(/<(?:SKILL)[ ](\d+)[ ](?:COOLDOWN):[ ](\d+)>/i)) {
        obj.cooldown[parseInt(RegExp.$1)] = parseInt(RegExp.$2);
      } else if (line.match(/<(?:SKILL)[ ](.*)[ ](?:COOLDOWN):[ ](\d+)>/i)) {
        var name = String(RegExp.$1).toUpperCase();
        if (Yanfly.SkillIdRef[name]) {
          var id = Yanfly.SkillIdRef[name];
        } else {
          continue;
        }
        obj.cooldown[id] = parseInt(RegExp.$2);
      } else if (line.match(/<(?:STYPE)[ ](\d+)[ ](?:COOLDOWN):[ ](\d+)>/i)) {
        obj.stypeCooldown[parseInt(RegExp.$1)] = parseInt(RegExp.$2);
      } else if (line.match(/<(?:GLOBAL COOLDOWN):[ ](\d+)>/i)) {
        obj.globalCooldown = parseInt(RegExp.$1);
      } else if (line.match(/<(?:BYPASS COOLDOWN)>/i)) {
        obj.bypassCooldown = true;
      } else if (line.match(/<(?:COOLDOWN EVAL)>/i)) {
        obj.cooldown[obj.id] = obj.cooldown[obj.id] || 0;
        evalMode = 'cooldown';
      } else if (line.match(/<\/(?:COOLDOWN EVAL)>/i)) {
        evalMode = 'none';
      } else if (line.match(/<(?:WARMUP EVAL)>/i)) {
        evalMode = 'warmup';
      } else if (line.match(/<\/(?:WARMUP EVAL)>/i)) {
        evalMode = 'none';
      } else if (evalMode === 'cooldown') {
        obj.cooldownEval = obj.cooldownEval + line + '\n';
      } else if (evalMode === 'warmup') {
        obj.warmupEval = obj.warmupEval + line + '\n';
      }
    }
  }
};

DataManager.processSCDNotetags2 = function(group) {
  var note1 = /<(?:SKILL)[ ](\d+)[ ](?:COOLDOWN):[ ]([\+\-]\d+)>/i;
  var note1a = /<(?:SKILL)[ ](.*)[ ](?:COOLDOWN):[ ]([\+\-]\d+)>/i;
  var note2 = /<(?:STYPE)[ ](\d+)[ ](?:COOLDOWN):[ ]([\+\-]\d+)>/i;
  var note3 = /<(?:GLOBAL COOLDOWN):[ ]([\+\-]\d+)>/i;
  var note4 = /<(?:SKILL)[ ](\d+)[ ](?:WARMUP):[ ]([\+\-]\d+)>/i;
  var note4a = /<(?:SKILL)[ ](.*)[ ](?:WARMUP):[ ]([\+\-]\d+)>/i;
  var note5 = /<(?:STYPE)[ ](\d+)[ ](?:WARMUP):[ ]([\+\-]\d+)>/i;
  var note6 = /<(?:GLOBAL WARMUP):[ ]([\+\-]\d+)>/i;
  for (var n = 1; n < group.length; n++) {
    var obj = group[n];
    var notedata = obj.note.split(/[\r\n]+/);

    obj.cooldownChange = {};
    obj.stypeCooldownChange = {};
    obj.globalCooldownChange = 0;
    obj.warmupChange = {};
    obj.stypeWarmupChange = {};
    obj.globalWarmupChange = 0;

    for (var i = 0; i < notedata.length; i++) {
      var line = notedata[i];
      if (line.match(note1)) {
        obj.cooldownChange[parseInt(RegExp.$1)] = parseInt(RegExp.$2);
      } else if (line.match(note1a)) {
        var name = String(RegExp.$1).toUpperCase();
        if (Yanfly.SkillIdRef[name]) {
          var id = Yanfly.SkillIdRef[name];
        } else {
          continue;
        }
        obj.cooldownChange[id] = parseInt(RegExp.$2);
      } else if (line.match(note2)) {
        obj.stypeCooldownChange[parseInt(RegExp.$1)] = parseInt(RegExp.$2);
      } else if (line.match(note3)) {
        obj.globalCooldownChange = parseInt(RegExp.$1);
      } else if (line.match(note4)) {
        obj.warmupChange[parseInt(RegExp.$1)] = parseInt(RegExp.$2);
      } else if (line.match(note4a)) {
        var name = String(RegExp.$1).toUpperCase();
        if (Yanfly.SkillIdRef[name]) {
          var id = Yanfly.SkillIdRef[name];
        } else {
          continue;
        }
        obj.warmupChange[id] = parseInt(RegExp.$2);
      } else if (line.match(note5)) {
        obj.stypeWarmupChange[parseInt(RegExp.$1)] = parseInt(RegExp.$2);
      } else if (line.match(note6)) {
        obj.globalWarmupChange = parseInt(RegExp.$1);
      }
    }
  }
};

DataManager.processSCDNotetags3 = function(group) {
  var note1 = /<(?:SKILL)[ ](\d+)[ ](?:COOLDOWN DURATION):[ ](\d+)([%％])>/i;
  var note1a = /<(?:SKILL)[ ](.*)[ ](?:COOLDOWN DURATION):[ ](\d+)([%％])>/i;
  var note2 = /<(?:STYPE)[ ](\d+)[ ](?:COOLDOWN DURATION):[ ](\d+)([%％])>/i;
  var note3 = /<(?:GLOBAL COOLDOWN DURATION):[ ](\d+)([%％])>/i;
  var note4 = /<(?:SKILL)[ ](\d+)[ ](?:COOLDOWN RATE):[ ](\d+)([%％])>/i;
  var note4a = /<(?:SKILL)[ ](.*)[ ](?:COOLDOWN RATE):[ ](\d+)([%％])>/i;
  var note5 = /<(?:STYPE)[ ](\d+)[ ](?:COOLDOWN RATE):[ ](\d+)([%％])>/i;
  var note6 = /<(?:GLOBAL COOLDOWN RATE):[ ](\d+)([%％])>/i;
  for (var n = 1; n < group.length; n++) {
    var obj = group[n];
    var notedata = obj.note.split(/[\r\n]+/);

    obj.cooldownDuration = {};
    obj.stypeCooldownDuration = {};
    obj.globalCooldownDuration = 1.0;
    obj.cooldownRate = {};
    obj.stypeCooldownRate = {};
    obj.globalCooldownRate = 1.0;

    for (var i = 0; i < notedata.length; i++) {
      var line = notedata[i];
      if (line.match(note1)) {
        obj.cooldownDuration[parseInt(RegExp.$1)] = 
          parseFloat(RegExp.$2) * 0.01;
      } else if (line.match(note1a)) {
        var name = String(RegExp.$1).toUpperCase();
        if (Yanfly.SkillIdRef[name]) {
          var id = Yanfly.SkillIdRef[name];
        } else {
          continue;
        }
        obj.cooldownDuration[id] = parseFloat(RegExp.$2) * 0.01;
      } else if (line.match(note2)) {
        obj.stypeCooldownDuration[parseInt(RegExp.$1)] = 
          parseFloat(RegExp.$2) * 0.01;
      } else if (line.match(note3)) {
        obj.globalCooldownDuration = parseFloat(RegExp.$1) * 0.01;
      } else if (line.match(note4)) {
        obj.cooldownRate[parseInt(RegExp.$1)] = parseFloat(RegExp.$2) * 0.01;
      } else if (line.match(note4a)) {
        var name = String(RegExp.$1).toUpperCase();
        if (Yanfly.SkillIdRef[name]) {
          var id = Yanfly.SkillIdRef[name];
        } else {
          continue;
        }
        obj.cooldownRate[id] = parseFloat(RegExp.$2) * 0.01;
      } else if (line.match(note5)) {
        obj.stypeCooldownRate[parseInt(RegExp.$1)] = 
          parseFloat(RegExp.$2) * 0.01;
      } else if (line.match(note6)) {
        obj.globalCooldownRate = parseFloat(RegExp.$1 * 0.01);
      }
    }
  }
};

//=============================================================================
// BattleManager
//=============================================================================

Yanfly.SCD.BattleManager_endBattle = BattleManager.endBattle;
BattleManager.endBattle = function(result) {
    Yanfly.SCD.BattleManager_endBattle.call(this, result);
    $gameParty.endBattleCooldowns();
};

BattleManager.timeBasedCooldowns = function() {
  if (!$gameParty.inBattle()) return false;
  if (!Imported.YEP_BattleEngineCore) return false;
  if (this.isTurnBased()) return false;
  if (this._timeBasedCooldowns !== undefined) return this._timeBasedCooldowns;
  this._timeBasedCooldowns = eval(Yanfly.Param.CDTimeBased);
  return this._timeBasedCooldowns;
};

if (Imported.YEP_BattleEngineCore) {
Yanfly.SCD.BattleManager_processActionSequence =
  BattleManager.processActionSequence;
  BattleManager.processActionSequence = function(actionName, actionArgs) {
    // GLOBAL COOLDOWN
    if (actionName === 'GLOBAL COOLDOWN') {
      return this.actionGlobalCooldown(actionArgs);
    }
    // SKILL COOLDOWN
    if (actionName.match(/SKILL[ ](\d+)[ ]COOLDOWN/i)) {
      return this.actionSkillCooldown(parseInt(RegExp.$1), actionArgs);
    }
    // SKILL TYPE COOLDOWN
    if (actionName.match(/SKILL[ ]TYPE[ ](\d+)[ ]COOLDOWN/i)) {
      return this.actionSTypeCooldown(parseInt(RegExp.$1), actionArgs);
    }
    // STYPE COOLDOWN
    if (actionName.match(/STYPE[ ](\d+)[ ]COOLDOWN/i)) {
      return this.actionSTypeCooldown(parseInt(RegExp.$1), actionArgs);
    }
    return Yanfly.SCD.BattleManager_processActionSequence.call(this,
      actionName, actionArgs);
  };
};

BattleManager.actionGlobalCooldown = function(actionArgs) {
    var targets = this.makeActionTargets(actionArgs[0]);
    if (targets.length < 1) return true;
    var cmd = actionArgs[1];
    if (cmd.match(/([\+\-]\d+)/i)) {
      var value = parseInt(RegExp.$1);
      for (var t = 0; t < targets.length; ++t) {
        var target = targets[t];
        for (var i = 0; i < target.allSkills().length; ++i) {
          var skill = target.allSkills()[i];
          if (skill) {
            target.addCooldown(skill.id, value);
          }
        }
      }
    } else if (cmd.match(/(\d+)/i)) {
      var value = parseInt(RegExp.$1);
      for (var t = 0; t < targets.length; ++t) {
        var target = targets[t];
        for (var i = 0; i < target.allSkills().length; ++i) {
          var skill = target.allSkills()[i];
          if (skill) {
            target.setCooldown(skill.id, value);
          }
        }
      }
    } else {
      return true;
    }
    return true;
};

BattleManager.actionSkillCooldown = function(skillId, actionArgs) {
    var targets = this.makeActionTargets(actionArgs[0]);
    if (targets.length < 1) return true;
    var cmd = actionArgs[1];
    if (cmd.match(/([\+\-]\d+)/i)) {
      var value = parseInt(RegExp.$1);
      for (var t = 0; t < targets.length; ++t) {
        var target = targets[t];
        target.addCooldown(skillId, value);
      }
    } else if (cmd.match(/(\d+)/i)) {
      var value = parseInt(RegExp.$1);
      for (var t = 0; t < targets.length; ++t) {
        var target = targets[t];
        target.setCooldown(skillId, value);
      }
    } else {
      return true;
    }
    return true;
};

BattleManager.actionSTypeCooldown = function(stypeId, actionArgs) {
    var targets = this.makeActionTargets(actionArgs[0]);
    if (targets.length < 1) return true;
    var cmd = actionArgs[1];
    if (cmd.match(/([\+\-]\d+)/i)) {
      var value = parseInt(RegExp.$1);
      for (var t = 0; t < targets.length; ++t) {
        var target = targets[t];
        for (var i = 0; i < target.allSkills().length; ++i) {
          var skill = target.allSkills()[i];
          if (skill && skill.stypeId === stypeId) {
            target.addCooldown(skill.id, value);
          }
        }
      }
    } else if (cmd.match(/(\d+)/i)) {
      var value = parseInt(RegExp.$1);
      for (var t = 0; t < targets.length; ++t) {
        var target = targets[t];
        for (var i = 0; i < target.allSkills().length; ++i) {
          var skill = target.allSkills()[i];
          if (skill && skill.stypeId === stypeId) {
            target.setCooldown(skill.id, value);
          }
        }
      }
    } else {
      return true;
    }
    return true;
};

//=============================================================================
// Game_BattlerBase
//=============================================================================

Yanfly.SCD.Game_BattlerBase_initMembers =
    Game_BattlerBase.prototype.initMembers;
Game_BattlerBase.prototype.initMembers = function() {
    Yanfly.SCD.Game_BattlerBase_initMembers.call(this);
    this.clearCooldowns();
    this.clearWarmups();
};

Game_BattlerBase.prototype.clearCooldowns = function() {
    this._cooldownTurns = {};
};

Game_BattlerBase.prototype.clearWarmups = function() {
    this._warmupTurns = {};
};

Game_BattlerBase.prototype.cooldown = function(skillId) {
    if (this._cooldownTurns === undefined) this.clearCooldowns();
    if (this._cooldownTurns[skillId] === undefined) {
      this._cooldownTurns[skillId] = 0;
    }
    return this._cooldownTurns[skillId];
};

Game_BattlerBase.prototype.warmup = function(skillId) {
    if (this._warmupTurns === undefined) this.clearWarmups();
    if (this._warmupTurns[skillId] === undefined) {
      this._warmupTurns[skillId] = 0;
    }
    return this._warmupTurns[skillId];
};

Game_BattlerBase.prototype.setCooldown = function(skillId, value) {
    if (!$dataSkills[skillId]) return;
    if ($dataSkills[skillId].bypassCooldown) return;
    if (this._cooldownTurns === undefined) this.clearCooldowns();
    this._cooldownTurns[skillId] = value;
};

Game_BattlerBase.prototype.addCooldown = function(skillId, value) {
    if (!$dataSkills[skillId]) return;
    if ($dataSkills[skillId].bypassCooldown) return;
    if (this._cooldownTurns === undefined) this.clearCooldowns();
    if (!this._cooldownTurns[skillId]) this._cooldownTurns[skillId] = 0;
    this._cooldownTurns[skillId] += value;
};

Game_BattlerBase.prototype.setWarmup = function(skillId, value) {
    if (!$dataSkills[skillId]) return;
    if ($dataSkills[skillId].bypassCooldown) return;
    if (this._warmupTurns === undefined) this.clearWarmups();
    this._warmupTurns[skillId] = value;
};

Game_BattlerBase.prototype.startWarmups = function() {
    if (this._warmupTurns === undefined) this.clearWarmups();
    for (var i = 0; i < this.allSkills().length; ++i) {
      var skill = this.allSkills()[i];
      if (!skill) continue;
      var warmup = skill.warmup;
      if (skill.warmupEval.length > 0) {
        var item = skill;
        var a = this;
        var user = this;
        var subject = this;
        var s = $gameSwitches._data;
        var v = $gameVariables._data;
        var code = skill.warmupEval;
        try {
          eval(code);
        } catch (e) {
          Yanfly.Util.displayError(e, code, 'CUSTOM WARMUP EVAL ERROR');
        }
      }
      warmup *= this.cooldownDuration(skill);
      warmup += this.getWarmupMods(skill);
      this.setWarmup(skill.id, warmup);
    }
};

Game_BattlerBase.prototype.updateCooldowns = function() {
    if (this._cooldownTurns === undefined) this.clearCooldowns();
    for (var skillId in this._cooldownTurns) {
      var skill = $dataSkills[skillId];
      if (!skill) continue;
      this._cooldownTurns[skillId] -= this.cooldownRate(skill);
    }
};

Game_BattlerBase.prototype.updateWarmups = function() {
    if (this._warmupTurns === undefined) this.clearWarmups();
    for (var skillId in this._warmupTurns) {
      var skill = $dataSkills[skillId];
      if (!skill) continue;
      this._warmupTurns[skillId] -= this.cooldownRate(skill);
    }
};

Game_BattlerBase.prototype.cooldownRateTick = function(skill) {
    this._cooldownTickRate = this._cooldownTickRate || {};
    if (!this._cooldownTickRate[skill.id]) {
      this._cooldownTickRate[skill.id] = this.cooldownRate(skill);
    }
    var rate = this._cooldownTickRate[skill.id];
    rate *= BattleManager.tickRate() / Yanfly.Param.CDTurnTime;
    return rate;
};

Game_BattlerBase.prototype.updateCooldownTicks = function() {
    if (this._cooldownTurns === undefined) this.clearCooldowns();
    for (var skillId in this._cooldownTurns) {
      var skill = $dataSkills[skillId];
      if (!skill) continue;
      if (this._cooldownTurns[skillId] <= 0) continue;
      this._cooldownTurns[skillId] -= this.cooldownRateTick(skill);
      this._cooldownTurns[skillId] = Math.max(0, this._cooldownTurns[skillId]);
    }
};

Game_BattlerBase.prototype.updateWarmupTicks = function() {
    if (this._warmupTurns === undefined) this.clearWarmups();
    for (var skillId in this._warmupTurns) {
      var skill = $dataSkills[skillId];
      if (!skill) continue;
      if (this._warmupTurns[skillId] <= 0) continue;
      this._warmupTurns[skillId] -= this.cooldownRateTick(skill);
      this._warmupTurns[skillId] = Math.max(0, this._warmupTurns[skillId]);
    }
};

Yanfly.SCD.Game_BattlerBase_meetsSkillConditions =
    Game_BattlerBase.prototype.meetsSkillConditions;
Game_BattlerBase.prototype.meetsSkillConditions = function(skill) {
    if (this.cooldown(skill.id) > 0) return false;
    if (this.warmup(skill.id) > 0) return false;
    return Yanfly.SCD.Game_BattlerBase_meetsSkillConditions.call(this, skill);
};

Yanfly.SCD.Game_BattlerBase_paySkillCost = 
    Game_BattlerBase.prototype.paySkillCost;
Game_BattlerBase.prototype.paySkillCost = function(skill) {
    Yanfly.SCD.Game_BattlerBase_paySkillCost.call(this, skill);
    this.payGlobalCooldown(skill);
    this.payStypeCooldownCost(skill);
    this.payCooldownCost(skill);
    this.applyCooldownMods(skill);
};

Game_BattlerBase.prototype.payGlobalCooldown = function(mainSkill) {
    for (var i = 0; i < this.allSkills().length; ++i) {
      var skill = this.allSkills()[i];
      if (!skill) continue;
      var value = mainSkill.globalCooldown;
      value *= this.cooldownDuration(mainSkill);
      value = Math.max(value, this.cooldown(skill.id));
      this.setCooldown(skill.id, value);
    }
};

Game_BattlerBase.prototype.payStypeCooldownCost = function(mainSkill) {
    for (var stypeId in mainSkill.stypeCooldown) {
      stypeId = parseInt(stypeId);
      for (var i = 0; i < this.allSkills().length; ++i) {
        var skill = this.allSkills()[i];
        if (!skill) continue;
        if (skill.stypeId !== stypeId) continue;
        var value = mainSkill.stypeCooldown[stypeId];
        value *= this.cooldownDuration(mainSkill);
        value = Math.max(value, this.cooldown(skill.id));
        this.setCooldown(skill.id, value);
      }
    }
};

Game_BattlerBase.prototype.payCooldownCost = function(skill) {
    for (var skillId in skill.cooldown) {
      skillId = parseInt(skillId);
      if (!$dataSkills[skillId]) continue;
      var cooldown = skill.cooldown[skillId];
      if (skill.id === skillId) {
        if (skill.cooldownEval.length > 0) {
          var item = skill;
          var a = this;
          var user = this;
          var subject = this;
          var s = $gameSwitches._data;
          var v = $gameVariables._data;
          var code = skill.cooldownEval;
          try {
            eval(code);
          } catch (e) {
            Yanfly.Util.displayError(e, code, 'CUSTOM COOLDOWN EVAL ERROR');
          }
        }
      }
      cooldown *= this.cooldownDuration(skill);
      cooldown = Math.max(cooldown, this.cooldown(skillId));
      this.setCooldown(skillId, cooldown);
    }
};

Game_BattlerBase.prototype.endBattleCooldowns = function() {
    this.resetCooldownTickRates();
    for (var skillId in this._cooldownTurns) {
      this._cooldownTurns[skillId] += $dataSkills[skillId].afterBattleCooldown;
    }
};

Game_BattlerBase.prototype.resetCooldownTickRates = function() {
    this._cooldownTickRate = {};
};

Game_BattlerBase.prototype.updateCooldownSteps = function() {
    for (var skillId in this._cooldownTurns) {
      var skill = $dataSkills[skillId];
      if (skill) {
        if ($gameParty.steps() % skill.cooldownSteps === 0) {
          this._cooldownTurns[skillId] -= this.cooldownRate(skill);
        }
      }
    }
};

Game_BattlerBase.prototype.applyCooldownEffect = function(skill) {
    this.applyGlobalCooldownChange(skill);
    this.applyStypeCooldownChange(skill);
    this.applyCooldownChange(skill);
};

Game_BattlerBase.prototype.applyCooldownChange = function(skill) {
    for (var skillId in skill.cooldownChange) {
      skillId = parseInt(skillId);
      if (!$dataSkills[skillId]) continue;
      if (!skill.cooldownChange[skillId]) continue;
      var value = skill.cooldownChange[skillId];
      this.addCooldown(skillId, value);
    }
};

Game_BattlerBase.prototype.applyStypeCooldownChange = function(mainSkill) {
    for (var stypeId in mainSkill.stypeCooldownChange) {
      stypeId = parseInt(stypeId);
      for (var i = 0; i < this.allSkills().length; ++i) {
        var skill = this.allSkills()[i];
        if (!skill) continue;
        if (skill.stypeId !== stypeId) continue;
        if (!mainSkill.stypeCooldownChange[stypeId]) continue;
        var value = mainSkill.stypeCooldownChange[stypeId];
        this.addCooldown(skill.id, value);
      }
    }
};

Game_BattlerBase.prototype.applyGlobalCooldownChange = function(mainSkill) {
    for (var i = 0; i < this.allSkills().length; ++i) {
      var skill = this.allSkills()[i];
      if (!skill) continue;
      var value = mainSkill.globalCooldownChange;
      this.addCooldown(skill.id, value);
    }
};

Game_BattlerBase.prototype.getWarmupMods = function(skill) {
    var value = 0;
    value += this.flatWarmupChange(skill);
    return value;
};

Game_BattlerBase.prototype.applyCooldownMods = function(skill) {
    var value = this.cooldown(skill.id);
    value += this.flatCooldownChange(skill);
    this.setCooldown(skill.id, Math.max(0, value));
};

//=============================================================================
// Game_Battler
//=============================================================================

Game_Battler.prototype.onBattleStartCooldowns = function() {
    this.resetCooldownTickRates();
    this.startWarmups();
};

Game_Battler.prototype.cooldownDuration = function(skill) {
    var skillId = skill.id;
    var stypeId = skill.stypeId;
    var value = 1.0;
    for (var i = 0; i < this.states().length; ++i) {
      var state = this.states()[i];
      if (!state) continue;
      if (state.cooldownDuration[skillId] !== undefined) {
        value *= state.cooldownDuration[skillId];
      }
      if (state.stypeCooldownDuration[stypeId] !== undefined) {
        value *= state.stypeCooldownDuration[stypeId];
      }
      value *= state.globalCooldownDuration;
    }
    return value;
};

Game_Battler.prototype.cooldownRate = function(skill) {
    var skillId = skill.id;
    var stypeId = skill.stypeId;
    var value = 1;
    for (var i = 0; i < this.states().length; ++i) {
      var state = this.states()[i];
      if (!state) continue;
      if (state.cooldownRate[skillId] !== undefined) {
        value *= state.cooldownRate[skillId];
      }
      if (state.stypeCooldownRate[stypeId] !== undefined) {
        value *= state.stypeCooldownRate[stypeId];
      }
      value *= state.globalCooldownRate;
    }
    return value;
};

Game_Battler.prototype.flatCooldownChange = function(skill) {
    var skillId = skill.id;
    var stypeId = skill.stypeId;
    var value = 0;
    for (var i = 0; i < this.states().length; ++i) {
      var state = this.states()[i];
      if (!state) continue;
      if (state.cooldownChange[skillId] !== undefined) {
        value += state.cooldownChange[skillId];
      }
      if (state.stypeCooldownChange[stypeId] !== undefined) {
        value += state.stypeCooldownChange[stypeId];
      }
      value += state.globalCooldownChange;
    }
    return value;
};

Game_Battler.prototype.flatWarmupChange = function(skill) {
    var skillId = skill.id;
    var stypeId = skill.stypeId;
    var value = 0;
    for (var i = 0; i < this.states().length; ++i) {
      var state = this.states()[i];
      if (!state) continue;
      if (state.warmupChange[skillId] !== undefined) {
        value += state.warmupChange[skillId];
      }
      if (state.stypeWarmupChange[stypeId] !== undefined) {
        value += state.stypeWarmupChange[stypeId];
      }
      value += state.globalWarmupChange;
    }
    return value;
};

Yanfly.SCD.Game_Battler_refresh = Game_Battler.prototype.refresh;
Game_Battler.prototype.refresh = function() {
    Yanfly.SCD.Game_Battler_refresh.call(this);
    this.resetCooldownTickRates();
};

if (Imported.YEP_BattleEngineCore) {

  Yanfly.SCD.Game_Battler_onTurnStart = Game_Battler.prototype.onTurnStart;
  Game_Battler.prototype.onTurnStart = function() {
    Yanfly.SCD.Game_Battler_onTurnStart.call(this);
    if (BattleManager.isTickBased() && !BattleManager.timeBasedCooldowns()) {
      this.updateCooldowns();
      this.updateWarmups();
    }
  };

  Yanfly.SCD.Game_Battler_updateTick = Game_Battler.prototype.updateTick;
  Game_Battler.prototype.updateTick = function() {
    Yanfly.SCD.Game_Battler_updateTick.call(this);
    if (BattleManager.isTickBased() && BattleManager.timeBasedCooldowns()) {
      this.updateCooldownTicks();
      this.updateWarmupTicks();
    };
  };

}; // Imported.YEP_BattleEngineCore

Game_Battler.prototype.allSkills = function() {
  var prevCase = $gameTemp._disableBattleSkills;
  $gameTemp._disableBattleSkills = true;
  var skills = this.skills();
  $gameTemp._disableBattleSkills = prevCase;
  return skills;
};

//=============================================================================
// Game_Actor
//=============================================================================

Game_Actor.prototype.cooldownDuration = function(skill) {
    var value = Game_Battler.prototype.cooldownDuration.call(this, skill);
    var skillId = skill.id;
    var stypeId = skill.stypeId;
    if (this.actor().cooldownDuration[skillId] !== undefined) {
      value *= this.actor().cooldownDuration[skillId];
    }
    if (this.currentClass().cooldownDuration[skillId] !== undefined) {
      value *= this.currentClass().cooldownDuration[skillId];
    }
    if (this.actor().stypeCooldownDuration[stypeId] !== undefined) {
      value *= this.actor().stypeCooldownDuration[stypeId];
    }
    if (this.currentClass().stypeCooldownDuration[stypeId] !== undefined) {
      value *= this.currentClass().stypeCooldownDuration[stypeId];
    }
    value *= this.actor().globalCooldownDuration;
    value *= this.currentClass().globalCooldownDuration;
    for (var i = 0; i < this.equips().length; ++i) {
      var equip = this.equips()[i];
      if (!equip) continue;
      if (equip.cooldownDuration !== undefined) {
        if (equip.cooldownDuration[skillId] !== undefined) {
          value *= equip.cooldownDuration[skillId];
        }
      }
      if (equip.stypeCooldownDuration !== undefined) {
        if (equip.stypeCooldownDuration[stypeId] !== undefined) {
          value *= equip.stypeCooldownDuration[stypeId];
        }
      }
      if (equip.globalCooldownDuration !== undefined) {
        value *= equip.globalCooldownDuration;
      }
    }
    return value;
};

Game_Actor.prototype.cooldownRate = function(skill) {
    var value = Game_Battler.prototype.cooldownRate.call(this, skill);
    var skillId = skill.id;
    var stypeId = skill.stypeId;
    if (this.actor().cooldownRate[skillId] !== undefined) {
      value *= this.actor().cooldownRate[skillId];
    }
    if (this.currentClass().cooldownRate[skillId] !== undefined) {
      value *= this.currentClass().cooldownRate[skillId];
    }
    if (this.actor().stypeCooldownRate[stypeId] !== undefined) {
      value *= this.actor().stypeCooldownRate[stypeId];
    }
    if (this.currentClass().stypeCooldownRate[stypeId] !== undefined) {
      value *= this.currentClass().stypeCooldownRate[stypeId];
    }
    value *= this.actor().globalCooldownRate;
    value *= this.currentClass().globalCooldownRate;
    for (var i = 0; i < this.equips().length; ++i) {
      var equip = this.equips()[i];
      if (!equip) continue;
      if (equip.cooldownRate !== undefined) {
        if (equip.cooldownRate[skillId] !== undefined) {
          value *= equip.cooldownRate[skillId];
        }
      }
      if (equip.stypeCooldownRate !== undefined) {
        if (equip.stypeCooldownRate[stypeId] !== undefined) {
          value *= equip.stypeCooldownRate[stypeId];
        }
      }
      if (equip.globalCooldownRate !== undefined) {
        value *= equip.globalCooldownRate;
      }
    }
    return value;
};

Game_Actor.prototype.flatCooldownChange = function(skill) {
    var skillId = skill.id;
    var stypeId = skill.stypeId;
    var value = Game_Battler.prototype.flatCooldownChange.call(this, skill);
    if (this.actor().cooldownChange[skillId] !== undefined) {
      value += this.actor().cooldownChange[skillId];
    }
    if (this.currentClass().cooldownChange[skillId] !== undefined) {
      value += this.currentClass().cooldownChange[skillId];
    }
    if (this.actor().stypeCooldownChange[stypeId] !== undefined) {
      value += this.actor().stypeCooldownChange[stypeId];
    }
    if (this.currentClass().stypeCooldownChange[stypeId] !== undefined) {
      value += this.currentClass().stypeCooldownChange[stypeId];
    }
    value += this.actor().globalCooldownChange;
    value += this.currentClass().globalCooldownChange;
    for (var i = 0; i < this.equips().length; ++i) {
      var equip = this.equips()[i];
      if (!equip) continue;
      if (equip.cooldownChange === undefined) continue;
      if (equip.cooldownChange[skillId] !== undefined) {
        value += equip.cooldownChange[skillId];
      }
      if (equip.stypeCooldownChange[stypeId] !== undefined) {
        value += equip.stypeCooldownChange[stypeId];
      }
      value += equip.globalCooldownChange;
    }
    return value;
};

Game_Actor.prototype.flatWarmupChange = function(skill) {
    var skillId = skill.id;
    var stypeId = skill.stypeId;
    var value = Game_Battler.prototype.flatWarmupChange.call(this, skill);
    if (this.actor().warmupChange[skillId] !== undefined) {
      value += this.actor().warmupChange[skillId];
    }
    if (this.currentClass().warmupChange[skillId] !== undefined) {
      value += this.currentClass().warmupChange[skillId];
    }
    if (this.actor().stypeWarmupChange[stypeId] !== undefined) {
      value += this.actor().stypeWarmupChange[stypeId];
    }
    if (this.currentClass().stypeWarmupChange[stypeId] !== undefined) {
      value += this.currentClass().stypeWarmupChange[stypeId];
    }
    value += this.actor().globalWarmupChange;
    value += this.currentClass().globalWarmupChange;
    for (var i = 0; i < this.equips().length; ++i) {
      var equip = this.equips()[i];
      if (!equip) continue;
      if (equip.warmupChange === undefined) continue;
      if (equip.warmupChange[skillId] !== undefined) {
        value += equip.warmupChange[skillId];
      }
      if (equip.stypeWarmupChange[stypeId] !== undefined) {
        value += equip.stypeWarmupChange[stypeId];
      }
      value += equip.globalWarmupChange;
    }
    return value;
};

//=============================================================================
// Game_Enemy
//=============================================================================

if (!Game_Enemy.prototype.skills) {
    Game_Enemy.prototype.skills = function() {
      var skills = []
      for (var i = 0; i < this.enemy().actions.length; ++i) {
        var skill = $dataSkills[this.enemy().actions[i].skillId]
        if (skill) skills.push(skill);
      }
      return skills;
    }
};

Game_Enemy.prototype.cooldownDuration = function(skill) {
    var value = Game_Battler.prototype.cooldownDuration.call(this, skill);
    var skillId = skill.id;
    var stypeId = skill.stypeId;
    if (this.enemy().cooldownDuration[skillId] !== undefined) {
      value *= this.enemy().cooldownDuration[skillId];
    }
    if (this.enemy().stypeCooldownDuration[stypeId] !== undefined) {
      value *= this.enemy().stypeCooldownDuration[stypeId];
    }
    value *= this.enemy().globalCooldownDuration;
    return value;
};

Game_Enemy.prototype.cooldownRate = function(skill) {
    var value = Game_Battler.prototype.cooldownRate.call(this, skill);
    var skillId = skill.id;
    var stypeId = skill.stypeId;
    if (this.enemy().cooldownRate[skillId] !== undefined) {
      value *= this.enemy().cooldownRate[skillId];
    }
    if (this.enemy().stypeCooldownRate[stypeId] !== undefined) {
      value *= this.enemy().stypeCooldownRate[stypeId];
    }
    value *= this.enemy().globalCooldownRate;
    return value;
};

Game_Enemy.prototype.flatCooldownChange = function(skill) {
    var skillId = skill.id;
    var stypeId = skill.stypeId;
    var value = Game_Battler.prototype.flatCooldownChange.call(this, skill);
    if (this.enemy().cooldownChange[skillId] !== undefined) {
      value += this.enemy().cooldownChange[skillId];
    }
    if (this.enemy().stypeCooldownChange[stypeId] !== undefined) {
      value += this.enemy().stypeCooldownChange[stypeId];
    }
    value += this.enemy().globalCooldownChange;
    return value;
};

Game_Enemy.prototype.flatWarmupChange = function(skill) {
    var skillId = skill.id;
    var stypeId = skill.stypeId;
    var value = Game_Battler.prototype.flatWarmupChange.call(this, skill);
    if (this.enemy().warmupChange[skillId] !== undefined) {
      value += this.enemy().warmupChange[skillId];
    }
    if (this.enemy().stypeWarmupChange[stypeId] !== undefined) {
      value += this.enemy().stypeWarmupChange[stypeId];
    }
    value += this.enemy().globalWarmupChange;
    return value;
};

//=============================================================================
// Game_Action
//=============================================================================

Yanfly.SCD.Game_Action_applyItemUserEffect =
    Game_Action.prototype.applyItemUserEffect;
Game_Action.prototype.applyItemUserEffect = function(target) {
    Yanfly.SCD.Game_Action_applyItemUserEffect.call(this, target);
    if (target) target.applyCooldownEffect(this.item());
};

//=============================================================================
// Game_Unit
//=============================================================================

Yanfly.SCD.Game_Unit_onBattleStart = Game_Unit.prototype.onBattleStart;
Game_Unit.prototype.onBattleStart = function() {
  Yanfly.SCD.Game_Unit_onBattleStart.call(this);
  this.onBattleStartCooldowns();
};

Game_Unit.prototype.onBattleStartCooldowns = function() {
  var members = this.cooldownMembers();
  var length = members.length
  for (var i = 0; i < length; ++i) {
    var member = members[i];
    if (member) member.onBattleStartCooldowns();
  }
};

Game_Unit.prototype.updateCooldowns = function() {
  if (BattleManager.timeBasedCooldowns()) return;
  var members = this.members();
  var length = members.length
  for (var i = 0; i < length; ++i) {
    var member = members[i];
    if (member) {
      member.updateCooldowns();
      member.updateWarmups();
    }
  }
};

Game_Unit.prototype.endBattleCooldowns = function() {
  var members = this.members();
  var length = members.length
  for (var i = 0; i < length; ++i) {
    var member = members[i];
    if (member) {
      member.endBattleCooldowns();
      member.clearWarmups();
    }
  }
};

Game_Unit.prototype.cooldownMembers = function() {
  return this.members();
};

//=============================================================================
// Game_Party
//=============================================================================

Yanfly.SCD.Game_Party_increaseSteps = Game_Party.prototype.increaseSteps;
Game_Party.prototype.increaseSteps = function() {
    Yanfly.SCD.Game_Party_increaseSteps.call(this);
    this.updateCooldownSteps();
};

Game_Party.prototype.updateCooldownSteps = function() {
    return this.members().forEach(function(member) {
        return member.updateCooldownSteps();
    });
};

Game_Party.prototype.cooldownMembers = function() {
  return this.allMembers();
};

//=============================================================================
// Game_Troop
//=============================================================================

Yanfly.SCD.Game_Troop_increaseTurn = Game_Troop.prototype.increaseTurn;
Game_Troop.prototype.increaseTurn = function() {
    Yanfly.SCD.Game_Troop_increaseTurn.call(this);
    if (Imported.YEP_BattleEngineCore) {
      if (BattleManager.isTurnBased()) {
        this.updateCooldowns();
        $gameParty.updateCooldowns();
      }
    } else {
      this.updateCooldowns();
      $gameParty.updateCooldowns();
    }
};

//=============================================================================
// Window_SkillList
//=============================================================================

Yanfly.SCD.Window_SkillList_drawCost = Window_SkillList.prototype.drawSkillCost;
Window_SkillList.prototype.drawSkillCost = function(skill, wx, wy, ww) {
    if (this._actor.warmup(skill.id) > 0) {
      return this.drawWarmup(skill, wx, wy, ww);
    } else if (this._actor.cooldown(skill.id) > 0) {
      return this.drawCooldown(skill, wx, wy, ww);
    } else {
      return Yanfly.SCD.Window_SkillList_drawCost.call(this, skill, wx, wy, ww);
    }
};

Window_SkillList.prototype.drawCooldown = function(skill, wx, wy, dw) {
    if (Yanfly.Icon.Cooldown > 0) {
      var iw = wx + dw - Window_Base._iconWidth;
      this.drawIcon(Yanfly.Icon.Cooldown, iw, wy + 2);
      dw -= Window_Base._iconWidth + 2;
    }
    this.changeTextColor(this.textColor(Yanfly.Param.CDTextColor));
    var fmt = Yanfly.Param.CDFmt;
    var value = this._actor.cooldown(skill.id);
    if (value % 1 !== 0) value = value.toFixed(2);
    if (value <= 0.009) value = 0.01;
    var text = fmt.format(Yanfly.Util.toGroup(value));
    this.contents.fontSize = Yanfly.Param.CDFontSize;
    this.drawText(text, wx, wy, dw, 'right');
    var returnWidth = dw - this.textWidth(text) - Yanfly.Param.SCCCostPadding;
    this.resetFontSettings();
    return returnWidth;
};

Window_SkillList.prototype.drawWarmup = function(skill, wx, wy, dw) {
    if (Yanfly.Icon.Warmup > 0) {
      var iw = wx + dw - Window_Base._iconWidth;
      this.drawIcon(Yanfly.Icon.Warmup, iw, wy + 2);
      dw -= Window_Base._iconWidth + 2;
    }
    this.changeTextColor(this.textColor(Yanfly.Param.WUTextColor));
    var fmt = Yanfly.Param.WUFmt;
    var value = this._actor.warmup(skill.id);
    if (value % 1 !== 0) value = value.toFixed(2);
    if (value <= 0.009) value = 0.01;
    var text = fmt.format(Yanfly.Util.toGroup(value));
    this.contents.fontSize = Yanfly.Param.WUFontSize;
    this.drawText(text, wx, wy, dw, 'right');
    var returnWidth = dw - this.textWidth(text) - Yanfly.Param.SCCCostPadding;
    this.resetFontSettings();
    return returnWidth;
};

//=============================================================================
// Utilities
//=============================================================================

Yanfly.Util = Yanfly.Util || {};

if (!Yanfly.Util.toGroup) {
    Yanfly.Util.toGroup = function(inVal) {
        return inVal;
    }
};

//=============================================================================
// End of File
//=============================================================================
};
