/*:
 * @target MZ
 * @plugindesc (v1.0) イベントのメモに特定語がある場合、出現から指定フレーム後にセルフスイッチを自動で切り替えます。
 * @author 
 *
 * @param DefaultKeyword
 * @text デフォルト判定ワード
 * @type string
 * @default AUTO_SS
 * @desc イベントのメモ欄にこの文字列を含んでいれば対象と見なします。空にするとメモ文字列判定を無効化します（代わりにメタタグを使ってください）。
 *
 * @param DefaultFrames
 * @text デフォルト待機フレーム数
 * @type number
 * @min 1
 * @default 60
 * @desc イベント出現からこのフレーム数経過でセルフスイッチ切替（1秒=60フレーム想定）
 *
 * @param DefaultSelfSwitch
 * @text デフォルトのセルフスイッチ
 * @type select
 * @option A
 * @value A
 * @option B
 * @value B
 * @option C
 * @value C
 * @option D
 * @value D
 * @default A
 *
 * @param DefaultValue
 * @text デフォルトの切替値
 * @type boolean
 * @on ON
 * @off OFF
 * @default true
 * @desc true -> ON にする、false -> OFF にする
 *
 * @param DefaultOnlyOnce
 * @text デフォルトは一度だけ?
 * @type boolean
 * @on 一度だけ
 * @off 繰り返す
 * @default true
 * @desc trueなら一度切り替えたらそのイベントは追跡をやめます。
 *
 * @help
 * ---------------------------------------------------------------
 * 概要
 * ---------------------------------------------------------------
 * イベントのメモ欄(note)に指定ワード（デフォルト: <DefaultKeyword>）が含まれる
 * か、または下記メタタグを持つイベントを検出し、そのイベントが"出現"
 * （マップ上に存在する状態）になってから指定フレーム数を経過したら、
 * 指定したセルフスイッチをON/OFFします。
 *
 * メタタグ方式（イベントのメモ欄に以下を追加）
 *  ※どれも任意。未指定部分はプラグインパラメータのデフォルトを使います。
 *
 * <AutoSS>                   // 単にこのタグがあるだけで対象
 * <AutoSSFrames:120>        // 待機フレーム数（数値）
 * <AutoSSSS:A>              // セルフスイッチ(A/B/C/D)
 * <AutoSSValue:OFF>         // ON または OFF を指定
 * <AutoSSOnce:false>        // 一度切り替えたら追跡を止めるか (true/false)
 *
 * 例）メモ欄に：
 * <AutoSS>
 * <AutoSSFrames:90>
 * <AutoSSSS:B>
 * <AutoSSValue:OFF>
 *
 * あるいは、メモにデフォルトワード（例：AUTO_SS）を含めれば
 * パラメータのデフォルト値で動作します。
 *
 * 注意点
 * - 「出現」の判定はマップ上にそのイベントが存在し（$gameMap.event(id)が存在）
 *   かつ _erased が false の状態を基準にしています。
 * - イベントが消滅（erase）した場合は追跡を自動で終了します。
 * - 同じマップでのイベントID単位で管理します（マップ切替時はマップIDをキーにします）。
 *
 * 利用方法
 * 1. このスクリプトファイルをプロジェクトの plugins フォルダへ置く
 * 2. プラグインマネージャで有効化し、パラメータを調整
 * 3. 対象イベントのメモ欄に上記タグか、デフォルトワードを記載
 *
 * バグ報告・要望があればエリスにどうぞ。余計な手間は取らせないけど、文句は受け付ける。
 */

(() => {
    const PLUGIN_NAME = 'AutoSelfSwitchByNote';
    const params = PluginManager.parameters(PLUGIN_NAME) || {};

    const defaultKeyword = String(params['DefaultKeyword'] || 'AUTO_SS');
    const defaultFrames = Number(params['DefaultFrames'] || 60);
    const defaultSS = String(params['DefaultSelfSwitch'] || 'A');
    const defaultValue = (String(params['DefaultValue'] || 'true') === 'true');
    const defaultOnlyOnce = (String(params['DefaultOnlyOnce'] || 'true') === 'true');

    // 内部データ管理（mapId_eventId -> { timer, config })
    const trackers = {};

    // ヘルパー: イベントのメモ欄からメタ情報を読み取る
    function parseMetaEvent(ev) {
        if (!ev || !ev.event) return null;
        const note = ev.event().note || '';
        const meta = ev.event().meta || {};
        const out = {};

        // メタタグ形式優先
        if ('AutoSS' in meta) {
            out.enabled = true;
            out.frames = meta['AutoSSFrames'] ? Number(meta['AutoSSFrames']) : undefined;
            out.ss = meta['AutoSSSS'] ? String(meta['AutoSSSS']) : undefined;
            if (meta['AutoSSValue']) out.value = String(meta['AutoSSValue']).toUpperCase() === 'ON';
            if (meta['AutoSSOnce']) out.once = String(meta['AutoSSOnce']).toLowerCase() === 'true';
            return out;
        }

        // note文字列にデフォルトキーワードが含まれるか
        if (defaultKeyword && defaultKeyword.length > 0 && note.indexOf(defaultKeyword) >= 0) {
            out.enabled = true;
            // その他はプラグインパラメータに従う
            return out;
        }
        return null;
    }

    function makeKey(mapId, eventId) {
        return mapId + '_' + eventId;
    }

    function clearTracker(mapId, eventId) {
        const key = makeKey(mapId, eventId);
        if (trackers[key]) delete trackers[key];
    }

    // 毎フレーム呼ばれる更新処理
    function updateAutoSS() {
        const mapId = $gameMap._mapId;
        if (!mapId) return;
        const events = $gameMap.events();
        // まず、存在するイベントをチェックして新規トラッキングを追加
        events.forEach(ev => {
            const id = ev.eventId();
            const key = makeKey(mapId, id);
            // すでに追跡済みならスキップ（ただし消滅チェックは下で）
            if (trackers[key]) return;

            // メモ/メタ判定
            const cfg = parseMetaEvent(ev);
            if (cfg && cfg.enabled) {
                // 補完
                cfg.frames = (typeof cfg.frames === 'number' && cfg.frames > 0) ? cfg.frames : defaultFrames;
                cfg.ss = cfg.ss || defaultSS;
                cfg.value = (typeof cfg.value === 'boolean') ? cfg.value : defaultValue;
                cfg.once = (typeof cfg.once === 'boolean') ? cfg.once : defaultOnlyOnce;

                trackers[key] = { timer: 0, cfg: cfg };
            }
        });

        // 次に既存トラッカーを進め、条件達成でセルフスイッチを操作
        Object.keys(trackers).forEach(key => {
            const parts = key.split('_');
            const mId = Number(parts[0]);
            const eId = Number(parts[1]);
            if (mId !== $gameMap._mapId) {
                // マップが切り替わってたらトラッカー削除
                delete trackers[key];
                return;
            }
            const ev = $gameMap.event(eId);
            if (!ev || ev._erased) {
                // イベントが消えた場合は追跡終了
                delete trackers[key];
                return;
            }
            const data = trackers[key];
            data.timer++;
            if (data.timer >= data.cfg.frames) {
                // セルフスイッチを切り替え
                const ssKey = [$gameMap._mapId, eId, data.cfg.ss];
                $gameSelfSwitches.setValue(ssKey, data.cfg.value);
                // 一度だけならトラッカー削除
                if (data.cfg.once) delete trackers[key];
                else data.timer = 0; // 繰り返す場合はタイマーをリセットして繰り返し可能
            }
        });
    }

    // Scene_Map 更新にフック
    const _Scene_Map_update = Scene_Map.prototype.update;
    Scene_Map.prototype.update = function() {
        _Scene_Map_update.call(this);
        try {
            updateAutoSS();
        } catch (e) {
            // 何かあってもゲームを落とさない
            console.error('AutoSelfSwitchByNote update error:', e);
        }
    };

})();
