/*:
 * @target MZ
 * @plugindesc Bomb timer: set a self-switch for events with <bomb> after specified frames. Starts on map load AND when an event with <bomb> is created. (Eris) - fixed for id-reuse
 * @author Eris
 *
 * @param DefaultFrames
 * @text DefaultFrames
 * @type number
 * @desc Default frames until switch flip (frames, not seconds)
 * @default 180
 *
 * @param DefaultSelfSwitch
 * @text DefaultSelfSwitch
 * @type select
 * @option A
 * @option B
 * @option C
 * @option D
 * @desc Default self switch to set when timer expires
 * @default A
 *
 * @param AutoStartOnMapLoad
 * @text AutoStartOnMapLoad
 * @type boolean
 * @desc Automatically scan current map for <bomb> events and start timers on map load
 * @default true
 *
 * @param DebugLog
 * @text DebugLog
 * @type boolean
 * @default false
 *
 * @command StartBombTimer
 * @text StartBombTimer
 * @desc Start a bomb timer for a specific event on the current map
 *
 * @arg eventId
 * @type number
 * @text EventId
 * @desc Event ID on the current map (use 0 for current running event in an event script)
 * @default 0
 *
 * @arg frames
 * @type number
 * @text Frames
 * @desc Frames until the self-switch is set
 * @default 180
 *
 * @arg selfSwitch
 * @type select
 * @option A
 * @option B
 * @option C
 * @option D
 * @text SelfSwitch
 * @desc Which self switch to set when time expires
 * @default A
 *
 * @command StopBombTimer
 * @text StopBombTimer
 * @desc Stop and remove a bomb timer for a specific event (current map)
 *
 * @arg eventId
 * @type number
 * @text EventId
 * @desc Event ID on the current map (use 0 for current running event in an event script)
 * @default 0
 *
 * @help
 * Same usage as before. This version includes protections against event-ID reuse:
 * - timers store a snapshot of the event note and (if available) instanceId when started,
 *   and before setting a self-switch on expiry, the plugin verifies the slot hasn't been
 *   replaced by a different event (by comparing note / instanceId). If the slot is empty,
 *   it still sets the switch (preserving the documented behavior).
 */

(() => {
    'use strict';

    const PLUGIN_NAME = 'BombSelfSwitchTimer';
    const params = PluginManager.parameters(PLUGIN_NAME);
    const DEFAULT_FRAMES = Math.max(1, Number(params['DefaultFrames'] || 180));
    const DEFAULT_SS = String(params['DefaultSelfSwitch'] || 'A').toUpperCase();
    const AUTO_START = String(params['AutoStartOnMapLoad'] || 'true') === 'true';
    const DEBUG = String(params['DebugLog'] || 'false') === 'true';

    const log = (...a) => { if (DEBUG) console.log('[BombSelfSwitchTimer]', ...a); };

    // ---------------- helpers ----------------
    const escapeRegExp = s => String(s).replace(/[.*+?^${}()|[\]\\]/g, '\\$&');

    const hasNoteTag = (eventData, tag) => {
        if (!eventData || !eventData.note) return false;
        const t = String(tag).trim();
        const re = new RegExp('<\\s*' + escapeRegExp(t) + '\\s*>', 'i');
        return re.test(eventData.note);
    };

    const getNoteMeta = (eventData, key) => {
        if (!eventData || !eventData.note) return null;
        const re = new RegExp(`<\\s*${escapeRegExp(key)}\\s*[:：]\\s*([^>]+?)\\s*>`, 'i');
        const m = eventData.note.match(re);
        return m ? m[1].trim() : null;
    };

    function ensureBombTimers() {
        if (!$gameMap._bombTimers) $gameMap._bombTimers = [];
        return $gameMap._bombTimers;
    }

    function findTimerIndex(mapId, eventId) {
        const list = ensureBombTimers();
        return list.findIndex(e => e.mapId === mapId && e.eventId === eventId);
    }

    // unified set/get for self-switch, tolerant of environments
    function getSelfSwitch(key) {
        if (!$gameSelfSwitches) return null;
        if (typeof $gameSelfSwitches.getValue === 'function') {
            try { return $gameSelfSwitches.getValue(key); } catch (e) { /* ignore */ }
        }
        if (typeof $gameSelfSwitches.getVariableValue === 'function') {
            try { return $gameSelfSwitches.getVariableValue(key); } catch (e) { /* ignore */ }
        }
        try { return $gameSelfSwitches._data && $gameSelfSwitches._data[key]; } catch (e) { /* ignore */ }
        return null;
    }

    function setSelfSwitch(key, value) {
        if (!$gameSelfSwitches) return;
        if (typeof $gameSelfSwitches.setValue === 'function') {
            try { $gameSelfSwitches.setValue(key, value); return; } catch (e) { /* ignore */ }
        }
        if (typeof $gameSelfSwitches.setVariableValue === 'function') {
            try { $gameSelfSwitches.setVariableValue(key, value); return; } catch (e) { /* ignore */ }
        }
        try {
            if (!$gameSelfSwitches._data) $gameSelfSwitches._data = {};
            // fall back to a deterministic key to avoid ambiguous object keys
            $gameSelfSwitches._data[JSON.stringify(key)] = value;
        } catch (e) { /* ignore */ }
    }

    // ---------------- BombTimer API ----------------
    const BombTimer = {
        startTimer(mapId, eventId, frames, selfSwitch) {
            if (!Number.isFinite(mapId) || !Number.isFinite(eventId)) return;
            const f = Math.max(1, Math.floor(Number(frames || DEFAULT_FRAMES)));
            const ss = (String(selfSwitch || DEFAULT_SS) || DEFAULT_SS).toUpperCase();
            const list = ensureBombTimers();
            const idx = findTimerIndex(mapId, eventId);

            // Gather snapshot info to detect ID reuse
            let noteSnapshot = null;
            let instanceId = null;
            try {
                const dataEvent = $dataMap && $dataMap.events && $dataMap.events[eventId];
                if (dataEvent && typeof dataEvent.note === 'string') noteSnapshot = dataEvent.note.trim();
                const mapEvent = $gameMap && typeof $gameMap.event === 'function' ? $gameMap.event(eventId) : null;
                if (mapEvent && typeof mapEvent._instanceId !== 'undefined') instanceId = mapEvent._instanceId;
            } catch (e) { /* ignore */ }

            if (idx >= 0) {
                // replace existing, update snapshot & instance
                list[idx].frames = f;
                list[idx].selfSwitch = ss;
                list[idx].noteSnapshot = noteSnapshot;
                list[idx].instanceId = instanceId;
                list[idx].startedAt = Date.now();
                log('Timer updated for', mapId, eventId, 'frames=', f, 'ss=', ss, 'inst=', instanceId);
                return;
            }

            const entry = {
                mapId: mapId,
                eventId: eventId,
                frames: f,
                selfSwitch: ss,
                // additional safety metadata:
                noteSnapshot: noteSnapshot,   // string or null
                instanceId: instanceId,       // number or null
                startedAt: Date.now()
            };
            list.push(entry);
            log('Timer started for', mapId, eventId, 'frames=', f, 'ss=', ss, 'inst=', instanceId);
        },

        stopTimer(mapId, eventId) {
            const list = ensureBombTimers();
            const idx = findTimerIndex(mapId, eventId);
            if (idx >= 0) list.splice(idx, 1);
        },

        scanAndStartForCurrentMap() {
            const mapId = $gameMap.mapId();
            if (!mapId) return;
            const evs = $gameMap.events();
            for (const ev of evs) {
                if (!ev || ev._erased) continue;
                const ed = ev.event();
                if (!ed) continue;
                if (hasNoteTag(ed, 'bomb')) {
                    const t = getNoteMeta(ed, 'bombTime');
                    const ss = getNoteMeta(ed, 'bombSS');
                    const frames = t ? Math.max(1, Number(t)) : DEFAULT_FRAMES;
                    BombTimer.startTimer(mapId, ev.eventId(), frames, ss || DEFAULT_SS);
                }
            }
        },

        // for debug
        _list() {
            return ensureBombTimers().slice();
        }
    };

    // expose
    window.BombTimer = BombTimer;

    // ---------------- process timers each frame ----------------
    const _Game_Map_update = Game_Map.prototype.update;
    Game_Map.prototype.update = function (sceneActive) {
        _Game_Map_update.apply(this, arguments);

        const list = ensureBombTimers();
        if (!list || list.length === 0) return;

        for (let i = list.length - 1; i >= 0; i--) {
            const e = list[i];
            if (e.mapId !== $gameMap.mapId()) continue;
            e.frames--;
            if (e.frames <= 0) {
                const key = [e.mapId, e.eventId, String(e.selfSwitch || DEFAULT_SS)];
                // Safely decide whether to write the self-switch:
                try {
                    // Fetch the current data-map event (static data at that slot) and the currently instantiated event (if any)
                    const dataEvent = $dataMap && $dataMap.events ? $dataMap.events[e.eventId] : null;
                    const mapEvent = $gameMap && typeof $gameMap.event === 'function' ? $gameMap.event(e.eventId) : null;

                    let allowSet = true;
                    // If slot is empty in data (null) and no mapEvent -> OK (setting has no effect)
                    // If slot has an event but it's different from the snapshot -> suspect reuse -> do NOT set
                    if (dataEvent && typeof dataEvent.note === 'string' && e.noteSnapshot != null) {
                        // compare trimmed notes — if different, assume different event/template
                        const currentNote = dataEvent.note.trim();
                        if (currentNote !== e.noteSnapshot) {
                            allowSet = false;
                            log('Timer expiry skipped due to note mismatch (possible id reuse):', e.eventId, 'expectedNoteHash=', e.noteSnapshot.slice(0,80), 'currentNoteHash=', currentNote.slice(0,80));
                        }
                    }
                    // If instanceId was recorded and there's a live mapEvent with an instance id, compare
                    if (typeof e.instanceId === 'number' && mapEvent && typeof mapEvent._instanceId !== 'undefined') {
                        if (mapEvent._instanceId !== e.instanceId) {
                            // different instance currently occupying slot -> skip
                            allowSet = false;
                            log('Timer expiry skipped due to instanceId mismatch (possible reuse):', e.eventId, 'expectedInst=', e.instanceId, 'currentInst=', mapEvent._instanceId);
                        }
                    }

                    // If mapEvent exists and looks identical by note (or no snapshot), we allow set.
                    // If mapEvent doesn't exist (slot free because event erased), allow set (preserve documented behavior).
                    if (allowSet) {
                        // only set if not already true
                        const cur = getSelfSwitch(key);
                        if (!cur) {
                            setSelfSwitch(key, true);
                            log('Self-switch set for', key);
                        } else {
                            log('Self-switch already true for', key);
                        }
                    } else {
                        log('Skipped setting self-switch for', key);
                    }
                } catch (err) {
                    // fallback: set anyway to avoid losing intended behaviour in exotic envs
                    try {
                        setSelfSwitch(key, true);
                        log('Fallback: Self-switch set for', key, '(due to error)', err);
                    } catch (e2) {
                        console.error('[BombSelfSwitchTimer] failed to set self-switch on fallback', e2);
                    }
                }
                // remove timer entry no matter what (avoid infinite retries)
                list.splice(i, 1);
            }
        }
    };

    // ---------------- start timers on map loaded (auto) ----------------
    const _Scene_Map_onMapLoaded = Scene_Map.prototype.onMapLoaded;
    Scene_Map.prototype.onMapLoaded = function () {
        _Scene_Map_onMapLoaded.apply(this, arguments);
        if (AUTO_START) {
            try {
                // delay scanning by 0 frames is fine; map data should be available now
                BombTimer.scanAndStartForCurrentMap();
            } catch (e) {
                console.error('[BombSelfSwitchTimer] scanAndStart error', e);
            }
        }
    };

    // ---------------- start timers when Game_Event is created (spawned) ----------------
    // Hook Game_Event.initialize so that when an event is created/constructed we start timer if it has <bomb>.
    const _Game_Event_initialize = Game_Event.prototype.initialize;
    Game_Event.prototype.initialize = function (mapId, eventId) {
        _Game_Event_initialize.apply(this, arguments);
        try {
            const ed = this.event();
            if (ed && ed.note && /<\s*bomb\s*>/i.test(ed.note)) {

                // snapshot data for safety
                const ss = (getNoteMeta(ed, 'bombSS') || DEFAULT_SS).toUpperCase();
                const actualEventId = typeof eventId === 'number' && eventId > 0 ? eventId : (this.eventId ? this.eventId() : 0);

                // Force-reset the target self-switch on creation (keeps behavior)
                if (actualEventId) {
                    const key = [mapId, actualEventId, ss];
                    try { setSelfSwitch(key, false); } catch (e) { /* ignore */ }
                }

                const t = getNoteMeta(ed, 'bombTime');
                const frames = t ? Math.max(1, Number(t)) : DEFAULT_FRAMES;

                // prepare snapshot note
                let noteSnapshot = null;
                try {
                    const dataEv = $dataMap && $dataMap.events ? $dataMap.events[actualEventId] : null;
                    if (dataEv && typeof dataEv.note === 'string') noteSnapshot = dataEv.note.trim();
                } catch (e) { /* ignore */ }

                // instanceId if available
                const inst = typeof this._instanceId !== 'undefined' ? this._instanceId : null;

                if (actualEventId) {
                    // Use extended startTimer API (we store noteSnapshot & instance)
                    const list = ensureBombTimers();
                    const idx = findTimerIndex(mapId, actualEventId);
                    if (idx >= 0) {
                        list[idx].frames = frames;
                        list[idx].selfSwitch = ss;
                        list[idx].noteSnapshot = noteSnapshot;
                        list[idx].instanceId = inst;
                        list[idx].startedAt = Date.now();
                        log('Timer updated (initialize) for', mapId, actualEventId, 'frames=', frames, 'ss=', ss, 'inst=', inst);
                    } else {
                        list.push({
                            mapId: mapId,
                            eventId: actualEventId,
                            frames: frames,
                            selfSwitch: ss,
                            noteSnapshot: noteSnapshot,
                            instanceId: inst,
                            startedAt: Date.now()
                        });
                        log('Timer started (initialize) for', mapId, actualEventId, 'frames=', frames, 'ss=', ss, 'inst=', inst);
                    }
                }
            }
        } catch (e) {
            // don't break game if something unexpected happens
            console.error('[BombSelfSwitchTimer] initialize hook error', e);
        }
    };

    // ---------------- plugin commands ----------------
    PluginManager.registerCommand(PLUGIN_NAME, "StartBombTimer", function (args) {
        let eventId = Number(args.eventId || 0);
        const frames = Number(args.frames || DEFAULT_FRAMES);
        const ss = String(args.selfSwitch || DEFAULT_SS).toUpperCase();
        let mapId = $gameMap ? $gameMap.mapId() : 0;
        if (eventId === 0) {
            if (this && this._eventId) eventId = this._eventId;
        }
        if (!mapId || !eventId) return;
        BombTimer.startTimer(mapId, eventId, frames, ss);
    });

    PluginManager.registerCommand(PLUGIN_NAME, "StopBombTimer", function (args) {
        let eventId = Number(args.eventId || 0);
        let mapId = $gameMap ? $gameMap.mapId() : 0;
        if (eventId === 0) {
            if (this && this._eventId) eventId = this._eventId;
        }
        if (!mapId || !eventId) return;
        BombTimer.stopTimer(mapId, eventId);
    });

})();
