/*:
 * @target MZ
 * @plugindesc (ミナ最終改修版v21 + 衝突誤爆修正) 近接ターゲットの棒立ち強制脱出 ＋ 1ページ目限定追跡
 * @author エリス (改: ミナ)
 * @help ★V21は、追跡処理をイベントの**1ページ目限定**で実行するよう変更しました。ページが切り替わると追跡が停止し、1ページ目に戻ると再開されます。★
 *
 * @param DefaultTargetTag
 * @text デフォルトターゲットタグ
 * @type string
 * @desc 追跡者が<Target:～>を持たない場合に使用されるターゲットタグ。
 * @default box
 *
 * @param PlayerKeyword
 * @text プレイヤーターゲットキーワード
 * @type string
 * @desc ターゲットとしてプレイヤーを指定する際のキーワード。
 * @default player
 *
 * @param MoverNoteTag
 * @text 追跡者メモタグ（このタグを持つイベントが追跡者になる）
 * @type string
 * @default Destruction
 *
 * @param RandomMoveType
 * @text ターゲット不在時の移動タイプ
 * @type select
 * @option ランダム移動
 * @value 1
 * @option プレイヤーに近づく
 * @value 2
 * @option プレイヤーから遠ざかる
 * @value 3
 * @option 動かない
 * @value 0
 * @default 1
 *
 * @param ChaseInterval
 * @text 追跡処理間隔(フレーム)
 * @type number
 * @min 1
 * @default 15
 *
 * @param NearTargetThreshold
 * @text 近接ターゲット距離 (タイル数)
 * @type number
 * @min 2
 * @desc この距離(マンハッタン距離)以下のターゲットに対してA*が失敗した場合、強制脱出を最優先します。U字型障害物の対策です。
 * @default 5
 *
 * @param CollisionSelfSwitch
 * @text デフォルト衝突時セルフスイッチ
 * @type select
 * @option A
 * @option B
 * @option C
 * @option D
 * @default A
 *
 * @param DebugLog
 * @text デバッグログ出力
 * @type boolean
 * @default false
 *
 * @param PlacementCooldownFrames
 * @text 配置クールダウン（フレーム）
 * @type number
 * @min 0
 * @default 6
 * @desc 置かれてから何フレームの間、追跡者がそのイベントを無視するか（デフォルト6）。推奨: 20（約0.33秒）
 *
 * @param PendingSelfSwitchDelay
 * @text 保留セルフスイッチ遅延（フレーム）
 * @type number
 * @min 0
 * @default 1
 * @desc 衝突判定でセルフスイッチを保留した場合、何フレーム後に実際に書き込むか（デフォルト1）。推奨: 2
 *
 * @command StartChase
 * @text 追跡開始（全体）
 * @desc 追跡処理を全体的に開始します（全体再開に使います）
 *
 * @command StopChase
 * @text 追跡停止（全体）
 * @desc 追跡処理を全体的に停止します
 *
 * @command StopSelfChase
 * @text 個別追跡停止（実行イベント）
 * @desc このイベントの追跡処理を停止します。再開はマップ移動などが必要です。
 */

(() => {
    const pluginName = "FindNearestTarget";
    const params = PluginManager.parameters(pluginName);
    const defaultTargetTag = String(params['DefaultTargetTag'] || 'box');
    const playerKeyword = String(params['PlayerKeyword'] || 'player');
    const moverTag = String(params['MoverNoteTag'] || 'Destruction');
    const ignoreTag = 'Ignore';
    const randomMoveType = Number(params['RandomMoveType'] || 1);
    const chaseIntervalDefault = Number(params['ChaseInterval'] || 15);
    const NEAR_THRESHOLD = Number(params['NearTargetThreshold'] || 5);
    const defaultCollisionSS = String(params['CollisionSelfSwitch'] || 'A');

    const debugLogParam = String(params['DebugLog'] || 'false') === 'true';
    const INDIVIDUAL_CHASE_KEY = 'chaseActive';

    // パラメータ化された設定値
    const PLACEMENT_COOLDOWN_FRAMES = Number(params['PlacementCooldownFrames'] || 6); // 置かれてから無視するフレーム数
    const PENDING_SS_DELAY = Number(params['PendingSelfSwitchDelay'] || 1); // pending SS の遅延（フレーム）

    let _frameCount = 0;

    // --- 新規: イベントインスタンスIDカウンタ & pending self-switch 書き込みキュー ---
    let _eventInstanceCounter = 0;
    // pending items: { mapId, eventId, letter, instanceId, scheduledAt }
    const _pendingSelfSwitches = [];

    let chaseActive = true;
    let _chaseCounter = 0;
    let chaseInterval = chaseIntervalDefault;

    const log = (...args) => {
        if (debugLogParam) console.log('[FindNearestTarget]', ...args);
    };

    const escapeRegExp = (s) => String(s).replace(/[.*+?^${}()|[\]\\]/g, '\\$&');

    const hasNoteTag = (eventData, tag) => {
        if (!eventData || !eventData.note) return false;
        const t = String(tag).trim();
        const re = new RegExp('<\\s*' + escapeRegExp(t) + '\\s*>', 'iu');
        return re.test(eventData.note);
    };

    const getNoteMeta = (eventData, key) => {
        if (!eventData || !eventData.note) return null;
        const re = new RegExp(`<\\s*${escapeRegExp(key)}\\s*[:：]\\s*([^>]+?)\\s*>`, 'iu');
        const m = eventData.note.match(re);
        return m ? m[1].trim() : null;
    };

    const manhattanDistance = (x1, y1, x2, y2) => Math.abs(x1 - x2) + Math.abs(y1 - y2);

    const dirToDelta = (dir) => {
        switch (dir) {
            case 2: return [0, 1];
            case 4: return [-1, 0];
            case 6: return [1, 0];
            case 8: return [0, -1];
            default: return [0, 0];
        }
    };

    const _Game_CharacterBase_canPass = Game_CharacterBase.prototype.canPass;
    const _Game_CharacterBase_isMapPassable = Game_CharacterBase.prototype.isMapPassable;

    // ---- ここから変更点 ----
    // グローバルフラグ：true のとき canPassForMover はプレイヤー位置を「通れる」とみなす
    let _ignorePlayerBlocking = false;

    // <Ignore>タグを持つイベントをすり抜けるようにする一時的なcanPass
    const canPassForMover = function (x, y, dir) {
        const x2 = $gameMap.roundXWithDirection(x, dir);
        const y2 = $gameMap.roundYWithDirection(y, dir);

        // 1. 地形・タイルセットの判定 (元の判定)
        if (!_Game_CharacterBase_canPass.call(this, x, y, dir)) {
            return false;
        }

        // 追加：フラグが立っていれば、プレイヤー位置を障害物として扱わない（通れる扱いにする）
        if (_ignorePlayerBlocking && $gamePlayer.x === x2 && $gamePlayer.y === y2) {
            // プレイヤータイルは「通れる」と判断
            return true;
        }

        // 2. イベントの判定
        const events = $gameMap.eventsXy(x2, y2);
        for (const event of events) {
            if (event.eventId() === this.eventId()) continue;
            const eventData = event.event();
            if ((eventData && hasNoteTag(eventData, ignoreTag)) || event.isThrough()) {
                continue;
            }
            return false;
        }

        return true;
    };
    // ---- ここまで変更点 ----

    // ヘルパー: オブジェクトが Game_Event かどうか安全に判定
    const isGameEvent = (obj) => {
        try { return obj instanceof Game_Event; } catch (e) {
            return obj && typeof obj.eventId === 'function';
        }
    };

    // pending queue を処理する（スケジュールされたフレームに到達したらチェックして書き込む）
    function processPendingSelfSwitches() {
        if (_pendingSelfSwitches.length === 0) return;
        const now = _frameCount;
        for (let i = _pendingSelfSwitches.length - 1; i >= 0; i--) {
            const item = _pendingSelfSwitches[i];
            if (now < item.scheduledAt) continue;

            // 追加の安全チェック: map と event が有効か・instanceId 一致かを確認
            const mapEvent = $gameMap.event(item.eventId);
            if (!mapEvent || mapEvent._erased) {
                log(` -> Pending SS cleared (event missing/erased) for event ${item.eventId} (expectedInst=${item.instanceId}) at frame ${now}`);
                _pendingSelfSwitches.splice(i, 1);
                continue;
            }

            // インスタンスIDの不一致チェック: 古い予約なので破棄
            if (typeof item.instanceId === 'number' && mapEvent._instanceId !== item.instanceId) {
                log(` -> Pending SS cleared (inst mismatch) for event ${item.eventId}. expectedInst=${item.instanceId}, mapEventInst=${mapEvent._instanceId}`);
                _pendingSelfSwitches.splice(i, 1);
                continue;
            }

            // 最終チェック: 該当セルフスイッチが既に true でないか
            const key = [item.mapId, item.eventId, item.letter];
            if (!$gameSelfSwitches.value(key)) {
                $gameSelfSwitches.setValue(key, true);
                log(` -> Pending SS set for event ${item.eventId} (inst ${item.instanceId}) at frame ${now}`);
            } else {
                log(` -> Pending SS already true for event ${item.eventId}`);
            }
            _pendingSelfSwitches.splice(i, 1);
        }
    }

    function scheduleSelfSwitch(mapId, eventId, letter, instanceId, delayFrames) {
        // delayFrames が与えられなければパラメータ PENDING_SS_DELAY を使う
        const delay = (typeof delayFrames === 'number' && delayFrames >= 0) ? Math.floor(delayFrames) : PENDING_SS_DELAY;
        const scheduledAt = _frameCount + Math.max(0, delay);

        // 安全策1: すでに同じ map,event,letter の予約があれば重複登録を避ける
        if (_pendingSelfSwitches.some(p => p.mapId === mapId && p.eventId === eventId && p.letter === letter)) {
            if (debugLogParam) log(`Duplicate schedule skipped for Event ${eventId} (map ${mapId} ss ${letter})`);
            return false;
        }

        // 安全策2: 実際のマップイベントが存在するか確認。存在すればその instanceId を採用。
        const ev = $gameMap.event(eventId);
        if (!ev || ev._erased) {
            if (debugLogParam) log(`Schedule skipped: target event ${eventId} not found or is erased on map ${mapId}`);
            return false;
        }
        const resolvedInst = (typeof ev._instanceId === 'number') ? ev._instanceId : instanceId;
        // もし resolvedInst が undefined / -1 のままなら、安全のため登録をスキップ
        if (typeof resolvedInst !== 'number' || resolvedInst < 0) {
            if (debugLogParam) log(`Schedule skipped: no valid instanceId for event ${eventId}`);
            return false;
        }

        _pendingSelfSwitches.push({ mapId, eventId, letter, instanceId: resolvedInst, scheduledAt });
        if (debugLogParam) log(`Scheduled pending SS for event ${eventId} inst ${resolvedInst} at ${scheduledAt} (delay ${delay})`);
        return true;
    }

    function handleMoverMovement(moverEvent) {
        if (!moverEvent || moverEvent._erased) return;

        if (moverEvent.isMoving()) return;

        const moverX = moverEvent.x;
        const moverY = moverEvent.y;

        const moverEd = moverEvent.event();
        const moverSpecificTarget = getNoteMeta(moverEd, 'Target');
        const moverCollisionSS = getNoteMeta(moverEd, 'CollisionSS') || defaultCollisionSS;
        const targetTag = (moverSpecificTarget && moverSpecificTarget.length > 0) ? moverSpecificTarget : defaultTargetTag;

        let nearestTarget = null;
        let minDistance = Infinity;

        // ターゲット検索ロジック...
        if (targetTag.toLowerCase() === playerKeyword.toLowerCase()) {
            nearestTarget = $gamePlayer;
            minDistance = manhattanDistance(moverX, moverY, nearestTarget.x, nearestTarget.y);
        } else {
            for (const target of $gameMap.events()) {
                if (!target || target._erased || target.eventId() === moverEvent.eventId()) continue;
                const ted = target.event();
                if (ted && hasNoteTag(ted, targetTag) && !target.isThrough()) {
                    const d = manhattanDistance(moverX, moverY, target.x, target.y);
                    if (d < minDistance) {
                        minDistance = d;
                        nearestTarget = target;
                    }
                }
            }
        }

        if (nearestTarget) {
            // --- ★ミナ追記(1)：ターゲットイベントの有効性チェック（消去済みなら追跡中止）★ ---
            // ターゲットがイベントであり、かつ消去済み (erased) の場合は追跡を中止し、ターゲット不在として扱う
            if (isGameEvent(nearestTarget) && nearestTarget._erased) {
                log(` -> Target event ${nearestTarget.eventId()} is erased. Treating as target-less.`);
                nearestTarget = null; // ターゲットを無効化し、ターゲット不在処理へ
            }
            // -----------------------------------------------------------------------------------
        }

        if (nearestTarget) { // ★修正：ここで再度チェック！★
            // 1. 衝突時のSelf-Switch ON 処理 (Player)
            if (minDistance <= 1) {
                if (nearestTarget === $gamePlayer && moverCollisionSS) {
                    const mapId = $gameMap.mapId();
                    const key = [mapId, moverEvent.eventId(), moverCollisionSS];
                    if (!$gameSelfSwitches.value(key)) {
                        $gameSelfSwitches.setValue(key, true);
                        return;
                    }
                }
            }

            const tx = nearestTarget.x;
            const ty = nearestTarget.y;
            let dir = 0;

            // 2. A*探索の実行 (ターゲット近傍検索を組み込み)
            _ignorePlayerBlocking = (targetTag.toLowerCase() !== playerKeyword.toLowerCase());
            Game_CharacterBase.prototype.canPass = canPassForMover;
            try {
                dir = moverEvent.findDirectionTo(tx, ty);

                // A*失敗時のターゲット近傍検索 
                if (dir === 0) {
                    const adjacentDirs = [2, 4, 6, 8];
                    for (const d of adjacentDirs) {
                        const [dx, dy] = dirToDelta(d);
                        const ax = tx + dx;
                        const ay = ty + dy;
                        if (moverX === ax && moverY === ay) continue;

                        const altDir = moverEvent.findDirectionTo(ax, ay);
                        if (altDir > 0) {
                            dir = altDir;
                            break;
                        }
                    }
                }
            } finally {
                Game_CharacterBase.prototype.canPass = _Game_CharacterBase_canPass;
                _ignorePlayerBlocking = false; // 忘れずリセット
            }


            // 3. 移動実行/失敗時のフォールバック
            if (dir > 0) {
                // 衝突予測のロジックは維持
                // 安全にイベントかどうか判定してから eventId() を使う
                if (isGameEvent(nearestTarget)) {
                    const targetEventId = nearestTarget.eventId();
                    // 置かれた直後のターゲットは無視する（クールダウン）
                    const birth = typeof nearestTarget._birthFrame === 'number' ? nearestTarget._birthFrame : -Infinity;
                    const age = (_frameCount - birth);

                    // PLACEMENT_COOLDOWN_FRAMES によるガード
                    if (birth > -Infinity && age < PLACEMENT_COOLDOWN_FRAMES) {
                        log(` -> Ignoring recently placed target ${targetEventId} (age ${age} frames). CollisionSS skipped.`);
                        return;
                    }

                    const [dx, dy] = dirToDelta(dir);
                    const predictedX = moverX + dx;
                    const predictedY = moverY + dy;

                    if (predictedX === tx && predictedY === ty) {
                        // 方向転換を強制実行してからセルフスイッチの発動（ただし即時書き込みではなく保留／検証）
                        moverEvent.setDirection(dir);
                        log(` -> Forced direction set to: ${dir} before collision SS. targetEventId=${targetEventId}`);

                        const mapId = $gameMap.mapId();

                        // **ここが修正点**: 直接 $gameSelfSwitches.setValue しないで、保留リストに登録して数フレーム後にインスタンスを確認してから設定する
                        // --- ★ミナ追記(2)：ターゲットのインスタンスIDのチェックをより厳密に★ ---
                        const instId = typeof nearestTarget._instanceId === 'number' ? nearestTarget._instanceId : -1;

                        // 追加の安全: 実体がない / インスタンスが未確定なら登録しない
                        if (instId < 0) {
                            log(` -> Skip scheduling: target event ${targetEventId} has no instanceId (Value: ${instId})`);
                            return;
                        }
                        // ----------------------------------------------------------------------

                        // すでに同一ターゲットに予約があるかチェックして登録
                        if (_pendingSelfSwitches.some(p => p.mapId === mapId && p.eventId === targetEventId && p.letter === moverCollisionSS)) {
                            if (debugLogParam) log(`Duplicate schedule skipped for Event ${targetEventId}`);
                            return;
                        }

                        // delay は scheduleSelfSwitch 内で解決される
                        scheduleSelfSwitch(mapId, targetEventId, moverCollisionSS, instId);
                        return;
                    }
                } else {
                    // nearestTarget がイベントでない（安全に無視）なら通常移動続行
                    const [dx, dy] = dirToDelta(dir);
                    const predictedX = moverX + dx;
                    const predictedY = moverY + dy;
                    if (predictedX === tx && predictedY === ty && nearestTarget === $gamePlayer) {
                        const mapId = $gameMap.mapId();
                        const key = [mapId, moverEvent.eventId(), moverCollisionSS];
                        if (!$gameSelfSwitches.value(key)) {
                            $gameSelfSwitches.setValue(key, true);
                            return;
                        }
                    }
                }
                moverEvent.moveStraight(dir); // 追跡移動
            } else {
                // ★★★ V20: A*失敗時のフォールバック分岐ロジック ★★★
                const isNearTarget = minDistance <= NEAR_THRESHOLD;
                let skipUltimateEscape = false; // 強制脱出をスキップするかどうかのフラグ

                if (!isNearTarget) {
                    // ターゲットが遠い場合: MZ標準の賢い移動を試す
                    log(' -> A* failed far from target (Distance: ' + minDistance + '). Falling back to goal-oriented moveTowardCharacter (MZ standard).');

                    // ここでも、ターゲットがプレイヤー以外ならプレイヤー無視フラグを立てる
                    _ignorePlayerBlocking = (nearestTarget !== $gamePlayer);
                    Game_CharacterBase.prototype.canPass = canPassForMover;
                    try {
                        moverEvent.moveTowardCharacter(nearestTarget);
                    } finally {
                        Game_CharacterBase.prototype.canPass = _Game_CharacterBase_canPass;
                        _ignorePlayerBlocking = false;
                    }

                    // moveTowardCharacterが成功してisMoving()がtrueなら、ここで終了
                    if (moverEvent.isMoving()) {
                        log(' -> moveTowardCharacter succeeded. Character is moving around the obstacle.');
                        skipUltimateEscape = true; // 強制脱出は不要
                    } else {
                        log(' -> moveTowardCharacter also failed. Proceeding to ULTIMATE ESCAPE.');
                    }
                } else {
                    // ターゲットが近い場合: moveTowardCharacterをスキップして即座に強制脱出
                    log(' -> A* failed near target (Distance: ' + minDistance + '). Skipping moveTowardCharacter, proceeding to ULTIMATE ESCAPE.');
                }

                if (skipUltimateEscape) return;

                // ★★★ V20: ULTIMATE ESCAPE (タイルのみチェック) - 全ての賢い移動が失敗した後の最終手段 ★★★
                log(' -> Applying ultimate TILE-ONLY forced escape.');

                let ultimateFallbackDir = 0;
                const directions = [2, 4, 6, 8];

                // 他のイベントを完全に無視し、マップタイルのみをチェックする (isMapPassableを使用)
                for (const d of directions) {
                    if (_Game_CharacterBase_isMapPassable.call(moverEvent, moverX, moverY, d)) {
                        ultimateFallbackDir = d;
                        break;
                    }
                }

                if (ultimateFallbackDir > 0) {
                    log(` -> ULTIMATE ESCAPE executed to direction: ${ultimateFallbackDir} (ignoring events).`);
                    moverEvent.moveStraight(ultimateFallbackDir);
                } else {
                    // タイル判定でも4方向全て不可の場合、設定されたRandomMoveTypeを実行
                    log(' -> ULTIMATE ESCAPE failed (surrounded by impassable tiles). Applying RandomMoveType.');

                    // ここはランダム移動なのでプレイヤー無視フラグは立てない（通常判定）
                    Game_CharacterBase.prototype.canPass = canPassForMover;
                    try {
                        switch (randomMoveType) {
                            case 1: moverEvent.moveRandom(); break;
                            case 2: moverEvent.moveTowardPlayer(); break;
                            case 3: moverEvent.moveAwayFromPlayer(); break;
                            case 0: default: break;
                        }
                    } finally {
                        Game_CharacterBase.prototype.canPass = _Game_CharacterBase_canPass;
                    }
                }
            }
        }

        // --- ターゲット不在時の処理 ---
        else {
            Game_CharacterBase.prototype.canPass = canPassForMover;
            try {
                switch (randomMoveType) {
                    case 1: moverEvent.moveRandom(); break;
                    case 2: moverEvent.moveTowardPlayer(); break;
                    case 3: moverEvent.moveAwayFromPlayer(); break;
                    case 0: default: break;
                }
            } finally {
                Game_CharacterBase.prototype.canPass = _Game_CharacterBase_canPass;
            }
        }
    }

    const _Scene_Map_update = Scene_Map.prototype.update;
    Scene_Map.prototype.update = function () {
        _Scene_Map_update.call(this);
        _frameCount++; // 全体フレームカウンタ (プラグイン内で出生時刻を測るため)

        // まず保留のセルフスイッチを処理
        processPendingSelfSwitches();

        if (!chaseActive) return;
        _chaseCounter++;
        if (_chaseCounter < chaseInterval) return;
        _chaseCounter = 0;
        if (debugLogParam) log('chase tick — scanning movers...');

        for (const event of $gameMap.events()) {
            if (!event || event._erased) continue;

            // --- 新規: イベント初期化（出生フレームとインスタンスIDを記録） ---
            if (typeof event._birthFrame === 'undefined') {
                event._birthFrame = _frameCount;
                if (debugLogParam) log(`Event ${event.eventId()} birthFrame set to ${event._birthFrame}`);
            }
            if (typeof event._instanceId === 'undefined' || event._instanceId < 0) {
                // instanceId が未定義か不正値の場合に割り当てる（マップロード時など）
                _eventInstanceCounter++;
                event._instanceId = _eventInstanceCounter;
                if (debugLogParam) log(`Event ${event.eventId()} instanceId assigned ${event._instanceId}`);
            }
            // --- 初期化ここまで ---

            const ed = event.event();
            if (ed && hasNoteTag(ed, moverTag)) {

                // ページ判定 (1ページ目=インデックス0 のみ追跡を有効にする)
                if (event._pageIndex !== 0) {
                    if (debugLogParam) log(`Event ${event.eventId()} is NOT on page 1 (index ${event._pageIndex}). Skipping chase.`);
                    continue; // 追跡処理をスキップ
                }

                const eventChaseActive = event._customData ? event._customData[INDIVIDUAL_CHASE_KEY] : true;
                if (!eventChaseActive) continue;
                handleMoverMovement(event);
            }
        }
    };

    PluginManager.registerCommand(pluginName, "StopSelfChase", function (args) {
        const eventId = this.eventId();
        if (eventId > 0) {
            const event = $gameMap.event(eventId);
            if (event) {
                if (!event._customData) event._customData = {};
                event._customData[INDIVIDUAL_CHASE_KEY] = false;
                if (debugLogParam) log(`StopSelfChase: Event ${eventId} set ${INDIVIDUAL_CHASE_KEY} to false.`);
            }
        }
    });

    PluginManager.registerCommand(pluginName, "StartChase", args => {
        chaseActive = true;
        _chaseCounter = 0;
        chaseInterval = chaseIntervalDefault;
        if (debugLogParam) log("StartChase called. chaseActive=", chaseActive, "interval=", chaseInterval);
    });

    PluginManager.registerCommand(pluginName, "StopChase", args => {
        chaseActive = false;
        if (debugLogParam) log("StopChase called. chaseActive=", chaseActive);
    });

    if (debugLogParam) log('plugin loaded. defaultTargetTag=', defaultTargetTag, ' moverTag=', moverTag, ' interval=', chaseInterval,
        ' placementCooldown=', PLACEMENT_COOLDOWN_FRAMES, ' pendingDelay=', PENDING_SS_DELAY);
})();