/*:
 * @target MZ
 * @plugindesc アクターのHPを■でグラデーション表示し、減少分を指定記号で表示するプラグインに加え、中間HP用シンボルおよびMP用中間ブロックシンボルをパラメータ指定可能にした ver1.7.5
 *
 * @param actorId
 * @text 対象アクターID
 * @desc HP/MP表示対象のアクターID（例: 1）
 * @default 1
 *
 * @param hpIconIndex
 * @text HPアイコン
 * @desc HPのラベルとして表示するアイコンのインデックス。アイコンセットで指定します。
 * @type number
 * @default 16
 *
 * @param hpIconScale
 * @text HPアイコン倍率
 * @desc HPアイコンの表示倍率（例: 0.5で半分、1.0で等倍、2.0で2倍）。
 * @type number
 * @decimals 2
 * @min 0.1
 * @default 1.0
 *
 * @param blockSymbol
 * @text HPフルブロックシンボル
 * @desc HPが十分にある部分を表すシンボル。例：「■」
 * @default ▮
 *
 * @param blockPerHP
 * @text 1ブロックのHP量
 * @desc 1つのブロックが何HP分か（例: 5）
 * @default 5
 *
 * @param midBlockSymbol
 * @text HP中間ブロックシンボル
 * @desc HPが中間範囲（1～指定値）残っている部分を表すシンボル。例：「▩」
 * @default ◘
 *
 * @param midBlockThreshold
 * @text 中間ブロック上限HP
 * @desc 何HP以下の時に中間ブロックを表示するか（1以上）。例:2
 * @default 2
 *
 * @param midBlockColorOverride
 * @text 中間ブロック色変更
 * @desc ONの場合、中間ブロックの色をグラデーションではなく指定色にします。
 * @type boolean
 * @on ON
 * @off OFF
 * @default false
 *
 * @param midBlockOverrideColor
 * @text 中間ブロック指定色
 * @desc 「中間ブロック色変更」がONの場合に適用される色（16進数カラーコード）。
 * @default #808080
 *
 * @param emptyBlockSymbol
 * @text HP空ブロックシンボル
 * @desc HPが残っていない部分を表す記号。例：「□」
 * @default ▯
 *
 * @param hpTextColor
 * @text HPラベル色
 * @desc HPの「HP」文字部分の色（16進数カラーコード）。例：#1E90FF (アイコンの場合はこの色は適用されません)
 * @default #1E90FF
 *
 * @param displaySwitch
 * @text HP表示スイッチ
 * @desc このスイッチがONの場合にのみHP表示を行います。0の場合は常時表示。
 * @default 0
 *
 * @param x
 * @text HPウィンドウX座標
 * @desc HPウィンドウのX座標
 * @default 0
 *
 * @param y
 * @text HPウィンドウY座標
 * @desc HPウィンドウのY座標
 * @default 0
 *
 * @param fontSize
 * @text HPフォントサイズ
 * @desc HP表示ウィンドウのフォントサイズ
 * @default 24
 *
 * @param blockYOffset
 * @text 全ブロックY座標微調整
 * @desc HP/MPの全てのブロックシンボルに適用されるY座標微調整。正の値で下へ、負の値で上へ。
 * @type number
 * @min -99
 * @max 99
 * @default 0
 *
 * @param emptyBlockYOffset
 * @text 空HPブロックY座標微調整
 * @desc HP空ブロックシンボル（例: □）にのみ適用されるY座標微調整。
 * @type number
 * @min -99
 * @max 99
 * @default 0
 *
 * @param emptyBlockFontSize
 * @text 空HPブロックフォントサイズ
 * @desc HP空ブロックシンボルのフォントサイズ。0の場合は通常のフォントサイズを適用。
 * @type number
 * @default 0
 *
 * @param blockSpacing
 * @text HPブロック間隔
 * @desc HPブロック間の水平方向の間隔（負の値で重ねる）。
 * @type number
 * @min -99
 * @max 99
 * @default 2
 *
 * @param mpIconIndex
 * @text MPアイコン
 * @desc MPのラベルとして表示するアイコンのインデックス。アイコンセットで指定します。
 * @type number
 * @default 17
 *
 * @param mpIconScale
 * @text MPアイコン倍率
 * @desc MPアイコンの表示倍率（例: 0.5で半分、1.0で等倍、2.0で2倍）。
 * @type number
 * @decimals 2
 * @min 0.1
 * @default 1.0
 *
 * @param mpBlockSymbol
 * @text MPブロックシンボル
 * @desc MPがある部分を表すシンボル。例：「★」
 * @default ★
 *
 * @param mpBlockPerMP
 * @text 1ブロックのMP量
 * @desc 1つのブロックが何MP分か（例: 5）
 * @default 5
 *
 * @param mpMidBlockSymbol
 * @text MP中間ブロックシンボル
 * @desc MPが中間範囲（1～指定値）残っている部分を表すシンボル。例：「◐」
 * @default ◐
 *
 * @param mpMidBlockThreshold
 * @text 中間ブロックMP上限
 * @desc 何MP以下の時に中間ブロックを表示するか（1以上）。例:2
 * @default 2
 *
 * @param emptyMpBlockSymbol
 * @text MP空ブロックシンボル
 * @desc MPが残っていない部分を表す記号。例：「☆」
 * @default ☆
 *
 * @param mpLabelColor
 * @text MPラベル色
 * @desc MPの「MP」文字部分の色（16進数カラーコード）。例：#9932cc (アイコンの場合はこの色は適用されません)
 * @default #9932cc
 *
 * @param mpBlockColor
 * @text MPブロック塗り色
 * @desc MPのブロック部分の色（16進数カラーコード）。例：#00FF00
 * @default #00FF00
 *
 * @param mpDisplaySwitch
 * @text MP表示スイッチ
 * @desc このスイッチがONの場合にのみMP表示を行います。0の場合は常時表示。
 * @default 0
 *
 * @param mpX
 * @text MPウィンドウX座標
 * @desc MPウィンドウのX座標
 * @default 0
 *
 * @param mpY
 * @text MPウィンドウY座標
 * @desc MPウィンドウのY座標
 * @default 40
 *
 * @param mpFontSize
 * @text MPブロックフォントサイズ
 * @desc MP表示ウィンドウのフォントサイズ
 * @default 24
 *
 * @param emptyMpBlockYOffset
 * @text 空MPブロックY座標微調整
 * @desc MP空ブロックシンボル（例: ☆）にのみ適用されるY座標微調整。
 * @type number
 * @min -99
 * @max 99
 * @default 0
 *
 * @param emptyMpBlockFontSize
 * @text 空MPブロックフォントサイズ
 * @desc MP空ブロックシンボルのフォントサイズ。0の場合は通常のフォントサイズを適用。
 * @type number
 * @default 0
 *
 * @param mpBlockSpacing
 * @text MPブロック間隔
 * @desc MPブロック間の水平方向の間隔（負の値で重ねる）。
 * @type number
 * @min -99
 * @max 99
 * @default 2
 *
 * @help
 * このプラグインは、指定アクターのHP/MPをグラデーションブロックで表示します。
 * HPについては、HPフルブロック、中間ブロック（残量が midBlockThreshold 以下の場合）および空ブロックで表示されます。
 * MPについては、MPブロック、中間ブロック（残量が mpMidBlockThreshold 以下の場合）および空ブロックで表示します。
 *
 * HP/MPのラベルは、プラグインパラメータで指定したアイコンに置き換わります。
 * アイコンのサイズも個別に調整可能です。
 *
 * また、MP表示については「MP」ラベルの色とブロック部分の色を、個別に指定可能です。
 * （アイコンを使用する場合、ラベル色はアイコンの色に依存するため設定は無視されます）
 *
 * 各パラメーターを変更することで、表示の見た目（ブロックのシンボル、1ブロックあたりのHP/MP量、閾値等）を調整できます。
 *
 * 【重要】ブロックシンボルのY座標がアイコンや他のブロックとズレる場合、
 * 「全ブロックY座標微調整」パラメータを調整してください。正の値で下に、負の値で上に動きます。
 *
 * 【新機能】HP/MPの「空ブロックシンボル」（例: □, ☆）のみが他のブロックとY座標がずれる場合、
 * それぞれ「空HPブロックY座標微調整」「空MPブロックY座標微調整」パラメータで個別に調整できます。
 *
 * 【新機能】ブロック間の間隔を調整する「HPブロック間隔」および「MPブロック間隔」
 * パラメータを追加しました。負の値を指定することで、ブロックを重ねて表示できます。
 *
 * 【新機能】HP中間ブロックシンボルの色をグラデーションではなく指定色にするか、
 * プラグインパラメータで設定できるようになりました。「中間ブロック色変更」をONにすると、
 * 「中間ブロック指定色」で指定した色（デフォルトはグレー #808080）が適用されます。
 *
 * 【新機能】HP/MPの空ブロックシンボルのみのフォントサイズを個別に調整できる
 * 「空HPブロックフォントサイズ」および「空MPブロックフォントサイズ」パラメータを追加しました。
 * 0を指定すると、通常のフォントサイズが適用されます。
 */

(() => {
    const parameters = PluginManager.parameters("HPBlockDisplay");
    const actorId = Number(parameters["actorId"]);
    const hpIconIndex = Number(parameters["hpIconIndex"] || 0);
    const hpIconScale = Number(parameters["hpIconScale"] || 1.0);
    const blockSymbol = parameters["blockSymbol"];
    const blockPerHP = Number(parameters["blockPerHP"]);
    const midBlockSymbol = parameters["midBlockSymbol"];
    const midBlockThreshold = Math.max(1, Number(parameters["midBlockThreshold"]));
    const midBlockColorOverride = parameters["midBlockColorOverride"] === "true";
    const midBlockOverrideColor = parameters["midBlockOverrideColor"];
    const emptyBlockSymbol = parameters["emptyBlockSymbol"];
    const hpTextColor = parameters["hpTextColor"];
    const displaySwitch = Number(parameters["displaySwitch"]);
    const windowX = Number(parameters["x"]);
    const windowY = Number(parameters["y"]);
    const fontSize = Number(parameters["fontSize"]);
    const blockYOffset = Number(parameters["blockYOffset"] || 0);
    const emptyBlockYOffset = Number(parameters["emptyBlockYOffset"] || 0);
    const emptyBlockFontSize = Number(parameters["emptyBlockFontSize"] || 0); // 追加
    const blockSpacing = Number(parameters["blockSpacing"] || 2);

    const mpIconIndex = Number(parameters["mpIconIndex"] || 0);
    const mpIconScale = Number(parameters["mpIconScale"] || 1.0);
    const mpBlockSymbol = parameters["mpBlockSymbol"];
    const mpBlockPerMP = Number(parameters["mpBlockPerMP"]);
    const mpMidBlockSymbol = parameters["mpMidBlockSymbol"];
    const mpMidBlockThreshold = Math.max(1, Number(parameters["mpMidBlockThreshold"]));
    const emptyMpBlockSymbol = parameters["emptyMpBlockSymbol"];
    const mpLabelColor = parameters["mpLabelColor"];
    const mpBlockColor = parameters["mpBlockColor"];
    const mpDisplaySwitch = Number(parameters["mpDisplaySwitch"]);
    const mpX = Number(parameters["mpX"]);
    const mpY = Number(parameters["mpY"]);
    const mpFontSize = Number(parameters["mpFontSize"]);
    const emptyMpBlockYOffset = Number(parameters["emptyMpBlockYOffset"] || 0);
    const emptyMpBlockFontSize = Number(parameters["emptyMpBlockFontSize"] || 0); // 追加
    const mpBlockSpacing = Number(parameters["mpBlockSpacing"] || 2);

    const labelBlockGap = 6;

    const windowPaddingAddition = 26;

    function componentToHex(c) {
        const hex = c.toString(16);
        return hex.length === 1 ? "0" + hex : hex;
    }
    function rgbToHex(r, g, b) {
        const hexR = componentToHex(r);
        const hexG = componentToHex(g);
        const hexB = componentToHex(b);
        return "#" + hexR + hexG + hexB;
    }
    function getBlockColor(index, total) {
        const ratio = total > 1 ? index / (total - 1) : 0;
        const r = Math.floor(200 + (255 - 200) * ratio);
        const g = Math.floor(0 + (255 - 0) * ratio);
        return rgbToHex(r, g, 0);
    }

    class Window_HPBlockDisplay extends Window_Base {
        standardPadding() { return 0; }
        constructor() {
            super(new Rectangle(windowX, windowY, 300, fontSize * 3 + 16));
            this.opacity = 0;
            this.contents.fontSize = fontSize;
            this.visible = (displaySwitch === 0 || $gameSwitches.value(displaySwitch));
            this.adjustWindowWidth();
            this.refresh();
        }
        adjustWindowWidth() {
            const actor = $gameActors.actor(actorId);
            if (!actor) return;
            this.contents.fontSize = fontSize;
            const labelWidth = ImageManager.iconWidth * hpIconScale;
            const blockWidth = this.textWidth(blockSymbol);
            const count = Math.ceil(actor.mhp / blockPerHP);
            const newWidth = labelWidth + labelBlockGap + count * blockWidth + (count > 0 ? (count - 1) * blockSpacing : 0) + windowPaddingAddition;
            if (this.width !== newWidth) {
                this.width = newWidth;
                this.createContents();
                this.contents.fontSize = fontSize;
            }
        }
        update() {
            super.update();
            const should = (displaySwitch === 0 || $gameSwitches.value(displaySwitch));
            if (should !== this.visible) {
                this.contents.clear();
                this.visible = should;
            }
            if (this.visible) {
                this.adjustWindowWidth();
                this.refresh();
            }
        }
        refresh() {
            this.contents.clear();
            this.contents.fontSize = fontSize;
            const actor = $gameActors.actor(actorId);
            if (!actor) return;
            const cur = actor.hp;
            const totalBlocks = Math.ceil(actor.mhp / blockPerHP);
            const y = (this.contentsHeight() - (ImageManager.iconHeight * hpIconScale)) / 2;

            const iconW = ImageManager.iconWidth;
            const iconH = ImageManager.iconHeight;
            this.contents.paintOpacity = this.contentsOpacity;
            this.contents.blt(
                ImageManager.loadSystem("IconSet"),
                hpIconIndex % 16 * iconW,
                Math.floor(hpIconIndex / 16) * iconH,
                iconW,
                iconH,
                0,
                y,
                iconW * hpIconScale,
                iconH * hpIconScale
            );
            this.contents.paintOpacity = 255;

            const labelW = iconW * hpIconScale;
            this.contents.fontSize = fontSize;
            const bw = this.textWidth(blockSymbol);

            const baseBlockY = y + (iconH * hpIconScale) / 2 - this.contents.fontSize / 2;

            for (let i = 0; i < totalBlocks; i++) {
                const x = labelW + labelBlockGap + i * (bw + blockSpacing);
                const rem = cur - i * blockPerHP;
                
                const commonBlockY = baseBlockY + blockYOffset;

                if (rem <= 0) {
                    this.changeTextColor("#000000");
                    const originalFontSize = this.contents.fontSize;
                    if (emptyBlockFontSize > 0) {
                        this.contents.fontSize = emptyBlockFontSize;
                    }
                    this.drawText(emptyBlockSymbol, x, commonBlockY + emptyBlockYOffset, bw, "left");
                    this.contents.fontSize = originalFontSize; // 元に戻す
                } else if (rem <= midBlockThreshold) {
                    const color = midBlockColorOverride ? midBlockOverrideColor : getBlockColor(i, totalBlocks);
                    this.drawTextColor(midBlockSymbol, x, commonBlockY, bw, "left", color);
                } else {
                    this.drawTextColor(blockSymbol, x, commonBlockY, bw, "left", getBlockColor(i, totalBlocks));
                }
            }
        }
        drawTextColor(text, x, y, maxWidth, align, color) {
            const context = this.contents.context;
            context.save();
            context.fillStyle = color;
            context.font = `${this.contents.fontSize}px ${this.contents.fontFace}`;
            context.textAlign = align;
            context.fillText(text, x, y + this.contents.fontSize, maxWidth);
            context.restore();
        }
    }

    class Window_MPBlockDisplay extends Window_Base {
        standardPadding() { return 0; }
        constructor() {
            const windowHeight = mpFontSize * 3 + 16;
            super(new Rectangle(mpX, mpY, 300, windowHeight));
            this.opacity = 0;
            this.contents.fontSize = mpFontSize;
            this.visible = (mpDisplaySwitch === 0 || $gameSwitches.value(mpDisplaySwitch));
            this.adjustWindowWidth();
            this.refresh();
        }
        adjustWindowWidth() {
            const actor = $gameActors.actor(actorId);
            let newWidth = 300;
            if (actor) {
                this.contents.fontSize = mpFontSize;
                const labelWidth = ImageManager.iconWidth * mpIconScale;
                const blockWidth = this.textWidth(mpBlockSymbol);
                const maxCount = Math.ceil(actor.mmp / mpBlockPerMP);
                newWidth = labelWidth + labelBlockGap + maxCount * blockWidth + (maxCount > 0 ? (maxCount - 1) * mpBlockSpacing : 0) + windowPaddingAddition;
            }
            if (this.width !== newWidth) {
                this.width = newWidth;
                this.createContents();
                this.contents.fontSize = mpFontSize;
            }
        }
        update() {
            super.update();
            const should = (mpDisplaySwitch === 0 || $gameSwitches.value(mpDisplaySwitch));
            if (should !== this.visible) {
                this.contents.clear();
                this.visible = should;
            }
            if (this.visible) {
                this.adjustWindowWidth();
                this.refresh();
            }
        }
        refresh() {
            this.contents.clear();
            this.contents.fontSize = mpFontSize;
            const actor = $gameActors.actor(actorId);
            if (!actor) return;
            const cur = actor.mp;
            const totalBlocks = Math.ceil(actor.mmp / mpBlockPerMP);
            const y = (this.contentsHeight() - (ImageManager.iconHeight * mpIconScale)) / 2;

            const iconW = ImageManager.iconWidth;
            const iconH = ImageManager.iconHeight;
            this.contents.paintOpacity = this.contentsOpacity;
            this.contents.blt(
                ImageManager.loadSystem("IconSet"),
                mpIconIndex % 16 * iconW,
                Math.floor(mpIconIndex / 16) * iconH,
                iconW,
                iconH,
                0,
                y,
                iconW * mpIconScale,
                iconH * mpIconScale
            );
            this.contents.paintOpacity = 255;

            const labelW = iconW * mpIconScale;
            this.contents.fontSize = mpFontSize;
            const bw = this.textWidth(mpBlockSymbol);

            const baseBlockY = y + (iconH * mpIconScale) / 2 - this.contents.fontSize / 2;

            for (let i = 0; i < totalBlocks; i++) {
                const x = labelW + labelBlockGap + i * (bw + mpBlockSpacing);
                const rem = cur - i * mpBlockPerMP;

                const commonBlockY = baseBlockY + blockYOffset;

                if (rem <= 0) {
                    this.changeTextColor("#000000");
                    const originalMpFontSize = this.contents.fontSize;
                    if (emptyMpBlockFontSize > 0) {
                        this.contents.fontSize = emptyMpBlockFontSize;
                    }
                    this.drawText(emptyMpBlockSymbol, x, commonBlockY + emptyMpBlockYOffset, bw, "left");
                    this.contents.fontSize = originalMpFontSize; // 元に戻す
                } else if (rem <= mpMidBlockThreshold) {
                    this.drawTextColor(mpMidBlockSymbol, x, commonBlockY, bw, "left", mpBlockColor);
                } else {
                    this.drawTextColor(mpBlockSymbol, x, commonBlockY, bw, "left", mpBlockColor);
                }
            }
        }
        drawTextColor(text, x, y, maxWidth, align, color) {
            const context = this.contents.context;
            context.save();
            context.fillStyle = color;
            context.font = `${this.contents.fontSize}px ${this.contents.fontFace}`;
            context.textAlign = align;
            context.fillText(text, x, y + this.contents.fontSize, maxWidth);
            context.restore();
        }
    }

    const _Scene_Map_create = Scene_Map.prototype.createAllWindows;
    Scene_Map.prototype.createAllWindows = function() {
        _Scene_Map_create.call(this);
        this._hpBlockWindow = new Window_HPBlockDisplay();
        this._mpBlockWindow = new Window_MPBlockDisplay();
        this.addChild(this._hpBlockWindow);
        this.addChild(this._mpBlockWindow);
    };
})();