/*:
 * @target MZ
 * @plugindesc 文字列を1文字ずつジャンプさせるアニメーションを表示します。（制御文字対応版）
 * @author 
 * @version 1.3.0
 * @help JumpingTextAnimation.js
 *
 * プラグインコマンドで指定した文字列を、1文字ずつジャンプさせながら
 * 表示します。
 * 制御文字 (例: \V[n], \C[n] など) にも対応しています。
 *
 * 【新機能: 文字装飾】
 * アウトライン（縁取り）の設定を追加しました。
 * 文字の太さや色を調整して、より見栄えのする表示が可能です。
 *
 * 利用規約:
 * 特にありません。自由にお使いください。
 *
 * 更新履歴:
 * 2024/05/21 Ver 1.2.1 - TypeError修正
 * 2025/05/24 Ver 1.2.2 - 制御文字対応を追加
 * 2025/09/30 Ver 1.3.0 - アウトライン(太さ/色)のパラメータを追加・反映
 *
 * @param defaultText
 * @text デフォルト表示文字列
 * @desc アニメーションさせるデフォルトの文字列です。
 * @type string
 * @default LEVEL UP!
 *
 * @param defaultX
 * @text デフォルトX座標
 * @desc 表示位置のX座標。(文字列中央)
 * @type number
 * @default Math.round(Graphics.boxWidth / 2)
 *
 * @param defaultY
 * @text デフォルトY座標
 * @desc 表示位置のY座標。(文字列中央)
 * @type number
 * @default Math.round(Graphics.boxHeight / 2)
 *
 * @param defaultFontSize
 * @text デフォルトフォントサイズ
 * @type number
 * @min 1
 * @default 32
 *
 * @param defaultFontColor
 * @text デフォルトフォント色
 * @desc フォントのメインカラー。
 * @type string
 * @default #FFFFFF
 *
 * @param defaultOutlineWidth
 * @text デフォルトアウトライン太さ
 * @desc 文字のアウトライン（縁取り）の太さ。0で非表示。
 * @type number
 * @min 0
 * @default 4
 *
 * @param defaultOutlineColor
 * @text デフォルトアウトライン色
 * @desc アウトラインの色。CSSの色指定 (例: rgba(0,0,0,0.8))
 * @type string
 * @default rgba(0, 0, 0, 0.8)
 *
 * @param defaultShadowColor
 * @text デフォルトシャドウ色 (未実装)
 * @desc 文字の影（シャドウ）の色。空欄で非表示。現状は非対応です。
 * @type string
 * @default
 *
 * @param defaultJumpHeight
 * @text デフォルトジャンプの高さ
 * @type number
 * @default 24
 *
 * @param defaultJumpDuration
 * @text デフォルトジャンプ時間
 * @type number
 * @min 1
 * @default 30
 *
 * @param defaultCharDelay
 * @text デフォルト文字間遅延
 * @type number
 * @min 0
 * @default 5
 *
 * @param defaultAutoRemove
 * @text デフォルト自動消去
 * @type boolean
 * @default true
 *
 * @param defaultRemoveDelay
 * @text デフォルト消去遅延
 * @type number
 * @min 0
 * @default 60
 *
 * @param defaultRemoveAnimation
 * @text デフォルト消去アニメーション
 * @type boolean
 * @default false
 *
 * @param defaultRemoveSlideHeight
 * @text 消去時のスライド高さ
 * @type number
 * @min 0
 * @default 32
 *
 * @param defaultRemoveSlideDuration
 * @text 消去時のアニメーション時間
 * @type number
 * @min 1
 * @default 45
 *
 * @param defaultRemoveCharInterval
 * @text 消去時の文字間隔
 * @type number
 * @min 0
 * @default 5
 *
 * @command showJumpingText
 * @text ジャンプ文字表示
 * @desc 指定した設定で文字列をジャンプさせながら表示します。
 *
 * @arg text
 * @text 表示文字列
 * @type string
 * @default
 *
 * @arg x
 * @text X座標
 * @type number
 * @default
 *
 * @arg y
 * @text Y座標
 * @type number
 * @default
 *
 * @arg fontSize
 * @text フォントサイズ
 * @type number
 * @min 1
 * @default
 *
 * @arg fontColor
 * @text フォント色
 * @type string
 * @default
 *
 * @arg outlineWidth
 * @text アウトライン太さ
 * @desc 文字のアウトライン（縁取り）の太さ。0で非表示。
 * @type number
 * @min 0
 * @default
 *
 * @arg outlineColor
 * @text アウトライン色
 * @desc アウトラインの色。
 * @type string
 * @default
 *
 * @arg shadowColor
 * @text シャドウ色 (未実装)
 * @desc 文字の影（シャドウ）の色。現状は非対応です。
 * @type string
 * @default
 *
 * @arg jumpHeight
 * @text ジャンプの高さ
 * @type number
 * @default
 *
 * @arg jumpDuration
 * @text ジャンプ時間
 * @type number
 * @min 1
 * @default
 *
 * @arg charDelay
 * @text 文字間遅延
 * @type number
 * @min 0
 * @default
 *
 * @arg autoRemove
 * @text 自動消去
 * @type boolean
 * @default
 *
 * @arg removeDelay
 * @text 消去までの遅延
 * @type number
 * @min 0
 * @default
 *
 * @arg removeAnimation
 * @text 消去アニメーション
 * @type boolean
 * @default
 *
 * @arg removeSlideHeight
 * @text 消去時のスライド高さ
 * @type number
 * @min 0
 * @default
 *
 * @arg removeSlideDuration
 * @text 消去時のアニメーション時間
 * @type number
 * @min 1
 * @default
 *
 * @arg removeCharInterval
 * @text 消去時の文字間隔
 * @type number
 * @min 0
 * @default
 */

(() => {
    'use strict';

    const pluginName = 'JumpingTextAnimation';
    const parameters = PluginManager.parameters(pluginName);

    const paramFromEval = (paramName, defaultValue) => {
        const value = parameters[paramName];
        if (value === undefined || value === "") return defaultValue;
        try {
            JSON.parse(value);
            return eval(value);
        } catch {
            try { return eval(value); } catch { return value; }
        }
    };
    const getStringParam = (param, defaultValue) =>
        param === undefined || param === "" ? defaultValue : String(param);
    const getNumberParam = (param, defaultValue) => {
        if (param === undefined || param === "") return defaultValue;
        const num = Number(param);
        return isNaN(num) ? defaultValue : num;
    };
    const getBoolParam = (param, defaultValue) =>
        param === undefined || param === "" ? defaultValue : String(param).toLowerCase() === 'true';

    const defaultText = getStringParam(parameters.defaultText, "LEVEL UP!");
    const defaultX = getNumberParam(paramFromEval('defaultX', null), Math.round(Graphics.boxWidth / 2));
    const defaultY = getNumberParam(paramFromEval('defaultY', null), Math.round(Graphics.boxHeight / 2));
    const defaultFontSize = getNumberParam(parameters.defaultFontSize, 32);
    const defaultFontColor = getStringParam(parameters.defaultFontColor, "#FFFFFF");
    // ↓↓↓ 追加したパラメーター ↓↓↓
    const defaultOutlineWidth = getNumberParam(parameters.defaultOutlineWidth, 4);
    const defaultOutlineColor = getStringParam(parameters.defaultOutlineColor, "rgba(0, 0, 0, 0.8)");
    const defaultShadowColor = getStringParam(parameters.defaultShadowColor, "");
    // ↑↑↑ 追加したパラメーター ↑↑↑
    const defaultJumpHeight = getNumberParam(parameters.defaultJumpHeight, 24);
    const defaultJumpDuration = getNumberParam(parameters.defaultJumpDuration, 30);
    const defaultCharDelay = getNumberParam(parameters.defaultCharDelay, 5);
    const defaultAutoRemove = getBoolParam(parameters.defaultAutoRemove, true);
    const defaultRemoveDelay = getNumberParam(parameters.defaultRemoveDelay, 60);
    const defaultRemoveAnimation = getBoolParam(parameters.defaultRemoveAnimation, false);
    const defaultRemoveSlideHeight = getNumberParam(parameters.defaultRemoveSlideHeight, 32);
    const defaultRemoveSlideDuration = getNumberParam(parameters.defaultRemoveSlideDuration, 45);
    const defaultRemoveCharInterval = getNumberParam(parameters.defaultRemoveCharInterval, 5);

    PluginManager.registerCommand(pluginName, 'showJumpingText', function(args) {
        let text = getStringParam(args.text, defaultText);
        const scene = SceneManager._scene;
        if (scene && scene._messageWindow) {
            text = scene._messageWindow.convertEscapeCharacters(text);
        }

        const x = getNumberParam(args.x, defaultX);
        const y = getNumberParam(args.y, defaultY);
        const fontSize = getNumberParam(args.fontSize, defaultFontSize);
        const fontColor = getStringParam(args.fontColor, defaultFontColor);
        // ↓↓↓ 追加した引数 ↓↓↓
        const outlineWidth = getNumberParam(args.outlineWidth, defaultOutlineWidth);
        const outlineColor = getStringParam(args.outlineColor, defaultOutlineColor);
        const shadowColor = getStringParam(args.shadowColor, defaultShadowColor);
        // ↑↑↑ 追加した引数 ↑↑↑
        const jumpHeight = getNumberParam(args.jumpHeight, defaultJumpHeight);
        const jumpDuration = getNumberParam(args.jumpDuration, defaultJumpDuration);
        const charDelay = getNumberParam(args.charDelay, defaultCharDelay);
        const autoRemove = getBoolParam(args.autoRemove, defaultAutoRemove);
        const removeDelay = getNumberParam(args.removeDelay, defaultRemoveDelay);
        const removeAnimation = getBoolParam(args.removeAnimation, defaultRemoveAnimation);
        const removeSlideHeight = getNumberParam(args.removeSlideHeight, defaultRemoveSlideHeight);
        const removeSlideDuration = getNumberParam(args.removeSlideDuration, defaultRemoveSlideDuration);
        const removeCharInterval = getNumberParam(args.removeCharInterval, defaultRemoveCharInterval);

        if (scene) {
            const parent = scene._windowLayer || scene._spriteset || scene;
            const sprite = new Sprite_JumpingText(
                text,
                x,
                y,
                fontSize,
                fontColor,
                // ↓↓↓ 追加した引数 ↓↓↓
                outlineWidth,
                outlineColor,
                shadowColor,
                // ↑↑↑ 追加した引数 ↑↑↑
                jumpHeight,
                jumpDuration,
                charDelay,
                autoRemove,
                removeDelay,
                removeAnimation,
                removeSlideHeight,
                removeSlideDuration,
                removeCharInterval
            );
            parent.addChild(sprite);
        }
    });

    function Sprite_JumpingText() { this.initialize(...arguments); }
    Sprite_JumpingText.prototype = Object.create(Sprite.prototype);
    Sprite_JumpingText.prototype.constructor = Sprite_JumpingText;

    Sprite_JumpingText.prototype.initialize = function(
        text,
        x,
        y,
        fontSize,
        fontColor,
        // ↓↓↓ 追加した引数 ↓↓↓
        outlineWidth,
        outlineColor,
        shadowColor,
        // ↑↑↑ 追加した引数 ↑↑↑
        jumpHeight,
        jumpDuration,
        charDelay,
        autoRemove,
        removeDelay,
        removeAnimation,
        removeSlideHeight,
        removeSlideDuration,
        removeCharInterval
    ) {
        Sprite.prototype.initialize.call(this);
        this._text = text;
        this.x = x;
        this.y = y;
        this._fontSize = fontSize;
        this._fontColor = fontColor;
        // ↓↓↓ プロパティとして保存 ↓↓↓
        this._outlineWidth = outlineWidth;
        this._outlineColor = outlineColor;
        this._shadowColor = shadowColor;
        // ↑↑↑ プロパティとして保存 ↑↑↑
        this._jumpHeight = jumpHeight;
        this._jumpDuration = Math.max(1, jumpDuration);
        this._charDelay = Math.max(0, charDelay);
        this._autoRemove = autoRemove;
        this._removeCounter = Math.max(0, removeDelay);
        this._removeAnimation = removeAnimation;
        this._removeSlideHeight = removeSlideHeight;
        this._removeSlideDuration = Math.max(1, removeSlideDuration);
        this._removeCharInterval = Math.max(0, removeCharInterval);

        this._charSprites = [];
        this._animationTimers = [];
        this._charJumpStates = [];
        this._charAnimFrame = [];
        this._charRemoveTimers = [];
        this._charRemoveStates = [];

        this._elapsedFrames = 0;
        this._allAnimationCompleted = false;
        this._removeAnimationStarted = false;

        this.anchor.x = 0.5;
        this.anchor.y = 0.5;

        this.createCharSprites();
    };

    Sprite_JumpingText.prototype.createCharSprites = function() {
        let totalWidth = 0;
        const tempBmp = new Bitmap(1, 1);
        tempBmp.fontSize = this._fontSize;
        for (const c of this._text) {
            totalWidth += tempBmp.measureTextWidth(c);
        }
        tempBmp.destroy();

        this.pivot.x = totalWidth / 2;
        this.pivot.y = this._fontSize / 2;

        let offsetX = 0;
        for (let i = 0; i < this._text.length; i++) {
            const char = this._text[i];
            const metricsBmp = new Bitmap(1, 1);
            metricsBmp.fontSize = this._fontSize;
            const w = metricsBmp.measureTextWidth(char) || this._fontSize;
            metricsBmp.destroy();

            // ↓↓↓ アウトラインの太さを考慮して描画領域を調整 ↓↓↓
            const outlineSize = this._outlineWidth;
            // 余裕を持たせるためのパディング + アウトライン分の太さ * 2
            const padding = Math.ceil(this._fontSize * 0.2) + outlineSize * 2;
            const bmpW = Math.max(w, this._fontSize) + padding;
            const bmpH = this._fontSize + padding;
            const offX = padding / 2; // drawTextの開始位置
            const offY = padding / 2;
            // ↑↑↑ アウトラインの太さを考慮して描画領域を調整 ↑↑↑

            const sp = new Sprite();
            sp.bitmap = new Bitmap(bmpW, bmpH);
            sp.bitmap.fontSize = this._fontSize;
            sp.bitmap.textColor = this._fontColor;
            
            // ↓↓↓ アウトラインの設定を反映 ↓↓↓
            if (this._outlineWidth > 0) {
                sp.bitmap.outlineWidth = this._outlineWidth;
                sp.bitmap.outlineColor = this._outlineColor;
            } else {
                // アウトラインなしの場合はデフォルト値に戻しておく
                sp.bitmap.outlineWidth = 0;
                sp.bitmap.outlineColor = 'rgba(0, 0, 0, 0)'; 
            }
            // ↑↑↑ アウトラインの設定を反映 ↑↑↑
            
            sp.bitmap.drawText(char, offX, offY, bmpW - offX * 2, this._fontSize, 'left');
            sp.anchor.x = 0;
            sp.anchor.y = 0.5;
            sp.x = offsetX - totalWidth / 2;
            sp.y = 0;

            this.addChild(sp);
            this._charSprites.push(sp);
            this._animationTimers.push(i * this._charDelay);
            this._charJumpStates.push(0);
            this._charAnimFrame.push(0);
            this._charRemoveTimers.push(i * this._removeCharInterval);
            this._charRemoveStates.push(0);

            offsetX += w;
        }
    };

    Sprite_JumpingText.prototype.update = function() {
        Sprite.prototype.update.call(this);
        if (!this._allAnimationCompleted) {
            this.updateJumpingAnimation();
        } else if (this._autoRemove) {
            this.updateRemoveAnimation();
        }
    };

    Sprite_JumpingText.prototype.updateJumpingAnimation = function() {
        this._elapsedFrames++;
        let allDone = true;
        const half = this._jumpDuration / 2;

        for (let i = 0; i < this._charSprites.length; i++) {
            if (this._charJumpStates[i] === 3) continue;
            allDone = false;
            if (this._charJumpStates[i] === 0 && this._elapsedFrames >= this._animationTimers[i]) {
                this._charJumpStates[i] = 1;
                this._charAnimFrame[i] = 0;
            }
            if (this._charJumpStates[i] === 1) {
                this._charAnimFrame[i]++;
                let progress = this._charAnimFrame[i] / half;
                if (progress > 1) progress = 1;
                this._charSprites[i].y = -this._jumpHeight * Math.sin(progress * (Math.PI / 2));
                if (this._charAnimFrame[i] >= half) {
                    this._charJumpStates[i] = 2;
                    this._charAnimFrame[i] = 0;
                }
            } else if (this._charJumpStates[i] === 2) {
                this._charAnimFrame[i]++;
                let progress = this._charAnimFrame[i] / half;
                if (progress > 1) progress = 1;
                this._charSprites[i].y = -this._jumpHeight * Math.cos(progress * (Math.PI / 2));
                if (this._charAnimFrame[i] >= half) {
                    this._charJumpStates[i] = 3;
                }
            }
        }

        if (allDone || this._charSprites.length === 0) {
            this._allAnimationCompleted = true;
        }
    };

    Sprite_JumpingText.prototype.updateRemoveAnimation = function() {
        if (!this._removeAnimationStarted) {
            this._removeCounter--;
            if (this._removeCounter <= 0) {
                this._removeAnimationStarted = true;
                // 消去アニメーション用フレームをリセットして滑らかに開始
                this._elapsedFrames = 0;
            }
            return;
        }
        if (this._removeAnimation) {
            this._elapsedFrames++;
            let allRemoved = true;
            for (let i = 0; i < this._charSprites.length; i++) {
                const sp = this._charSprites[i];
                if (!sp) continue;
                if (this._charRemoveStates[i] === 2) continue;
                allRemoved = false;
                if (this._charRemoveStates[i] === 0 && this._elapsedFrames >= this._charRemoveTimers[i]) {
                    this._charRemoveStates[i] = 1;
                }
                if (this._charRemoveStates[i] === 1) {
                    const progress = Math.min(1, (this._elapsedFrames - this._charRemoveTimers[i]) / this._removeSlideDuration);
                    sp.y = -this._removeSlideHeight * progress;
                    sp.opacity = 255 * (1 - progress);
                    if (progress >= 1) {
                        this._charRemoveStates[i] = 2;
                        sp.parent.removeChild(sp);
                        sp.bitmap.destroy();
                        sp.destroy();
                        this._charSprites[i] = null;
                    }
                }
            }
            if (allRemoved) {
                this.parent.removeChild(this);
                this.destroy();
            }
        } else if (this._removeCounter < 0) {
            this.parent.removeChild(this);
            this.destroy();
        }
    };

    Sprite_JumpingText.prototype.destroy = function(options) {
        this._charSprites.forEach(sp => {
            if (sp && sp.bitmap) sp.bitmap.destroy();
            if (sp) sp.destroy();
        });
        this._charSprites = [];
        Sprite.prototype.destroy.call(this, options);
    };

    // 初回フォント反映用
    const _Scene_Boot_create = Scene_Boot.prototype.create;
    Scene_Boot.prototype.create = function() {
        _Scene_Boot_create.call(this);
        if (typeof FontManager !== 'undefined' && FontManager.mainFontSize) {
            const bmp = new Bitmap(1, 1);
            bmp.fontSize = FontManager.mainFontSize();
            bmp.fontFace = FontManager.mainFontName();
            // アウトライン設定をここで一度行うことで、フォントが確実にロードされる
            bmp.outlineWidth = defaultOutlineWidth;
            bmp.destroy();
        }
    };
})();