/*:
 * @target MZ
 * @plugindesc Qキーでスクショ背景のポーズを開閉。開始後クールダウンで誤反応防止。スイッチで機能のON/OFFが可能。
 * @author 
 *
 * @help PauseWithQ_ToggleCooldown.js
 *
 * Qキー（pageup）でポーズシーンを開き、もう一度Qで閉じられます。
 * 開始直後は一定フレーム入力を無視して、1回で開いて閉じる事故を防ぎます。
 *
 * 【機能ON/OFFについて】
 * 「機能ON/OFFスイッチID」に1以上のスイッチ番号を設定すると、
 * そのスイッチがONの時のみポーズ機能が有効になります。
 * スイッチがOFFの時はQキーを押してもポーズ画面は開きません。
 *
 * @param CooldownFrames
 * @text クールダウンフレーム数
 * @type number
 * @min 0
 * @default 12
 * @desc ポーズ開始直後に入力を無視するフレーム数（60fps基準）。例: 12 ≒ 0.2秒
 *
 * @param ControlSwitchId
 * @text 機能ON/OFFスイッチID
 * @type switch
 * @default 0
 * @desc 指定したスイッチがONの時のみポーズ機能が有効になります。0で常に有効。
 */

(() => {
    const parameters = PluginManager.parameters("PauseWithQ_ToggleCooldown");
    const COOLDOWN = Number(parameters["CooldownFrames"] || 12);
    const CONTROL_SWITCH_ID = Number(parameters["ControlSwitchId"] || 0);

    const _Scene_Map_update = Scene_Map.prototype.update;
    Scene_Map.prototype.update = function () {
        _Scene_Map_update.call(this);

        if (Input.isTriggered("pageup")) {
            if (CONTROL_SWITCH_ID > 0 && !$gameSwitches.value(CONTROL_SWITCH_ID)) {
                return; // スイッチOFFなら無効
            }
            SceneManager.snapForBackground();
            SceneManager.push(Scene_Pause);
        }
    };



  // --- ポーズシーン定義 (変更なし) ---
  function Scene_Pause() {
    this.initialize(...arguments);
  }

  Scene_Pause.prototype = Object.create(Scene_Base.prototype);
  Scene_Pause.prototype.constructor = Scene_Pause;

  Scene_Pause.prototype.initialize = function() {
    Scene_Base.prototype.initialize.call(this);
    this._cooldown = COOLDOWN; // 開始時クールダウン
  };

  Scene_Pause.prototype.create = function() {
    Scene_Base.prototype.create.call(this);
    this.createBackground();
    this.createMessage();
  };

  Scene_Pause.prototype.start = function() {
    Scene_Base.prototype.start.call(this);
    Input.clear();
    TouchInput.clear();
    this._cooldown = COOLDOWN;
  };

  Scene_Pause.prototype.createBackground = function() {
    const bg = SceneManager.backgroundBitmap();
    this._backgroundSprite = new Sprite(bg);
    this._backgroundSprite.setColorTone([-64, -64, -64, 0]); // 少し暗く
    this.addChild(this._backgroundSprite);
  };

  Scene_Pause.prototype.createMessage = function() {
    const text = new Sprite(new Bitmap(Graphics.width, Graphics.height));
    text.bitmap.fontSize = 48;
    text.bitmap.textColor = "#FFFFFF";
    text.bitmap.outlineColor = "rgba(0,0,0,0.6)";
    text.bitmap.outlineWidth = 6;
    text.bitmap.drawText("PAUSE", 0, Graphics.height/2 - 24, Graphics.width, 48, "center");
    this.addChild(text);
  };

  // 入力処理：クールダウン中は完全無視。Q・Esc・右クリックで閉じる。
  Scene_Pause.prototype.update = function() {
    Scene_Base.prototype.update.call(this);
    if (this._cooldown > 0) {
      this._cooldown--;
      return; // 反応しない期間
    }
    if (Input.isTriggered("pageup") || Input.isTriggered("cancel") || TouchInput.isCancelled()) {
      SceneManager.pop();
    }
  };
})();