/*:
 * @target MZ
 * @plugindesc タイトルのニューゲームに確認ダイアログを挟む（中央配置・縦並び・初期選択は「いいえ」・描画安定化）
 * @author 
 */

(() => {
    // =========================
    // 設定（ここを変えれば位置調整できる）
    // =========================
    const SIDE_MARGIN = 20;       // 画面左右に最低確保する余白(px)
    const GLOBAL_OFFSET_X = -150;    // 中央計算後に一括で横にずらす（負で左、正で右）
    // =========================

    // タイトルのコマンドウィンドウの newGame ハンドラを差し替え
    const _Scene_Title_createCommandWindow = Scene_Title.prototype.createCommandWindow;
    Scene_Title.prototype.createCommandWindow = function() {
        _Scene_Title_createCommandWindow.call(this);
        if (this._commandWindow) {
            this._commandWindow.setHandler('newGame', () => {
                SceneManager.push(Scene_NGConfirm);
            });
        }
    };

    // 確認用の選択ウィンドウ（縦並び・中央揃え・描画安定化）
    class Window_NGConfirm extends Window_Command {
        constructor(rect) {
            super(rect);
        }
        makeCommandList() {
            // enabled を true にしておく
            this.addCommand('はい', 'yes', true);
            this.addCommand('いいえ', 'no', true);
        }
        maxCols() { return 1; } // 縦並び
        numVisibleRows() { return this.maxItems(); }
        itemTextAlign() { return 'center'; }

        // 明示的な描画で安定させる
        drawItem(index) {
            const rect = this.itemLineRect(index);
            const name = this.commandName(index);
            this.resetTextColor();
            this.changePaintOpacity(this.isCommandEnabled(index));
            // 横余白を8px入れて枠と干渉しないようにする
            this.drawText(name, rect.x + 8, rect.y, rect.width - 16, this.itemTextAlign());
        }
    }

    // 確認シーン
    class Scene_NGConfirm extends Scene_Base {
        constructor() { super(); }

        create() {
            super.create();
            this.createWindowLayer();
            this._createMessageAndChoice();
        }

        _createMessageAndChoice() {
            const lines = [
                "ニューゲームを開始しますか？",
                "（現在のオートセーブデータが上書きされます）"
            ];

            const pad = 18;
            const textLineHeight = Window_Base.prototype.lineHeight.call(this);
            const totalTextHeight = textLineHeight * lines.length;

            // ============ メッセージウィンドウ =============
            // 最大幅は画面幅と SIDE_MARGIN に基づく
            const maxMsgW = Math.min(800, Graphics.width - SIDE_MARGIN * 2);
            let msgW = maxMsgW;
            let msgH = totalTextHeight + pad * 2 + 12;

            // 中央配置起点（ここでGLOBAL_OFFSET_Xを反映）
            let msgX = Math.floor((Graphics.width - msgW) / 2) + GLOBAL_OFFSET_X;
            let msgY = Math.floor((Graphics.height - msgH) / 2 - 40);

            // 右端・左端の安全チェック（SIDE_MARGIN確保）
            if (msgX + msgW > Graphics.width - SIDE_MARGIN) {
                msgX = Graphics.width - msgW - SIDE_MARGIN;
            }
            if (msgX < SIDE_MARGIN) {
                msgX = SIDE_MARGIN;
            }

            this._msgWindow = new Window_Base(new Rectangle(msgX, msgY, msgW, msgH));
            this.addWindow(this._msgWindow);

            // メッセージ描画（中央揃え）
            this._msgWindow.contents.clear();
            const innerW = this._msgWindow.contents.width - pad * 2;
            let curY = Math.floor((this._msgWindow.contents.height - totalTextHeight) / 2);
            for (let i = 0; i < lines.length; i++) {
                this._msgWindow.drawText(lines[i], pad, curY + i * textLineHeight, innerW, 'center');
            }

            // ============ 選択ウィンドウ（メッセージ幅に対して中央） ============
            let choiceW = Math.min(360, Math.floor(msgW * 0.8)); // メッセージ幅の80%以内
            // safety：もし choiceW が文字幅に足りないと感じたらここを増やして
            const tmpWindowForHeight = new Window_NGConfirm(new Rectangle(0,0,choiceW,1));
            const choiceH = tmpWindowForHeight.fittingHeight(tmpWindowForHeight.numVisibleRows());

            let choiceX = msgX + Math.floor((msgW - choiceW) / 2);
            let choiceY = this._msgWindow.y + this._msgWindow.height + 16;

            // 画面端安全チェック
            if (choiceX + choiceW > Graphics.width - SIDE_MARGIN) {
                choiceX = Graphics.width - choiceW - SIDE_MARGIN;
            }
            if (choiceX < SIDE_MARGIN) {
                choiceX = SIDE_MARGIN;
            }

            // 生成・ハンドラ登録・初期選択
            this._choiceWindow = new Window_NGConfirm(new Rectangle(choiceX, choiceY, choiceW, 1));
            this._choiceWindow.setHandler('yes', this.onYes.bind(this));
            this._choiceWindow.setHandler('no', this.onNo.bind(this));
            this._choiceWindow.select(1); // 初期選択「いいえ」
            this.addWindow(this._choiceWindow);

            // 実際の高さにして contents を再作成して refresh
            this._choiceWindow.move(choiceX, choiceY, choiceW, choiceH);
            this._choiceWindow.createContents();
            this._choiceWindow.refresh();

            // フォーカスを選択ウィンドウへ
            this._choiceWindow.activate();

            // デバッグ用ログ（不要ならコメントアウトして）
            // console.log('NGConfirm pos:', { msgX, msgW, choiceX, choiceW, choiceH, gw: Graphics.width, gh: Graphics.height });
        }

        onYes() {
            DataManager.setupNewGame();
            SceneManager.goto(Scene_Map);
        }

        onNo() {
            SceneManager.goto(Scene_Title);
        }

        update() {
            super.update();
            if (Input.isTriggered('cancel')) {
                this.onNo();
            }
        }
    }

    window.Scene_NGConfirm = Scene_NGConfirm;
})();
