/*:
 * @target MZ
 * @plugindesc 座標と設定を指定して、装飾付きの文字を画面に固定表示します（セーブ/ロード対応版） 
 * @author 
 * @version 1.2.0
 * @help
 * プラグインコマンドで指定した文字列を、指定座標に装飾付きで表示します。
 * 表示状態は $gameSystem に保存されるため、セーブ/ロードやシーン切替後も復元されます。
 *
 * @param defaultFontSize
 * @text デフォルトフォントサイズ
 * @type number
 * @default 36
 *
 * @param defaultFontColor
 * @text デフォルトフォント色
 * @type string
 * @default #FFFF00
 *
 * @param defaultOutlineWidth
 * @text デフォルトアウトライン太さ
 * @type number
 * @default 5
 *
 * @param defaultOutlineColor
 * @text デフォルトアウトライン色
 * @type string
 * @default rgba(0,0,0,1.0)
 *
 * @command showFixedText
 * @text 固定文字表示
 * @arg id
 * @type string
 * @default score
 * @arg text
 * @type string
 * @default 123456
 * @arg x
 * @type number
 * @default 100
 * @arg y
 * @type number
 * @default 50
 * @arg align
 * @type select
 * @option left
 * @option center
 * @option right
 * @default left
 * @arg fontSize
 * @type number
 * @default
 * @arg fontColor
 * @type string
 * @default
 * @arg outlineWidth
 * @type number
 * @default
 * @arg outlineColor
 * @type string
 * @default
 *
 * @command updateFixedText
 * @text 固定文字更新
 * @arg id
 * @type string
 * @default score
 * @arg text
 * @type string
 * @default
 *
 * @command removeFixedText
 * @text 固定文字消去
 * @arg id
 * @type string
 * @default score
 */

(() => {
    'use strict';
    const pluginName = 'SimpleTextDisplay';
    const params = PluginManager.parameters(pluginName);

    const defaultFontSize = Number(params.defaultFontSize || 36);
    const defaultFontColor = String(params.defaultFontColor || "#FFFF00");
    const defaultOutlineWidth = Number(params.defaultOutlineWidth || 5);
    const defaultOutlineColor = String(params.defaultOutlineColor || "rgba(0,0,0,1.0)");

    // --- 永続データ ---
    const fixedTextData = () => {
        if (!$gameSystem._fixedTextData) $gameSystem._fixedTextData = {};
        return $gameSystem._fixedTextData;
    };

    // --- スプライト定義 ---
    function Sprite_FixedText() { this.initialize(...arguments); }
    Sprite_FixedText.prototype = Object.create(Sprite.prototype);
    Sprite_FixedText.prototype.constructor = Sprite_FixedText;

    Sprite_FixedText.prototype.initialize = function(id, text, x, y, align, fontSize, fontColor, outlineWidth, outlineColor) {
        Sprite.prototype.initialize.call(this);
        this._id = id;
        this._x = x; this._y = y;
        this._align = align;
        this._fontSize = fontSize;
        this._fontColor = fontColor;
        this._outlineWidth = outlineWidth;
        this._outlineColor = outlineColor;
        this.updateText(text);
    };

    Sprite_FixedText.prototype.updateText = function(newText) {
        const msgWin = SceneManager._scene && SceneManager._scene._messageWindow;
        const text = msgWin ? msgWin.convertEscapeCharacters(newText) : newText;
        if (this._currentText === text) return;
        this._currentText = text;

        if (this.bitmap) this.bitmap.destroy();
        if (!text) return;

        const temp = new Bitmap(1,1);
        temp.fontSize = this._fontSize;
        const textWidth = temp.measureTextWidth(text);
        temp.destroy();

        const pad = this._outlineWidth * 2 + 4;
        const bmpW = Math.ceil(textWidth + pad);
        const bmpH = Math.ceil(this._fontSize + pad);

        this.bitmap = new Bitmap(bmpW, bmpH);
        this.bitmap.fontSize = this._fontSize;
        this.bitmap.textColor = this._fontColor;
        this.bitmap.outlineWidth = this._outlineWidth;
        this.bitmap.outlineColor = this._outlineColor;
        this.bitmap.drawText(text, 0, 0, bmpW, bmpH, this._align);

        this.anchor.x = (this._align === 'center') ? 0.5 : (this._align === 'right' ? 1.0 : 0);
        this.anchor.y = 0.5;
        this.x = this._x; this.y = this._y;
    };

    // --- Spriteset_Map 拡張 ---
    const _Spriteset_Map_initialize = Spriteset_Map.prototype.initialize;
    Spriteset_Map.prototype.initialize = function() {
        _Spriteset_Map_initialize.call(this);
        this._fixedTextSprites = {};
        this._fixedTextLayer = new Sprite();
        this.addChild(this._fixedTextLayer);
        this.restoreFixedTexts();
    };

    Spriteset_Map.prototype.restoreFixedTexts = function() {
        const data = fixedTextData();
        for (const id in data) {
            const d = data[id];
            const sprite = new Sprite_FixedText(id, d.text, d.x, d.y, d.align, d.fontSize, d.fontColor, d.outlineWidth, d.outlineColor);
            this._fixedTextLayer.addChild(sprite);
            this._fixedTextSprites[id] = sprite;
        }
    };

    // --- コマンド ---
    PluginManager.registerCommand(pluginName, 'showFixedText', function(args) {
        const id = args.id || 'score';
        const text = args.text || '';
        const x = Number(args.x || 100);
        const y = Number(args.y || 50);
        const align = args.align || 'left';
        const fontSize = Number(args.fontSize || defaultFontSize);
        const fontColor = args.fontColor || defaultFontColor;
        const outlineWidth = Number(args.outlineWidth || defaultOutlineWidth);
        const outlineColor = args.outlineColor || defaultOutlineColor;

        fixedTextData()[id] = { text, x, y, align, fontSize, fontColor, outlineWidth, outlineColor };

        const spriteset = SceneManager._scene._spriteset;
        if (spriteset && spriteset._fixedTextLayer) {
            if (spriteset._fixedTextSprites[id]) {
                const s = spriteset._fixedTextSprites[id];
                Object.assign(s, { _x:x, _y:y, _align:align, _fontSize:fontSize, _fontColor:fontColor, _outlineWidth:outlineWidth, _outlineColor:outlineColor });
                s.updateText(text);
            } else {
                const s = new Sprite_FixedText(id, text, x, y, align, fontSize, fontColor, outlineWidth, outlineColor);
                spriteset._fixedTextLayer.addChild(s);
                spriteset._fixedTextSprites[id] = s;
            }
        }
    });

    PluginManager.registerCommand(pluginName, 'updateFixedText', function(args) {
        const id = args.id || 'score';
        const text = args.text || '';
        if (fixedTextData()[id]) fixedTextData()[id].text = text;
        const spriteset = SceneManager._scene._spriteset;
        if (spriteset && spriteset._fixedTextSprites[id]) spriteset._fixedTextSprites[id].updateText(text);
    });

    PluginManager.registerCommand(pluginName, 'removeFixedText', function(args) {
        const id = args.id || 'score';
        delete fixedTextData()[id];
        const spriteset = SceneManager._scene._spriteset;
        if (spriteset && spriteset._fixedTextSprites[id]) {
            const s = spriteset._fixedTextSprites[id];
            if (s.parent) s.parent.removeChild(s);
            s.destroy();
            delete spriteset._fixedTextSprites[id];
        }
    });
})();
