/*:
 * @target MZ
 * @plugindesc TIME表示プラグイン（開始/停止/一時停止/再開）表示位置・サイズ調整可（1時間以上は時:分:秒表示対応）/ 表示状態を保持する修正版 + デバッグ用コマンド追加
 * @author 
 *
 * @param PosX
 * @text 表示X位置
 * @type number
 * @default 10
 *
 * @param PosY
 * @text 表示Y位置
 * @type number
 * @default 10
 *
 * @param FontSize
 * @text フォントサイズ
 * @type number
 * @default 24
 *
 * @param FontColor
 * @text フォント色
 * @type string
 * @default #FFFFFF
 *
 * @command StartTIME
 * @text TIME開始
 * @desc TIMEカウントを開始します
 *
 * @command StopTIME
 * @text TIME停止
 * @desc TIMEカウントを停止しリセットします
 *
 * @command PauseTIME
 * @text TIME一時停止
 * @desc タイマーを停止します（秒数は保持）
 *
 * @command ResumeTIME
 * @text TIME再開
 * @desc 停止していたタイマーを再開します
 *
 * @command AddTIME
 * @text TIME加算（デバッグ用）
 * @desc 指定した秒数を強制的に加算します（デバッグ用）
 * @arg seconds
 * @text 加算する秒数
 * @type number
 * @default 60
 *
 * @command SetTIME
 * @text TIME設定（デバッグ用）
 * @desc 指定した秒数に強制的に設定します（デバッグ用）
 * @arg seconds
 * @text 設定する秒数
 * @type number
 * @default 0
 */

(() => {
    const pluginName = "TIME_Display";

    const params = PluginManager.parameters(pluginName);
    const posX = Number(params['PosX'] || 10);
    const posY = Number(params['PosY'] || 10);
    const fontSize = Number(params['FontSize'] || 24);
    const fontColor = String(params['FontColor'] || '#FFFFFF');

    const FRAME_PER_SECOND = 60;
    let frameCount = 0;
    let totalSeconds = 0;
    let timeActive = false;
    let displayOn = false; // 表示状態を保持
    let _timeSprite = null;

    Number.prototype.padZero = function(len) {
        return this.toString().padStart(len, '0');
    };

    // --- Scene_Map にスプライトを追加 ---
    const _Scene_Map_createDisplayObjects = Scene_Map.prototype.createDisplayObjects;
    Scene_Map.prototype.createDisplayObjects = function() {
        _Scene_Map_createDisplayObjects.call(this);

        // displayOn が true のときだけスプライトを作成
        if (displayOn) {
            _timeSprite = new Sprite(new Bitmap(240, 32));
            _timeSprite.x = posX;
            _timeSprite.y = posY;
            _timeSprite.visible = true;
            this.addChild(_timeSprite);
        } else {
            _timeSprite = null;
        }
    };

    // --- 更新処理 ---
    const _Scene_Map_update = Scene_Map.prototype.update;
    Scene_Map.prototype.update = function() {
        _Scene_Map_update.call(this);

        // カウント処理
        if (timeActive) {
            frameCount++;
            if (frameCount >= FRAME_PER_SECOND) {
                frameCount = 0;
                totalSeconds++;
                $gameVariables.setValue(500, totalSeconds);
            }
        }

        // 描画処理
        if (_timeSprite && _timeSprite.bitmap) {
            if (displayOn) {
                const bmp = _timeSprite.bitmap;
                bmp.clear();
                bmp.fontFace = 'GameFont';
                bmp.fontSize = fontSize;
                bmp.textColor = fontColor;

                const hour = Math.floor(totalSeconds / 3600);
                let timeText = "";
                if (hour > 0) {
                    const min = Math.floor((totalSeconds % 3600) / 60).padZero(2);
                    const sec = (totalSeconds % 60).padZero(2);
                    timeText = `${hour}:${min}:${sec}`;
                } else {
                    const min = Math.floor(totalSeconds / 60).padZero(2);
                    const sec = (totalSeconds % 60).padZero(2);
                    timeText = `${min}:${sec}`;
                }

                bmp.drawText(`TIME: ${timeText}`, 0, 0, bmp.width, bmp.height, 'left');
                _timeSprite.visible = true;
            } else {
                _timeSprite.bitmap.clear();
                _timeSprite.visible = false;
            }
        }
    };

    // --- プラグインコマンド ---
    PluginManager.registerCommand(pluginName, "StartTIME", args => {
        timeActive = true;
        displayOn = true;
        if (!_timeSprite) {
            _timeSprite = new Sprite(new Bitmap(240, 32));
            _timeSprite.x = posX;
            _timeSprite.y = posY;
            SceneManager._scene.addChild(_timeSprite);
        }
        _timeSprite.visible = true;
    });

    PluginManager.registerCommand(pluginName, "StopTIME", args => {
        timeActive = false;
        frameCount = 0;
        totalSeconds = 0;
        displayOn = false;
        if (_timeSprite) {
            _timeSprite.visible = false;
            _timeSprite.bitmap.clear();
        }
        $gameVariables.setValue(500, 0);
    });

    PluginManager.registerCommand(pluginName, "PauseTIME", args => {
        timeActive = false; // 秒数は保持
    });

    PluginManager.registerCommand(pluginName, "ResumeTIME", args => {
        timeActive = true;
        displayOn = true;
        if (!_timeSprite) {
            _timeSprite = new Sprite(new Bitmap(240, 32));
            _timeSprite.x = posX;
            _timeSprite.y = posY;
            SceneManager._scene.addChild(_timeSprite);
        }
        _timeSprite.visible = true;
    });

    // --- デバッグ用: 時間を進める（AddTIME） ---
    PluginManager.registerCommand(pluginName, "AddTIME", args => {
        const add = Number(args.seconds) || 0;
        totalSeconds += add;
        // 変数500に反映
        $gameVariables.setValue(500, totalSeconds);
    });

    // --- デバッグ用: 時間を設定する（SetTIME） ---
    PluginManager.registerCommand(pluginName, "SetTIME", args => {
        const val = Number(args.seconds) || 0;
        totalSeconds = val;
        // 変数500に反映
        $gameVariables.setValue(500, totalSeconds);
    });

})();
