/*:
 * @target MZ
 * @plugindesc ベストタイム記録＆表示（ステージ別）
 * @author 
 *
 * @param PosX
 * @text 表示X位置
 * @type number
 * @default 10
 *
 * @param PosY
 * @text 表示Y位置
 * @type number
 * @default 50
 *
 * @param FontSize
 * @text フォントサイズ
 * @type number
 * @default 24
 *
 * @param FontColor
 * @text フォント色
 * @type string
 * @default #00FF00
 *
 * @command SaveRecord
 * @text ベスト記録を保存
 * @desc 現在のタイムを指定ステージのベストタイムとして保存
 * 
 * @arg stage
 * @text ステージ番号
 * @type number
 * @default 1
 *
 * @command ShowRecord
 * @text ベスト記録を表示
 * @desc 指定ステージのベストタイムを表示
 * 
 * @arg stage
 * @text ステージ番号
 * @type number
 * @default 1
 *
 * @command HideRecord
 * @text 表示を消す
 * @desc ベスト記録の表示をオフにする
 */

(() => {
    const pluginName = "TIME_Record_Stage";

    const params = PluginManager.parameters(pluginName);
    const posX = Number(params['PosX'] || 10);
    const posY = Number(params['PosY'] || 50);
    const fontSize = Number(params['FontSize'] || 24);
    const fontColor = String(params['FontColor'] || '#00FF00');

    let showRecord = false;
    let currentStage = 1; // 今表示するステージ
    let _recordSprite = null;

    // $gameSystemにステージごとのベストタイム保存
    Game_System.prototype.bestTimes = function() {
        if (!this._bestTimes) this._bestTimes = {};
        return this._bestTimes;
    };

    Game_System.prototype.setBestTime = function(stage, time) {
    const bests = this.bestTimes();
     if (!bests[stage] || time > bests[stage]) { // ★大きい方を保存
        bests[stage] = time;
    }

    };

    Game_System.prototype.getBestTime = function(stage) {
        const bests = this.bestTimes();
        return bests[stage] || 0;
    };

    // ★プラグインコマンド★
    PluginManager.registerCommand(pluginName, "SaveRecord", args => {
        const stage = Number(args.stage || 1);
        const current = $gameVariables.value(500); 
        if (current > 0) {
            $gameSystem.setBestTime(stage, current);
        }
    });

    PluginManager.registerCommand(pluginName, "ShowRecord", args => {
        currentStage = Number(args.stage || 1);
        showRecord = true;
    });

    PluginManager.registerCommand(pluginName, "HideRecord", args => {
        showRecord = false;
        if (_recordSprite) {
            _recordSprite.visible = false;
            _recordSprite.bitmap.clear();
        }
    });

    // マップに表示用スプライト追加
    const _Scene_Map_createDisplayObjects = Scene_Map.prototype.createDisplayObjects;
    Scene_Map.prototype.createDisplayObjects = function() {
        _Scene_Map_createDisplayObjects.call(this);
        _recordSprite = new Sprite(new Bitmap(360, 32));
        _recordSprite.x = posX;
        _recordSprite.y = posY;
        this.addChild(_recordSprite);
    };

    // 更新処理（表示を時:分:秒対応に変更）
    const _Scene_Map_update = Scene_Map.prototype.update;
    Scene_Map.prototype.update = function() {
        _Scene_Map_update.call(this);

        if (_recordSprite && _recordSprite.bitmap) {
            if (showRecord) {
                const best = $gameSystem.getBestTime(currentStage);
                if (best > 0) {
                    // ここを時:分:秒対応に変更
                    const hour = Math.floor(best / 3600);
                    let timeText = "";
                    if (hour > 0) {
                        const min = Math.floor((best % 3600) / 60).padZero(2);
                        const sec = (best % 60).padZero(2);
                        timeText = `${hour}:${min}:${sec}`;
                    } else {
                        const min = Math.floor(best / 60).padZero(2);
                        const sec = (best % 60).padZero(2);
                        timeText = `${min}:${sec}`;
                    }

                    const bmp = _recordSprite.bitmap;
                    bmp.clear();
                    bmp.fontFace = 'GameFont';
                    bmp.fontSize = fontSize;
                    bmp.textColor = fontColor;
                    bmp.drawText(`STAGE ${currentStage} BEST: ${timeText}`, 
                                 0, 0, bmp.width, bmp.height, 'left');
                    _recordSprite.visible = true;
                } else {
                    _recordSprite.visible = false;
                }
            } else {
                _recordSprite.visible = false;
            }
        }
    };

    // 便利関数
    Number.prototype.padZero = function(len) {
        return this.toString().padStart(len, '0');
    };
})();
