/*:
 * @target MZ
 * @plugindesc Trigger self switch on contact between events with specific tags
 * @author 
 *
 * @param TriggerTag
 * @text TriggerTag
 * @type string
 * @desc Event tag that triggers the contact effect
 * @default trap
 *
 * @param TargetTag
 * @text TargetTag
 * @type string
 * @desc Event tag that will have its self switch changed
 * @default enemy
 *
 * @param ExcludeTag
 * @text 除外タグ
 * @type string
 * @desc TargetTagが付いていても、このタグが付いていたらスイッチを切り替えません。空欄の場合は機能しません。
 * @default invincible
 * * @param SelfSwitch
 * @text SelfSwitch
 * @type select
 * @option A
 * @option B
 * @option C
 * @option D
 * @default A
 *
 */


(() => {
    const pluginName = "TrapSwitch";
    const params = PluginManager.parameters(pluginName);

    // パラメータを読み込みます
    const ssToSet = String(params['SelfSwitch'] || 'A');
    const trapTag = String(params['TriggerTag'] || 'trap');
    const enemyTag = String(params['TargetTag'] || 'enemy');
    // 除外タグを読み込みます
    const excludeTag = String(params['ExcludeTag'] || '').trim(); // ★trim()で空白を除去し、空欄かをチェックしやすくします

    // ノートタグを持っているかチェックする関数
    const hasNoteTag = (eventData, tag) => {
        // ★タグが空文字列（または未定義/null）なら、タグは無いものとして false を返す
        if (!tag || tag === '') return false;
        
        if (!eventData || !eventData.note) return false;
        
        const t = String(tag);
        // 大文字小文字を区別しない正規表現で、<タグ>形式をチェック
        const re = new RegExp('<\\s*' + t + '\\s*>', 'iu');
        return re.test(eventData.note);
    };

    function handleTrapCollisions() {
        for (const trap of $gameMap.events()) {
            if (!trap || trap._erased) continue;
            // トラップタグのチェック
            if (!hasNoteTag(trap.event(), trapTag)) continue;

            const tx = trap.x;
            const ty = trap.y;

            for (const enemy of $gameMap.events()) {
                if (!enemy || enemy._erased) continue;
                // ターゲットタグのチェック
                if (!hasNoteTag(enemy.event(), enemyTag)) continue;
                
                // ★除外タグのチェック！
                // excludeTagが設定されており（空欄でなく）、かつイベントにそのタグが付いていたらスキップ
                if (excludeTag !== '' && hasNoteTag(enemy.event(), excludeTag)) {
                    // console.log(`Event ${enemy.eventId()} has <${excludeTag}> tag and is excluded.`);
                    continue; // スイッチは切り替えずに次のループへ
                }

                const ex = enemy.x;
                const ey = enemy.y;

                // 接触判定: 座標が完全に一致する場合
                if (tx === ex && ty === ey) {
                    const key = [$gameMap.mapId(), enemy.eventId(), ssToSet];

                    if (!$gameSelfSwitches.value(key)) {
                        $gameSelfSwitches.setValue(key, true);
                        console.log(`Trap triggered! Event ${enemy.eventId()} SS ${ssToSet} ON`);
                    }
                }
            }
        }
    }

    const _Scene_Map_update = Scene_Map.prototype.update;
    Scene_Map.prototype.update = function () {
        _Scene_Map_update.call(this);
        handleTrapCollisions();
    };
})();