//=============================================================================
// SimpleVoiceGeneralVolume.js
//  ※ SimpleVoice.js のボイス音量キーを `voiceVolumeGeneral` に変更した版
//=============================================================================

/*:
 * @plugindesc 簡易ボイスプラグイン（音量キー独立：voiceVolumeGeneral） @target MZ
 * @url https://github.com/triacontane/RPGMakerMV/tree/mz_master/SimpleVoice.js
 * @base PluginCommonBase
 * @author トリアコンタン
 *
 * @param optionName
 * @text オプション名称
 * @type string
 * @desc オプション画面に表示されるボイス音量の設定項目名称です。
 * @default ボイス 音量
 *
 * @param optionValue
 * @text オプション初期値
 * @type number
 * @desc ボイス音量の初期値です。
 * @default 100
 *
 * @command PLAY_VOICE
 * @text ボイスの演奏
 * @desc ボイスを演奏します。
 * @arg name
 * @text ファイル名称
 * @type file
 * @dir audio/se
 * @default
 *
 * @arg volume
 * @text 音量
 * @type number
 * @min 0
 * @max 100
 * @default 90
 *
 * @arg pitch
 * @text ピッチ
 * @type number
 * @default 100
 *
 * @arg pan
 * @text 左右バランス
 * @type number
 * @min -100
 * @max 100
 * @default 0
 *
 * @arg channel
 * @text チャンネル番号
 * @type number
 * @default 0
 *
 * @arg loop
 * @text ループ有無
 * @type boolean
 * @default false
 *
 * @command STOP_VOICE
 * @text ボイスの停止
 * @desc 演奏中のボイスを停止します。
 * @arg name
 * @text ファイル名称
 * @type file
 * @dir audio/se
 * @default
 *
 * @arg channel
 * @text チャンネル番号
 * @type number
 * @default 0
 *
 * @help
 * このプラグインは、SimpleVoice.js の音量設定キーを
 * `voiceVolumeGeneral` に変更したバージョンです。
 * これにより、BattleVoiceMZ.js などと音量設定が衝突しません。
 *
 * 元プラグインと同様にボイスの再生・停止をプラグインコマンドで制御します。
 *
 * 【変更点】
 * ・ConfigManager.voiceVolume → ConfigManager.voiceVolumeGeneral に変更
 * ・AudioManager._voiceVolume → AudioManager._voiceVolumeGeneral に変更
 *
 * 利用規約は原作と同様に MIT ライセンスです。
 */

(function () {
    'use strict';
    const script = document.currentScript;
    const param = PluginManagerEx.createParameter(script);

    PluginManagerEx.registerCommand(script, 'PLAY_VOICE', args => {
        AudioManager.playVoice(args, args.loop, args.channel);
    });

    PluginManagerEx.registerCommand(script, 'STOP_VOICE', args => {
        if (args.name) {
            AudioManager.stopVoice(args.name, null);
        } else if (args.channel) {
            AudioManager.stopVoice(null, args.channel);
        } else {
            AudioManager.stopVoice(null, null);
        }
    });

    //=============================================================================
    // ConfigManager - 音量設定キー変更
    //=============================================================================
    Object.defineProperty(ConfigManager, 'voiceVolumeGeneral', {
        get: function () {
            return AudioManager._voiceVolumeGeneral;
        },
        set: function (value) {
            AudioManager.voiceVolumeGeneral = value;
        }
    });

    const _ConfigManager_makeData = ConfigManager.makeData;
    ConfigManager.makeData = function () {
        const config = _ConfigManager_makeData.apply(this, arguments);
        config.voiceVolumeGeneral = this.voiceVolumeGeneral;
        return config;
    };

    const _ConfigManager_applyData = ConfigManager.applyData;
    ConfigManager.applyData = function (config) {
        _ConfigManager_applyData.apply(this, arguments);
        const symbol = 'voiceVolumeGeneral';
        this.voiceVolumeGeneral = config.hasOwnProperty(symbol) ? this.readVolume(config, symbol) : param.optionValue;
    };

    //=============================================================================
    // Window_Options - オプション表示
    //=============================================================================
    const _Window_Options_addVolumeOptions = Window_Options.prototype.addVolumeOptions;
    Window_Options.prototype.addVolumeOptions = function () {
        _Window_Options_addVolumeOptions.apply(this, arguments);
        this.addCommand(param.optionName, 'voiceVolumeGeneral');
    };

    const _Scene_Options_maxCommands = Scene_Options.prototype.maxCommands;
    Scene_Options.prototype.maxCommands = function () {
        return _Scene_Options_maxCommands.apply(this, arguments) + 1;
    };

    //=============================================================================
    // AudioManager - 音量管理機能（キー変更）
    //=============================================================================
    Object.defineProperty(AudioManager, 'voiceVolumeGeneral', {
        get: function () {
            return this._voiceVolumeGeneral;
        },
        set: function (value) {
            this._voiceVolumeGeneral = value;
        }
    });

    AudioManager._voiceBuffers = [];
    AudioManager._voiceVolumeGeneral = param.optionValue || 100;

    AudioManager.updateVoiceParameters = function (buffer, voice) {
        this.updateBufferParameters(buffer, this._voiceVolumeGeneral, voice);
    };

    AudioManager.playVoice = function (voice, loop, channel) {
        const voicePath = PluginManagerEx.convertEscapeCharacters(voice.name);
        if (voicePath) {
            const path = ('se/' + voicePath).split('/');
            const name = path.pop();
            const folder = path.join('/') + '/';
            this.stopVoice(voice.name, channel);
            const buffer = this.createBuffer(folder, name);
            this.updateVoiceParameters(buffer, voice);
            buffer.play(loop, 0);
            buffer.path = voicePath;
            buffer.channel = channel;
            this._voiceBuffers.push(buffer);
        }
    };

    AudioManager.stopVoice = function (name, channel) {
        const voicePath = name ? PluginManagerEx.convertEscapeCharacters(name) : null;
        this._voiceBuffers.forEach(function (buffer) {
            if (!name && !channel || buffer.path === voicePath || buffer.channel === channel) {
                buffer.stop();
            }
        });
        this.filterPlayingVoice();
    };

    AudioManager.filterPlayingVoice = function () {
        this._voiceBuffers = this._voiceBuffers.filter(function (buffer) {
            const playing = buffer.isPlaying() || !buffer.isReady();
            if (!playing) buffer.stop();
            return playing;
        });
    };

    const _AudioManager_stopAll = AudioManager.stopAll;
    AudioManager.stopAll = function () {
        _AudioManager_stopAll.apply(this, arguments);
        this.stopVoice();
    };

    //=============================================================================
    // Scene_Base - フェードアウト時にボイスも停止
    //=============================================================================
    const _Scene_Base_fadeOutAll = Scene_Base.prototype.fadeOutAll;
    Scene_Base.prototype.fadeOutAll = function () {
        _Scene_Base_fadeOutAll.apply(this, arguments);
        AudioManager.stopVoice();
    };
})();
