/*:
 * @target MZ
 * @plugindesc 重複なしランダムSE再生スクリプト
 * @author 
 * @help
 * SEを重複なくランダムに再生する関数を提供します。
 * イベントコマンドの「スクリプト」で以下のコードを実行してください。
 * 
 * ■使い方（引数の順番）
 * playRandomSe("ファイル名", 開始, 終了, 音量, ピッチ, パン, 桁数);
 * 
 * ▼ 基本（音量90, ピッチ100, パン0, 桁数0）
 * playRandomSe("Scene_", 1, 20);
 * 
 * ▼ 音量などを指定する場合
 * playRandomSe("Scene_", 1, 20, 80, 100, -50);
 * （音量80、ピッチ100、パン-50[左] で再生）
 * 
 * ▼ ゼロ埋め(01, 02...)も含める場合
 * playRandomSe("Scene_", 1, 20, 90, 100, 0, 2);
 * （最後の「2」が桁数です）
 * 
 * ■引数の説明
 * 音量：0～100 (省略時90)
 * ピッチ：50～150 (省略時100)
 * パン：-100(左) ～ 100(右) (省略時0)
 * 桁数：0ならそのまま、2なら01,02... (省略時0)
 * 
 * ■バッグのリセット
 * clearSeBags();
 */

// ▼ 重複防止用の「くじ引き箱」を保存する場所
var _seShuffleBags = {};

/**
 * 指定範囲のSEをランダム（重複なしシャッフル）再生する関数
 * @param {string} baseName - ファイル名の共通部分
 * @param {number} min      - 開始番号
 * @param {number} max      - 終了番号
 * @param {number} volume   - 音量 (省略可: 90)
 * @param {number} pitch    - ピッチ (省略可: 100)
 * @param {number} pan      - 位相 (省略可: 0)
 * @param {number} digits   - 桁数 (省略可: 0)
 */
function playRandomSe(baseName, min, max, volume, pitch, pan, digits) {
    // 1. そのシーン用の「くじ引き箱」がなければ作る
    if (!_seShuffleBags[baseName] || _seShuffleBags[baseName].length === 0) {
        _seShuffleBags[baseName] = [];
        for (var i = min; i <= max; i++) {
            _seShuffleBags[baseName].push(i);
        }
        _seShuffleBags[baseName].sort(() => Math.random() - 0.5);
    }

    // 2. 箱から1つ数字を取り出す
    var num = _seShuffleBags[baseName].pop();

    // 3. 数字を文字列に変換（必要ならゼロ埋め）
    var numString = String(num);
    if (digits && digits > 0) {
        numString = numString.padStart(digits, '0');
    }

    // 4. ファイル名を合成して再生
    var seName = baseName + numString;
    
    // 引数が省略された場合のデフォルト値
    var vol = (volume === undefined) ? 90 : volume;
    var pit = (pitch === undefined) ? 100 : pitch;
    var pn  = (pan === undefined) ? 0 : pan;

    AudioManager.playSe({ name: seName, volume: vol, pitch: pit, pan: pn });
    
    // デバッグ用
    console.log("再生SE: " + seName + " Vol:" + vol + " Pan:" + pn);
}

/**
 * くじ引き箱をリセットする関数
 */
function clearSeBags() {
    _seShuffleBags = {};
}