/*:
 * @target MZ
 * @plugindesc ボイス・BGV・システム・通常SEの4系統で音量を管理＆ボイス単音化
 * @author
 *
 * @help
 * ■機能
 * 1. 内部的に「ボイス音量」「BGV音量」「システム音量」の管理機能を追加します。
 * 2. 以下のルールで自動的に音量を適用します。
 *    - ファイル名に "Voice" を含む → 「ボイス音量」 (＆前のボイス停止)
 *    - ファイル名に "BGV" を含む   → 「BGV音量」
 *    - [システム]で設定されているSE → 「システム音量」
 *    - それ以外 → 「SE音量」
 *
 * ■NUUN_OptionExでの設定方法
 * NUUN側の設定で以下の「コマンドシンボル」を指定してください。
 * 
 * ・ボイス音量   : voiceVolume
 * ・BGV音量      : bgvVolume
 * ・システム音量 : systemSeVolume
 */

(() => {
    // ▼ キーワード設定（小文字で記入）
    const voiceKeyword = "voice"; 
    const bgvKeyword   = "bgv";
    // ▲ ----------------------------------------------------------

    // ------------------------------------------------------------
    // 1. ConfigManagerに設定項目を追加
    // ------------------------------------------------------------
    ConfigManager.voiceVolume = 100;
    ConfigManager.bgvVolume = 100;     // 
    ConfigManager.systemSeVolume = 100;

    const _ConfigManager_makeData = ConfigManager.makeData;
    ConfigManager.makeData = function() {
        const config = _ConfigManager_makeData.call(this);
        config.voiceVolume = this.voiceVolume;
        config.bgvVolume = this.bgvVolume; // 
        config.systemSeVolume = this.systemSeVolume;
        return config;
    };

    const _ConfigManager_applyData = ConfigManager.applyData;
    ConfigManager.applyData = function(config) {
        _ConfigManager_applyData.call(this, config);
        this.voiceVolume = this.readVolume(config, "voiceVolume");
        this.bgvVolume = this.readVolume(config, "bgvVolume"); // 
        this.systemSeVolume = this.readVolume(config, "systemSeVolume");
    };

    // ------------------------------------------------------------
    // 2. オプション画面にバーを追加（NUUN未使用時用）
    // ------------------------------------------------------------
    const _Window_Options_addVolumeOptions = Window_Options.prototype.addVolumeOptions;
    Window_Options.prototype.addVolumeOptions = function() {
        _Window_Options_addVolumeOptions.call(this);
        this.addCommand("システムSE 音量", "systemSeVolume");
        this.addCommand("BGV 音量", "bgvVolume"); // 
        this.addCommand("ボイス 音量", "voiceVolume");
    };

    // ------------------------------------------------------------
    // 3. 判定ロジック
    // ------------------------------------------------------------
    function getVolumeType(seName) {
        if (!seName) return 'normal';
        const name = seName.toLowerCase();

        // A. ボイス判定
        if (name.includes(voiceKeyword)) {
            return 'voice';
        }

        // B. BGV判定
        if (name.includes(bgvKeyword)) {
            return 'bgv';
        }

        // C. システムSE判定
        if ($dataSystem && $dataSystem.sounds) {
            const isSystem = $dataSystem.sounds.some(sysSound => 
                sysSound && sysSound.name && sysSound.name.toLowerCase() === name
            );
            if (isSystem) return 'system';
        }

        // D. それ以外
        return 'normal';
    }

    // ------------------------------------------------------------
    // 4. 再生処理の改変
    // ------------------------------------------------------------
    
    // 通常のSE再生
    const _AudioManager_playSe = AudioManager.playSe;
    AudioManager.playSe = function(se) {
        if (!se.name) return;

        const type = getVolumeType(se.name);
        const realSeVolume = this._seVolume; // 元の設定を退避

        if (type === 'voice') {
            // ボイス停止処理
            this._seBuffers.forEach(buffer => {
                if (buffer.isPlaying() && buffer._url) {
                    const decodedName = decodeURIComponent(buffer._url).toLowerCase();
                    if (decodedName.includes(voiceKeyword)) {
                        buffer.stop();
                    }
                }
            });
            // 音量すり替え
            this._seVolume = ConfigManager.voiceVolume;

        } else if (type === 'bgv') {
            // BGV音量にすり替え（★追加）
            this._seVolume = ConfigManager.bgvVolume;

        } else if (type === 'system') {
            // システム音量にすり替え
            this._seVolume = ConfigManager.systemSeVolume;
        }

        // 再生
        _AudioManager_playSe.call(this, se);

        // 音量を戻す
        this._seVolume = realSeVolume;
    };

    // 静的SE再生
    const _AudioManager_playStaticSe = AudioManager.playStaticSe;
    AudioManager.playStaticSe = function(se) {
        if (!se.name) return;

        const type = getVolumeType(se.name);
        const realSeVolume = this._seVolume;

        if (type === 'system') {
            this._seVolume = ConfigManager.systemSeVolume;
        } else if (type === 'voice') {
            this._seVolume = ConfigManager.voiceVolume;
        } else if (type === 'bgv') {
            this._seVolume = ConfigManager.bgvVolume;
        }

        _AudioManager_playStaticSe.call(this, se);
        this._seVolume = realSeVolume;
    };

})();