//=============================================================================
// AddPlugin_Base
// バージョン: 1.0.0
//=============================================================================
// Copyright (c) 2026 とりぬー
// Released under the MIT license
// http://opensource.org/licenses/mit-license.php
//=============================================================================

/*:
 * @target MZ
 * @plugindesc 共通処理
 * @author とりぬー
 * 
 * @help 
 * 汎用的な記述を行う共通処理セット
 * プラグインコマンドはありません。
 * よく使うスクリプトを実装するためのものです。
 * 
 * 0. キー入力の追加
 *    追加したいキーがあれば、このコードに追加する
 *    条件分岐のスクリプトで以下の記載でキー判定が可能になる。
 * 
 *    Input.isTriggered('○○') ・・・ 押した瞬間だけ条件を満たす
 *    Input.isPressed('○○')   ・・・ 押している長さに関係なく押してる間ずっと条件を満たす
 *    Input.isRepeated('○○')  ・・・ 24フレーム以上押し続けていると条件を満たす
 * 
 * 1. actorEquipTraitsRet
 *    指定アクターの装備一覧を取得し、指定したcodeのvalueを合算。
 *    結果は配列形式で返ってきます。
 *    例として、命中率や回避率は同じcodeでもdataIdが違うので
 *    　　　　　命中率が0番目の配列、回避率が1番目の配列のような形式で返ります。
 * 
 *    使用例(スクリプト):
 *    let traitsRet = actorEquipTraitsRet(actorId,code); //plugin Base
 * 
 * 2. actorClassTraitsRet
 *    指定アクターの職業を取得し、指定したcodeのvalueを合算。
 *    結果は配列形式で返ってきます。
 * 
 *    使用例(スクリプト):
 *    let traitsRet = actorClassTraitsRet(actorId,code); //plugin Base
 * 
 * 3. actorStateTraitsRet
 *    指定アクターのかかっているステートを取得し、指定したcodeのvalueを合算。
 *    結果は配列形式で返ってきます。
 * 
 *    使用例(スクリプト):
 *    let traitsRet = actorStateTraitsRet(actorId,code); //plugin Base
 * 
 * 4. actorAllTraitsRet
 *    指定アクターの装備一覧、職業、かかっているステートを取得し、
 *    指定したcodeのvalueを合算。結果は配列形式で返ってきます。
 *    (1~3,6の処理をまとめたもの)
 * 
 *    使用例(スクリプト):
 *    let traitsRet = actorAllTraitsRet(actorId,code); //plugin Base
 * 
 * 5. actorAllmetaRet
 *    指定アクターのアクターのゲーム開始前データベース、
 *    装備一覧、職業、かかっているステートを取得し、
 *    指定したmetaNameのvalueを合算。結果は数値形式で返ってきます。
 *    metaNameにはあなたがメモ欄に設定しているメタの文字列を入力します。
 *    例：<testTag:1> とあるなら、装備と職業とステートのメモを見て、
 *        testTagの合計を加算し、数値を返します。
 * 
 *    使用例(スクリプト):
 *    let metaRet = actorAllMetaRet(actorId,metaName); //plugin Base
 * 
 * 6. actorTraitsRet
 *    アクターのゲーム開始前データベースを取得し、指定したcodeのvalueを合算。
 *    結果は配列形式で返ってきます。
 * 
 *    使用例(スクリプト):
 *    let traitsRet = actorTraitsRet(actorId,code); //plugin Base
 * 
 * 11.スキルDBからダメージ計算を取得して計算結果を返す
 * 
 * 12.スキルDBからステートを取得して対象に掛けられるか結果を返す
 * 
 * 13.アイテムDBからダメージ計算を取得して計算結果を返す
 * 
 * 14.DBのタグがついているものから自動抽選スクリプト
 *    以下のタグがついていると、抽選対象になる
 *    抽選対象。Xは1や2が入る。1を指定すると1のみ抽選対象になる
 *    <RandomRank:X> 
 *    抽選時の重み。数字が高いほど選ばれやすい。1などの整数を想定
 *    <RandomPercent:X></RandomPercent:X>
 * 
 * 15.$gameTemp._forceItemReturnToMenu = true;
 *    SceneManager.push(Scene_Item);
 *    上記2行をスクリプトとして呼び出すことで、
 *    アイテム画面からメニュー画面に戻る動作が出来るコードです。
 * 
 * 
 * ※指定するcode一覧例
 * 11 → 属性有効度
 * 13 → ステート有効度
 * 14 → ステート無効化
 * 22 → 追加能力値:回避や命中など
 * 31 → 攻撃時属性
 * 32 → 攻撃時ステート
 * 
 */

(() => {
    "use strict";

    //キャラスムーズ強制無効、画面ズーム時の変な線対策。
    //他プラグインでも実施されているが、再定義。
    const _Bitmap_startLoading = Bitmap.prototype._startLoading;
    Bitmap.prototype._startLoading = function() {
        this._smooth = false;
        _Bitmap_startLoading.apply(this);
    };

    //0.
    (function () {
        //使用可能になるキー登録
        Input.keyMapper[65] = 'A';
        // Input.keyMapper[66] = 'B';
        Input.keyMapper[67] = 'C';
        // Input.keyMapper[68] = 'D';
        // Input.keyMapper[69] = 'E';
        Input.keyMapper[70] = 'F';
        // Input.keyMapper[71] = 'G';
        // Input.keyMapper[72] = 'H';
        // Input.keyMapper[73] = 'I';
        // Input.keyMapper[74] = 'J';
        // Input.keyMapper[75] = 'K';
        // Input.keyMapper[76] = 'L';
        // Input.keyMapper[77] = 'M';
        // Input.keyMapper[78] = 'N';
        // Input.keyMapper[79] = 'O';
        Input.keyMapper[80] = 'P';
        Input.keyMapper[81] = 'Q';
        // Input.keyMapper[82] = 'R';
        Input.keyMapper[83] = 'S';
        // Input.keyMapper[84] = 'T';
        // Input.keyMapper[85] = 'U';
        Input.keyMapper[86] = 'V';
        // Input.keyMapper[87] = 'W';
        // Input.keyMapper[88] = 'X';
        // Input.keyMapper[89] = 'Y';
        // Input.keyMapper[90] = 'Z';
    })();

    //1.
    window.actorEquipTraitsRet = function(actorId,code) {
        const codeNumber = code;
        const actorEquips = $gameActors.actor(actorId).equips();
        let filteredEquips = actorEquips;
        
        // 武器切り替えシステムプラグインが導入されている場合
        if(PluginManager._scripts.includes("AddPlugin_SecondWeaponChange")){
            // `$dataActors` からアクターの特性を取得
            const isDualWield = $dataActors[actorId].traits.some(trait => trait.code === 55);

            // equips() から二刀流のときだけ 2番目の装備を削除
            filteredEquips = actorEquips.filter((item, index) => {
                return !(isDualWield && index === 1); // 二刀流なら2番目の装備を取り除く
            });
        }

        const filteredData = filteredEquips.map(item => {
            if (item && item.traits) {
                return item.traits.filter(trait => trait.code === codeNumber);
            }
            return item;
        }).filter(item => item !== null);

        const mergedData = dataMerage(filteredData);

        return mergedData;
    }
    //2.
    window.actorClassTraitsRet = function(actorId,code) {
        const codeNumber = code;
        const actorClass = $dataClasses[$gameActors.actor(actorId)._classId].traits;
        const filteredData = actorClass.filter(item => item.code === codeNumber);

        const mergedData = dataMerage(filteredData);

        return mergedData;
    }
    //3.
    window.actorStateTraitsRet = function(actorId,code) {
        const codeNumber = code;
        const actorState = $gameActors.actor(actorId)._states;
        let stateArray = [];

        actorState.forEach((item, index) => {
            let pushTarget = $dataStates[item].traits;
            pushTarget.forEach((item, index) => {
                stateArray.push(item)
            });
        });

        const filteredData = stateArray.filter(item => item.code === codeNumber);

        const mergedData = dataMerage(filteredData);

        return mergedData;
    }
    //4.
    window.actorAllTraitsRet = function(actorId,code) {
        const actorLet = actorTraitsRet(actorId,code);
        const classLet = actorClassTraitsRet(actorId,code);
        const equipLet = actorEquipTraitsRet(actorId,code);
        const stateLet = actorStateTraitsRet(actorId,code);
        
        const combinedArray = actorLet.concat(classLet, equipLet, stateLet )
        const mergedData = dataMerage(combinedArray);
        
        return mergedData;
    }
    //5.
    window.actorAllMetaRet = function(actorId,metaName) {
        let totalValue = 0;
        const actor = $gameActors.actor(actorId);

        const actorMeta = $dataActors[actorId].meta;
        totalValue += parseInt(actorMeta[metaName] || 0,10);
        
        let actorEquips = actor.equips();
        let filteredEquips = actorEquips;

        // 武器切り替えシステムプラグインが導入されている場合
        if(PluginManager._scripts.includes("AddPlugin_SecondWeaponChange")){
            // `$dataActors` からアクターの特性を取得
            const isDualWield = $dataActors[actorId].traits.some(trait => trait.code === 55);

            // equips() から二刀流のときだけ 2番目の装備を削除
            filteredEquips = actorEquips.filter((item, index) => {
                return !(isDualWield && index === 1); // 二刀流なら2番目の装備を取り除く
            });
        }

        filteredEquips.forEach(equip => {
            if (equip && equip.meta[metaName]) {
                totalValue += parseInt(equip.meta[metaName] || 0,10);
            }
        });

        let classId = actor._classId;
        let classData = $dataClasses[classId];
        if (classData && classData.meta[metaName]) {
            totalValue += parseInt(classData.meta[metaName] || 0,10);
        }

        let states = actor._states;
        states.forEach(stateId => {
            let stateData = $dataStates[stateId];
            if (stateData && stateData.meta[metaName]) {
                totalValue += parseInt(stateData.meta[metaName] || 0,10);
            }
        });

        return totalValue;
    }
    //6.
    window.actorTraitsRet = function(actorId,code) {
        const codeNumber = code;
        const actor = $dataActors[actorId].traits;
        const filteredData = actor.filter(item => item.code === codeNumber);

        const mergedData = dataMerage(filteredData);

        return mergedData;
    }

    //値を合算
    function dataMerage(filteredData) {
        const traitMap = new Map();

        filteredData.flat().forEach(trait => {
            const key = `${trait.code}_${trait.dataId}`;
            const value = trait.value;

            if (traitMap.has(key)) {
                const existing = traitMap.get(key);

                // 有効度(code11と13)は計算方法を切り替える
                if (value < 1 && (trait.code === 11 || trait.code === 13)) {
                    existing.value -= (1 - value);
                }else if (value >= 1 && (trait.code === 11 || trait.code === 13)) {
                    existing.value += (value - 1);
                }else {
                    existing.value += value;
                }
            } else {
                traitMap.set(key, { ...trait });
            }
        });

        // 小数第2位で四捨五入
        for (const trait of traitMap.values()) {
            trait.value = Math.round(trait.value * 100) / 100;
        }

        return Array.from(traitMap.values());
    }

    //11.
    window.skillDamageRet = function(skillId,actorId1,actorId2) {
        //スキルIDを参照し計算式を取得
        const skillDamageRate = $dataSkills[skillId].damage.formula;
        //スキルの属性を取得
        const skillElementId = $dataSkills[skillId].damage.elementId;
        //ダメージ計算
        const a = $gameActors.actor(actorId1)
        const b = $gameActors.actor(actorId2)
        const skillDamageResult = skillNumberJoin(skillDamageRate, {a, b});

        //攻撃する側の、攻撃属性を取得
        const attackElement = actorAllTraitsRet(actorId1,31);

        //スキルの攻撃属性をマージ
        const skillElementObject = {code: 31, dataId: skillElementId, value: 1};
        const combinedArray = attackElement.concat(skillElementObject)
        const attackElementMerge = dataMerage(combinedArray);

        //攻撃される側の、属性有効度を取得
        const guardElement = actorAllTraitsRet(actorId2,11);

        //ダメージの基本倍率
        let totalRate = 1.0;
        //全ての耐性を考慮した計算 (例えば2種類の属性で攻撃した際、軽減も2種類の値を用いて計算する)
        for (let i = 0; i < attackElementMerge.length; i++) {
            const attackId = attackElementMerge[i].dataId;

            for (let j = 0; j < guardElement.length; j++) {
                const guard = guardElement[j];

                if (guard.dataId === attackId) {
                    totalRate *= guard.value;
                }
            }
        }
        // 倍率込みのダメージ計算
        let result = Math.round(skillDamageResult * totalRate);

        //最小と最大の定義
        const minResult = 0;
        const maxResult = 999999;
        if(minResult >= result){
            result = minResult;
        }
        if(maxResult <= result){
            result = maxResult;
        }

        return result;
    }
    //12.
    window.attackStateRet = function(skillId,actorId1,actorId2) {
        //スキルIDを参照し特徴を取得
        const skillEffects = $dataSkills[skillId].effects;
        
        //スキルのステート付加を取得
        const skillEffects21 = skillEffects.filter(effect => effect.code === 21);
        //マージ用に加工する
        let skillEffectsState = [];
        skillEffects21.forEach(effect => {
            let skillElementObject = {code: 32, dataId: effect.dataId, value: effect.value1};
            skillEffectsState.push(skillElementObject);
        });

        //攻撃する側の、攻撃時状態異常を取得
        const attackState = actorAllTraitsRet(actorId1,32);
        //スキルのステート付加をマージ
        const combinedArray = attackState.concat(skillEffectsState)
        let attackElementMerge = dataMerage(combinedArray);

        // dataIdが0のものを除外(ツクールMZ標準の通常攻撃はステートDBを参照するものではない)
        attackElementMerge = attackElementMerge.filter(item => item.dataId !== 0);

        //攻撃される側の、ステート有効度を取得
        const guardState13 = actorAllTraitsRet(actorId2,13);
        
        //攻撃される側の、ステート無効を取得
        const guardState14 = actorAllTraitsRet(actorId2,14);

        let stateArray = [];
        attackElementMerge.forEach(state => {
            const stateId = state.dataId; // 状態異常のステートID
            let statePercent = state.value; // 状態異常の確率
            
            //ステート有効度の減算
            const value13 = guardState13.filter(state13 => state13.dataId === stateId);
            if(value13.length != 0 && 0 < value13[0].value){
                statePercent = Math.round(statePercent * value13[0].value * 100) / 100;
            }
            //ステート無効化の反映
            const value14 = guardState14.filter(state14 => state14.dataId === stateId);
            if(value14.length != 0 && 1 === value14[0].value){
                statePercent = 0;
            }
            
            // statePercent以下ならtrue(かかる)、以上ならfalse(かからない)
            const random = Math.random() < statePercent ? true : false;

            if (random) {
                stateArray.push(stateId);
            }
        });

        return stateArray;

    }
    //13.
    window.itemDamageRet = function(itemId,actorId1,actorId2) {
        //アイテムIDを参照し計算式を取得
        const itemDamageRate = $dataItems[itemId].damage.formula;
        //アイテムの属性を取得
        const itemElementId = $dataItems[itemId].damage.elementId;
        //ダメージ計算
        const a = $gameActors.actor(actorId1)
        const b = $gameActors.actor(actorId2)
        const skillDamageResult = skillNumberJoin(itemDamageRate, {a, b});

        //アイテムの攻撃属性を取得
        const itemElementObject = [];
        itemElementObject.push({code: 31, dataId: itemElementId, value: 1});

        //攻撃される側の、属性有効度を取得
        const guardElement = actorAllTraitsRet(actorId2,11);

        //ダメージの基本倍率
        let totalRate = 1.0;
        //全ての耐性を考慮した計算 (例えば2種類の属性で攻撃した際、軽減も2種類の値を用いて計算する)
        for (let i = 0; i < itemElementObject.length; i++) {
            const attackId = itemElementObject[i].dataId;

            for (let j = 0; j < guardElement.length; j++) {
                const guard = guardElement[j];

                if (guard.dataId === attackId) {
                    totalRate *= guard.value;
                }
            }
        }
        // 軽減後のダメージ計算
        let result = Math.floor(skillDamageResult * totalRate);

        //最小と最大の定義
        const minResult = 0;
        const maxResult = 9999;
        if(minResult >= result){
            result = minResult;
        }
        if(maxResult <= result){
            result = maxResult;
        }

        return result;
    }
    //ダメージ計算式の再評価用
    window.skillNumberJoin = function(obj, scope) {
        // 正規表現で数式のみを許可する
        if (!/^[0-9a-zA-Z.*+\-/() ]+$/.test(obj)) {
            throw new Error('Invalid expression');
        }
        // スコープを関数内で展開する
        return Function('"use strict";' + 
            Object.keys(scope).map(key => `var ${key} = this.${key};`).join('') + 
            'return (' + obj + ')').call(scope);
    }
    //14.
    window.itemTagRandomGet = function(itemType, targetRank) {
        let data = [];

        if (itemType === 0) {
            data = $dataItems;
        } else if (itemType === 1) {
            data = $dataWeapons;
        } else if (itemType === 2) {
            data = $dataArmors;
        }

        // RankとPersentの両方をチェック
        const weightedItems = [];

        for (const item of data) {
            if (!item || !item.note) continue;

            const rankMatch = item.note.match(/<RandomRank:(\d+)>/);
            const percentMatch = item.note.match(/<RandomPercent:(\d+)>/);

            if (rankMatch && Number(rankMatch[1]) === targetRank && percentMatch) {
                const weight = Number(percentMatch[1]);
                // 重み分だけ配列に追加
                for (let i = 0; i < weight; i++) {
                    weightedItems.push(item);
                }
            }
        }

        // 重み付き抽選
        const randomItem = weightedItems[Math.floor(Math.random() * weightedItems.length)];
        // idを返す、何もなければ7を返す(アイテム指定なし対策)
        return randomItem ? randomItem.id : 7;
    }
    
    //15.
    const _Scene_Item_popScene_AddPlugin = Scene_Item.prototype.popScene;
    Scene_Item.prototype.popScene = function() {
        if ($gameTemp._forceItemReturnToMenu) {
            $gameTemp._forceItemReturnToMenu = false;
            SceneManager.goto(Scene_Menu);
        } else {
            _Scene_Item_popScene_AddPlugin.call(this);
        }
    };

})();
