//=============================================================================
// AddPlugin_ChangeBasicSpeed - プレイヤーやイベントの基本速度を変更
// バージョン: 1.0.0
//=============================================================================
// Copyright (c) 2026 とりぬー
// Released under the MIT license
// http://opensource.org/licenses/mit-license.php
//=============================================================================

/*:
 * @target MZ
 * @plugindesc プレイヤーとイベントの基本速度を変更するプラグイン
 * @author とりぬー
 * 
 * @help 
 * プレイヤーとイベントの基本速度を変更できます。
 * プラグインコマンドで呼び出して、ゲーム中に変更も出来ます。
 * 
 * @command SetPlayerBasicSpeed
 * @text プレイヤーの基本速度を設定
 * @desc プレイヤーの基本速度を設定します。
 *
 * @arg speedRate
 * @type number
 * @min 1
 * @max 10000
 * @default 100
 * @text 速度割合
 * @desc プレイヤーの基本速度の割合を設定します。通常は100です。
 * 
 * @command SetEventBasicSpeed
 * @text イベントの基本速度を設定
 * @desc イベントの基本速度を設定します。
 *
 * @arg speedRate
 * @type number
 * @min 1
 * @max 10000
 * @default 100
 * @text 速度割合
 * @desc イベントの基本速度の割合を設定します。通常は100です。
 * 
 * @param <basicSpeed>
 * @text ＜基本速度＞
 * 
 * @param initPlayerBasicSpeed
 * @parent <initPlayerBasicSpeed>
 * @text プレイヤーの基本速度
 * @desc プレイヤーの基本速度の割合を設定します。通常は100です。
 * @type number
 * @min 1
 * @max 10000
 * @default 100
 * 
 * @param initEventBasicSpeed
 * @parent <initEventBasicSpeed>
 * @text イベントの基本速度
 * @desc イベントの基本速度の割合を設定します。通常は100です。
 * @type number
 * @min 1
 * @max 10000
 * @default 100
 */

(() => {
    "use strict";
    const pluginName = document.currentScript.src.match(/^.*\/(.*).js$/)[1];

    PluginManager.registerCommand(pluginName, "SetPlayerBasicSpeed", args => {
        const speedRate = Number(args.speedRate) || 100;
        Game_Player.prototype.distancePerFrame = function() {
            const ret = Game_CharacterBase.prototype.distancePerFrame.apply(this, arguments);
            return ret * speedRate / 100;
        };

        Game_Follower.prototype.distancePerFrame = function() {
            const ret = Game_CharacterBase.prototype.distancePerFrame.apply(this, arguments);
            return ret * speedRate / 100;
        };
    });

    PluginManager.registerCommand(pluginName, "SetEventBasicSpeed", args => {
        const speedRate = Number(args.speedRate) || 100;
        Game_Event.prototype.distancePerFrame = function() {
            const ret = Game_CharacterBase.prototype.distancePerFrame.apply(this, arguments);
            return ret * speedRate / 100;
        };
    });

    const parameters = PluginManager.parameters(pluginName);
    const initPlayerBasicSpeed = Number(parameters["initPlayerBasicSpeed"] || 100);
    const initEventBasicSpeedRate = Number(parameters["initEventBasicSpeed"] || 100);

    // 初期設定
    if (initPlayerBasicSpeed != 100) {
        Game_Player.prototype.distancePerFrame = function() {
            const ret = Game_CharacterBase.prototype.distancePerFrame.apply(this, arguments);
            return ret * initPlayerBasicSpeed / 100;
        };

        Game_Follower.prototype.distancePerFrame = function() {
            const ret = Game_CharacterBase.prototype.distancePerFrame.apply(this, arguments);
            return ret * initPlayerBasicSpeed / 100;
        };
    }

    if (initEventBasicSpeedRate != 100) {
        Game_Event.prototype.distancePerFrame = function() {
            const ret = Game_CharacterBase.prototype.distancePerFrame.apply(this, arguments);
            return ret * initEventBasicSpeedRate / 100;
        };
    }
})();
