//=============================================================================
// AddPlugin_IconShortcut
// バージョン: 1.0.0
//=============================================================================
// Copyright (c) 2026 とりぬー
// Released under the MIT license
// http://opensource.org/licenses/mit-license.php
//=============================================================================

/*:
 * @target MZ
 * @plugindesc アイコンを表示してコモンイベントと紐づける
 * @author とりぬー
 * 
 * @help 
 * アイテムやスキルのショートカットを提供するプラグインです。
 * アイコンと所持数量を表示します。
 * スキルを登録した場合は、消費MPが表示されます。
 * 
 * 1.プラグインコマンド「アイコンを追加」で配列に情報を追加します。
 * 　 { type: "item", id: 1, num: 2 }, { type: "weapon", id: 2, num: 4 },{ type: "skill", id: 3, num: 7 }, { type: "armor", id: 2, num: 4 }
 * 　例として、上記のような形で入っていきます。10個以上になると、
 * 　最初に入ったものから順に押し出される形で消去されます。
 *   同じtypeとidが配列に登録される場合、自動で数量を上書きします。
 * 
 * 2.プラグインコマンド「アイコンの表示」で画面に表示します。
 *   キーボードの1~9,0を押下することで、コモンイベントを呼び出せます。
 *   どのアイコンが呼び出されたかも変数に入ります。
 * 　 { type: "item", id: 1, num: 2 }のような形式なのでコモンイベント内で
 * 　スクリプトで取り出して判定することで個別の処理を作成出来ます。
 *   コモンイベント終了後に制御スイッチをOFFにしないと、
 *   次のコモンイベント呼び出しは起きません。
 * 
 * 3.プラグインコマンド「アイコンの非表示」で画面から非表示にします。
 *   ショートカットキーも無効化されます。
 * 
 * 4.プラグインコマンド「アイコンのクリア」で追加したアイコンを全て消去します。
 *   
 * 
 * @param iconVariable
 * @text 表示するアイコン情報が入る配列変数
 * @desc アイコン情報が入る変数を設定します。
 *       デフォルトは1です。
 * @default 1
 * @type variable
 * 
 * @param iconReturnVariable
 * @text 起動したアイコンの情報が入る配列変数
 * @desc 起動したアイコンの情報が入る変数を設定します。
 *       デフォルトは2です。
 * @default 2
 * @type variable
 * 
 * @param commonCallVariable
 * @text アイコンを押下したときに起動するコモンイベントID
 * @desc 起動するコモンイベントIDを設定します。
 *       デフォルトは1です。
 * @default 1
 * @type common_event
 * 
 * @param commonCallSwitch
 * @text コモンイベント起動中、このプラグインの起動を制御するスイッチ
 * @desc スイッチIDを設定します。デフォルトは1です。
 *       コモンイベント終了後に、イベント内でスイッチOFF推奨です。
 * @default 1
 * @type switch
 * 
 * @param spriteCallSwitch
 * @text アイコンの表示ON・OFFを管理するスイッチ
 * @desc スイッチIDを設定します。デフォルトは2です。
 *       ゲーム中にプラグインコマンドでON・OFFを変更出来ます。
 * @default 2
 * @type switch
 * 
 * @param iconSpriteX
 * @type number
 * @min 0
 * @max 9999
 * @default 0
 * @text アイコンの表示位置 横
 * @desc アイコンの表示位置 横
 * 
 * @param iconSpriteY
 * @type number
 * @min 0
 * @max 9999
 * @default 0
 * @text アイコンの表示位置 縦
 * @desc アイコンの表示位置 縦
 * 
 * @param iconLimit
 * @type number
 * @min 0
 * @max 10
 * @default 9
 * @text アイコンの表示数
 * @desc アイコンの表示数(MAX 10)
 * 
 * @param touchIcon
 * @default false
 * @type boolean
 * @text アイコンタッチで起動
 * @desc アイコンタッチでキーを押下した時と同様の
 *       動作をします。無効にする場合はfalse
 * 
 * @command iconSpriteDisplay
 * @text アイコンの表示
 * @desc アイコンを表示する
 * 
 * @command iconSpriteClose
 * @text アイコンの非表示
 * @desc アイコンを非表示する
 * 
 * @command iconSpritePush
 * @text アイコンを追加
 * @desc アイコンを追加する
 * 
 * @arg targetType
 * @type targetType
 * @text 種別ID
 * @desc "item","weapon","skill","armor"のいずれかを入力してください。
 *       アイコンIDがどのDBに該当するかを決めるものです。
 * 
 * @arg targetIconId
 * @type targetIconId
 * @text アイコンID
 * @desc アイコンIDの設定をします。
 *       例：[1]など、1だと数字と判定しゲーム変数で判定します。
 * 
 * @command iconSpriteClear
 * @text アイコンのクリア
 * @desc アイコンをクリアする
 * 
 */
// iconData = [
//     { type: "item", id: 1, num: 2 }, { type: "weapon", id: 2, num: 4 },
//     { type: "skill", id: 3, num: 7 }, { type: "armor", id: 2, num: 4 },
//     { type: "item", id: 5, num: 0 }
// ];

(() => {
    "use strict";
    const pluginName = document.currentScript.src.match(/^.*\/(.*).js$/)[1];

    const params = PluginManager.parameters(pluginName);
    const iconVariable = Number(params["iconVariable"]);
    const iconReturnVariable = Number(params["iconReturnVariable"]);
    const commonCallVariable = Number(params["commonCallVariable"]);
    const iconSpriteX = Number(params["iconSpriteX"]);
    const iconSpriteY = Number(params["iconSpriteY"]);
    const iconLimit = Number(params["iconLimit"]);
    const touchIcon = Boolean(params["touchIcon"] === 'true' || false);
    let commonCallSwitchValue = Number(params["commonCallSwitch"]);

    let pluginSwitchIconHistory = Number(params["spriteCallSwitch"]);;
    let iconData = [];
    let iconSprites = [];
    let labelStr = "";

    PluginManager.registerCommand(pluginName, "iconSpriteDisplay", () => {
        iconData = $gameVariables.value(iconVariable);
        
        //iconLimitに応じてラベルスプライト作成
        labelStr = "";
        for (let i = 1; i <= iconLimit; i++){
            if(i == 10){
                labelStr = labelStr + String(0);
            }else{
                labelStr = labelStr + String(i);
            }
        }

        $gameSwitches.setValue(pluginSwitchIconHistory, true);
        displayIconSprites();
    });

    PluginManager.registerCommand(pluginName, "iconSpriteClose", () => {
        $gameSwitches.setValue(pluginSwitchIconHistory, false);
        clearIconSprites();
    });

    PluginManager.registerCommand(pluginName, "iconSpritePush",args => {
        let targetType = "item";
        let targetIconId = [];
        iconData = $gameVariables.value(iconVariable);
        if (iconData == 0) {
            iconData = []; // 初期値なら配列に変換
        }
        targetType = args.targetType;
        targetIconId = args.targetIconId ? JSON.parse(args.targetIconId) : [];

        if (!Array.isArray(targetIconId)) {
            //数字の場合はゲーム変数から取得
            targetIconId = [];
            targetIconId.push($gameVariables.value(Number(args.targetIconId)));
        }
        targetIconId.forEach(iconId => {
            iconSpritePush(targetType,iconId,iconData);
        });
        if($gameSwitches.value(pluginSwitchIconHistory)){
            displayIconSprites();
        }
    });

    PluginManager.registerCommand(pluginName, "iconSpriteClear", () => {
        iconData = [];
        if($gameSwitches.value(pluginSwitchIconHistory)){
            displayIconSprites();
        }
    });
    
    function iconSpritePush(targetTypeArg, iconIdArg, iconDataArg) {
        const item = getItem(targetTypeArg, iconIdArg); // アイテムデータを取得
        if (item) {
            // 数量を更新
            const updatedQuantity = updateItemQuantity({ type: targetTypeArg, id: iconIdArg });
    
            // 既存データの確認
            let existingIcon = iconDataArg.find(icon => icon.type === targetTypeArg && icon.id === iconIdArg);
            if (existingIcon) {
                // 既存のアイコンデータを更新
                existingIcon.num = updatedQuantity;
            } else {
                // 新しいデータを追加
                let pushBox = { type: targetTypeArg, id: iconIdArg, num: updatedQuantity };
                iconDataArg.push(pushBox);
    
                // 上限（9個）の確認
                if (iconDataArg.length > iconLimit) {
                    iconDataArg.shift(); // 古いデータを削除
                }
            }
    
            // 更新後のデータを変数に保存
            $gameVariables.setValue(iconVariable, iconDataArg);
        } else {
            console.log("指定されたDBは存在しません");
        }
    }

    function updateItemQuantity(element) {
        if (element.type === "item") {
            const allItems = $gameParty.items();
            const matchingItem = allItems.find(item => item.id === element.id);
            return matchingItem ? $gameParty.numItems($dataItems[matchingItem.id]) : 0;
        } else if (element.type === "skill") {
            const item = getItem(element.type, element.id);
            const actor = $gameParty.members()[0]; // 先頭アクターを取得
            const actualMpCost = actor.skillMpCost(item); // 実際の消費MPを取得
            return item ? actualMpCost : 0;
        } else{
            //武器、防具の数量更新はなし
        }
        return 0;
    }

    class ClickableSprite extends Sprite {
        constructor() {
            super();
            this.setClickHandler(null);
        }

        setClickHandler(handler) {
            this._clickHandler = handler;
        }

        update() {
            super.update();
            if (this.isClicked() && this._clickHandler) {
                this._clickHandler();
            }
        }

        isClicked() {
            if(touchIcon){
                return TouchInput.isTriggered() &&
                   TouchInput.x >= this.x && TouchInput.x < this.x + this.width &&
                   TouchInput.y >= this.y && TouchInput.y < this.y + this.height;
            }
        }
    }

    function getItem(type, id) {
        const dataMap = {
            "item": $dataItems,
            "weapon": $dataWeapons,
            "skill": $dataSkills,
            "armor": $dataArmors
        };
        return dataMap[type] ? dataMap[type][id] : null;
    }

    function createSprite(iconIndex, x, y, handler = null) {
        const iconSize = 32;
        const bitmap = ImageManager.loadSystem('IconSet');
        const iconX = (iconIndex % 16) * iconSize;
        const iconY = Math.floor(iconIndex / 16) * iconSize;

        const sprite = new ClickableSprite();
        sprite.bitmap = new Bitmap(iconSize, iconSize);
        sprite.bitmap.blt(bitmap, iconX, iconY, iconSize, iconSize, 0, 0);
        sprite.x = x;
        sprite.y = y;
        if (handler) sprite.setClickHandler(handler);

        SceneManager._scene.addChild(sprite);
        return sprite;
    }

    function createInitialsSprite(num, x, y, coler) {
        const sprite = new Sprite(new Bitmap(64, 32));
        sprite.bitmap.textColor = coler;
        sprite.bitmap.drawText(num.toString(), 0, 0, 64, 32, 'center');
        sprite.x = x - 17;
        sprite.y = y - 8;

        SceneManager._scene.addChild(sprite);
        return sprite;
    }

    function createNumSprite(num, x, y, coler) {
        const sprite = new Sprite(new Bitmap(64, 32));
        sprite.bitmap.textColor = coler;
        sprite.bitmap.drawText(num.toString(), 0, 0, 64, 32, 'center');
        sprite.x = x - 5;
        sprite.y = y + 10;

        SceneManager._scene.addChild(sprite);
        return sprite;
    }

    let keyHandlerSetup = false;
    function setupKeyHandler() {
        if (keyHandlerSetup) return;
        keyHandlerSetup = true;
        document.addEventListener('keydown', event => {
            if (!$gameSwitches.value(commonCallSwitchValue) && $gameSwitches.value(pluginSwitchIconHistory)) {
                const keyMappings = '1234567890'.split('').reduce((map, key, idx) => {
                    map[key] = idx;
                    return map;
                }, {});
    
                const index = keyMappings[event.key];
                
                if (index !== undefined && index < iconData.length) {
                    const element = iconData[index];
                    $gameSwitches.setValue(commonCallSwitchValue, true);
                    $gameVariables.setValue(iconReturnVariable, { type: element.type, id: element.id, num: element.num });
                    $gameTemp.reserveCommonEvent(commonCallVariable);
                }
            }
        });
    }

    function createWindow(x, y, width, height, text) {
        const window = new Window_SceneExtra({
            x, y, width, height,
            Text: text,
            LineHeight: 36,
            WindowSkin: "Window",
            ShowOpenAnimation: false,
        });
        SceneManager._scene.addChild(window);
        return window;
    }

    function createTextSprite(text, x, y, spacing, color) {
        const characters = text.split('');
        const totalWidth = characters.length * 32 + (characters.length - 1) * spacing;
        const sprite = new Sprite(new Bitmap(totalWidth, 32));
        sprite.bitmap.textColor = color;
        characters.forEach((char, index) => {
            sprite.bitmap.drawText(char, index * (32 + spacing), 0, 32, 32, 'center');
        });
        sprite.x = x;
        sprite.y = y + 25;

        SceneManager._scene.addChild(sprite);
        return sprite;
    }

    function clearIconSprites() {
        const scene = SceneManager._scene;
        if (scene && iconSprites.length > 0) {
            iconSprites.forEach(sprite => {
            if (sprite.bitmap) {
                sprite.bitmap.destroy();
            }
            scene.removeChild(sprite);
            });
            iconSprites = [];
        }
    }

    function addIconSprite(item, index, num, type) {
        const iconSprite = createSprite(item.iconIndex, iconSpriteX + (index * 50), iconSpriteY, () => {
            $gameVariables.setValue(iconReturnVariable, { type: type, id: item.id, num: num });
            $gameTemp.reserveCommonEvent(commonCallVariable);
        });
        iconSprites.push(iconSprite); // スプライトを配列に追加
    
        let textColor = "#FFFFFF";
        let name = type === "skill" ? $dataSkills[item.id].name : $dataItems[item.id].name;
        name = name.substring(0, 2);
        const initialsSprite = createInitialsSprite(name, iconSprite.x, iconSprite.y, textColor);
        iconSprites.push(initialsSprite); // 頭文字スプライトを配列に追加

        if (num >= 0) {
            textColor = type === "skill" ? "#41BFEF" : "#FFFFFF";
            const numSprite = createNumSprite(num, iconSprite.x, iconSprite.y, textColor);
            iconSprites.push(numSprite); // 数字スプライトを配列に追加
        }
    }

    function displayIconSprites() {
        clearIconSprites(); // 既存のスプライトをクリア
    
        // ウィンドウとラベルを表示
        const window = createWindow(iconSpriteX - 9, iconSpriteY - 9, 50 * iconLimit, 50, "");
        iconSprites.push(window);

        const textSprite = createTextSprite(labelStr, window.x + 10, window.y + 9, 18, '#ffd700');
        iconSprites.push(textSprite);
    
        // アイコンデータを更新＆描画
        if (iconData) {
            iconData.forEach((element, index) => {
                // 数量の更新
                element.num = updateItemQuantity(element);
    
                // スプライトを追加
                const item = getItem(element.type, element.id);
                if (item) {
                    addIconSprite(item, index, element.num, element.type);
                }
            });
    
            // 更新したアイコンデータをゲーム変数に保存
            $gameVariables.setValue(iconVariable, iconData);
            setupKeyHandler(); // キーハンドラーを設定
        }
    }
    
    const _Scene_Map_onMapLoaded = Scene_Map.prototype.onMapLoaded;
    Scene_Map.prototype.onMapLoaded = function() {
        _Scene_Map_onMapLoaded.call(this);
        if($gameSwitches.value(pluginSwitchIconHistory)){
            displayIconSprites(); // マップ読み込み時にスプライトを表示
        }
    };

})();