//=============================================================================
// AddPlugin_SecondWeaponChange
// バージョン: 1.0.0
//=============================================================================
// Copyright (c) 2026 とりぬー
// Released under the MIT license
// http://opensource.org/licenses/mit-license.php
//=============================================================================

/*:
 * @target MZ
 * @plugindesc 二刀流時の2つ目の武器のステータスを無効化する
 * @author 
 * @help 
 * このプラグインは以下の効果をもちます。
 * 
 * 1.二刀流のアクターが2つ目の武器を装備した際に
 *   その武器のステータス変動を無効化します。
 *   また、二つ目の武器ラベルを「武器2」に変更します。
 * 
 * 2.下記の既存メソッドの二刀流用のコードに分岐するようになります。
 *   AddPlugin_Base.js
 *   AddPlugin_WeaponsAreaChecker.js
 * 
 * 3.プラグインコマンドで1番目の武器と2番目の武器の
 *   持ち替えを行います。
 *   変更結果は以下のように変数に入ります。
 *   状況によって演出を変えたいときなどの分岐条件で使用してください。
 *   0 → 武器を持っていなかったので入れ替えなし
 *   1 → 武器をどちらも持っている状態で入れ替え
 *   2 → 1番目の武器のみ持っている状態で入れ替え(1が装備なし、2が装備ありになった状態)
 *   3 → 2番目の武器のみ持っている状態で入れ替え(1が装備あり、2が装備なしになった状態)
 *   ※先頭アクターが取得出来ない、先頭アクターが二刀流でない場合は、入れ替えなし。
 * 
 * @param changeResult
 * @text 武器入れ替えの結果
 * @desc 武器入れ替えの結果が入る変数を設定します。
 *       詳細は3.の説明を読むこと
 * @default 1
 * @type variable
 * 
 * @command SwapEquip
 * @text 先頭アクターの武器を入れ替える
 * @desc 先頭アクターの武器を入れ替える
 * 
 */

(() => {
    "use strict";
    const pluginName = document.currentScript.src.match(/^.*\/(.*).js$/)[1];
    let parameters = PluginManager.parameters(pluginName);
    let change_result = Number(parameters['changeResult']);

    PluginManager.registerCommand(pluginName, "SwapEquip", () => {
        const actor = $gameParty.members()[0]; // 先頭アクターを取得
        if (!actor) return;

        const actorId = actor.actorId();
        const isDualWield = $dataActors[actorId].traits.some(trait => trait.code === 55);
        if (!isDualWield) return; // 二刀流でなければ処理しない

        let slotResult = 0;
        const slot1 = actor.equips()[0];
        const slot2 = actor.equips()[1];

        if (slot1) {
            actor.changeEquipById(1, 0);
            if(slot2){
                actor.changeEquipById(2, 0);
                actor.changeEquipById(1, slot2.id);
                actor.changeEquipById(2, slot1.id);
                slotResult = 1;
            }else{
                actor.changeEquipById(2, slot1.id);
                slotResult = 2;
            }
        }else if(slot2){
            actor.changeEquipById(2, 0);
            actor.changeEquipById(1, slot2.id);
            slotResult = 3;
        }

        $gameVariables.setValue(change_result,slotResult);
    });

    const _Game_Actor_paramPlus = Game_Actor.prototype.paramPlus;
    Game_Actor.prototype.paramPlus = function(paramId) {
        let value = _Game_Actor_paramPlus.call(this, paramId);
        if (this.isDualWield()) {
            const secondWeapon = this.equips()[1];
            if (secondWeapon && secondWeapon.etypeId === 1) {
                const baseWeapon = $dataWeapons[secondWeapon.id]; // **DBの武器データを取得**
                value -= baseWeapon.params[paramId]; // **元々の武器設定値のみ減算**
            }
        }
        return value;
    };

    const _Game_Actor_xparam = Game_Actor.prototype.xparam;
    Game_Actor.prototype.xparam = function(xparamId) {
        let value = _Game_Actor_xparam.call(this, xparamId);
        if (this.isDualWield()) {
            const secondWeapon = this.equips()[1];
            if (secondWeapon && secondWeapon.etypeId === 1) {
                const baseWeapon = $dataWeapons[secondWeapon.id]; // **DBの武器データ**
                value -= baseWeapon.traits.filter(t => t.code === 22 && t.dataId === xparamId)
                                          .reduce((sum, t) => sum + t.value, 0); // **DB登録済みの特徴のみ減算**
            }
        }
        return value;
    };

    const _Game_Actor_sparam = Game_Actor.prototype.sparam;
    Game_Actor.prototype.sparam = function(sparamId) {
        let value = _Game_Actor_sparam.call(this, sparamId);
        if (this.isDualWield()) {
            const secondWeapon = this.equips()[1];
            if (secondWeapon && secondWeapon.etypeId === 1) {
                const baseWeapon = $dataWeapons[secondWeapon.id]; // **DBの武器データ**
                value -= baseWeapon.traits.filter(t => t.code === 23 && t.dataId === sparamId)
                                          .reduce((sum, t) => sum + t.value, 0); // **DB登録済みの特徴のみ減算**
            }
        }
        return value;
    };

    const _Window_EquipSlot_drawItem = Window_EquipSlot.prototype.drawItem;
    Window_EquipSlot.prototype.drawItem = function(index) {
        const rect = this.itemRect(index);
        this.changeTextColor(ColorManager.systemColor());

        const actor = this._actor;
        const item = actor ? actor.equips()[index] : null;
        const etypeId = actor ? actor.equipSlots()[index] : null;

        // 装備タイプ名を取得
        let equipTypeName = etypeId ? $dataSystem.equipTypes[etypeId] : "";

        // 武器スロットの判定（二刀流対応）
        if (etypeId === 1) {
            if (index === 0) {
                equipTypeName = "武器";
            } else {
                equipTypeName = "武器2";
            }
        }

        this.drawText(equipTypeName, rect.x, rect.y, rect.width / 2);
        this.resetTextColor();

        // 装備品がある場合、アイコンと名前を表示
        if (item) {
            const iconX = rect.x + rect.width / 6; // アイコンの開始位置
            const nameX = iconX + 32; // アイコンの横幅を考慮し、名前の開始位置を調整

            this.drawIcon(item.iconIndex, iconX, rect.y); // アイコン描画
            this.drawText(item.name, nameX, rect.y, rect.width / 2 - 32);
        }
    };

})();