//=============================================================================
// AddPlugin_StateDisplayTarget.js
// バージョン: 1.0.0
//=============================================================================
// Copyright (c) 2026 とりぬー
// Released under the MIT license
// http://opensource.org/licenses/mit-license.php
//=============================================================================

/*:ja
 * @target MZ
 * @plugindesc マップにステートを表示するプラグイン
 * @author とりぬー
 * 
 * @help
 * 1.アクターが状態異常にかかっているとき、
 *   アクターの頭上にアイコンを表示します。
 *   この機能はプラグインコマンドの「アクターの頭上表示スイッチ」で
 *   ON,OFFが出来ます。デフォルトはONです。
 * 
 * 2.イベントIDを指定して、ステートアイコンを表示出来ます。
 *   この機能はプラグインコマンドの「ステートを表示するイベントIDの設定」で
 *   設定出来ます。
 *   ターン数は空でも問題ありません。アイコンの上に数字を表示したい時に
 *   使用してください。
 *   
 *   設定したイベントIDに一括でアイコン消去も出来ます。
 *   プラグインコマンドの「ステートを表示するイベントIDを全て消去」で可能です。
 * 
 * 
 * @command setActorStateDisplay
 * @text アクターの頭上表示スイッチの切り替え
 * @desc アクターの頭上表示スイッチの切り替え
 * 
 * @arg actorFrg
 * @type boolean
 * @desc アクターの頭上表示スイッチ 表示する: true / 表示しない:false
 * @default true
 * 
 * @command targetEventIdClear
 * @text ステートを表示するイベントIDを全て消去
 * @desc ステートを表示するイベントIDを全て消去
 * 
 * @command targetEventIdSet
 * @text ステートを表示するイベントIDの設定
 * @desc ステートを表示するイベントIDの設定
 *
 * @arg textEventId
 * @type targetEventId
 * @text イベントID
 * @desc ステートを表示するイベントIDの設定をします。
 *       例：ひとつの場合、[1]　複数の場合、[1,2,3] など
 * 
 * @arg textStateId
 * @type targetStateId
 * @text ステートID
 * @desc 表示するステートIDの設定をします。
 *       例：ひとつの場合、[1]　複数の場合、[1,2,3] 非表示の場合、[] など
 * 
 * @arg textTurnId
 * @type targetTurnId
 * @text ターン数
 * @desc 表示するステートIDの上に数字を表示します。
 *       例：ひとつの場合、[1]　複数の場合、[1,2,3] 非表示の場合、[] など
 * 
 * @command notSplitePlugin
 * @text スプライト更新処理を切り替える
 * @desc trueにすると、毎フレーム更新される処理を一時的に停止する。
 *       処理が重い場合使用。
 *
 * @arg notPlugin
 * @type notPlugin
 * @text スプライト更新処理を切り替える。
 * @default true
 * @type boolean
 * 
 * @command clearMapStateSprites
 * @text スプライトの完全破棄、メモリリーク対策
 * @desc 保持したままでは重いので適宜呼び出し推奨。
 *       スプライトを使用しないマップや場所移動前に。
 * 
 * @param actorStateVariable
 * @text アクターのステータスが入る配列変数
 * @desc アクターのステータスが入る変数を設定します。
 *       AddPlugin_StateRLMoveの同じ項目と合わせると連携出来ます。
 * @default 1
 * @type variable
 * 
 */

(function() {
    "use strict";
    
    const pluginName = document.currentScript.src.match(/^.*\/(.*).js$/)[1];
    let parameters = PluginManager.parameters(pluginName);

    let event_state_map = new Map();
    let event_state_turn_map = new Map();
    let actor_state_display_enabled = true;
    let actor_state_variable = Number(parameters["actorStateVariable"]);
    let actor_state_not = false;

    PluginManager.registerCommand(pluginName, "setActorStateDisplay", args => {
        actor_state_display_enabled = Boolean(parseArray(args.actorFrg));
        const characterSprites = SceneManager._scene._spriteset._characterSprites;
        characterSprites.forEach(sprite => {
            if (sprite._character) {
                sprite.setCharacter(sprite._character);
            }
        });
    });
    
    PluginManager.registerCommand(pluginName, "targetEventIdClear", () => {
        event_state_map.clear();
        event_state_turn_map.clear();
        updateEventStateIcons();
    });

    PluginManager.registerCommand(pluginName, "targetEventIdSet", args => {
        const newEventIds = parseArray(args.textEventId)
        const newStateIds = parseArray(args.textStateId);
        const newTurnIds = parseArray(args.textTurnId);

        newEventIds.forEach(eventId => {
            event_state_map.set(eventId, newStateIds);
            event_state_turn_map.set(eventId, newTurnIds);
        });
        updateEventStateIcons();
    });

    PluginManager.registerCommand(pluginName, "notSplitePlugin", args => {
        actor_state_not = Boolean(parseArray(args.notPlugin));
    });

    PluginManager.registerCommand(pluginName, "clearMapStateSprites", () => {
        clearMapStateSprites();
        updateEventStateIcons();
    });

    function parseArray(arg) {
        return JSON.parse(arg || '[]');
    }

    function updateEventStateIcons() {
        const scene = SceneManager._scene;
        if (!scene || !scene._spriteset) return;

        const characterSprites = scene._spriteset._characterSprites;
        characterSprites.forEach(sprite => {
            const mapState = sprite._mapStateSprite;
            if (!mapState) return;

            let setupParams;
            const char = sprite._character;
            if (!(char instanceof Game_Event || char instanceof Game_Player)) return;
            
            if (char instanceof Game_Player) {
                setupParams = {
                    states: actor_state_display_enabled
                        ? () => $gameParty.leader().states().sort((a, b) => a.id - b.id)
                        : () => [],
                    turns: actor_state_display_enabled
                        ? () => $gameVariables.value(actor_state_variable)
                        : () => []
                };
            } else if (char instanceof Game_Event) {
                const id = char.eventId();
                setupParams = event_state_map.has(id)
                    ? {
                        states: () => (event_state_map.get(id) || [])
                            .map(i => $dataStates[i]).sort((a, b) => a.id - b.id),
                        turns: () => event_state_turn_map.get(id) || []
                    }
                    : { states: () => [], turns: () => [] };
            }

            mapState.setup(setupParams);
            mapState.updateIcon();
        });
    }

    function clearMapStateSprites() {
        const scene = SceneManager._scene;
        if (!scene || !scene._spriteset) return;

        const characterSprites = scene._spriteset._characterSprites;
        if (!characterSprites) return;

        characterSprites.forEach(sprite => {
            const mapState = sprite._mapStateSprite;
            if (mapState) {
                if (Array.isArray(mapState._iconSprites)) {
                    mapState._iconSprites.forEach(s => {
                        if (s) {
                            s.destroy({ children: true });
                        }
                    });
                    mapState._iconSprites = [];
                }

                if (Array.isArray(mapState._turnSprites)) {
                    mapState._turnSprites.forEach(s => {
                        if (s) {
                            s.destroy({ children: true });
                        }
                    });
                    mapState._turnSprites = [];
                }

                mapState.removeChildren();
                mapState.destroy({ children: true });
            }

            sprite._mapStateSprite = null;
        });
        event_state_map.clear();
        event_state_turn_map.clear();
        
    }

    let _Sprite_Character_setCharacter = Sprite_Character.prototype.setCharacter;
    Sprite_Character.prototype.setCharacter = function(character) {
        _Sprite_Character_setCharacter.call(this, character);
        if (!this._mapStateSprite) {
            this._mapStateSprite = new Sprite_MapState();
            this.addChild(this._mapStateSprite);
        }
    };
    
    function Sprite_MapState() {
        Sprite_StateIcon.call(this);
        this._iconSprites = [];
        this._turnSprites = [];
    }

    Sprite_MapState.prototype = Object.create(Sprite_StateIcon.prototype);
    Sprite_MapState.prototype.constructor = Sprite_MapState;

    Sprite_MapState.prototype.update = function() {
        if (actor_state_not) return;
        Sprite_StateIcon.prototype.update.call(this);
    };

    Sprite_MapState.prototype.updateIcon = function() {
        this.removeChildren();

        const states = this._battler ? this._battler.states() : [];
        let stateTurns = Array.isArray(this._battler?.turns?.()) ? this._battler.turns() : [];

        if (stateTurns.length > 0 && typeof stateTurns[0] === 'object') {
            stateTurns = stateTurns.map(item => item.turn);
        }

        this._iconSprites = [];
        this._turnSprites = [];

        const iconSpacing = 28;
        const startX = -(states.length - 1) * iconSpacing / 2;

        for (let i = 0; i < states.length; i++) {
            const state = states[i];
            const turn = stateTurns[i] ?? "";
            this.createIconSprite(state.iconIndex, startX + i * iconSpacing, turn);
        }
    };

    Sprite_MapState.prototype.createIconSprite = function(iconIndex, x, turn) {
        const sprite = new Sprite_StateIcon();
        sprite._iconIndex = iconIndex;
        sprite.updateFrame();
        sprite.x = x;
        sprite.y = -64;
        this.addChild(sprite);
        this._iconSprites.push(sprite);

        const turnSprite = new Sprite(new Bitmap(32, 32));
        turnSprite.bitmap.drawText(turn.toString(), 0, 0, 32, 32, 'center');
        turnSprite.x = sprite.x - 16;
        turnSprite.y = sprite.y - 30;
        this.addChild(turnSprite);
        this._turnSprites.push(turnSprite);
    };

})();