//=============================================================================
// AddPlugin_StateRLMove - ステートの状態を変数で管理する
// バージョン: 1.0.0
//=============================================================================
// Copyright (c) 2026 とりぬー
// Released under the MIT license
// http://opensource.org/licenses/mit-license.php
//=============================================================================

/*:
 * @target MZ
 * @plugindesc ステートの状態を変数で管理するプラグイン
 * @author とりぬー
 * 
 * @help
 * アクターやイベントのステートを変数で管理するプラグインです。
 * プラグインコマンドを呼び出して、使用します。
 * (製作者が定めたターン処理ごとにプラグインコマンドが呼び出されるイメージです)
 * 
 * 管理出来るアクターは先頭の1人までです。
 * 
 * 以下の3~4はイベントに対してステート処理出来ますが、処理に条件があります。
 * 条件1
 *   イベントそれぞれの状態異常はセルフ変数で管理してください。
 *   セルフ変数を使用出来るプラグインとの併用を想定しています。
 * 条件2
 *   セルフ変数の都合上、このプラグインを呼び出したイベントのセルフ変数限定で
 *   処理が可能です。
 * 
 * 1.プラグインコマンド「先頭のアクターにステートを与える」で、
 *   先頭のアクターにステートを付加します。
 *   その際、プラグインで指定した変数に、ステートIDとステート継続ターンの最小値が配列に入ります。
 * 
 * 2.プラグインコマンド「アクターのステート継続ターン数を-1」で、
 *   先頭のアクターのステート継続ターン数を-1します。
 *   その際ターン数が0になったらステートを解除します。
 * 
 * 3.プラグインコマンド「イベントにステートを与える」で、
 *   このコマンドを呼び出したイベントに対してステートを与えます。
 *   その際、プラグインで指定した変数に、ステートIDとステート継続ターンの最小値が配列に入ります。
 *   プラグイン連携がONの場合、画面に状態を反映します。
 * 
 * 4.プラグインコマンド「イベントのステート継続ターン数を-1」で、
 *   このコマンドを呼び出したイベントのステート継続ターン数を-1します。
 *   その際ターン数が0になったらステートを解除します。
 *   プラグイン連携がONの場合、画面に状態を反映します。
 * 
 * 5.プラグインコマンド「先頭アクターの指定ステート解除」で、
 *   現在アクターがかかっているステートを解除します。
 * 
 * 6.プラグインコマンド「イベントの指定ステート解除」で、
 *   このコマンドを呼び出したイベントのステートを解除します。
 * 
 * 7.プラグインコマンド「指定したアクターのステート確率算出」で、
 *   指定したアクターIDの指定アクターの装備一覧、職業、かかっているステートを
 *   取得し、そのステートにかかる確率を指定した変数に返します。
 *   他のコマンドは耐性に関わらず付与するので、
 *   このコマンドで確率を算出した結果から、条件分岐を用いることを想定しています。
 * 
 * 8.プラグインコマンド「アクターステートを元に調整」で、
 *   アクターのかかっているステートと変数内容に差異がある場合以下のように調整します。
 *   ・アクターがかかっているステートに変数を合わせます。
 *     例えば、アクターの毒が装備のステート無効によって解除された場合、
 *     このコマンドを呼び出すと変数からも毒の情報を削除します。
 *     また、アクターの毒が何かの処理によって増えた場合、
 *     このコマンドを呼び出すと変数に毒の情報を追加します。
 *     その際のターン数はそのステートデータベースの最小ターンです。
 * 
 * @param actorStateVariable
 * @text アクターのステータスが入る配列変数
 * @desc アクターのステータスが入る変数を設定します。
 *       デフォルトは1です。
 * @default 1
 * @type variable
 * 
 * @param eventStateVariable
 * @text イベントのステータスが入る配列変数
 * @desc イベントのステータスが入る変数を設定します。
 *       デフォルトは2です。
 * @default 2
 * @type variable
 * 
 * @param addTurnVariable
 * @text ステート追加ターン
 * @desc ステートを与える時、ステートDBの最小ターンで与えますが、
 *       更に値を追加したい時、変数に入っている値だけ追加します。
 * @default 0
 * @type variable
 * 
 * @param pluginAutoStateUpdateFlag
 * @text 自動でイベントのアイコン表示をするか
 * @desc AddPlugin_StateDisplayTarget.jsの機能と連携して
 *       自動でステート表示を更新するか
 * @default false
 * @type boolean
 * 
 * @command stateActorGive
 * @text 先頭のアクターにステートを与える
 * @desc 先頭のアクターにステートを与える
 * 
 * @arg targetActorStateId
 * @type targetActorStateId
 * @text ステートID
 * @desc 先頭のアクターに与えるステートIDの設定をします。
 *       例：配列の場合[1、2]　数字だとゲーム変数で判定します。
 * 
 * @command stateActorMove
 * @text アクターのステート継続ターン数を-1
 * @desc アクターのステート継続ターン数を-1
 * 
 * @command stateEventGive
 * @text イベントにステートを与える
 * @desc イベントにステートを与える
 * 
 * @arg targetEventStateId
 * @type targetEventStateId
 * @text ステートID
 * @desc イベントに与えるステートIDの設定をします。
 *       例：配列の場合[1、2]　数字だとゲーム変数で判定します。
 * 
 * @command stateEventMove
 * @text イベントのステート継続ターン数を-1
 * @desc イベントのステート継続ターン数を-1
 * 
 * @command stateActorRefresh
 * @text 先頭アクターの指定ステート解除
 * @desc 先頭アクターの指定ステート解除
 * 
 * @arg targetActorRefreshStateId
 * @type targetActorRefreshStateId
 * @text 解除ステートID
 * @desc 先頭アクターの解除するステートIDの設定をします。
 *       例：配列の場合[1、2]　数字だとゲーム変数で判定します。
 * 
 * @command stateEventRefresh
 * @text イベントの指定ステート解除
 * @desc イベントの指定ステート解除
 * 
 * @arg targetEventRefreshStateId
 * @type targetEventRefreshStateId
 * @text 解除ステートID
 * @desc イベントの解除するステートIDの設定をします。
 *       例：配列の場合[1、2]　数字だとゲーム変数で判定します。
 * 
 * @command stateActorPercent
 * @text 指定したアクターのステート確率算出
 * @desc 指定したアクターのステート確率算出
 * 
 * @arg targetActorId
 * @type targetActorId
 * @text アクターID
 * @desc 算出したいアクターIDを設定します。一人まで対応です。
 *       例：配列の場合[1]　数字だとゲーム変数で判定します。
 * 
 * @arg targetStateId
 * @type targetStateId
 * @text ステートID
 * @desc 算出したいステートIDを設定をします。一つまで対応です。
 *       例：配列の場合[1]　数字だとゲーム変数で判定します。
 * 
 * @arg returnVariable
 * @type returnVariable
 * @text ステートの確率を格納する変数
 * @desc ステートの確率を格納する変数を指定します。
 *       デフォルトは3です。0~100以上で返ってくる想定です。
 * @default 3
 * @type variable
 * 
 * @command stateActorAdjust
 * @text アクターステートを元に調整
 * @desc アクターステートを元に調整
 * 
 */
(() => {  
    "use strict";
    const pluginName = document.currentScript.src.match(/^.*\/(.*).js$/)[1];
    let parameters = PluginManager.parameters(pluginName);

    let actor_state_variable = Number(parameters["actorStateVariable"]);
    let event_state_variable = Number(parameters["eventStateVariable"]);
    let add_turn_variable = Number(parameters["addTurnVariable"]);
    let auto_state_update_flg = Boolean(parameters["pluginAutoStateUpdateFlag"] === 'true' || false);
    let state_turn_limit = 999; //ステートの限界ターン数

    //1
    PluginManager.registerCommand(pluginName, "stateActorGive", args => {
        let target_state_id = args.targetActorStateId ? JSON.parse(args.targetActorStateId) :[];
        if (typeof target_state_id === 'number') {
            //数字の場合はゲーム変数から取得
            target_state_id = $gameVariables.value(Number(JSON.parse(args.targetActorStateId)));
        }
        
        if(!Array.isArray(target_state_id)){
            target_state_id = [];
        }
        //最小ターンが未設定の状態異常は省く
        let minTurnEach = [];
        target_state_id.forEach(id => {
            if ($dataStates[id] && typeof $dataStates[id].minTurns !== 'undefined') {
                minTurnEach.push(id);
            }
        });
        minTurnEach.sort((a, b) => a.id - b.id);
        minTurnEach.forEach(minEach => {
            stateActorGive(minEach);
        });
    });

    //2
    PluginManager.registerCommand(pluginName, "stateActorMove", () => {
        stateActorMove();
    });

    //3
    PluginManager.registerCommand(pluginName, "stateEventGive", args => {
        let target_state_id = args.targetEventStateId ? JSON.parse(args.targetEventStateId) :[];
        if (typeof target_state_id === 'number') {
            //数字の場合はゲーム変数から取得
            target_state_id = $gameVariables.value(Number(JSON.parse(args.targetEventStateId)));
        }
        
        if(!Array.isArray(target_state_id)){
            target_state_id = [];
        }
        //最小ターンが未設定の状態異常は省く
        let minTurnEach = [];
        target_state_id.forEach(id => {
            if ($dataStates[id] && typeof $dataStates[id].minTurns !== 'undefined') {
                minTurnEach.push(id);
            }
        });
        minTurnEach.sort((a, b) => a.id - b.id);
        minTurnEach.forEach(minEach => {
            stateEventGive(minEach);
        });
    });

    //4
    PluginManager.registerCommand(pluginName, "stateEventMove", () => {
        stateEventMove();
    });

    //5
    PluginManager.registerCommand(pluginName, "stateActorRefresh", args => {
        let target_state_refresh_id = args.targetActorRefreshStateId ? JSON.parse(args.targetActorRefreshStateId) :[];
        if (typeof target_state_refresh_id === 'number') {
            //数字の場合はゲーム変数から取得
            target_state_refresh_id = $gameVariables.value(Number(JSON.parse(args.targetActorRefreshStateId)));
        }
        
        if(!Array.isArray(target_state_refresh_id)){
            target_state_refresh_id = [];
        }
        //最小ターンが未設定の状態異常は省く
        let minTurnEach = [];
        target_state_refresh_id.forEach(id => {
            if ($dataStates[id] && typeof $dataStates[id].minTurns !== 'undefined') {
                minTurnEach.push(id);
            }
        });
        minTurnEach.sort((a, b) => a.id - b.id);
        stateActorRefresh(minTurnEach);
    });

    //6
    PluginManager.registerCommand(pluginName, "stateEventRefresh", args => {
        let target_state_refresh_id = args.targetEventRefreshStateId ? JSON.parse(args.targetEventRefreshStateId) :[];
        if (typeof target_state_refresh_id === 'number') {
            //数字の場合はゲーム変数から取得
            target_state_refresh_id = $gameVariables.value(Number(JSON.parse(args.targetEventRefreshStateId)));
        }
        
        if(!Array.isArray(target_state_refresh_id)){
            target_state_refresh_id = [];
        }
        //最小ターンが未設定の状態異常は省く
        let minTurnEach = [];
        target_state_refresh_id.forEach(id => {
            if ($dataStates[id] && typeof $dataStates[id].minTurns !== 'undefined') {
                minTurnEach.push(id);
            }
        });
        minTurnEach.sort((a, b) => a.id - b.id);
        stateEventRefresh(minTurnEach);
    });

    //7
    PluginManager.registerCommand(pluginName, "stateActorPercent", args => {
        let target_actor_id = args.targetActorId ? JSON.parse(args.targetActorId) :[];
        if (typeof target_actor_id === 'number') {
            //数字の場合はゲーム変数から取得
            target_actor_id = $gameVariables.value(Number(JSON.parse(args.targetActorId)));
        }
        if(Array.isArray(target_actor_id)){
            target_actor_id = target_actor_id[0];
        }

        let target_state_id = args.targetStateId ? JSON.parse(args.targetStateId) :[];
        if (typeof target_state_id === 'number') {
            //数字の場合はゲーム変数から取得
            target_state_id = $gameVariables.value(Number(JSON.parse(args.targetStateId)));
        }
        if(Array.isArray(target_state_id)){
            target_state_id = target_state_id[0];
        }

        let return_variable = Number(args.returnVariable);

        //ステート有効度を取得
        const guardState13 = actorAllTraitsRet(target_actor_id,13);
        //ステート無効を取得
        const guardState14 = actorAllTraitsRet(target_actor_id,14);
        // 状態異常の確率
        let statePercent = 1.0;
        //ステート有効度の減算
        const value13 = guardState13.filter(state13 => state13.dataId === target_state_id);
        if(value13.length != 0 && 0 < value13[0].value){
            statePercent = statePercent * value13[0].value;
        }
        //ステート無効化の反映
        const value14 = guardState14.filter(state14 => state14.dataId === target_state_id);
        if(value14.length != 0 && 1 === value14[0].value){
            statePercent = 0;
        }
        $gameVariables.setValue(return_variable, statePercent * 100);
    });

    //8
    PluginManager.registerCommand(pluginName, "stateActorAdjust", () => {
        //・アクターステートを主軸とする。(actorStates)
        //  現在格納変数(currentStates)と同じIDがあれば、turn数はステートDB取得せず
        //  現在格納変数にあるturn数と同じ数を入れる 
        //・アクターステートだけにidがある場合は新しいステートのminTurnsをステートDB取得
        //・現在格納変数のみにあるステータスIDは削除する

        let actor = $gameParty.members()[0];
        let actorStates = actor._states.slice();
        actorStates.sort((a, b) => a - b);

        let previousStates = $gameVariables.value(actor_state_variable) || [];
        let previousStateMap = {};
        for (let state of previousStates) {
            previousStateMap[state.id] = state.turn;
        }

        let currentStates = actorStates.map(id => {
            return {
                id: id,
                turn: previousStateMap.hasOwnProperty(id) ? previousStateMap[id] : $dataStates[id].minTurns
            };
        });

        // 結果を変数に保存
        $gameVariables.setValue(actor_state_variable, currentStates);

        //プラグイン連携機能
        if(auto_state_update_flg){
            pluginStateDisplay()
        }
        
    });

    function stateActorGive(minEach) {
        // 先頭のアクターID
        let actorId = $gameParty.members()[0].actorId();
        let addTurn = 0;
        if(add_turn_variable !=0){
            addTurn = $gameVariables.value(add_turn_variable);
        }
        // 追加または更新したいステート
        let newState = { id: minEach, turn: $dataStates[minEach].minTurns + addTurn };
        // 変数内のステートを取得
        let currentStates = $gameVariables.value(actor_state_variable) || [];

        // ステートIDが既に存在するか確認し、存在する場合はターン数を更新し、存在しない場合は追加する
        if (Array.isArray(currentStates)) {
            let existingState = currentStates.find(state => state.id === newState.id);
            if (existingState) {
                existingState.turn = existingState.turn + newState.turn;  // ターン数を更新
                if(existingState.turn > state_turn_limit){
                    existingState.turn = state_turn_limit;
                }
            } else {
                $gameActors.actor(actorId).addState(newState.id); // ステートの付加
                currentStates.push(newState);  // 新しいステートを追加
            }
            currentStates.sort((a, b) => a.id - b.id);
            // 更新されたステートをゲーム変数に設定
            $gameVariables.setValue(actor_state_variable, currentStates);
        } else {
            console.error("currentStates が配列ではありません");
        }
    }

    function stateActorMove() {
        let actorId = $gameParty.members()[0].actorId(); //先頭のアクター取得
        let currentStates = $gameVariables.value(actor_state_variable) || [];
        let removedStates = []; // 0になったステートIDを格納する配列
        
        //念のため、計算できない文字列を持っていたら消去。計算不能のエラー対策
        currentStates = currentStates.filter(item => 
            typeof item.id !== "string" && typeof item.turn !== "string"
        );

        for (let i = 0; i < currentStates.length; i++) {
            currentStates[i].turn--;
            if (currentStates[i].turn === 0) {
                removedStates.push(currentStates[i]); // 0になったステートを追加
                currentStates.splice(i, 1); // 現在のステート配列から削除
                i--; // spliceでインデックスがずれるため、インデックスを調整
            }
        }
        
        // 更新されたステートをゲーム変数に設定
        $gameVariables.setValue(actor_state_variable, currentStates);
        
        // 状態異常の削除を実行
        removedStates.forEach(state => {
            $gameActors.actor(actorId).removeState(state.id); // アクターから状態異常を削除
        });
    }

    function stateEventGive(minEach) {
        // 追加または更新したいステート
        let addTurn = 0;
        if(add_turn_variable !=0){
            addTurn = $gameVariables.value(add_turn_variable);
        }
        let newState = { id: minEach, turn: $dataStates[minEach].minTurns + addTurn };
        // 変数内のステートを取得
        let currentStates = $gameVariables.value(event_state_variable) || [];

        // ステートIDが既に存在するか確認し、存在する場合はターン数を更新し、存在しない場合は追加する
        if (Array.isArray(currentStates)) {
            let existingState = currentStates.find(state => state.id === newState.id);
            if (existingState) {
                existingState.turn = existingState.turn + newState.turn;  // ターン数を更新
                if(existingState.turn > state_turn_limit){
                    existingState.turn = state_turn_limit;
                }
            } else {
                currentStates.push(newState);  // 新しいステートを追加
            }
            currentStates.sort((a, b) => a.id - b.id);
            // 更新されたステートをゲーム変数に設定
            $gameVariables.setValue(event_state_variable, currentStates);
        } else {
            console.error("currentStates が配列ではありません");
        }
        
        //プラグイン連携機能
        if(auto_state_update_flg){
            pluginStateDisplay()
        }
    }

    function stateEventMove() {
        let currentStates = $gameVariables.value(event_state_variable) || [];
        let removedStates = []; // 0になったステートIDを格納する配列

        //念のため、計算できない文字列を持っていたら消去。計算不能のエラー対策
        currentStates = currentStates.filter(item => 
            typeof item.id !== "string" && typeof item.turn !== "string"
        );
        
        for (let i = 0; i < currentStates.length; i++) {
            currentStates[i].turn--;
            if (currentStates[i].turn === 0) {
                removedStates.push(currentStates[i]); // 0になったステートを追加
                currentStates.splice(i, 1); // 現在のステート配列から削除
                i--; // spliceでインデックスがずれるため、インデックスを調整
            }
        }
        
        // 更新されたステートをゲーム変数に設定
        $gameVariables.setValue(event_state_variable, currentStates);

        //プラグイン連携機能
        if(auto_state_update_flg){
            pluginStateDisplay()
        }
    }

    function stateActorRefresh(minTurnEach){
        // 先頭のアクターID
        let actorId = $gameParty.members()[0].actorId();
        // 変数内のステートを取得
        let currentStates = $gameVariables.value(actor_state_variable) || [];
        // 削除したいIDのリスト
        let targetIds = minTurnEach; 

        // 配列であることを確認し、処理を実行
        if (Array.isArray(currentStates)) {
            // targetIdsに含まれていない要素だけを残す
            currentStates = currentStates.filter(item => !targetIds.includes(item.id));
        }
        targetIds.forEach(id => {
            $gameActors.actor(actorId).removeState(id);
        });

        $gameVariables.setValue(actor_state_variable,currentStates);
    }

    function stateEventRefresh(minTurnEach){
        // 変数内のステートを取得
        let currentStates = $gameVariables.value(event_state_variable) || [];
        // 削除したいIDのリスト
        let targetIds = minTurnEach; 

        // 配列であることを確認し、処理を実行
        if (Array.isArray(currentStates)) {
            // targetIdsに含まれていない要素だけを残す
            currentStates = currentStates.filter(item => !targetIds.includes(item.id));
        }

        $gameVariables.setValue(event_state_variable,currentStates);

        //プラグイン連携機能
        if(auto_state_update_flg){
            pluginStateDisplay()
        }
    }

    function pluginStateDisplay() {
        let statelArray = $gameVariables.value(event_state_variable);
        
        let eventId = [];
        eventId.push($gameMap._interpreter._eventId);

        let idArray = [];
        if(statelArray.length == 0){
            idArray = [];
        }else{
            for (let i = 0; i < statelArray.length; i++) {
                idArray.push(statelArray[i].id)
            }
        }
        
        //状態異常の削除は呼び出し元のイベントIDに対して実施される事が前提
        const args = {textEventId: JSON.stringify(eventId) , textStateId: JSON.stringify(idArray), textTurnId: JSON.stringify(statelArray)};
        PluginManager.callCommand($gameMap._interpreter,"AddPlugin_StateDisplayTarget","targetEventIdSet",args);
    }
})();