//=============================================================================
// AddPlugin_WeaponsAreaChecker - 攻撃範囲を表示するプラグイン
// バージョン: 1.0.0
//=============================================================================
// Copyright (c) 2026 とりぬー
// Released under the MIT license
// http://opensource.org/licenses/mit-license.php
//=============================================================================

/*:
 * @target MZ
 * @plugindesc 攻撃範囲を表示するプラグイン
 * @author とりぬー
 * 
 * @help
 * 指定した武器の情報を用いて攻撃範囲を表示するプラグインです。
 * プラグインコマンドを呼び出して、使用します。
 * 
 * 武器のメモ欄に以下のタグを設定する事が前提です。
 * 武器の前方範囲を決めるタグ　　　
 * →　<attackRangeForward>
 * 武器の前方斜め範囲を決めるタグ　
 * →　<attackRangeDiagonal>
 * 武器が貫通(一番近い敵に当たったら処理を止めるかどうか)するかを決めるタグ　
 * →　<attackStop>
 * 
 * プレイヤーの先頭アクターが武器を持っていない場合、
 * DBの2番目の武器を参照します。
 * 
 * 攻撃範囲の座標のみ取得して、他の処理に使用したい場合は以下のスクリプトを使用します。
 * 使用例(スクリプト):
 * let positions = updatePositions(); //攻撃範囲の座標を取得する。
 *
 * @command showPassableTiles
 * @text 攻撃範囲を表示する
 * @desc 攻撃範囲を表示する
 * 
 * @command hidePassableTiles
 * @text 攻撃範囲を非表示にする
 * @desc 攻撃範囲を非表示にする
 * 
 * @param 色
 * @type string
 * @default blue
 * @desc 色を設定します。
 * カラー名やカラーコードが指定できます。
 *
 * @param 不透明度
 * @type string
 * @default 0.5
 * @desc 色の不透明度を設定します。
 * 0.0（透明）～1.0（不透明）で指定できます。
 * 
 */

(() => {  
    "use strict";
    const pluginName = document.currentScript.src.match(/^.*\/(.*).js$/)[1];
    let parameters = PluginManager.parameters(pluginName);

    let userParameter_color = parameters['色'];
    let userParameter_alpha = Number(parameters['不透明度']);

    PluginManager.registerCommand(pluginName, "showPassableTiles", () => {
        showPassableTiles();
    });

    PluginManager.registerCommand(pluginName, "hidePassableTiles", () => {
        hidePassableTiles();
    });

    class Sprite_Passable extends Sprite {
        constructor(x, y) {
            super();
            this._character = new Game_CharacterBase();
            this._character.setPosition(x, y);
            this.bitmap = new Bitmap($gameMap.tileWidth(), $gameMap.tileHeight());
            let rect = this.getRect();
            this.bitmap.fillRect(rect.x, rect.y, rect.width, rect.height, userParameter_color);
            this.alpha = userParameter_alpha;
            this.updatePosition();
        }

        getRect() {
            // 区切りを作る
            // 矩形の初期サイズ（タイルサイズそのもの）
            let rect = { 
                x: 0, 
                y: 0, 
                width: $gameMap.tileWidth(), 
                height: $gameMap.tileHeight() 
            };
    
            rect.x += 1; // 左側の区切り
            rect.y += 1; // 上側の区切り
            rect.width -= 2; // 両側で幅を縮める
            rect.height -= 2; // 両側で高さを縮める
    
            return rect; // 調整後の矩形を返す
        }

        update() {
            this.updatePosition();
            super.update();
        }

        updatePosition() {
            this.x = this._character.screenX() - $gameMap.tileWidth() / 2;
            this.y = this._character.screenY() - $gameMap.tileHeight() + this._character.shiftY();
        }
    }

    let __passableSpriteArea = null;

    function showPassableTiles() {
        let positions = updatePositions();
        if (!__passableSpriteArea) {
            __passableSpriteArea = new Sprite();
        } else {
            __passableSpriteArea.removeChildren();
        }

        if (Array.isArray(positions)) {
            for (let i = 0; i < positions.length; i++) {
                let tile = positions[i];
                let px = tile.x;
                let py = tile.y;
                __passableSpriteArea.addChild(new Sprite_Passable(px, py));
            }
            SceneManager._scene._spriteset.addChild(__passableSpriteArea);
        } else {
            console.error('変数のデータが配列ではありません');
        }
    }

    function hidePassableTiles() {
        if (__passableSpriteArea) {
            SceneManager._scene._spriteset.removeChild(__passableSpriteArea);
            __passableSpriteArea = null;
        }
    }

    window.updatePositions = function() {
        let d = $gamePlayer.direction();
        let x = $gamePlayer.x;
        let y = $gamePlayer.y;
        let positions = [];
    
        let allEquips = $gameParty.members()[0].equips();

        // 武器切り替えシステムプラグインが導入されている場合
        if(PluginManager._scripts.includes("AddPlugin_SecondWeaponChange")){
            // `$dataActors` からアクターの特性を取得
            const isDualWield = $dataActors[$gameParty.members()[0].actorId()].traits.some(trait => trait.code === 55);
            // equips() から二刀流のときだけ 2番目の装備を削除
            allEquips = allEquips.filter((item, index) => {
                return !(isDualWield && index === 1); // 二刀流なら2番目の装備を取り除く
            });
        }

        let distance1 = 0;
        let distance2 = 0;
        let distance3 = 0;

        if (allEquips) {
            for (let equip of allEquips) {
                if (equip && equip.meta) {
                    distance1 += parseInt(equip.meta.attackRangeForward || 0, 10);
                    distance2 += parseInt(equip.meta.attackRangeDiagonal || 0, 10);
                    distance3 += parseInt(equip.meta.attackStop || 0, 10);
                }
            }
        } else {
            // 装備がない場合、デフォルト武器（ID2）を使用
            distance1 = parseInt($dataWeapons[2].meta.attackRangeForward, 10);
            distance2 = parseInt($dataWeapons[2].meta.attackRangeDiagonal, 10);
            distance3 = parseInt($dataWeapons[2].meta.attackStop, 10);
        }
    
        const directions = {
            2: { dx: 0, dy: 1 },   // 下
            4: { dx: -1, dy: 0 },  // 左
            6: { dx: 1, dy: 0 },   // 右
            8: { dx: 0, dy: -1 }   // 上
        };
    
        if (directions[d]) {
            // 前方
            for (let i = 1; i <= distance1; i++) {
                let checkX1 = x + directions[d].dx * i;
                let checkY1 = y + directions[d].dy * i;
                if($gameMap.isAirshipLandOk(checkX1, checkY1) || distance3 != 0){
                    positions.push({ x: checkX1, y: checkY1 });
                }else{
                    break;
                }
            }

             // 斜め
            let breakCountSwitch1 = false;
            let breakCountSwitch2 = false;
            for (let i = 1; i <= distance2; i++) {
                if (distance2 > 0) {
                    if (d === 2 || d === 8) { 
                        let checkX2_1 = x - i;
                        let checkX2_2 = x + i;
                        let checkY2 = y + directions[d].dy * i;
                        if(($gameMap.isAirshipLandOk(checkX2_1, checkY2) || distance3 != 0) && !breakCountSwitch1){
                            positions.push({ x: checkX2_1, y: checkY2 }); // 斜め左
                        }else{
                            breakCountSwitch1 = true;
                        }
                        if(($gameMap.isAirshipLandOk(checkX2_2, checkY2) || distance3 != 0) && !breakCountSwitch2){
                            positions.push({ x: checkX2_2, y: checkY2 }); // 斜め右
                        }else{
                            breakCountSwitch2 = true;
                        }

                    } else {
                        let checkY3_1 = y - i;
                        let checkY3_2 = y + i;
                        let checkX3 = x + directions[d].dx * i;
                        if(($gameMap.isAirshipLandOk(checkX3, checkY3_1) || distance3 != 0) && !breakCountSwitch1){
                            positions.push({ x: checkX3, y: checkY3_1 }); // 斜め左
                        }else{
                            breakCountSwitch1 = true;
                        }
                        if(($gameMap.isAirshipLandOk(checkX3, checkY3_2) || distance3 != 0) && !breakCountSwitch2){
                            positions.push({ x: checkX3, y: checkY3_2 }); // 斜め右
                        }else{
                            breakCountSwitch2 = true;
                        }
                    }
                }
            }
            
        }
    
        return positions;
    }
    

})();