// ===================================================
// ARTM_TMGreedShopMZ
// Copyright (c) 2021 Artemis
// This software is released under the MIT license.
// http://opensource.org/licenses/mit-license.php
// ===================================================
// [Version]
// 1.0.0 初版
// 1.0.1 素材ウインドウの表示位置を調整
// 1.0.2 マウススクロール時の素材ウインドウ調整
//=============================================================================
// TMPlugin - 欲張りショップ
// バージョン: 2.2.0
// 最終更新日: 2018/07/17
// 配布元    : http://hikimoki.sakura.ne.jp/
//-----------------------------------------------------------------------------
// Copyright (c) 2016 tomoaky
// Released under the MIT license.
// http://opensource.org/licenses/mit-license.php
//=============================================================================
/*:ja
 * @target MZ
 * @plugindesc お金以外にアイテムも要求されるショップ機能を追加します。
 * @author Artemis
 *
 * @help ARTM_TMGreedShopMZ
 * tomoaky様作「欲張りショップ」プラグインのMZ移植版です。
 * 基本的な動作は変わっておりません。
 *
 * @command greedShop
 * @text 欲張りショップ
 * @desc 欲張りショップを開きます。
 *
 * @command setGreedCommand
 * @text コマンド名を変更
 * @desc 購入するコマンド名を変更します
 *
 * @arg commandName
 * @type string
 * @default
 * @text 変更するコマンド名
 *
 * @command changeShowPrice
 * @text プライス表示を変更
 * @desc 欲張りショップのプライス表示設定を変更します
 *
 * @arg flag
 * @type boolean
 * @default true
 * @text プライスを表示する
 *
 * @command changeSeGreedBuy
 * @text 購入時に鳴らす効果音を変更
 * @desc 欲張りショップの購入時に鳴らす効果音を変更します
 *
 * @arg seGreedBuy
 * @type struct<SoundEffect>
 * @desc 効果音
 * @default {"name":"Shop1","volume":"90","pitch":"100","pan":"0"}
 *
 * @param materialWindowPosition
 * @type select
 * @option 商品名の下
 * @value 0
 * @option 商品名の右
 * @value 1
 * @option 購入ウィンドウの下
 * @value 2
 * @desc 素材ウィンドウの表示位置
 * @default 0
 *
 * @param materialWindowWidth
 * @type number
 * @desc 素材ウィンドウの幅
 * 初期値: 408
 * @default 408
 *
 * @param buyWindowWidth
 * @type number
 * @desc 購入ウィンドウの幅
 * 初期値: 456
 * @default 456
 *
 * @param buyWindowHeight
 * @type number
 * @desc 購入ウィンドウの高さ
 * 0なら自動で決定
 * @default 0
 *
 * @param materialMax
 * @type number
 * @desc 設定できる素材の最大数。
 * 初期値: 5
 * @default 5
 *
 * @param fontRate
 * @type number
 * @decimals 2
 * @desc 素材情報のフォント倍率
 * @default 0.8
 *
 * @param greedCommand
 * @desc 欲張りショップの購入コマンド名。
 * 初期値: 購入する
 * @default 購入する
 *
 * @param needText
 * @desc 素材一覧のトップに表示するテキスト
 * ( 何も設定しなければ省略 )
 * @default 必要なもの
 *
 * @param showSellCommand
 * @type boolean
 * @desc 購入のみの場合にも売却コマンドを表示する。
 * @default true
 *
 * @param showMaterialWindow
 * @type boolean
 * @desc 素材ウィンドウを表示する。
 * @default true
 *
 * @param overlaid
 * @type boolean
 * @desc 素材ウィンドウを他のウィンドウと違うレイヤーに表示する。
 * 四隅が欠ける問題は解決しますが、可読性は低下します。
 * @default true
 *
 * @param backOpacity
 * @type number
 * @desc 素材ウィンドウの背景の不透明度
 * @default 192
 *
 * @param showMaterialFromNumberWindow
 * @type boolean
 * @desc 個数選択ウィンドウに素材情報を表示する。
 * @default true
 *
 * @param showPrice
 * @type boolean
 * @desc 商品ウィンドウに価格を表示する。
 * @default true
 *
 * @param seGreedBuy
 * @type struct<SoundEffect>
 * @desc 欲張りショップで購入時に鳴らす効果音
 * @default {"name":"Shop1","volume":"90","pitch":"100","pan":"0"}
 *
 * @help
 * TMPlugin - 欲張りショップ ver2.2.0
 *
 * 使い方:
 *
 *   アイテム、武器、防具にメモ欄タグ（後述）を使って素材を設定します。
 *
 *   イベントコマンド『プラグインコマンド』で greedShop を実行し、
 *   直後にイベントコマンド『ショップの処理』で、素材を設定したアイテムを
 *   商品として販売してください。
 *
 *   このプラグインは RPGツクールMV Version 1.6.1 で動作確認をしています。
 *
 *
 * プラグインコマンド:
 *
 *   greedShop
 *     このコマンドが実行された直後にショップの処理を
 *     実行することで欲張りショップになります。
 *
 *   greedCommand 買っちゃう
 *     欲張りショップの購入コマンド名を『買っちゃう』に変更します。
 *     この変更はセーブデータには保存されません。
 *
 *
 * メモ欄タグ（アイテム、武器、防具）:
 *
 *   <mat1:I1*3>
 *     お金以外にアイテム１番が３個必要になります。
 *     mat2, mat3... と素材を追加していくことができます。
 *     I の部分が W なら武器、A なら防具になります。
 *
 *   <matKey:1>
 *     mat1 タグに設定されている素材をキー素材として扱います。
 *     キー素材を所持していない場合、商品リストから除外されます。
 *     <matKey:1 2> というように半角スペースで区切り、複数のキー素材を
 *     設定することもできます。（この場合、mat1 と mat2 がキー素材になる）
 *     このタグを使うことで、レシピを所持していないと
 *     ショップに並ばない商品などを表現することができます。
 *
 *   <matG:50>
 *     価格を50に設定します、この設定は欲張りショップが
 *     有効になっている場合にのみ購入価格として反映されます。
 *
 *
 * メモ欄タグ（武器、防具）:
 *
 *   <noConsume>
 *     このタグを指定した武器、防具は素材として設定しても
 *     消費されなくなります。
 *
 *     消耗設定が『しない』になっているアイテムを素材にした場合、
 *     消耗しないが必要なものとして機能します。
 *
 *
 * ==============================================================
 * [追加機能]
 * ・グループ指定の機能を追加
 *   -> 素材指定時に 'G' を使うことでグループ指定が可能になります
 *  例: <mat1:G1*3> アイテムグループ１のアイテムを3個
 * ・パラメータ消費の機能を追加
 *   -> 素材指定時に 'P' を使うことで各種パラメータを消費します
 *  例: <mat1:P1*10> パラメータ１を10消費
 *  パラメータは以下に対応
 *    P1: 体力
 *
 * INAZUMA GAMES
 * ==============================================================
 */
/*~struct~SoundEffect:
 *
 * @param name
 * @type file
 * @dir audio/se/
 * @desc 効果音のファイル名
 * @default
 * @require 1
 *
 * @param volume
 * @type number
 * @max 100
 * @desc 効果音の音量
 * 初期値: 90
 * @default 90
 *
 * @param pitch
 * @type number
 * @min 50
 * @max 150
 * @desc 効果音のピッチ
 * 初期値: 100
 * @default 100
 *
 * @param pan
 * @type number
 * @min -100
 * @max 100
 * @desc 効果音の位相
 * 初期値: 0
 * @default 0
 *
 */
// @ts-ignore
var Imported = Imported || {};
Imported.TMGreedShop = true;
var ARTM;
(function (ARTM) {
    // NOTE:
    // [暫定] 欲張りショップで買うものは１回につき１つだけにする.
    ARTM.BuyLimitOne = true;
    ARTM.MaterialType = {
        Item: 'I',
        Weapon: 'W',
        Armor: 'A',
        Group: 'G',
        Param: 'P',
    };
    ARTM.Param = {
        materialMax: 5,
        materialWindowWidth: 408,
        needText: "",
        fontRate: 0.8,
        backOpacity: 192,
        showMaterialWindow: true,
    };
    ARTM.CharacterParamId = {
        Mp: 1,
    };
    ARTM.SelectMaterials = [];
    ARTM.Amount = 1;
    ARTM.SelectMaterialIndex = 0;
    ARTM.GREED_SHOP_DESC_TEXT_KEY = "GREED_SHOP_DESC";
    function isDataItem(item) {
        return ("note" in item);
    }
    ARTM.isDataItem = isDataItem;
    function setSelectMaterials(materials) {
        ARTM.SelectMaterials = [];
        materials.forEach(m => {
            ARTM.SelectMaterials.push(Object.assign({}, m));
        });
        // 同じアイテムを重複して選択しない
        const excludeList = ARTM.SelectMaterials.filter(m => m.type !== ARTM.MaterialType.Param)
            .filter(m => m.groupId <= 0)
            .map(m => m.id);
        ARTM.SelectMaterials.filter(m => m.groupId > 0).forEach(m => {
            const list = ARTM.allItems()
                .filter(data => Iz.Life.getItemGroups(data.id).find(group => group === m.groupId))
                // 最低1個は作れる
                .filter(data => ARTM.numItems(data.id) >= m.need)
                // 重複はしないように
                .filter(data => !excludeList.find(id => id === data.id))
                // 安い順で
                .sort((a, b) => a.price - b.price);
            if (list.length > 0) {
                m.id = list[0].id;
                excludeList.push(m.id);
            }
        });
    }
    ARTM.setSelectMaterials = setSelectMaterials;
    class Window_MaterialItemList extends Window_Selectable {
        get group() {
            return this._group;
        }
        initialize(rect, ...args) {
            super.initialize(rect, ...args);
            this._group = 0;
            this._itemList = [];
            this.hide();
        }
        setup(group, index) {
            this._group = group;
            this.makeItemList(index);
            this.refresh();
        }
        maxCols() {
            return 2;
        }
        colSpacing() {
            return 16;
        }
        maxItems() {
            return this._itemList.length;
        }
        makeItemList(index) {
            if (this._group <= 0) {
                this._itemList = [];
            }
            else {
                // 素材として既に選択されているものは除外
                const excludeList = ARTM.SelectMaterials.filter((m, i) => i !== index).filter(m => m.id > 0).map(m => m.id);
                const list = ARTM.allItems();
                this._itemList = list.filter(data => Iz.Life.getItemGroups(data.id).find(group => group === this._group))
                    .filter(data => !excludeList.find(id => id === data.id));
            }
        }
        itemAt(index) {
            return this._itemList[index];
        }
        item() {
            return this.itemAt(this.index());
        }
        numberWidth() {
            return this.textWidth("000");
        }
        isEnabled(item) {
            return ARTM.numItems(item.id) >= ARTM.SelectMaterials[ARTM.SelectMaterialIndex].need;
        }
        isCurrentItemEnabled() {
            return ARTM.numItems(this.item().id) >= ARTM.SelectMaterials[ARTM.SelectMaterialIndex].need;
        }
        drawItem(index) {
            const item = this.itemAt(index);
            if (item) {
                const numberWidth = this.numberWidth();
                const rect = this.itemLineRect(index);
                this.changePaintOpacity(this.isEnabled(item));
                this.drawItemName(item, rect.x, rect.y, rect.width - numberWidth);
                this.drawItemNumber(item, rect.x, rect.y, rect.width);
                this.changePaintOpacity(true);
            }
        }
        drawItemNumber(item, x, y, width) {
            this.drawText(":", x, y, width - this.textWidth("00"), "right");
            this.drawText(ARTM.numItems(item.id) + "", x, y, width, "right");
        }
        updateHelp() {
            this._helpWindow?.setItem(this.item());
        }
    }
    ARTM.Window_MaterialItemList = Window_MaterialItemList;
    class Window_MaterialSelect extends Window_Selectable {
        initialize(rect, ...args) {
            super.initialize(rect);
            this.hide();
        }
        select(index) {
            if (index >= 0) {
                const material = ARTM.SelectMaterials[index];
                if (!material)
                    return;
                if (material.groupId <= 0)
                    return;
            }
            super.select(index);
        }
        maxItems() {
            return ARTM.SelectMaterials.length;
        }
        drawItem(index) {
            const rect = this.itemLineRect(index);
            const material = ARTM.SelectMaterials[index];
            this.drawGreedMaterial(rect.x, rect.y + 4, material, ARTM.Amount, ARTM.Param.fontRate);
        }
        drawItemBackground(index) {
            if (ARTM.SelectMaterials[index].groupId <= 0)
                return;
            super.drawItemBackground(index);
        }
        updateHelp() {
            this._helpWindow?.setText(Iz.Text.get(ARTM.GREED_SHOP_DESC_TEXT_KEY));
        }
    }
    ARTM.Window_MaterialSelect = Window_MaterialSelect;
    class Window_Material extends Window_Base {
        initialize() {
            this._materials = [];
            const width = this.windowWidth();
            const height = this.windowHeight();
            super.initialize(new Rectangle(0, 0, width, height));
            this.hide();
        }
        windowWidth() {
            return ARTM.Param.materialWindowWidth;
        }
        windowHeight() {
            let n = this._materials.length;
            if (ARTM.Param.needText) {
                n += 1;
            }
            if (this._price) {
                n += 1;
            }
            return Math.floor(this.fittingHeight(n) * ARTM.Param.fontRate);
        }
        standardBackOpacity() {
            return ARTM.Param.backOpacity;
        }
        show() {
            super.show();
            if (this._materials.length === 0) {
                this.hide();
            }
        }
        materials() {
            return this._materials;
        }
        setShopItem(item, price) {
            if (this._shopItem !== item) {
                this._shopItem = item;
                this._price = price;
                this._materials = DataManager.getGreedShopMaterials(item);
                if (this._materials.length > 0) {
                    this.refresh();
                }
                if (ARTM.Param.showMaterialWindow) {
                    this.show();
                }
            }
        }
        refresh() {
            this.move(this.x, this.y, this.width, this.windowHeight());
            this.createContents();
            this.drawGreedMaterialsFromItem(0, 0, this._shopItem, 1, ARTM.Param.fontRate);
        }
        ;
    }
    ARTM.Window_Material = Window_Material;
})(ARTM || (ARTM = {}));
(function (ARTM) {
    let _owners = [];
    function allItems() {
        return Iz.Life.ItemHolder.items(_owners).map(item => $dataItems[item]);
    }
    ARTM.allItems = allItems;
    function numItems(itemId) {
        return Iz.Life.ItemHolder.numItems(itemId, _owners);
    }
    ARTM.numItems = numItems;
    function numGroupItems(groupId) {
        return Iz.Life.ItemHolder.numGroupItems(groupId, _owners);
    }
    ARTM.numGroupItems = numGroupItems;
    function loseItems(itemId, num) {
        Iz.Life.ItemHolder.loseItems(itemId, num, _owners);
    }
    ARTM.loseItems = loseItems;
    function setOwner(owner) {
        _owners = owner;
    }
    ARTM.setOwner = setOwner;
    function initializeOwner() {
        _owners = [Iz.Life.ItemOwnership.Party];
    }
    ARTM.initializeOwner = initializeOwner;
    function getOwners() {
        return _owners;
    }
    ARTM.getOwners = getOwners;
    initializeOwner();
})(ARTM || (ARTM = {}));
var ARTMUtil;
(function (ARTMUtil) {
    function setOwnershipsOrder(owners) {
        ARTM.setOwner(owners);
    }
    ARTMUtil.setOwnershipsOrder = setOwnershipsOrder;
})(ARTMUtil || (ARTMUtil = {}));
var DataManager;
(function (DataManager) {
    function materialToItem(material) {
        const type = material.type.toUpperCase();
        if (type === ARTM.MaterialType.Item || type === ARTM.MaterialType.Group) {
            return $dataItems[material.id];
        }
        else if (type === ARTM.MaterialType.Weapon) {
            return $dataWeapons[material.id];
        }
        else if (type === ARTM.MaterialType.Armor) {
            return $dataArmors[material.id];
        }
        else if (type === ARTM.MaterialType.Param) {
            return { id: material.id };
        }
        return null;
    }
    DataManager.materialToItem = materialToItem;
    ;
    function getGreedShopMaterial(item, re, i) {
        const key = 'mat' + i;
        let material = null;
        const metaData = item.meta[key];
        if (metaData) {
            const match = re.exec(metaData);
            if (match) {
                const matType = match[1];
                const id = +match[2];
                material = {
                    type: matType,
                    groupId: matType === ARTM.MaterialType.Group ? id : 0,
                    id: matType === ARTM.MaterialType.Group ? 0 : id,
                    need: +match[3]
                };
            }
        }
        return material;
    }
    DataManager.getGreedShopMaterial = getGreedShopMaterial;
    ;
    function getGreedShopMaterials(item) {
        let result = [];
        if (item) {
            const re = /(i|w|a|g|p)(\d+)\*(\d+)/i;
            for (let i = 1; i <= ARTM.Param.materialMax; i++) {
                const mat = getGreedShopMaterial(item, re, i);
                if (mat) {
                    result.push(mat);
                }
            }
        }
        return result;
    }
    DataManager.getGreedShopMaterials = getGreedShopMaterials;
    ;
    function isConsumableMaterial(item) {
        // return item.consumable || (!this.isItem(item) && !item.meta.noConsume);
        if (!item)
            return false;
        return DataManager.isItem(item) || (!DataManager.isItem(item) && !item.meta.noConsume);
    }
    DataManager.isConsumableMaterial = isConsumableMaterial;
    ;
})(DataManager || (DataManager = {}));
(() => {
    const PLUGIN_NAME = "ARTM_TMGreedShopMZ";
    const parameters = PluginManager.parameters(PLUGIN_NAME);
    const materialWindowPosition = +(parameters['materialWindowPosition'] || 0);
    ARTM.Param.materialWindowWidth = +(parameters['materialWindowWidth'] || 408);
    const buyWindowWidth = +(parameters['buyWindowWidth'] || 456);
    const buyWindowHeight = +(parameters['buyWindowHeight'] || 0);
    ARTM.Param.materialMax = +(parameters['materialMax'] || 5);
    ARTM.Param.fontRate = +(parameters['fontRate'] || 0.8);
    let greedCommand = parameters['greedCommand'] || '購入する';
    ARTM.Param.needText = parameters['needText'] || '';
    const showSellCommand = JSON.parse(parameters['showSellCommand'] || 'true');
    ARTM.Param.showMaterialWindow = JSON.parse(parameters['showMaterialWindow'] || 'true');
    const overlaid = JSON.parse(parameters['overlaid'] || 'true');
    ARTM.Param.backOpacity = +(parameters['backOpacity'] || 192);
    const showMaterialFromNumberWindow = JSON.parse(parameters['showMaterialFromNumberWindow'] || 'true');
    let showPrice = JSON.parse(parameters['showPrice'] || 'true');
    let seGreedBuy = JSON.parse(parameters['seGreedBuy'] || '{}');
    const originalShowPrice = showPrice;
    //-----------------------------------------------------------------------------
    // PluginManager
    //
    PluginManager.registerCommand(PLUGIN_NAME, "greedShop", args => {
        $gameTemp.setGreedShop(true);
    });
    PluginManager.registerCommand(PLUGIN_NAME, "setGreedCommand", args => {
        greedCommand = args.commandName;
        const langText = Iz.Text.get(greedCommand);
        if (langText) {
            greedCommand = langText;
        }
    });
    PluginManager.registerCommand(PLUGIN_NAME, "changeShowPrice", args => {
        showPrice = args.flag === "true";
    });
    PluginManager.registerCommand(PLUGIN_NAME, "changeSeGreedBuy", args => {
        seGreedBuy = JSON.parse(args.seGreedBuy);
    });
    //-----------------------------------------------------------------------------
    // Game_Temp
    //
    const _Game_Temp_initialize = Game_Temp.prototype.initialize;
    Game_Temp.prototype.initialize = function () {
        _Game_Temp_initialize.call(this);
        this._greedShop = false;
        this._greedShopScrolling = false;
    };
    Game_Temp.prototype.setGreedShop = function (flag) {
        this._greedShop = flag;
    };
    Game_Temp.prototype.isGreedShop = function () {
        return this._greedShop;
    };
    Game_Temp.prototype.startGreedShopScroll = function () {
        this._greedShopScrolling = true;
    };
    Game_Temp.prototype.isGreedShopScrolling = function () {
        return this._greedShopScrolling;
    };
    //-----------------------------------------------------------------------------
    // Window_Base
    //
    Window_Base.prototype.drawGreedNeedText = function (y, rate) {
        const lh = Math.floor(this.lineHeight() * rate);
        this.changeTextColor(this.systemColor());
        this.contents.drawText(ARTM.Param.needText, 0, y, this.contents.width, lh, 'center');
        return y + lh;
    };
    Window_Base.prototype.drawStretchIcon = function (x, y, index, rate) {
        const bitmap = ImageManager.loadSystem('IconSet');
        const pw = ImageManager.iconWidth;
        const ph = ImageManager.iconHeight;
        const sx = index % 16 * pw;
        const sy = Math.floor(index / 16) * ph;
        const dw = Math.floor(pw * rate);
        const dh = Math.floor(ph * rate);
        this.contents.blt(bitmap, sx, sy, pw, ph, x, y, dw, dh);
    };
    Window_Base.prototype.drawGreedPrice = function (y, amount, rate) {
        const x = this.itemPadding();
        const lh = Math.floor(this.lineHeight() * rate);
        const width = this.contents.width - x * 2;
        const unitWidth = Math.min(80, this.textWidth(TextManager.currencyUnit));
        this.resetTextColor();
        // @ts-ignore
        this.contents.drawText(this._price * amount, x, y, width - unitWidth - 6, lh, 'right');
        this.changeTextColor(this.systemColor());
        this.contents.drawText(TextManager.currencyUnit, x + width - unitWidth, y, unitWidth, lh, 'right');
    };
    Window_Base.prototype.drawGreedMaterial = function (x, y, material, amount, rate) {
        const x2 = x + Math.floor((ImageManager.iconWidth + 4) * rate);
        const lh = Math.floor(this.lineHeight() * rate);
        const paddingOffsetX = -10;
        // グループ
        if (material.groupId > 0 && material.id <= 0) {
            const need = material.need * amount;
            this.changeTextColor(ColorManager.normalColor());
            // 最初にグループが一致するアイテム画像を暫定で設定する
            const firstHitGroupItem = $dataItems.filter(data => !!data)
                .find(data => Iz.Life.getItemGroups(data.id).find(group => group === material.groupId));
            if (firstHitGroupItem) {
                this.drawStretchIcon(x, y, firstHitGroupItem.iconIndex, rate);
            }
            this.contents.drawText(Iz.Life.getItemGroupName(material.groupId), x2, y, 312, lh, "left");
            this.contents.drawText(`${need}`, x, y, this.contents.width - this.itemPadding() + paddingOffsetX, lh, 'right');
            return y + lh;
        }
        else {
            const materialItem = DataManager.materialToItem(material);
            if (!materialItem) {
                return y + lh;
            }
            if (ARTM.isDataItem(materialItem)) {
                const need = material.need * amount;
                const n = ARTM.numItems(materialItem.id);
                let text = DataManager.isConsumableMaterial(materialItem) ? '' + n + '/' : '--/';
                text += ('   ' + need).substr(-3);
                this.drawStretchIcon(x, y, materialItem.iconIndex, rate);
                this.changeTextColor(ColorManager.normalColor());
                let name = "";
                if (material.groupId <= 0) {
                    name = materialItem.name;
                }
                else {
                    name = `${materialItem.name} (${Iz.Life.getItemGroupName(material.groupId)})`;
                }
                // @ts-ignore
                this.contents.drawText(name, x2, y, 312, lh);
                this.contents.drawText(text, x, y, this.contents.width - this.itemPadding() + paddingOffsetX, lh, 'right');
                return y + lh;
            }
            else {
                const need = material.need * amount;
                this.changeTextColor(ColorManager.normalColor());
                this.contents.drawText(TextManager.basic(4), x2, y, 312, lh, "left");
                const consumeHpText = Iz.Life.isConsumeHp() ? `${Iz.Life.getHP()}/` : "--/";
                const numText = `${consumeHpText}${('   ' + need).substr(-3)}`;
                this.contents.drawText(numText, x, y, this.contents.width - this.itemPadding() + paddingOffsetX, lh, 'right');
                return y + lh;
            }
        }
    };
    Window_Base.prototype.drawGreedMaterials = function (x, y, materials, amount, rate) {
        this.resetFontSettings();
        this.contents.fontSize = Math.floor(this.contents.fontSize * rate);
        if (ARTM.Param.needText) {
            y = this.drawGreedNeedText(y, rate);
        }
        for (let i = 0; i < materials.length; i++) {
            y = this.drawGreedMaterial(x, y, materials[i], amount, rate);
        }
        // @ts-ignore
        if (this._price) {
            this.drawGreedPrice(y, amount, rate);
        }
        this.resetFontSettings();
    };
    Window_Base.prototype.drawGreedMaterialsFromItem = function (x, y, item, amount, rate) {
        const materials = DataManager.getGreedShopMaterials(item);
        this.drawGreedMaterials(x, y, materials, amount, rate);
    };
    //-----------------------------------------------------------------------------
    // Window_ShopCommand
    //
    const _Window_ShopCommand_maxCols = Window_ShopCommand.prototype.maxCols;
    Window_ShopCommand.prototype.maxCols = function () {
        return (showSellCommand || !this._purchaseOnly) ? _Window_ShopCommand_maxCols.call(this) : 2;
    };
    const _Window_ShopCommand_refresh = Window_ShopCommand.prototype.refresh;
    Window_ShopCommand.prototype.refresh = function () {
        _Window_ShopCommand_refresh.call(this);
        this.refreshCursor();
    };
    const _Window_ShopCommand_makeCommandList = Window_ShopCommand.prototype.makeCommandList;
    Window_ShopCommand.prototype.makeCommandList = function () {
        if ($gameTemp.isGreedShop()) {
            this.addCommand(greedCommand, 'buy');
            if (showSellCommand || !this._purchaseOnly) {
                this.addCommand(TextManager.sell, 'sell', !this._purchaseOnly);
            }
            this.addCommand(TextManager.cancel, 'cancel');
        }
        else {
            _Window_ShopCommand_makeCommandList.call(this);
        }
    };
    //-----------------------------------------------------------------------------
    // Window_ShopBuy
    //
    const _Window_ShopBuy_initialize = Window_ShopBuy.prototype.initialize;
    Window_ShopBuy.prototype.initialize = function (x, y, height, shopGoods) {
        if ($gameTemp.isGreedShop() && +buyWindowHeight > 0) {
            height = +buyWindowHeight;
        }
        _Window_ShopBuy_initialize.call(this, x, y, height, shopGoods);
    };
    // NOTE: 使ってなさそう
    /*
    const _Window_ShopBuy_windowWidth = Window_ShopBuy.prototype.windowWidth;
    Window_ShopBuy.prototype.windowWidth = function() {
        if ($gameTemp.isGreedShop()) {
            return buyWindowWidth;
        }
        return _Window_ShopBuy_windowWidth.call(this);
    };
    */
    const _Window_ShopBuy_price = Window_ShopBuy.prototype.price;
    Window_ShopBuy.prototype.price = function (item) {
        if ($gameTemp.isGreedShop() && (item && item.meta.matG)) {
            return +item.meta.matG;
        }
        return _Window_ShopBuy_price.call(this, item);
    };
    const _Window_ShopBuy_isEnabled = Window_ShopBuy.prototype.isEnabled;
    Window_ShopBuy.prototype.isEnabled = function (item) {
        const result = _Window_ShopBuy_isEnabled.call(this, item);
        if (result && $gameTemp.isGreedShop()) {
            const materials = DataManager.getGreedShopMaterials(item);
            for (let i = 0; i < materials.length; i++) {
                const material = materials[i];
                if (material.type === ARTM.MaterialType.Group) {
                    const num = ARTM.numGroupItems(material.groupId);
                    if (num < material.need) {
                        return false;
                    }
                }
                else {
                    const matItem = DataManager.materialToItem(material);
                    if (!matItem)
                        continue;
                    if (ARTM.isDataItem(matItem)) {
                        if (ARTM.numItems(matItem.id) < material.need) {
                            return false;
                        }
                    }
                    else {
                        if (matItem.id === ARTM.CharacterParamId.Mp) {
                            if (!Iz.Life.isConsumeHp())
                                continue;
                            if (Iz.Life.getHP() < material.need) {
                                return false;
                            }
                        }
                    }
                }
            }
        }
        return result;
    };
    const _Window_ShopBuy_update = Window_ShopBuy.prototype.update;
    Window_ShopBuy.prototype.update = function () {
        _Window_ShopBuy_update.call(this);
        if (this.active && this._materialWindow) {
            this._materialWindow.setShopItem(this.item(), this.price(this.item()));
            this.setMaterialWindowPosition();
        }
    };
    Window_ShopBuy.prototype.updateSmoothScroll = function () {
        Window_Scrollable.prototype.updateSmoothScroll.call(this);
        if ($gameTemp.isGreedShop()) {
            const bottom = this.y + this.height;
            const materialWindow = this._materialWindow;
            if (materialWindow.y + materialWindow.windowHeight() > bottom ||
                materialWindow.y < this.y) {
                materialWindow.hide();
                $gameTemp.startGreedShopScroll();
            }
            // @ts-ignore
            else if (!SceneManager._scene._numberWindow.visible) {
                materialWindow.show();
            }
        }
    };
    Window_ShopBuy.prototype.calcMaterialWindowPositionY = function (rect) {
        const testHeight = this.lineHeight() + this.itemPadding();
        const index = this.index() - this.topRow();
        return (this.y + this.padding + this.rowSpacing() / 2 +
            (index + 1) * rect.height +
            ((index + 1) * this.rowSpacing()) -
            this._scrollY % testHeight);
    };
    Window_ShopBuy.prototype.setMaterialWindowPosition = function () {
        const rect = this.itemRectWithPadding(this.index());
        const y = this.calcMaterialWindowPositionY(rect);
        // @ts-ignore
        const h_helpWindowNoinc = Graphics.boxHeight - this._helpWindow.height;
        let x = 0;
        switch (materialWindowPosition) {
            case 0: // item bottom
                x = this.x + this.width / 2;
                this._materialWindow.x = x - this._materialWindow.width / 2;
                this._materialWindow.y = y;
                if (this._materialWindow.y + this._materialWindow.height > h_helpWindowNoinc) {
                    this._materialWindow.y -= this._materialWindow.height + rect.height;
                }
                break;
            case 1: // item right
                x = this.x + this.width - this.padding;
                this._materialWindow.x = x;
                this._materialWindow.y = y;
                if (this._materialWindow.x + this._materialWindow.width > Graphics.boxWidth) {
                    this._materialWindow.x = Graphics.boxWidth - this._materialWindow.width;
                }
                if (this._materialWindow.y + this._materialWindow.height > h_helpWindowNoinc) {
                    this._materialWindow.y -= this._materialWindow.height + rect.height;
                }
                break;
            case 2: // container bottom
                this._materialWindow.x = this.x;
                this._materialWindow.y = this.y + this.height;
                if (this._materialWindow.y + this._materialWindow.height > h_helpWindowNoinc) {
                    this._materialWindow.y = h_helpWindowNoinc - this._materialWindow.height;
                }
                break;
        }
    };
    const _Window_ShopBuy_makeItemList = Window_ShopBuy.prototype.makeItemList;
    Window_ShopBuy.prototype.makeItemList = function () {
        _Window_ShopBuy_makeItemList.call(this);
        for (let i = this._data.length - 1; i >= 0; i--) {
            const item = this._data[i];
            if (item && item.meta.matKey) {
                this.makeItemListForMat(item, i);
            }
        }
    };
    Window_ShopBuy.prototype.makeItemListForMat = function (item, i) {
        const materials = DataManager.getGreedShopMaterials(item);
        // @ts-ignore
        const keys = item.meta.matKey.split(" ").map(Number);
        let valid = true;
        for (let j = 0; j < keys.length; j++) {
            const material = materials[keys[j] - 1];
            const matItem = DataManager.materialToItem(material);
            if (!matItem)
                continue;
            if (ARTM.isDataItem(matItem)) {
                if (ARTM.numItems(matItem.id) <= 0) {
                    valid = false;
                    break;
                }
            }
            else {
                // パラメータごとにチェック
                if (matItem.id === ARTM.CharacterParamId.Mp) {
                    if (!Iz.Life.isConsumeHp())
                        continue;
                    if (Iz.Life.getHP() < material.need) {
                        valid = false;
                        break;
                    }
                }
            }
        }
        if (!valid) {
            this._data.splice(i, 1);
            this._price.splice(i, 1);
        }
    };
    const _Window_ShopBuy_drawItem = Window_ShopBuy.prototype.drawItem;
    Window_ShopBuy.prototype.drawItem = function (index) {
        if ($gameTemp.isGreedShop() && !showPrice) {
            const item = this._data[index];
            const rect = this.itemRect(index);
            rect.width -= this.itemPadding();
            this.changePaintOpacity(this.isEnabled(item));
            this.drawItemName(item, rect.x, rect.y, rect.width);
            this.changePaintOpacity(true);
        }
        else {
            _Window_ShopBuy_drawItem.call(this, index);
        }
    };
    Window_ShopBuy.prototype.setMaterialWindow = function (materialWindow) {
        this._materialWindow = materialWindow;
    };
    Window_ShopBuy.prototype.select = function (index) {
        Window_Selectable.prototype.select.call(this, index);
        if (this._materialWindow) {
            this._materialWindow.show();
        }
    };
    //-----------------------------------------------------------------------------
    // Window_ShopStatus
    //
    const _Window_ShopStatus_initialize = Window_ShopStatus.prototype.initialize;
    Window_ShopStatus.prototype.initialize = function (rect) {
        _Window_ShopStatus_initialize.call(this, rect);
        this._toggleButtons = [];
    };
    Window_ShopStatus.prototype.onButtonClick = function (partyId) {
        // @ts-ignore
        const member = $gameParty.members[partyId];
        if (member) {
            alert(member.name());
        }
    };
    Window_ShopStatus.prototype.clearToggleButtons = function () {
        this._toggleButtons.forEach(b => b._pressedTgl = false);
        // @ts-ignore
        dubugNum = 0;
    };
    //-----------------------------------------------------------------------------
    // Window_ShopNumber
    //
    /* 使っていない？
    const _Window_ShopNumber_windowWidth = Window_ShopNumber.prototype.windowWidth;
    Window_ShopNumber.prototype.windowWidth = function() {
        if ($gameTemp.isGreedShop()) {
            return buyWindowWidth;
        }
        return _Window_ShopNumber_windowWidth.call(this);
    };
    */
    const _Window_ShopNumber_refresh = Window_ShopNumber.prototype.refresh;
    Window_ShopNumber.prototype.refresh = function () {
        _Window_ShopNumber_refresh.call(this);
        if ($gameTemp.isGreedShop()) {
            ARTM.Amount = this.number();
            // @ts-ignore
            SceneManager._scene.onChangeAmount();
        }
        /*
        if ($gameTemp.isGreedShop() && showMaterialFromNumberWindow) {
            this.drawGreedMaterialsFromItem(0, this.lineHeight() * 2, this._item, this._number, ARTM.Param.fontRate);
        }
        const scene = SceneManager._scene;
        // @ts-ignore
        scene._statusWindow.clearToggleButtons();
        */
    };
    const _Window_ShopNumber_drawTotalPrice = Window_ShopNumber.prototype.drawTotalPrice;
    Window_ShopNumber.prototype.drawTotalPrice = function () {
        if (!$gameTemp.isGreedShop()) {
            _Window_ShopNumber_drawTotalPrice.call(this);
        }
    };
    const _Window_ShopNumber_itemNameY = Window_ShopNumber.prototype.itemNameY;
    Window_ShopNumber.prototype.itemNameY = function () {
        if ($gameTemp.isGreedShop() && showMaterialFromNumberWindow) {
            return 0;
        }
        return _Window_ShopNumber_itemNameY.call(this);
    };
    /* 使っていない？　MVの実装のなごりっぽい
    const _Window_ShopNumber_priceY = Window_ShopNumber.prototype.priceY;
    Window_ShopNumber.prototype.priceY = function() {
        if ($gameTemp.isGreedShop() && showMaterialFromNumberWindow) {
            return this.lineHeight() * 1.5;
        }
        return _Window_ShopNumber_priceY.call(this);
    };
    */
    const _Window_ShopNumber_buttonY = Window_ShopNumber.prototype.buttonY;
    Window_ShopNumber.prototype.buttonY = function () {
        if ($gameTemp.isGreedShop() && showMaterialFromNumberWindow) {
            return this.contents.height - this.lineHeight() * 2.6;
        }
        return _Window_ShopNumber_buttonY.call(this);
    };
    const _Window_ShopNumber_updateHelp = Window_ShopNumber.prototype.updateHelp;
    Window_ShopNumber.prototype.updateHelp = function () {
        if ($gameTemp.isGreedShop()) {
            this._helpWindow?.setText(Iz.Text.get(ARTM.GREED_SHOP_DESC_TEXT_KEY));
        }
        else {
            _Window_ShopNumber_updateHelp.call(this);
        }
    };
    const _Window_ShopNumber_isCurrentItemEnabled = Window_ShopNumber.prototype.isCurrentItemEnabled;
    Window_ShopNumber.prototype.isCurrentItemEnabled = function () {
        if ($gameTemp.isGreedShop()) {
            const failed = ARTM.SelectMaterials.find(m => {
                const item = DataManager.materialToItem(m);
                if (!item)
                    return true;
                if (ARTM.isDataItem(item)) {
                    const requiredNum = m.need * ARTM.Amount;
                    if (ARTM.numItems(item.id) < requiredNum) {
                        return true;
                    }
                }
                else {
                    if (item.id === ARTM.CharacterParamId.Mp) {
                        if (Iz.Life.isConsumeHp()) {
                            if (Iz.Life.getHP() < m.need * ARTM.Amount) {
                                return true;
                            }
                        }
                    }
                }
                return false;
            });
            return !failed;
        }
        else {
            return _Window_ShopNumber_isCurrentItemEnabled.call(this);
        }
    };
    //-----------------------------------------------------------------------------
    // Scene_Shop
    //
    const _Scene_Shop_terminate = Scene_Shop.prototype.terminate;
    Scene_Shop.prototype.terminate = function () {
        _Scene_Shop_terminate.call(this);
        $gameTemp.setGreedShop(false);
        ARTM.initializeOwner();
        showPrice = originalShowPrice;
    };
    const _Scene_Shop_create = Scene_Shop.prototype.create;
    Scene_Shop.prototype.create = function () {
        _Scene_Shop_create.call(this);
        if ($gameTemp.isGreedShop()) {
            this._numberWindow.setHelpWindow(this._helpWindow);
            this.createMaterialWindow();
            this.createMaterialSelectWindow();
            this.createMaterialItemListWindow();
            ARTM.Amount = 1;
        }
    };
    Scene_Shop.prototype.createMaterialWindow = function () {
        // 
        this._materialWindow = new ARTM.Window_Material(new Rectangle(0, 0, 0, 0));
        this._buyWindow.setMaterialWindow(this._materialWindow);
        if (overlaid) {
            this.addChild(this._materialWindow);
        }
        else {
            this.addWindow(this._materialWindow);
        }
    };
    Scene_Shop.prototype.createMaterialSelectWindow = function () {
        const offset = 120;
        const rect = new Rectangle(this._numberWindow.x, this._numberWindow.y + offset, this._numberWindow.width, this._numberWindow.height - offset);
        this._materialSelectWindow = new ARTM.Window_MaterialSelect(rect);
        this._materialSelectWindow.opacity = 0;
        this._materialSelectWindow.setHandler("ok", this.onMaterialSelectOk.bind(this));
        this._materialSelectWindow.setHandler("cancel", this.onMaterialSelectCancel.bind(this));
        this._materialSelectWindow.setHelpWindow(this._helpWindow);
        this.addChild(this._materialSelectWindow);
    };
    Scene_Shop.prototype.createMaterialItemListWindow = function () {
        const rect = new Rectangle(this._numberWindow.x, this._numberWindow.y, this._numberWindow.width, this._numberWindow.height);
        this._materialItemListWindow = new ARTM.Window_MaterialItemList(rect);
        this._materialItemListWindow.opacity = 0;
        this._materialItemListWindow.setHandler("ok", this.onMaterialItemListOk.bind(this));
        this._materialItemListWindow.setHandler("cancel", this.onMaterialItemListCancel.bind(this));
        this._materialItemListWindow.setHelpWindow(this._helpWindow);
        this.addChild(this._materialItemListWindow);
    };
    const _Scene_Shop_onBuyOk = Scene_Shop.prototype.onBuyOk;
    Scene_Shop.prototype.onBuyOk = function () {
        if (this._materialWindow) {
            ARTM.setSelectMaterials(this._materialWindow.materials());
        }
        _Scene_Shop_onBuyOk.call(this);
        if (this._materialWindow) {
            this._materialWindow.hide();
            this._materialSelectWindow.refresh();
            this._materialSelectWindow.show();
        }
    };
    const _Scene_Shop_onBuyCancel = Scene_Shop.prototype.onBuyCancel;
    Scene_Shop.prototype.onBuyCancel = function () {
        _Scene_Shop_onBuyCancel.call(this);
        if (this._materialWindow) {
            this._materialWindow.setShopItem(null);
        }
    };
    const _Scene_Shop_onNumberOk = Scene_Shop.prototype.onNumberOk;
    Scene_Shop.prototype.onNumberOk = function () {
        if ($gameTemp.isGreedShop() && this._commandWindow.currentSymbol() === 'buy') {
            AudioManager.playSe(seGreedBuy);
            this.doBuy(this._numberWindow.number());
            this.endNumberInput();
            this._goldWindow.refresh();
            this._statusWindow.refresh();
            this._materialSelectWindow.hide();
        }
        else {
            _Scene_Shop_onNumberOk.call(this);
        }
    };
    const _Scene_Shop_onNumberCancel = Scene_Shop.prototype.onNumberCancel;
    Scene_Shop.prototype.onNumberCancel = function () {
        _Scene_Shop_onNumberCancel.call(this);
        if ($gameTemp.isGreedShop()) {
            this._materialSelectWindow?.hide();
        }
    };
    const _Scene_Shop_update = Scene_Shop.prototype.update;
    Scene_Shop.prototype.update = function () {
        _Scene_Shop_update.call(this);
        if ($gameTemp.isGreedShop()) {
            if (this._commandWindow.currentSymbol() === "buy" && this._numberWindow.active) {
                if (Input.isTriggered(Iz.IO.KEY_NAME_SHIFT)) {
                    this.onChangeMaterialSelect();
                }
            }
        }
    };
    Scene_Shop.prototype.onChangeMaterialSelect = function () {
        if (!ARTM.SelectMaterials.find(m => m.groupId > 0))
            return;
        SceneManager.updateInputData();
        SoundManager.playOk();
        this._numberWindow.select(-1);
        this._numberWindow.deactivate();
        this._materialSelectWindow.activate();
        const index = ARTM.SelectMaterials.findIndex(m => m.groupId > 0);
        this._materialSelectWindow.select(index);
    };
    Scene_Shop.prototype.onMaterialSelectOk = function () {
        const index = this._materialSelectWindow.index();
        ARTM.SelectMaterialIndex = index;
        const material = ARTM.SelectMaterials[index];
        this._materialItemListWindow.setup(material.groupId, this._materialSelectWindow.index());
        this._materialItemListWindow.show();
        this._materialItemListWindow.activate();
        this._materialItemListWindow.select(0);
        this._numberWindow.contentsOpacity = 0;
        this._numberWindow._contentsBackSprite.opacity = 0;
        this._materialSelectWindow.hide();
    };
    Scene_Shop.prototype.onMaterialSelectCancel = function () {
        this._materialSelectWindow.select(-1);
        this._numberWindow.activate();
        this._numberWindow.select(0);
    };
    Scene_Shop.prototype.onMaterialItemListOk = function () {
        const item = this._materialItemListWindow.item();
        if (item) {
            ARTM.SelectMaterials[this._materialSelectWindow.index()].id = item.id;
        }
        this._materialSelectWindow.activate();
        this._materialSelectWindow.show();
        this._materialSelectWindow.refresh();
        this._numberWindow.contentsOpacity = 255;
        this._numberWindow._contentsBackSprite.opacity = 255;
        this._numberWindow.setup(this._item, this.maxBuy(), this.buyingPrice());
        this._materialItemListWindow.hide();
    };
    Scene_Shop.prototype.onMaterialItemListCancel = function () {
        this._materialSelectWindow.activate();
        this._materialSelectWindow.show();
        this._numberWindow.contentsOpacity = 255;
        this._numberWindow._contentsBackSprite.opacity = 255;
        this._materialItemListWindow.hide();
    };
    Scene_Shop.prototype.onChangeAmount = function () {
        this._materialSelectWindow.refresh();
    };
    const _Scene_Shop_doBuy = Scene_Shop.prototype.doBuy;
    Scene_Shop.prototype.doBuy = function (number) {
        _Scene_Shop_doBuy.call(this, number);
        if (this._materialWindow) {
            const materials = ARTM.SelectMaterials;
            for (let i = 0; i < materials.length; i++) {
                const material = materials[i];
                const item = DataManager.materialToItem(material);
                if (!item)
                    continue;
                if (ARTM.isDataItem(item)) {
                    if (DataManager.isConsumableMaterial(item)) {
                        ARTM.loseItems(item.id, material.need * number);
                    }
                }
                else {
                    if (item.id === ARTM.CharacterParamId.Mp) {
                        Iz.Life.consumeHP(material.need, true);
                    }
                }
            }
            this._materialWindow.refresh();
        }
    };
    const _Scene_Shop_maxBuy = Scene_Shop.prototype.maxBuy;
    Scene_Shop.prototype.maxBuy = function () {
        let result = _Scene_Shop_maxBuy.call(this);
        if (this._materialWindow) {
            if (ARTM.BuyLimitOne) {
                result = 1;
            }
            else {
                const materials = ARTM.SelectMaterials;
                for (let i = 0; i < materials.length; i++) {
                    const material = materials[i];
                    const item = DataManager.materialToItem(material);
                    if (!item)
                        continue;
                    if (ARTM.isDataItem(item)) {
                        if (DataManager.isConsumableMaterial(item)) {
                            const n = ARTM.numItems(item.id);
                            result = Math.min(result, Math.floor(n / material.need));
                        }
                    }
                    else {
                        if (item.id === ARTM.CharacterParamId.Mp) {
                            if (!Iz.Life.isConsumeHp())
                                continue;
                            const n = Iz.Life.getHP();
                            result = Math.min(result, Math.floor(n / material.need));
                        }
                    }
                }
            }
        }
        return result;
    };
    const _Scene_Shop_endNumberInput = Scene_Shop.prototype.endNumberInput;
    Scene_Shop.prototype.endNumberInput = function () {
        _Scene_Shop_endNumberInput.call(this);
        if (this._materialWindow && ARTM.Param.showMaterialWindow) {
            this._materialWindow.show();
        }
    };
})();
//# sourceMappingURL=ARTM_TMGreedShopMZ.js.map