//===========================================================================
// CallMapEvent.js
//===========================================================================
/*:
 * @target MZ
 * @plugindesc プラグインコマンドでマップイベントを呼び出します
 * @author Shun
 *
 *
 * @param MapList
 * @type struct<MapData>[]
 * @desc CallMapEvent用マップ一覧
 * @default []
 *
 *
 * @command CallMapEvent
 * @text CallMapEvent
 * @desc マップイベントの呼び出し
 *
 * @arg name
 * @type string
 * @text キー名
 * @desc キー名
 *
 * @arg eventId
 * @text eventId
 * @desc イベントID
 *
 * @arg eventIdVariable
 * @type variable
 * @text eventIdVariable
 * @desc イベントID変数(eventIdより優先)
 *
 * @arg pageNumber
 * @type number
 * @text pageNumber
 * @desc ページ番号
 *
 *
 * @command CallSameMapEvent
 * @text CallSameMapEvent
 * @desc 同一イベント内の呼び出し(CallMapEvent or CallSameMapEvent先のイベント内でのみ使用可)
 *
 * @arg pageNumber
 * @type number
 * @text pageNumber
 * @desc ページ番号
 *
 *
 * @help
 * MapIdに呼び出したいイベントのマップIDを設定し、CallMapEventを使ってイベントを実行します
 *
 * プラグインコマンド詳細
 * CallMapEvent [eventId] [pageIndex]
 *
 */
/*~struct~MapData:
 * @param mapId
 * @type number
 * @desc マップID
 *
 * @param name
 * @type string
 * @desc CallMapEventコマンド時に使用する際のキー名
 *
 */
(function () {
    "use strict";
    const pluginName = "CallMapEvent";
    function commandCallMapEvent(args) {
        const name = args.name;
        let eventId = 0;
        const eventIdVariable = getEventId(args.eventIdVariable);
        if (eventIdVariable) {
            eventId = $gameVariables.value(eventIdVariable);
        }
        else {
            eventId = getEventId(args.eventId);
            if (!eventId) {
                const e = $gameEventMapList[name]
                    .events()
                    .find((e) => e.event().name === args.eventId);
                if (e) {
                    eventId = e.eventId();
                }
                else {
                    return;
                }
            }
        }
        const pageIndex = getPageIndex(args.pageNumber);
        $gameMap._interpreter.callMapEvent(name, eventId, pageIndex);
    }
    PluginManager.registerCommand(pluginName, "CallMapEvent", (args) => {
        commandCallMapEvent(args);
    });
    function commandCallSameMapEvent(args) {
        // @ts-ignore
        const thisObj = this;
        const pageIndex = getPageIndex(args.pageNumber);
        thisObj.callSameMapEvent(pageIndex);
    }
    PluginManager.registerCommand(pluginName, "CallSameMapEvent", commandCallSameMapEvent);
    function getEventId(strEventId) {
        if (strEventId === true)
            return 0;
        const eventId = Number.parseInt(convertEscapeCharacters(strEventId));
        return eventId;
    }
    function getPageIndex(strPageNumber) {
        if (strPageNumber === true)
            return -1;
        let pageIndex = Number.parseInt(convertEscapeCharacters(strPageNumber));
        if (!pageIndex) {
            pageIndex = -1;
        }
        return pageIndex;
    }
    function mapId2FileName(mapId) {
        return "Map" + ("000" + mapId).slice(-3) + ".json";
    }
    const convertEscapeCharacters = function (text_) {
        let text = text_;
        if (text == null) {
            text = "";
        }
        const window = SceneManager._scene?._windowLayer?.children[0];
        return window && window instanceof Window_Base
            ? window.convertEscapeCharacters(text)
            : text;
    };
    const parameters = PluginManager.parameters("CallMapEvent");
    let paramMapList = [];
    function getDataEventMapName(name) {
        return `$dataEventMap_${name}`;
    }
    function getDataEventMap(name) {
        // @ts-ignore
        return window[getDataEventMapName(name)];
    }
    function getDataEventMapById(mapId) {
        const target = paramMapList.find((p) => p.mapId === mapId);
        return getDataEventMap(target.name);
    }
    class MapData {
    }
    {
        if (DataManager.isBattleTest() || DataManager.isEventTest()) {
            return;
        }
        const list = JSON.parse(parameters.MapList).map((m) => JSON.parse(m));
        list.forEach((data) => {
            data.mapId = Number.parseInt(data.mapId);
        });
        paramMapList = list;
        paramMapList.forEach((data) => {
            DataManager._databaseFiles.push({
                name: getDataEventMapName(data.name),
                src: mapId2FileName(data.mapId),
            });
        });
    }
    const _DataManagerCreateGameObjects = DataManager.createGameObjects;
    DataManager.createGameObjects = function () {
        _DataManagerCreateGameObjects.call(this);
        window.$gameEventMapList = {};
        paramMapList.forEach((data) => {
            $gameEventMapList[data.name] = new Game_EventMap();
        });
    };
    const _DataManagerSetupNewGame = DataManager.setupNewGame;
    DataManager.setupNewGame = function () {
        _DataManagerSetupNewGame.call(this);
        paramMapList.forEach((data) => {
            const target = $gameEventMapList[data.name];
            target.setup(data.mapId);
        });
    };
    const _DataManagerEXtractSaveContents = DataManager.extractSaveContents;
    DataManager.extractSaveContents = function (contents) {
        _DataManagerEXtractSaveContents.call(this, contents);
        paramMapList.forEach((data) => {
            const target = $gameEventMapList[data.name];
            target.setup(data.mapId);
        });
    };
    Game_Interpreter.prototype.callMapEvent = function (name, eventId, pageIndex_) {
        const event = $gameEventMapList[name].event(eventId);
        if (!event)
            return;
        let pageIndex = pageIndex_;
        if (pageIndex < 0) {
            pageIndex = event.findProperPageIndex();
            if (pageIndex < 0) {
                return;
            }
        }
        else {
            pageIndex -= 1;
        }
        event._pageIndex = pageIndex;
        this.setupChild(event.list(), this._eventId);
        if (this._childInterpreter) {
            this._childInterpreter.callerInfo = {
                mapId: $gameEventMapList[name].mapId(),
                eventId: event.eventId(),
            };
        }
    };
    Game_Interpreter.prototype.callSameMapEvent = function (pageIndex_) {
        if (!this.callerInfo)
            return;
        let mapKey = undefined;
        for (const key in $gameEventMapList) {
            if ($gameEventMapList[key].mapId() === this.callerInfo.mapId) {
                mapKey = key;
                break;
            }
        }
        if (!mapKey)
            return;
        const event = $gameEventMapList[mapKey].event(this.callerInfo.eventId);
        if (!event)
            return;
        let pageIndex = pageIndex_;
        if (pageIndex < 0) {
            pageIndex = event.findProperPageIndex();
            if (pageIndex < 0) {
                return;
            }
        }
        else {
            pageIndex -= 1;
        }
        event._pageIndex = pageIndex;
        this.setupChild(event.list(), this._eventId);
        if (this._childInterpreter) {
            this._childInterpreter.callerInfo = {
                mapId: $gameEventMapList[mapKey].mapId(),
                eventId: event.eventId(),
            };
        }
    };
    class Game_EventMap extends Game_Map {
        setup(mapId) {
            this._mapId = mapId;
            this._displayX = 0;
            this._displayY = 0;
            this.setupEvents();
            this._needsRefresh = false;
        }
        data() {
            return getDataEventMapById(this._mapId).data;
        }
        setupEvents() {
            this._events = [];
            const dataEventMap = getDataEventMapById(this._mapId);
            for (let i = 0; i < dataEventMap.events.length; i++) {
                if (dataEventMap.events[i]) {
                    this._events[i] = new Game_MapEvent(this._mapId, i);
                }
            }
            this._commonEvents = this.parallelCommonEvents().map(function (commonEvent) {
                return new Game_CommonEvent(commonEvent.id);
            });
            this.refreshTileEvents();
        }
    }
    class Game_MapEvent extends Game_Event {
        event() {
            return getDataEventMapById(this._mapId).events[this._eventId];
        }
        initialize(mapId, eventId) {
            this._mapId = mapId;
            this._eventId = eventId;
            this.refresh();
        }
        loadTagParam() {
            return false;
        }
    }
})();
//# sourceMappingURL=CallMapEvent.js.map