/*:
 * @target MZ
 * @plugindesc 酪農
 *
 * @help
 *
 *
 *
 *
 * @command addLivestock
 * @text addLivestock
 * @desc 家畜を追加
 *
 * @arg id
 * @text id
 * @desc 家畜ID
 * @type number
 *
 * @arg shedId
 * @text shedId
 * @desc 小屋ID
 * @type number
 *
 *
 * @command removeLivestock
 * @text removeLivestock
 * @desc 家畜を削除
 *
 * @arg eventId
 * @text eventId
 * @desc 家畜のイベントID　指定なしor0はこのイベントになります
 * @type number
 *
 * @arg livestockIndex
 * @text livestockIndex
 * @desc 家畜Index　eventIdより優先されます
 * @type number
 *
 *
 * @command changeShed
 * @text changeShed
 * @desc 動物の所属小屋を変更します
 *
 * @arg eventId
 * @text eventId
 * @desc 家畜のイベントID　指定なしor0はこのイベントになります
 * @type number
 *
 * @arg livestockIndex
 * @text livestockIndex
 * @desc 家畜Index　eventIdより優先されます
 * @type number
 *
 * @arg shedId
 * @text shedId
 * @desc 小屋ID
 * @type number
 *
 *
 * @command newday
 * @text newday
 * @desc 日付経過します　餌の消費や生産品の産出を行います
 *
 *
 * @command copyShed
 * @text copyShed
 * @desc 小屋にいる家畜、飼葉、孵化器などを別の小屋にコピーします
 *
 * @arg srcShed
 * @text srcShed
 * @desc コピー元小屋ID
 * @type number
 *
 * @arg dstShed
 * @text dstShed
 * @desc コピー先小屋ID
 * @type number
 *
 * @arg livestockSpawnRegion
 * @text livestockSpawnRegion
 * @desc 家畜生成リージョン 未指定の場合ランダム
 * @type number
 *
 * @arg excludeLastLivestock
 * @text excludeLastLivestock
 * @desc 最後尾の家畜を除外
 * @type boolean
 *
 * @arg excludeHatchedLivestock
 * @text excludeHatchedLivestock
 * @desc 孵化した家畜を除外
 * @type boolean
 *
 * @arg excludeBirthedLivestock
 * @text excludeBirthedLivestock
 * @desc 産まれた家畜を除外
 * @type boolean
 *
 */
var Iz;
(function (Iz) {
    var Life;
    (function (Life) {
        const BrushTag = "brush";
        const MilkingTag = "milking";
        const PregnancyTag = "pregnancy";
        const BrushList = [];
        const MilkingList = [];
        const PregnancyList = [];
        Life.DairyTemplateMapId = 95;
        Life.DairyEventTemplateMapId = 96;
        Life.LivestockTemplateId_Egg = 31;
        Life.IncubatorEventId = 32;
        Life.FeedItemId = 190;
        Life.LastLivestockIndexVariableId = 1041;
        Life.HatchedLivestockIndexVariableId = 1042;
        Life.BirthedLivestockIndexVariableId = 1043;
        Life.AutoFeedMaxDays = 3;
        Life.IncubatorItemId = 438;
        Life.AutoFeederItemId = 437;
        Life.AutoFeederChargeItemId = 155;
        const ShedEnterRegion = 24;
        function getHappinessRating(shedId, shedLevel) {
            const target = $dataDairyShedLevel.find((data) => data && data.id === shedId && data.level === shedLevel);
            if (target) {
                return target.happiness_rate;
            }
            return 1;
        }
        function getLivestockLimit(shedId, shedLevel) {
            const target = $dataDairyShedLevel.find((data) => data && data.id === shedId && data.level === shedLevel);
            if (target) {
                return target.livestock_limit;
            }
            return 0; // データがない場合
        }
        Life.getLivestockLimit = getLivestockLimit;
        function isLivestockLimit(shedId, shedLevel, amountLivestock) {
            const limit = getLivestockLimit(shedId, shedLevel);
            return amountLivestock >= limit;
        }
        Life.isLivestockLimit = isLivestockLimit;
        // 餌箱の最大数
        function getLivestockFeedsMax(shedId, shedLevel) {
            return getLivestockLimit(shedId, shedLevel); // 家畜の数と同じ
        }
        // 1つの餌箱に入れられる餌の最大数
        function getLivestockFeedAmountLimit() {
            return 1;
        }
        function isBrush(itemId) {
            return BrushList.includes(itemId);
        }
        Life.isBrush = isBrush;
        function isMilking(itemId) {
            return MilkingList.includes(itemId);
        }
        Life.isMilking = isMilking;
        function isPregnancy(itemId) {
            return PregnancyList.includes(itemId);
        }
        Life.isPregnancy = isPregnancy;
        function isAutoFeeder(itemId) {
            return itemId === Life.AutoFeederItemId;
        }
        Life.isAutoFeeder = isAutoFeeder;
        function isAutoFeederChargeItem(itemId) {
            return itemId === Life.AutoFeederChargeItemId;
        }
        Life.isAutoFeederChargeItem = isAutoFeederChargeItem;
        function getLivestockMapId(shedId) {
            const shedLevel = getLivestockShedLevel(shedId);
            const data = $dataDairyShedLevel
                .filter((data) => data && data.id === shedId)
                .find((data) => data.level === shedLevel);
            if (!data)
                return 0;
            return data.map_id;
        }
        Life.getLivestockMapId = getLivestockMapId;
        function getLivestockShedId(mapId) {
            const data = $dataDairyShedLevel.find((data) => data && data.map_id === mapId);
            if (!data)
                return 0;
            return data.id;
        }
        Life.getLivestockShedId = getLivestockShedId;
        function getLivestockShedLevel(shedId) {
            const dataShed = $dataDairyShed[shedId];
            if (!dataShed)
                return 0;
            return $gameVariables.value(dataShed.level_variable) || 0;
        }
        Life.getLivestockShedLevel = getLivestockShedLevel;
        function getLivestockShedMaxLevel(shedId) {
            const sorted = $dataDairyShedLevel
                .filter((data) => data && data.id === shedId)
                .sort((a, b) => b.level - a.level);
            return sorted[0] ? sorted[0].level : 0;
        }
        Life.getLivestockShedMaxLevel = getLivestockShedMaxLevel;
        function isEventShed(shedId) {
            return !!$dataDairyShed[shedId]?.event;
        }
        Life.isEventShed = isEventShed;
        function getLivestockTemplateId(livestockId) {
            const data = $dataDairyLivestock[livestockId];
            if (!data)
                return 0;
            return data.template_id;
        }
        Life.getLivestockTemplateId = getLivestockTemplateId;
        function getLivestockProductRank(happiness) {
            if (happiness >= $parameter.dairy.happiness_threshold_l)
                return Life.LivestockProductRank.L;
            if (happiness >= $parameter.dairy.happiness_threshold_m)
                return Life.LivestockProductRank.M;
            return Life.LivestockProductRank.S;
        }
        Life.getLivestockProductRank = getLivestockProductRank;
        function getMilkList() {
            const MilkTag = "milk";
            return $dataItems
                .filter((data) => data)
                .filter((data) => data.meta[MilkTag])
                .sort((a, b) => Number(a.meta[MilkTag]) - Number(b.meta[MilkTag]));
        }
        Life.getMilkList = getMilkList;
        function getMilkItemId(happiness) {
            const milks = getMilkList();
            const rank = getLivestockProductRank(happiness);
            if (rank === Life.LivestockProductRank.L)
                return milks[2].id;
            if (rank === Life.LivestockProductRank.M)
                return milks[1].id;
            return milks[0].id;
        }
        Life.getMilkItemId = getMilkItemId;
        function getMilkSize(itemId) {
            const milks = getMilkList();
            if (milks[2]?.id === itemId)
                return Life.LivestockProductRank.L;
            if (milks[1]?.id === itemId)
                return Life.LivestockProductRank.M;
            return Life.LivestockProductRank.S;
        }
        Life.getMilkSize = getMilkSize;
        function getEggList() {
            const EggTag = "egg";
            return $dataItems
                .filter((data) => data)
                .filter((data) => data.meta[EggTag])
                .sort((a, b) => Number(a.meta[EggTag]) - Number(b.meta[EggTag]));
        }
        Life.getEggList = getEggList;
        function getEggItemId(happiness) {
            const eggs = getEggList();
            const rank = getLivestockProductRank(happiness);
            if (rank === Life.LivestockProductRank.L)
                return eggs[2].id;
            if (rank === Life.LivestockProductRank.M)
                return eggs[1].id;
            return eggs[0].id;
        }
        Life.getEggItemId = getEggItemId;
        function getEggSize(itemId) {
            const eggs = getEggList();
            if (eggs[2]?.id === itemId)
                return Life.LivestockProductRank.L;
            if (eggs[1]?.id === itemId)
                return Life.LivestockProductRank.M;
            return Life.LivestockProductRank.S;
        }
        Life.getEggSize = getEggSize;
        // 使用可能な最大数
        function getIncubatorUsableMax(shedId) {
            const limit = getIncubatorDataList(shedId).length; // 小屋に設置可能な最大数
            return Math.max(0, limit);
        }
        Life.getIncubatorUsableMax = getIncubatorUsableMax;
        function getIncubatorDataList(shedId) {
            const level = getLivestockShedLevel(shedId);
            return $dataDairyShedLevelIncubator
                .filter((data) => data && data.id === shedId && data.level === level)
                .sort((a, b) => a.index - b.index);
        }
        Life.getIncubatorDataList = getIncubatorDataList;
        function pickItem(key, mapId) {
            const shed = Life.Dairy.getShed(getLivestockShedId(mapId));
            if (!shed)
                return;
            const pickData = shed.getPickItem(key);
            if (!pickData)
                return;
            const item = $dataItems[pickData.itemId];
            if (!item)
                return;
            const amount = 1;
            $gameParty.gainItem(item, amount);
            shed.removePickItem(key);
            Iz.System.notifyGainItem(item.id, amount);
            Life.Status.addWorkLevelExpDairy(Iz.Life.Status.WorkDairyType.GetEgg);
            Life.Status.addGoddessPointEgg(getEggSize(item.id));
            Life.Stats.dairy.getShed(shed.id)?.addItem(item.id, 1);
            Iz.System.playPickSe();
        }
        Life.pickItem = pickItem;
        function pickItemFromLivestock(livestockIndex, notify) {
            const livestock = Life.Dairy.getLivestock(livestockIndex);
            if (!livestock)
                return;
            if (!livestock.isAdult())
                return;
            livestock.picked = true;
            let item = undefined;
            if (livestock.type === Life.Data.LivestockType.Cow) {
                item = $dataItems[getMilkItemId(livestock.happiness)];
                if (item) {
                    Life.Status.addWorkLevelExpDairy(Iz.Life.Status.WorkDairyType.Milking);
                    Life.Status.addGoddessPointMilk(getMilkSize(item.id));
                    Life.Stats.dairy.getShed(livestock.shedId)?.addItem(item.id, 1);
                }
            }
            if (item === undefined)
                return;
            $gameParty.gainItem(item, 1);
            Iz.System.playPickSe();
            if (notify) {
                Torigoya.NotifyMessage.Addons.GetItem.notifyGainItem(item, 1);
            }
        }
        Life.pickItemFromLivestock = pickItemFromLivestock;
        function getLivestockOverheadKey(livestockIndex) {
            return `livestockOverhead_${livestockIndex}`;
        }
        Life.getLivestockOverheadKey = getLivestockOverheadKey;
        function getChildId(livestockId) {
            const livestock = $dataDairyLivestock[livestockId];
            if (!livestock)
                return 0;
            let childId = livestock.id;
            while (true) {
                const target = $dataDairyLivestock.find((data) => data && data.evolved_to === childId);
                if (!target) {
                    break;
                }
                childId = target.id;
            }
            return childId;
        }
        Life.getChildId = getChildId;
        function isIncubatorEvent(event) {
            if (!event)
                return false;
            return event.event().meta.dairy === "incubator";
        }
        Life.isIncubatorEvent = isIncubatorEvent;
        function isFeedBox(event) {
            if (!event)
                return false;
            return PluginManagerEx.findMetaValue(event.event(), ["livestock"]) === "feed";
        }
        Life.isFeedBox = isFeedBox;
        const ExcludeSpawnRegions = [ShedEnterRegion, 241, 242, 243, 244, 245, 246, 247, 248];
        function getDairySpawnPoints(random) {
            const list = [];
            for (let y = 0; y < $gameMap.height(); y++) {
                for (let x = 0; x < $gameMap.width(); x++) {
                    if ($gamePlayer.pos(x, y))
                        continue;
                    if ($gameMap.eventsXy(x, y).length > 0)
                        continue;
                    const region = Iz.Life.getRegion(x, y);
                    if (ExcludeSpawnRegions.includes(region))
                        continue;
                    if ($gameMap.isPassable(x, y, 2) ||
                        $gameMap.isPassable(x, y, 4) ||
                        $gameMap.isPassable(x, y, 6) ||
                        $gameMap.isPassable(x, y, 8)) {
                        list.push({ x: x, y: y });
                    }
                }
            }
            if (random) {
                list.sort((a, b) => 0.5 - Math.random());
            }
            if (list.length === 0) {
                list.push({ x: 0, y: 0 });
            }
            return list;
        }
        Life.getDairySpawnPoints = getDairySpawnPoints;
        function getLastLivestockIndex() {
            return $gameVariables.value(Life.LastLivestockIndexVariableId);
        }
        Life.getLastLivestockIndex = getLastLivestockIndex;
        function getHatchedLivestockIndex() {
            return $gameVariables.value(Life.HatchedLivestockIndexVariableId);
        }
        Life.getHatchedLivestockIndex = getHatchedLivestockIndex;
        function getBirthedLivestockIndex() {
            return $gameVariables.value(Life.BirthedLivestockIndexVariableId);
        }
        Life.getBirthedLivestockIndex = getBirthedLivestockIndex;
        Life.LivestockProductRank = {
            S: 0,
            M: 1,
            L: 2,
        };
        class Livestock {
            constructor(data) {
                this._data = data;
            }
            get id() {
                return this._data.id;
            }
            get index() {
                return this._data.index;
            }
            get name() {
                return this._data.name;
            }
            set name(value) {
                this._data.name = value;
            }
            get templateId() {
                return getLivestockTemplateId(this.id);
            }
            get type() {
                return $dataDairyLivestock[this.id]?.type || Life.Data.LivestockType.None;
            }
            get shedId() {
                return this._data.shedId;
            }
            set shedId(value) {
                this._data.shedId = value;
            }
            get producted() {
                return this._data.producted;
            }
            set producted(value) {
                this._data.producted = value;
            }
            get picked() {
                return this._data.picked;
            }
            set picked(value) {
                this._data.picked = value;
            }
            get growth() {
                return this._data.growth;
            }
            set growth(value) {
                const growthMax = $dataDairyLivestock[this.id].growth_max;
                this._data.growth = Math.min(Math.max(Math.floor(value), 0), growthMax);
            }
            get happiness() {
                return this._data.happiness;
            }
            set happiness(value) {
                this._data.happiness = Math.min(Math.max(Math.floor(value), 0), $parameter.dairy.happiness_max);
            }
            get talked() {
                return this._data.talked;
            }
            set talked(value) {
                this._data.talked = value;
            }
            get cared() {
                return this._data.cared;
            }
            set cared(value) {
                this._data.cared = value;
            }
            get ageOfDays() {
                return this._data.ageOfDays;
            }
            set ageOfDays(value) {
                this._data.ageOfDays = value;
            }
            get startHaveMonth() {
                return this._data.startHaveMonth;
            }
            get startHaveDay() {
                return this._data.startHaveDay;
            }
            get pregnant() {
                return this._data.pregnant;
            }
            set pregnant(value) {
                this._data.pregnant = value;
            }
            get pregnancyDays() {
                return this._data.pregnancyDays;
            }
            set pregnancyDays(value) {
                this._data.pregnancyDays = value;
            }
            get pregnancyLeftDays() {
                const data = $dataDairyPregnancy[this.id];
                if (!data) {
                    return 0;
                }
                return Math.max(0, data.duration - this.pregnancyDays);
            }
            get noFeedDays() {
                return this._data.noFeedDays;
            }
            set noFeedDays(value) {
                this._data.noFeedDays = value;
            }
            get hatchedData() {
                return this._data.hatchedData;
            }
            get birthed() {
                return this._data.birthed;
            }
            set birthed(value) {
                this._data.birthed = value;
            }
            get gaveBirthed() {
                return this._data.gaveBirthed;
            }
            set gaveBirthed(value) {
                this._data.gaveBirthed = value;
            }
            get productRank() {
                return getLivestockProductRank(this.happiness);
            }
            get price() {
                const data = $dataDairyLivestock[this.id];
                if (data) {
                    const item = $dataItems[data.shop_item_id];
                    if (item) {
                        return item.price;
                    }
                }
                return 100;
            }
            get sellPrice() {
                if (this.productRank === Life.LivestockProductRank.S)
                    return Math.floor(this.price / 2);
                if (this.productRank === Life.LivestockProductRank.L)
                    return this.price * 2;
                return this.price;
            }
            isAdult() {
                // TOOD:
                // 進化先がないものを大人として扱う
                const data = $dataDairyLivestock[this.id];
                if (!data)
                    return false;
                if (data.evolved_to !== 0)
                    return false;
                return true;
            }
            checkEvolve() {
                const data = $dataDairyLivestock[this.id];
                if (!data)
                    return false;
                if (this.growth < data.growth_max)
                    return false;
                const evolvedData = $dataDairyLivestock[data.evolved_to];
                if (!evolvedData)
                    return false;
                this._data.id = evolvedData.id;
                this.growth = 0;
                return true;
            }
            canProducted() {
                return this.producted && !this.picked && !this.pregnant && this.isAdult();
            }
            forSale() {
                return this.isAdult() && !this.pregnant;
            }
            newday() {
                this.talked = false;
                this.producted = false;
                this.picked = false;
                this.cared = false;
                this.ageOfDays++;
                if (this.pregnant) {
                    this.pregnancyDays++;
                }
                this.clearHatched();
                this.clearBirthed();
            }
            clearHatched() {
                this._data.hatchedData = {
                    on: false,
                    incubatorIndex: 0,
                };
            }
            clearBirthed() {
                this._data.birthed = false;
                this._data.gaveBirthed = false;
            }
            cloneData() {
                const data = Object.assign({}, this._data);
                data.hatchedData = Object.assign({}, this._data.hatchedData);
                return data;
            }
        }
        Life.Livestock = Livestock;
        // export class LivestockIncubator {
        //     private _data: LivestockIncubatorData;
        //     constructor(data: LivestockIncubatorData) {
        //         this._data = data;
        //     }
        //     get id() { return this._data.id; }
        //     get itemId() { return this._data.itemId; }
        //     getData() {
        //         return Object.assign({}, this._data);
        //     }
        // }
        class LivestockShed {
            constructor(shedId) {
                this._id = shedId;
                this.clear();
            }
            get id() {
                return this._id;
            }
            get level() {
                return getLivestockShedLevel(this.id);
            }
            get mapId() {
                // 小屋とレベルからマップIDを求める
                return getLivestockMapId(this.id);
            }
            get feeds() {
                return this._feeds.values();
            }
            get incubators() {
                return this._incubators.values();
            }
            get enableIncubators() {
                return this._incubators.values().filter((ic) => ic.enable);
            }
            get pickItems() {
                return this._pickItems;
            }
            clear() {
                this._feeds = new Life.ExpansionList(() => {
                    return { index: 0, amount: 0, autoFeed: { enable: false, leftDays: 0 } };
                }, () => {
                    return getLivestockFeedsMax(this._id, getLivestockShedLevel(this._id));
                });
                this._incubators = new Life.ExpansionList(() => {
                    return { index: 0, enable: false, itemId: 0, elapsedDays: 0 };
                }, () => {
                    return getIncubatorUsableMax(this._id);
                });
                this._pickItems = new Map();
            }
            apply(data) {
                data.feeds
                    .filter((data) => data)
                    .forEach((data) => {
                    this.setFeed(data.index, data.amount);
                    const feed = this._feeds.get(data.index);
                    if (feed) {
                        Object.assign(feed, data);
                    }
                });
                data.incubators
                    .filter((data) => data)
                    .forEach((data) => {
                    this.setIncubator(data.index, data.enable);
                    this.setEgg(data.index, data.itemId, data.elapsedDays);
                });
                if (data.pickItems) {
                    data.pickItems.forEach((item) => {
                        if ($dataItems[item.itemId]) {
                            const newData = {
                                key: 0,
                                itemId: 0,
                            };
                            this._pickItems.set(item.key, Object.assign(newData, item));
                        }
                    });
                }
            }
            getFeed(feedIndex) {
                return this._feeds.get(feedIndex);
            }
            addFeed(feedIndex, amount) {
                const data = this._feeds.get(feedIndex);
                if (!data)
                    return;
                data.amount = Math.min(Math.max(data.amount + amount, 0), getLivestockFeedAmountLimit());
            }
            setFeed(feedIndex, amount) {
                const data = this._feeds.get(feedIndex);
                if (!data)
                    return;
                data.amount = Math.min(Math.max(amount, 0), getLivestockFeedAmountLimit());
            }
            isAutoFeed(feedIndex) {
                const data = this._feeds.get(feedIndex);
                if (!data)
                    return false;
                return data.autoFeed.enable;
            }
            setAutoFeed(feedIndex, flg) {
                const data = this._feeds.get(feedIndex);
                if (!data)
                    return;
                data.autoFeed.enable = flg;
            }
            setAutoFeedLeftDays(feedIndex, leftDays) {
                const data = this._feeds.get(feedIndex);
                if (!data)
                    return;
                data.autoFeed.leftDays = leftDays;
            }
            existIncubator(incubatorIndex) {
                return !!this._incubators.get(incubatorIndex)?.enable;
            }
            getIncubator(incubatorIndex) {
                return this._incubators.get(incubatorIndex);
            }
            setIncubator(incubatorIndex, enable) {
                const data = this._incubators.get(incubatorIndex);
                if (!data)
                    return false;
                data.enable = enable;
                return true;
            }
            setEgg(incubatorIndex, itemId, elapsedDays) {
                const data = this._incubators.get(incubatorIndex);
                if (!data)
                    return false;
                if (!data.enable)
                    return false;
                data.itemId = itemId;
                data.elapsedDays = elapsedDays ?? 0;
                return true;
            }
            pickEgg(incubatorIndex) {
                const data = this._incubators.get(incubatorIndex);
                if (!data || !data.enable || data.itemId <= 0)
                    return 0;
                const itemId = data.itemId;
                this.clearIncubator(incubatorIndex);
                return itemId;
            }
            existEgg(incubatorIndex) {
                const data = this._incubators.get(incubatorIndex);
                return !!data && data.enable && data.itemId > 0;
            }
            clearIncubator(incubatorIndex) {
                const data = this._incubators.get(incubatorIndex);
                if (!data)
                    return;
                data.itemId = 0;
                data.elapsedDays = 0;
            }
            isFeedLimit(feedIndex) {
                const feed = this.getFeed(feedIndex);
                if (!feed)
                    return true;
                return feed.amount >= getLivestockFeedAmountLimit();
            }
            getFeedAmount(feedIndex) {
                const feed = this.getFeed(feedIndex);
                return feed ? feed.amount : 0;
            }
            getPickItem(key) {
                return this._pickItems.get(key);
            }
            addPickItem(itemId) {
                let key = 1;
                while (true) {
                    if (this._pickItems.has(key)) {
                        key++;
                        continue;
                    }
                    break;
                }
                this._pickItems.set(key, {
                    key: key,
                    itemId: itemId,
                });
            }
            removePickItem(key) {
                this._pickItems.delete(key);
            }
            newday() {
                this._pickItems.clear();
            }
            getData() {
                const data = {
                    id: this._id,
                    feeds: this.feeds,
                    incubators: this.enableIncubators,
                    // autoFeed: this._autoFeed,
                    pickItems: [...this._pickItems.values()].sort((a, b) => a.key - b.key),
                };
                return data;
            }
        }
        Life.LivestockShed = LivestockShed;
        function createEmptyLivestockData() {
            const data = {
                id: 0,
                index: 0,
                name: "",
                growth: 0,
                happiness: 0,
                talked: false,
                cared: false,
                shedId: 0,
                producted: true,
                picked: true,
                ageOfDays: 0,
                startHaveMonth: 0,
                startHaveDay: 0,
                pregnant: false,
                pregnancyDays: 0,
                noFeedDays: 0,
                hatchedData: {
                    on: false,
                    incubatorIndex: 0,
                },
                birthed: false,
                gaveBirthed: false,
            };
            return data;
        }
        Life.createEmptyLivestockData = createEmptyLivestockData;
        class Game_Dairy {
            constructor() {
                this.clear();
            }
            get livestocks() {
                return this._livestocks.filter((s) => !!s);
            }
            get livestockSheds() {
                return this._livestockSheds.filter((s) => !!s);
            }
            get hatchedLivestock() {
                return this.livestocks.find((s) => !isEventShed(s.shedId) && s.hatchedData.on);
            }
            get birthedLivestock() {
                return this.livestocks.find((s) => !isEventShed(s.shedId) && s.birthed);
            }
            get gaveBirthedLivestock() {
                return this.livestocks.find((s) => !isEventShed(s.shedId) && s.gaveBirthed);
            }
            clear() {
                this._livestocks = [];
                this._livestockSheds = [];
            }
            initialize() {
                this.clear();
                $dataDairyShed
                    .filter((data) => data)
                    .forEach((data) => {
                    this._livestockSheds[data.id] = new LivestockShed(data.id);
                });
            }
            apply(list, shedList) {
                list.map((data) => {
                    const target = createEmptyLivestockData();
                    return new Livestock(Iz.Obj.apply(target, data));
                }).forEach((data) => {
                    this._livestocks[data.index] = data;
                });
                shedList.forEach((data) => {
                    const target = this._livestockSheds[data.id];
                    if (target) {
                        target.apply(data);
                    }
                });
            }
            getLivestock(index) {
                return this._livestocks[index];
            }
            getShedLivestocks(shedId) {
                return this.livestocks.filter((s) => s.shedId === shedId);
            }
            getShed(shedId) {
                return this._livestockSheds[shedId];
            }
            addLivestock(option, createEvent) {
                const shed = this._livestockSheds[option.shedId];
                if (!shed)
                    return -1;
                if (!$dataDairyLivestock[option.id])
                    return -1;
                if (this.isFull(shed.id))
                    return -1;
                const data = createEmptyLivestockData();
                data.id = option.id;
                data.index = this.generateLivestockIndex();
                data.name = option.name ?? "NoName";
                data.shedId = option.shedId;
                data.ageOfDays = option.ageOfDays ?? 0;
                data.startHaveMonth = Iz.System.getMonth();
                data.startHaveDay = Iz.System.getDay();
                const livestock = new Livestock(data);
                livestock.growth = option.growth ?? 0;
                livestock.happiness = option.happiness ?? 0;
                this._livestocks[data.index] = livestock;
                if (createEvent) {
                    const mapId = getLivestockMapId(data.shedId);
                    if (mapId === $gameMap.mapId()) {
                        const p = option.position ?? getDairySpawnPoints(true)[0];
                        const templateMapId = Iz.Life.isEventShed(data.shedId)
                            ? Life.DairyEventTemplateMapId
                            : Life.DairyTemplateMapId;
                        $gameMap.createLivestockEvent(data.index, livestock.templateId, p.x, p.y, templateMapId);
                    }
                }
                $gameVariables.setValue(Life.LastLivestockIndexVariableId, data.index);
                return data.index;
            }
            removeLivestock(livestockIndex) {
                const livestock = this.getLivestock(livestockIndex);
                if (!livestock)
                    return;
                delete this._livestocks[livestockIndex];
                const event = $gameMap.events().find((e) => e.getLivestockIndex() === livestockIndex);
                if (event) {
                    $gameMap.eraseEvent(event.eventId());
                }
            }
            getFeed(mapId, feedIndex) {
                const shedId = getLivestockShedId(mapId);
                const shed = this._livestockSheds[shedId];
                if (!shed)
                    return 0;
                return shed.getFeedAmount(feedIndex);
            }
            addFeed(mapId, feedIndex, amount) {
                const shedId = getLivestockShedId(mapId);
                const shed = this._livestockSheds[shedId];
                if (!shed)
                    return;
                // if(shed.isFeedLimit(feedId))
                //     return;
                shed.addFeed(feedIndex, amount);
                if (mapId === $gameMap.mapId()) {
                    // 飼い葉イベントのセルフスイッチを操作
                    const event = $gameMap
                        .events()
                        .find((e) => e.getLivestockFeedIndex() === feedIndex);
                    if (event) {
                        event.refreshLivestockFeed();
                    }
                }
            }
            generateLivestockIndex() {
                const emptyIndex = this._livestocks.findIndex((s) => !s);
                if (emptyIndex >= 0) {
                    return emptyIndex;
                }
                return this._livestocks.length > 0 ? this._livestocks.length : 0;
            }
            addGrowth(livestockIndex, addValue) {
                const livestock = this.getLivestock(livestockIndex);
                if (!livestock)
                    return;
                livestock.growth += addValue;
            }
            addHappiness(livestockIndex, addValue_, calculate = true) {
                const livestock = this.getLivestock(livestockIndex);
                if (!livestock)
                    return;
                // 小屋レベルに応じた係数
                let addValue = addValue_;
                if (calculate) {
                    const shedLevel = getLivestockShedLevel(livestock.shedId);
                    addValue *= getHappinessRating(livestock.shedId, shedLevel);
                }
                livestock.happiness += addValue;
                Life.Stats.dairy.addTotalHappiness(addValue);
            }
            care(livestockIndex) {
                const livestock = this.getLivestock(livestockIndex);
                if (!livestock)
                    return;
                livestock.cared = true;
                this.addHappiness(livestock.index, $parameter.dairy.happiness_add_cared);
            }
            talk(livestockIndex) {
                const livestock = this.getLivestock(livestockIndex);
                if (!livestock)
                    return;
                if (!livestock.talked) {
                    livestock.talked = true;
                    Life.Status.addWorkLevelExpDairy(Life.Status.WorkDairyType.Talk);
                    Life.Stats.dairy.addTalkedCount(1);
                }
            }
            changeShed(livestockIndex, shedId) {
                const livestock = this.getLivestock(livestockIndex);
                if (!livestock)
                    return;
                const shed = this.livestockSheds.find((shed) => shed.id === shedId);
                if (!shed)
                    return;
                livestock.shedId = shedId;
            }
            // 妊娠中の家畜の胎児、孵化器のたまごもカウント
            isFull(shedId) {
                const shedLevel = getLivestockShedLevel(shedId);
                let sum = 0;
                this.getShedLivestocks(shedId).forEach((s) => {
                    sum += s.pregnant ? 2 : 1;
                });
                this.livestockSheds.forEach((shed) => {
                    if (shedId !== shed.id)
                        return;
                    shed.enableIncubators
                        .filter((data) => !!data)
                        .forEach((data) => {
                        sum += data.itemId > 0 ? 1 : 0;
                    });
                });
                return isLivestockLimit(shedId, shedLevel, sum);
            }
            becomePregnant(livestockIndex) {
                const livestock = this.getLivestock(livestockIndex);
                if (!livestock)
                    return false;
                livestock.pregnant = true;
                livestock.pregnancyDays = 0;
            }
            ableToPregnant(livestockIndex) {
                const livestock = this.getLivestock(livestockIndex);
                if (!livestock)
                    return false;
                const dataPregnancy = $dataDairyPregnancy[livestock.id];
                if (!dataPregnancy)
                    return false;
                if (livestock.pregnant)
                    return false;
                if (this.isFull(livestock.shedId))
                    return false;
                return true;
            }
            existIncubator(shedId, incubatorIndex) {
                return !!this._livestockSheds[shedId]?.existIncubator(incubatorIndex);
            }
            setIncubator(shedId, incubatorIndex, enable) {
                const res = this._livestockSheds[shedId]?.setIncubator(incubatorIndex, enable);
                this.refreshIncubators();
                return !!res;
            }
            putEgg(shedId, incubatorIndex, itemId) {
                if (!$gameParty.hasItem($dataItems[itemId]))
                    return;
                this.pickEgg(shedId, incubatorIndex);
                if (this._livestockSheds[shedId].setEgg(incubatorIndex, itemId)) {
                    $gameParty.loseItem($dataItems[itemId], 1, false);
                }
                this.refreshIncubators();
            }
            pickEgg(shedId, incubatorIndex) {
                const itemId = this._livestockSheds[shedId].pickEgg(incubatorIndex);
                $gameParty.gainItem($dataItems[itemId], 1);
                this.refreshIncubators();
            }
            refreshIncubators() {
                this.livestockSheds
                    .filter((shed) => getLivestockMapId(shed.id) === $gameMap.mapId())
                    .forEach((shed) => {
                    $gameMap
                        .events()
                        .filter((e) => shed.incubators.find((ic) => ic.index === e.getIncubatorIndex()))
                        .forEach((e) => {
                        const mapId = $gameMap.mapId();
                        const eventId = e.eventId();
                        const icIndex = e.getIncubatorIndex();
                        // 孵化器
                        $gameSelfSwitches.setValue([mapId, eventId, "A"], this.existIncubator(shed.id, icIndex));
                        // タマゴ
                        $gameSelfSwitches.setValue([mapId, eventId, "B"], this.existEgg(shed.id, icIndex));
                    });
                });
            }
            existEgg(shedId, incubatorIndex) {
                return !!this._livestockSheds[shedId].existEgg(incubatorIndex);
            }
            autoFeed() {
                // 自動餌やり
                this.livestockSheds.forEach((shed) => {
                    const feeds = shed.feeds;
                    for (let i = 0; i < feeds.length; i++) {
                        const feed = feeds[i];
                        if (!feed.autoFeed.enable)
                            continue;
                        if (feed.autoFeed.leftDays <= 0)
                            continue;
                        while (true) {
                            const feedNum = $gameParty.numItems($dataItems[Life.FeedItemId]);
                            if (feedNum <= 0)
                                break;
                            if (shed.isFeedLimit(i))
                                break;
                            shed.addFeed(i, 1);
                            $gameParty.loseItem($dataItems[Life.FeedItemId], 1, false);
                        }
                    }
                });
            }
            checkAutoFeedLeftDays() {
                this.livestockSheds.forEach((shed) => {
                    const feeds = shed.feeds;
                    for (let i = 0; i < feeds.length; i++) {
                        const feed = feeds[i];
                        if (!feed.autoFeed.enable)
                            continue;
                        if (feed.autoFeed.leftDays > 0) {
                            feed.autoFeed.leftDays--;
                        }
                    }
                });
            }
            changeName(livestockIndex, name) {
                const livestock = this.getLivestock(livestockIndex);
                if (!livestock)
                    return;
                if (!name)
                    return;
                livestock.name = name;
            }
            copy(option) {
                const srcShed = this._livestockSheds[option.srcShed];
                const dstShed = this._livestockSheds[option.dstShed];
                if (!srcShed || !dstShed)
                    return;
                this.clearShed(option.dstShed);
                dstShed.apply(srcShed.getData());
                this.getShedLivestocks(option.srcShed).forEach((l) => {
                    if (option.excludeLastLivestock && l.index === getLastLivestockIndex()) {
                        return;
                    }
                    if (option.excludeHatchedLivestock && l.index === getHatchedLivestockIndex()) {
                        return;
                    }
                    if (option.excludeBirthedLivestock && l.index === getBirthedLivestockIndex()) {
                        return;
                    }
                    const data = l.cloneData();
                    const index = this.generateLivestockIndex();
                    data.index = index;
                    data.shedId = option.dstShed;
                    this._livestocks[index] = new Livestock(data);
                });
            }
            clearShed(shedId) {
                const shed = this._livestockSheds[shedId];
                if (!shed)
                    return;
                this.getShedLivestocks(shedId).forEach((l) => {
                    this.removeLivestock(l.index);
                });
                shed.clear();
            }
            checkFeed() {
                this.livestockSheds.forEach((shed) => {
                    // 餌の数だけ産出フラグをONにする. 余った餌はそのまま
                    {
                        const livestocks = this.livestocks.filter((s) => s.shedId === shed.id);
                        const feeds = shed.feeds.filter((f) => !!f && f.amount > 0);
                        const productNum = livestocks.length > feeds.length ? feeds.length : livestocks.length;
                        const mapId = getLivestockMapId(shed.id);
                        feeds
                            .filter((f, i) => i < productNum)
                            .forEach((f) => this.addFeed(mapId, f.index, -1));
                        // 餌が家畜よりも足りない場合はindex順に与える
                        const sortList = livestocks.sort((a, b) => a.index - b.index);
                        for (let i = 0; i < sortList.length; i++) {
                            const s = sortList[i];
                            let growthRate = 1;
                            if (i < productNum) {
                                s.producted = true;
                                this.addHappiness(s.index, $parameter.dairy.happiness_add_feed);
                                // 餌を与えている場合は成長度割り増し
                                growthRate = $parameter.dairy.growth_rate_feed;
                                s.noFeedDays = 0;
                            }
                            else {
                                // なつき度低下
                                const index = Math.min(Math.max(s.noFeedDays, 0), $parameter.dairy.happiness_decreases_nofeed.length - 1);
                                this.addHappiness(s.index, $parameter.dairy.happiness_decreases_nofeed[index], false);
                                s.noFeedDays++;
                            }
                            const baseValue = $dataDairyLivestock[s.id]?.growth_add_base ?? 0;
                            this.addGrowth(s.index, baseValue * growthRate);
                        }
                    }
                });
            }
            checkPickItem() {
                this.livestocks
                    .filter((s) => s.type === Life.Data.LivestockType.Chicken)
                    .filter((s) => s.canProducted())
                    .forEach((s) => {
                    const itemId = getEggItemId(s.happiness);
                    this.getShed(s.shedId)?.addPickItem(itemId);
                });
            }
            checkHatched() {
                let hatchedLivestockIndex = -1;
                this.livestockSheds.forEach((shed) => {
                    // たまごチェック
                    shed.enableIncubators
                        .filter((incubator) => incubator)
                        .forEach((incubator) => {
                        if (incubator.itemId > 0) {
                            incubator.elapsedDays++;
                        }
                        if (hatchedLivestockIndex >= 0) {
                            return;
                        }
                        if (incubator.elapsedDays >= $parameter.dairy.hatching_days) {
                            shed.clearIncubator(incubator.index);
                            const ChickId = 4;
                            // 同じ小屋にいる動物のなつき度の平均値を参照
                            let happiness = 0;
                            if (this.livestocks.length > 0) {
                                happiness = Math.max(0, Math.floor((this.livestocks
                                    .map((s) => s.happiness)
                                    .reduce((a, b) => a + b, 0) /
                                    this.livestocks.length) *
                                    $parameter.dairy.happiness_rate_egg));
                            }
                            const option = {
                                id: ChickId,
                                shedId: shed.id,
                                happiness: happiness,
                            };
                            const index = this.addLivestock(option, false);
                            if (index >= 0) {
                                hatchedLivestockIndex = index;
                                this._livestocks[index].hatchedData.on = true;
                                this._livestocks[index].hatchedData.incubatorIndex =
                                    incubator.index;
                            }
                        }
                    });
                });
                $gameVariables.setValue(Life.HatchedLivestockIndexVariableId, hatchedLivestockIndex);
                // this._hatched = hatchedLivestockIndex >= 0;
            }
            checkBirth(livestockIndex) {
                const parent = this.getLivestock(livestockIndex);
                if (!parent)
                    return false;
                if (!parent.pregnant)
                    return false;
                const dataPregnancy = $dataDairyPregnancy[parent.id];
                if (!dataPregnancy)
                    return false;
                if (parent.pregnancyDays < dataPregnancy.duration)
                    return false;
                // 出産は１日に１匹のみ
                if (this.birthedLivestock) {
                    return false;
                }
                parent.pregnant = false;
                parent.pregnancyDays = 0;
                // 親のなつき度を一部継承
                const option = {
                    id: getChildId(parent.id),
                    shedId: parent.shedId,
                    happiness: Math.max(0, Math.floor(parent.happiness * $parameter.dairy.happiness_inherited_rate)),
                };
                const index = this.addLivestock(option, false);
                const birthed = index >= 0;
                if (birthed) {
                    this._livestocks[index].birthed = true;
                    parent.gaveBirthed = true;
                }
                $gameVariables.setValue(Life.BirthedLivestockIndexVariableId, birthed ? index : -1);
                return birthed;
            }
            newday() {
                this.livestocks.forEach((s) => {
                    s.newday();
                });
                this.livestockSheds.forEach((shed) => {
                    shed.newday();
                });
                this.autoFeed();
                this.checkFeed();
                this.checkPickItem();
                this.checkHatched();
                // 出産・進化チェック
                this.livestocks.forEach((s) => {
                    if (s.pregnant) {
                        if (this.checkBirth(s.index)) {
                        }
                    }
                    else {
                        s.checkEvolve();
                    }
                });
                this.autoFeed();
                this.checkAutoFeedLeftDays();
            }
        }
        Life.Game_Dairy = Game_Dairy;
        Life.Dairy = new Game_Dairy();
        function initializeDairy() {
            Iz.Life.getToolItems(BrushTag).forEach((data) => {
                BrushList[data.level - 1] = data.id;
            });
            Iz.Life.getToolItems(MilkingTag).forEach((data) => {
                MilkingList[data.level - 1] = data.id;
            });
            Iz.Life.getToolItems(PregnancyTag).forEach((data) => {
                PregnancyList[data.level - 1] = data.id;
            });
            Life.Dairy.initialize();
        }
        Life.initializeDairy = initializeDairy;
    })(Life = Iz.Life || (Iz.Life = {}));
})(Iz || (Iz = {}));
(function (Iz) {
    var Life;
    (function (Life) {
        var DairyTemp;
        (function (DairyTemp) {
            const caredLivestocks = new Map();
            function cared(livestockId) {
                if (!caredLivestocks.get(livestockId)) {
                    caredLivestocks.set(livestockId, { livestockId: livestockId, count: 0 });
                }
            }
            DairyTemp.cared = cared;
            function updateCareInfo() {
                if (caredLivestocks.size <= 0)
                    return;
                if ($gameMap.isEventRunning() || $gameMessage.isBusy()) {
                    clearCared();
                    return;
                }
                const ar = [];
                caredLivestocks.forEach((info) => {
                    info.count++;
                    if (info.count >= 30) {
                        ar.push(info.livestockId);
                        if (info.livestockId === 1) {
                            Iz.System.playSe("Cow");
                        }
                        else if (info.livestockId === 2) {
                            Iz.System.playSe("Chicken2");
                        }
                    }
                });
                ar.forEach((id) => {
                    caredLivestocks.delete(id);
                });
            }
            DairyTemp.updateCareInfo = updateCareInfo;
            function clearCared() {
                caredLivestocks.clear();
            }
            DairyTemp.clearCared = clearCared;
        })(DairyTemp = Life.DairyTemp || (Life.DairyTemp = {}));
    })(Life = Iz.Life || (Iz.Life = {}));
})(Iz || (Iz = {}));
(() => {
    const _Scene_Map_update = Scene_Map.prototype.update;
    Scene_Map.prototype.update = function () {
        _Scene_Map_update.call(this);
        Iz.Life.DairyTemp.updateCareInfo();
    };
    const _Scene_Map_terminate = Scene_Map.prototype.terminate;
    Scene_Map.prototype.terminate = function () {
        _Scene_Map_terminate.call(this);
        Iz.Life.DairyTemp.clearCared();
    };
})();
(function (Iz) {
    var Life;
    (function (Life) {
        var DairyUtil;
        (function (DairyUtil) {
            function isCared(livestockIndex) {
                const livestock = Iz.Life.Dairy.getLivestock(livestockIndex);
                return livestock ? livestock.cared : false;
            }
            DairyUtil.isCared = isCared;
            // 各小屋の動物のなつき度の合計値
            function getHappinessShed(shedId) {
                return Life.Dairy.livestocks
                    .filter((s) => s.shedId === shedId)
                    .map((s) => s.happiness)
                    .reduce((sum, value) => sum + value, 0);
            }
            DairyUtil.getHappinessShed = getHappinessShed;
            // 動物全体のなつき度の合計値
            function getHappinessAll() {
                return Life.Dairy.livestocks.map((s) => s.happiness).reduce((sum, value) => sum + value, 0);
            }
            DairyUtil.getHappinessAll = getHappinessAll;
            // 小屋の家畜数
            function getShedLivestocks(shedId) {
                return Life.Dairy.getShedLivestocks(shedId).length;
            }
            DairyUtil.getShedLivestocks = getShedLivestocks;
            // 小屋の家畜が一杯か(胎児、たまごも含めてカウント)
            function isLivestockFull(shedId) {
                return Life.Dairy.isFull(shedId);
            }
            DairyUtil.isLivestockFull = isLivestockFull;
            function hasIncubator() {
                const item = $dataItems[Life.IncubatorItemId];
                return $gameParty.hasItem(item);
            }
            DairyUtil.hasIncubator = hasIncubator;
            function setIncubator(eventId, enable) {
                const mapId = $gameMap.mapId();
                const shedId = Life.getLivestockShedId(mapId);
                if (shedId <= 0)
                    return;
                const e = $gameMap.event(eventId);
                if (!e)
                    return;
                const res = Life.Dairy.setIncubator(shedId, e.getIncubatorIndex(), enable);
                if (res) {
                    const item = $dataItems[Life.IncubatorItemId];
                    $gameParty.gainItem(item, enable ? -1 : 1);
                }
            }
            DairyUtil.setIncubator = setIncubator;
            function hasEgg() {
                return !!Life.getEggList().find((data) => $gameParty.hasItem(data));
            }
            DairyUtil.hasEgg = hasEgg;
            function putEgg(shedId, incubatorIndex) {
                const egg = Life.getEggList().find((data) => $gameParty.hasItem(data));
                if (egg) {
                    Life.Dairy.putEgg(shedId, incubatorIndex, egg.id);
                }
            }
            DairyUtil.putEgg = putEgg;
            function isHatched() {
                return !!Life.Dairy.hatchedLivestock;
            }
            DairyUtil.isHatched = isHatched;
            function hatchedPosition() {
                const h = Life.Dairy.hatchedLivestock;
                if (h) {
                    const i = h.hatchedData.incubatorIndex;
                    const data = Life.getIncubatorDataList(h.shedId).find((d) => d.index === i);
                    if (data) {
                        return { x: data.x, y: data.y };
                    }
                }
                return { x: 0, y: 0 };
            }
            DairyUtil.hatchedPosition = hatchedPosition;
            function clearHatched() {
                Life.Dairy.livestocks.forEach((s) => {
                    s.clearHatched();
                });
            }
            DairyUtil.clearHatched = clearHatched;
            function isBirthed() {
                return !!Life.Dairy.birthedLivestock;
            }
            DairyUtil.isBirthed = isBirthed;
            function clearBirthed() {
                Life.Dairy.livestocks.forEach((s) => {
                    s.clearBirthed();
                });
            }
            DairyUtil.clearBirthed = clearBirthed;
            // 出産した動物の名前（生まれた動物ではない）
            function gaveBirthedLivestockName() {
                return Life.Dairy.gaveBirthedLivestock?.name ?? "";
            }
            DairyUtil.gaveBirthedLivestockName = gaveBirthedLivestockName;
            function pregnant(livestockIndex) {
                const s = Life.Dairy.getLivestock(livestockIndex);
                if (!s) {
                    return false;
                }
                return s.pregnant;
            }
            DairyUtil.pregnant = pregnant;
            function pregnancyLeftDays(livestockIndex) {
                const s = Life.Dairy.getLivestock(livestockIndex);
                if (!s) {
                    return 0;
                }
                return s.pregnancyLeftDays;
            }
            DairyUtil.pregnancyLeftDays = pregnancyLeftDays;
            function enableAutoFeed(mapId, feedIndex) {
                const shedId = Life.getLivestockShedId(mapId);
                const shed = Life.Dairy.livestockSheds.find((shed) => shed.id === shedId);
                if (!shed)
                    return false;
                shed.setAutoFeed(feedIndex, true);
                shed.setAutoFeedLeftDays(feedIndex, 0);
                return true;
            }
            DairyUtil.enableAutoFeed = enableAutoFeed;
            function disableAutoFeed(mapId, feedIndex) {
                const shedId = Life.getLivestockShedId(mapId);
                const shed = Life.Dairy.livestockSheds.find((shed) => shed.id === shedId);
                if (!shed)
                    return false;
                shed.setAutoFeed(feedIndex, false);
                shed.setAutoFeedLeftDays(feedIndex, 0);
                return true;
            }
            DairyUtil.disableAutoFeed = disableAutoFeed;
            function suppliableAutoFeed(mapId, feedIndex) {
                const shedId = Life.getLivestockShedId(mapId);
                const shed = Life.Dairy.livestockSheds.find((shed) => shed.id === shedId);
                if (!shed)
                    return false;
                const feed = shed.getFeed(feedIndex);
                if (!feed)
                    return false;
                if (feed.autoFeed.leftDays >= Life.AutoFeedMaxDays)
                    return false;
                return true;
            }
            DairyUtil.suppliableAutoFeed = suppliableAutoFeed;
            function supplyAutoFeed(mapId, feedIndex) {
                const shedId = Life.getLivestockShedId(mapId);
                const shed = Life.Dairy.livestockSheds.find((shed) => shed.id === shedId);
                if (!shed)
                    return;
                const feed = shed.getFeed(feedIndex);
                if (!feed)
                    return;
                feed.autoFeed.leftDays = Life.AutoFeedMaxDays;
            }
            DairyUtil.supplyAutoFeed = supplyAutoFeed;
            function livestocksNum(...livestockIds) {
                const ids = [...livestockIds].filter((id) => $dataDairyLivestock[id]);
                let num = 0;
                ids.forEach((id) => {
                    num += Life.Dairy.livestocks.filter((s) => s.id === id).length;
                });
                return num;
            }
            DairyUtil.livestocksNum = livestocksNum;
            function changeName(livestockIndex, name) {
                Iz.Life.Dairy.changeName(livestockIndex, name);
            }
            DairyUtil.changeName = changeName;
            function getName(livestockIndex) {
                const s = Iz.Life.Dairy.getLivestock(livestockIndex);
                if (!s) {
                    return "";
                }
                return s.name;
            }
            DairyUtil.getName = getName;
            function getDefaultNameChicken() {
                return Iz.Text.get("DAIRY_DEFAULT_NAME_CHICKEN");
            }
            DairyUtil.getDefaultNameChicken = getDefaultNameChicken;
            function getDefaultNameCow() {
                return Iz.Text.get("DAIRY_DEFAULT_NAME_COW");
            }
            DairyUtil.getDefaultNameCow = getDefaultNameCow;
            function copyShed(srcShed, dstShed, option) {
                Life.Dairy.copy({
                    srcShed: srcShed,
                    dstShed: dstShed,
                    excludeLastLivestock: option?.excludeLastLivestock,
                    excludeHatchedLivestock: option?.excludeHatchedLivestock,
                    excludeBirthedLivestock: option?.excludeBirthedLivestock,
                });
                const opt = {
                    livestockSpawnRegion: option?.livestockSpawnRegion,
                };
                $gameMap.setupLivestockEvents(opt);
                $gameMap.refreshDairy();
            }
            DairyUtil.copyShed = copyShed;
            function clearEventShed() {
                $dataDairyShed
                    .filter((data) => data)
                    .filter((data) => data.event)
                    .forEach((data) => {
                    Life.Dairy.clearShed(data.id);
                });
            }
            DairyUtil.clearEventShed = clearEventShed;
        })(DairyUtil = Life.DairyUtil || (Life.DairyUtil = {}));
    })(Life = Iz.Life || (Iz.Life = {}));
})(Iz || (Iz = {}));
(() => {
    const script = document.currentScript;
    PluginManagerEx.registerCommand(script, "addLivestock", (args) => {
        const option = {
            id: args.id,
            shedId: args.shedId,
        };
        Iz.Life.Dairy.addLivestock(option, true);
    });
    class SpriteAutoFeeder extends Sprite {
        constructor(e) {
            const bitmap = new Bitmap(32, 29);
            // bitmap.fillAll("red");
            super(bitmap);
            this._event = e;
            this._shedId = Iz.Life.getLivestockShedId(e.mapId());
            this.anchor.set(0.5, 0);
            this.request();
        }
        request() {
            this.bitmap = ImageManager.loadSystem("Feeder");
            this.bitmap.addLoadListener(() => {
                this.setFrame(0, 0, 32, 29);
            });
        }
        update() {
            super.update();
            const shed = Iz.Life.Dairy.getShed(this._shedId);
            const isAutoFeed = shed?.isAutoFeed(this._event.getLivestockFeedIndex()) ?? false;
            this.visible = isAutoFeed;
        }
    }
    function getLivestockIndexFromEvent(args) {
        let livestockIndex = -1;
        if (Number.isFinite(args.livestockIndex) && args.livestockIndex >= 0) {
            livestockIndex = args.livestockIndex;
        }
        else if (args.eventId > 0) {
            const event = $gameMap._events[args.eventId];
            if (event) {
                livestockIndex = event.getLivestockIndex();
            }
        }
        else {
            // @ts-ignore
            livestockIndex = this.character(0).getLivestockIndex();
        }
        return livestockIndex;
    }
    PluginManagerEx.registerCommand(script, "changeShed", changeShed);
    function changeShed(args) {
        // @ts-ignore
        const livestockIndex = getLivestockIndexFromEvent.bind(this)(args);
        Iz.Life.Dairy.changeShed(livestockIndex, args.shedId);
    }
    PluginManagerEx.registerCommand(script, "removeLivestock", removeLivestock);
    function removeLivestock(args) {
        // @ts-ignore
        const livestockIndex = getLivestockIndexFromEvent.bind(this)(args);
        Iz.Life.Dairy.removeLivestock(livestockIndex);
    }
    PluginManagerEx.registerCommand(script, "newday", (args) => {
        Iz.Life.Dairy.newday();
    });
    PluginManagerEx.registerCommand(script, "copyShed", (args) => {
        Iz.Life.DairyUtil.copyShed(args.srcShed, args.dstShed, {
            livestockSpawnRegion: args.livestockSpawnRegion,
            excludeLastLivestock: args.excludeLastLivestock,
            excludeHatchedLivestock: args.excludeHatchedLivestock,
            excludeBirthedLivestock: args.excludeBirthedLivestock,
        });
    });
    function getOverheadKey(livestockIndex) {
        return Iz.Life.getLivestockOverheadKey(livestockIndex);
    }
    const _createGameObjects = DataManager.createGameObjects;
    DataManager.createGameObjects = function () {
        _createGameObjects.call(this);
        Iz.Life.initializeDairy();
    };
    const _DataManager_makeSaveContents = DataManager.makeSaveContents;
    DataManager.makeSaveContents = function () {
        const contents = _DataManager_makeSaveContents.call(this);
        contents.dairy = {
            livestocks: Iz.Life.Dairy.livestocks.map((s) => s.cloneData()),
            livestockSheds: Iz.Life.Dairy.livestockSheds.map((s) => s.getData()),
        };
        return contents;
    };
    const _DataManager_extractSaveContents = DataManager.extractSaveContents;
    DataManager.extractSaveContents = function (contents) {
        _DataManager_extractSaveContents.call(this, contents);
        // TODO:
        // セーブデータをそのまま設定するのではなく、初期化したデータに適用するようにしたい
        // （※不正なデータが反映されないように）
        // Iz.Life.Dairy.initialize(contents.dairy.livestocks,contents.dairy.livestockSheds);
        Iz.Life.Dairy.apply(contents.dairy.livestocks, contents.dairy.livestockSheds);
    };
    const _Scene_Map_onMapLoaded = Scene_Map.prototype.onMapLoaded;
    Scene_Map.prototype.onMapLoaded = function () {
        _Scene_Map_onMapLoaded.call(this);
        $gameMap.refreshDairy();
    };
    Game_Map.prototype.refreshDairy = function () {
        const events = this.events().filter((e) => Iz.Life.isIncubatorEvent(e));
        events.forEach((e) => {
            const mapId = this.mapId();
            const eventId = e.eventId();
            const icIndex = e.getIncubatorIndex();
            const shedId = Iz.Life.getLivestockShedId(mapId);
            const existIncubator = Iz.Life.Dairy.existIncubator(shedId, icIndex);
            $gameSelfSwitches.setValue([mapId, eventId, "A"], existIncubator);
            const existEgg = Iz.Life.Dairy.existEgg(Iz.Life.getLivestockShedId(mapId), icIndex);
            $gameSelfSwitches.setValue([mapId, eventId, "B"], existEgg);
        });
        const feeds = this.events().filter((e) => Iz.Life.isFeedBox(e));
        feeds.forEach((e, i) => {
            e.refreshLivestockFeed();
        });
    };
    Game_Map.prototype.createLivestockEvent = function (index, templateId, x, y, templateMapId = Iz.Life.DairyTemplateMapId) {
        const event = this.spawnEventAndGet(templateId, x, y, true, templateMapId);
        if (event) {
            event._livestockIndex = index;
            const d = [2, 4, 6, 8].sort((a, b) => 0.5 - Math.random())[0];
            event.setDirection(d);
        }
        return event;
    };
    const _Game_Map_setupEvents = Game_Map.prototype.setupEvents;
    Game_Map.prototype.setupEvents = function () {
        _Game_Map_setupEvents.call(this);
        this.setupLivestockEvents();
    };
    Game_Map.prototype.setupLivestockEvents = function (option) {
        // 餌設定
        const feeds = this.events().filter((e) => Iz.Life.isFeedBox(e));
        feeds.forEach((e, i) => {
            e.setLivestockFeedIndex(i);
            e.refreshLivestockFeed();
        });
        const shedId = Iz.Life.getLivestockShedId(this.mapId());
        const templateMapId = Iz.Life.isEventShed(shedId)
            ? Iz.Life.DairyEventTemplateMapId
            : Iz.Life.DairyTemplateMapId;
        // 孵化器
        {
            const shed = Iz.Life.Dairy.livestockSheds.find((s) => s.id === shedId);
            if (shed) {
                const points = Iz.Life.getIncubatorDataList(shedId);
                const isEventShed = Iz.Life.isEventShed(shedId);
                shed.incubators
                    .filter((ic) => ic)
                    .map((ic) => points[ic.index])
                    .forEach((data) => {
                    const event = this.spawnEventAndGet(Iz.Life.IncubatorEventId, data.x, data.y, true, templateMapId);
                    event?.setIncubatorIndex(data.index);
                    // イベントマップでは設置、撤去できないように
                    if (!isEventShed) {
                        event?.setEventType(Iz.Life.EventType.Incubator);
                    }
                });
            }
        }
        // 家畜生成
        let spawnPoints = Iz.Life.getDairySpawnPoints(true);
        const spawnRegion = option?.livestockSpawnRegion;
        if (spawnRegion) {
            spawnPoints = spawnPoints.filter((p) => Iz.Life.isRegion(p.x, p.y, spawnRegion));
        }
        const livestocks = Iz.Life.Dairy.livestocks.filter((s) => Iz.Life.getLivestockMapId(s.shedId) === this._mapId);
        let spawnIndex = 0;
        livestocks.forEach((s) => {
            if (spawnIndex >= spawnPoints.length) {
                spawnIndex = 0;
            }
            const p = spawnPoints[spawnIndex++];
            const event = this.createLivestockEvent(s.index, s.templateId, p.x, p.y, templateMapId);
            if (!event)
                return;
            event._throughOnPlayerMove = $dataDairyLivestock[s.id]?.through_on_player_move ?? false;
            event._pushMoveData.enable = $dataDairyLivestock[s.id]?.push_move ?? false;
            if (Iz.Life.isEventShed(s.shedId))
                return;
            if (s.type === Iz.Life.Data.LivestockType.Cow && s.canProducted()) {
                const key = getOverheadKey(event.getLivestockIndex());
                const MilkBalloon = 11;
                this.getOverhead().setParam(key, {
                    balloonId: MilkBalloon,
                    duration: 20,
                });
            }
        });
        // pickItem
        const shed = Iz.Life.Dairy.getShed(Iz.Life.getLivestockShedId(this._mapId));
        if (!shed)
            return;
        const eggList = Iz.Life.getEggList();
        shed.pickItems.forEach((data) => {
            // たまご設置
            if (eggList.map((egg) => egg.id).includes(data.itemId)) {
                if (spawnIndex >= spawnPoints.length) {
                    spawnIndex = 0;
                }
                const p = spawnPoints[spawnIndex++];
                const egg = this.spawnEventAndGet(Iz.Life.LivestockTemplateId_Egg, p.x, p.y, true, templateMapId);
                if (egg) {
                    egg._livestockPickKey = data.key;
                }
                return;
            }
        });
    };
    Game_Event.prototype.getLivestockIndex = function () {
        return this._livestockIndex ?? -1;
    };
    Game_Event.prototype.getLivestockFeedIndex = function () {
        return this._livestockFeedIndex ?? -1;
    };
    Game_Event.prototype.setLivestockFeedIndex = function (index) {
        this._livestockFeedIndex = index;
    };
    Game_Event.prototype.gainLivestockFeed = function (value) {
        const index = this.getLivestockFeedIndex();
        if (index < 0)
            return;
        Iz.Life.Dairy.addFeed($gameMap.mapId(), index, value);
    };
    Game_Event.prototype.refreshLivestockFeed = function () {
        if (this._mapId !== $gameMap.mapId())
            return;
        const amount = Iz.Life.Dairy.getFeed(this._mapId, this.getLivestockFeedIndex());
        $gameSelfSwitches.setValue([this._mapId, this.eventId(), "D"], amount > 0);
    };
    Game_Event.prototype.getLivestockPickKey = function () {
        return this._livestockPickKey;
    };
    Game_Event.prototype.playOverhead = function (balloonId) {
        if (this.getLivestockIndex() < 0) {
            return;
        }
        const key = getOverheadKey(this.getLivestockIndex());
        $gameMap.getOverhead().playOneshot(key, balloonId);
    };
    Game_Event.prototype.getIncubatorIndex = function () {
        return this._incubatorIndex ?? -1;
    };
    Game_Event.prototype.setIncubatorIndex = function (index) {
        this._incubatorIndex = index;
    };
    const _Sprite_Character_initialize = Sprite_Character.prototype.initialize;
    Sprite_Character.prototype.initialize = function (character) {
        _Sprite_Character_initialize.call(this, character);
        if (character instanceof Game_Event) {
            if (character.getLivestockFeedIndex() >= 0) {
                const s = new SpriteAutoFeeder(character);
                s.x = 4;
                s.y = -64;
                s.visible = false;
                this.addChild(s);
            }
        }
    };
    const _Spriteset_Map_makePrefabEventSprite = Spriteset_Map.prototype.makePrefabEventSprite;
    Spriteset_Map.prototype.makePrefabEventSprite = function (event) {
        _Spriteset_Map_makePrefabEventSprite.call(this, event);
        const target = this._characterSprites[this._characterSprites.length - 1];
        if (event.getLivestockIndex() >= 0) {
            const key = getOverheadKey(event.getLivestockIndex());
            const sprite = new Sprite_OverheadBalloon(target, key);
            this._effectsContainer.addChild(sprite);
        }
    };
    const _Window_Base_convertEscapeCharacters = Window_Base.prototype.convertEscapeCharacters;
    Window_Base.prototype.convertEscapeCharacters = function (text_) {
        let text = text_;
        text = _Window_Base_convertEscapeCharacters.call(this, text);
        // 制御文字で孵化までの残り日数を表示
        const character = $gameMap.interpreter().character(0);
        if (character && character.getIncubatorIndex() >= 0) {
            const shed = Iz.Life.Dairy.livestockSheds.find((shed) => shed.id === Iz.Life.getLivestockShedId($gameMap.mapId()));
            if (shed) {
                const incubator = shed.getIncubator(character.getIncubatorIndex());
                if (incubator) {
                    const days = Math.max(0, $parameter.dairy.hatching_days - incubator.elapsedDays);
                    text = text.replace(/\x1bDAIRY_EGG/gi, `${days}`);
                }
            }
        }
        return text;
    };
})();
//# sourceMappingURL=Dairy.js.map