/*:
 * @target MZ
 * @plugindesc 施設関連
 *
 * @help
 *
 *
 *
 */
var Iz;
(function (Iz) {
    var Life;
    (function (Life) {
        const FacilityLevelMin = 1;
        const FacilityLevelMax = 3;
        class ConditionDayOfWeek {
            check(facilityId, index) {
                const data = $dataFacilitySchedule.find((data) => data && data.id === facilityId && data.index === index);
                if (!data) {
                    return false;
                }
                const dayOfWeek = data.condition_type_id;
                return Iz.System.getDayOfWeek() === dayOfWeek;
            }
        }
        class ConditionFacilityLevel {
            check(facilityId, index) {
                const data = $dataFacilitySchedule.find((data) => data && data.id === facilityId && data.index === index);
                if (!data) {
                    return false;
                }
                const facility = Life.Facility.get(facilityId);
                if (!facility) {
                    return false;
                }
                const level = data.condition_type_id;
                return facility.level === level;
            }
        }
        class FacilityInfo {
            constructor(data) {
                this._data = { id: 0, level: 0 };
                this.set(data);
            }
            get id() {
                return this._data.id;
            }
            get level() {
                return this._data.level;
            }
            set level(value) {
                this._data.level = Math.min(Math.max(value, FacilityLevelMin), FacilityLevelMax);
            }
            set(data) {
                this._data.id = data.id;
                this.level = data.level;
            }
            cloneData() {
                return Object.assign({}, this._data);
            }
        }
        Life.FacilityInfo = FacilityInfo;
        class FacilityProxy {
            constructor() {
                this._conditions = new Map();
                this._facilities = new Map();
                this._facilityMap = new Map();
            }
            get list() {
                return [...this._facilities.values()];
            }
            initialize() {
                this._conditions = new Map();
                this._conditions.set(Life.Data.FacilityConditionType.DayOfWeek, new ConditionDayOfWeek());
                this._conditions.set(Life.Data.FacilityConditionType.FacilityLevel, new ConditionFacilityLevel());
                this._facilities = new Map();
                [...$dataFacility.values()]
                    .sort((a, b) => a.id - b.id)
                    .forEach((data) => {
                    this._facilities.set(data.id, new FacilityInfo({ id: data.id, level: 1 }));
                });
                this._facilityMap = new Map();
                $dataFacilityMap
                    .filter((data) => data)
                    .forEach((data) => {
                    this._facilityMap.set(data.map, data.id);
                });
            }
            get(id) {
                return this._facilities.get(id);
            }
            getByMap(mapId) {
                const facilityId = this.getFacilityId(mapId);
                return this.get(facilityId);
            }
            getFacilityId(mapId) {
                const facilityId = this._facilityMap.get(mapId) ?? 0;
                return facilityId;
            }
            setData(data) {
                const info = this.get(data.id);
                if (info) {
                    info.set(data);
                }
            }
            isOpenFacility(id) {
                const info = this.get(id);
                if (!info) {
                    return false;
                }
                const schedule = this.getSchedule(id);
                if (!schedule) {
                    return false;
                }
                return isOpenTime(schedule);
            }
            isOpenShop(id) {
                const schedule = this.getShopSchedule(id);
                if (!schedule) {
                    return false;
                }
                // 定休日判定
                const closedDayofWeeks = this.getClosedDayofWeek(id);
                if (closedDayofWeeks.includes(Iz.System.getDayOfWeek())) {
                    return false;
                }
                // 臨時休業
                if (this.isTemporarilyClosed(id)) {
                    return false;
                }
                return isOpenTime(schedule);
            }
            isTemporarilyClosed(id) {
                const isTemporaryClosed = $dataShopClosedDay
                    .filter((data) => data && data.id === id)
                    .filter((data) => data.closed_type === Life.Data.ShopClosedType.Temporarily_Switch)
                    .some((data) => $gameSwitches.value(data.closed_type_id));
                return isTemporaryClosed;
            }
            getClosedDayofWeek(id) {
                const dayofWeeks = $dataShopClosedDay
                    .filter((data) => data && data.id === id)
                    .filter((data) => data.closed_type === Life.Data.ShopClosedType.DayOfWeek)
                    .map((data) => data.closed_type_id);
                return dayofWeeks;
            }
            getSchedule(id) {
                const info = this.get(id);
                if (!info) {
                    return undefined;
                }
                const schedules = $dataFacilitySchedule.filter((data) => data && data.id === id);
                for (let i = schedules.length - 1; i >= 0; i--) {
                    const schedule = schedules[i];
                    if (!schedule.condition_type) {
                        return schedule;
                    }
                    const conditionOk = this._conditions
                        .get(schedule.condition_type)
                        ?.check(schedule.id, schedule.index);
                    if (conditionOk) {
                        return schedule;
                    }
                }
                return undefined;
            }
            getShopSchedule(id) {
                const info = this.get(id);
                if (!info) {
                    return undefined;
                }
                const schedules = $dataShopSchedule.filter((data) => data && data.id === id);
                for (let i = schedules.length - 1; i >= 0; i--) {
                    const schedule = schedules[i];
                    if (!schedule.condition_type) {
                        return schedule;
                    }
                    const conditionOk = this._conditions
                        .get(schedule.condition_type)
                        ?.check(schedule.id, schedule.index);
                    if (conditionOk) {
                        return schedule;
                    }
                }
                return undefined;
            }
        }
        function isOpenTime(time) {
            const beginTime = Iz.System.toTime(time.begin_hour, time.begin_minute);
            const endTime = Iz.System.toTime(time.end_hour, time.end_minute);
            const currentTime = Iz.System.toTime(Iz.System.getHour(), Iz.System.getMinute());
            return currentTime >= beginTime && currentTime < endTime;
        }
        Life.isOpenTime = isOpenTime;
        Life.Facility = new FacilityProxy();
        function initializeFacility() {
            Life.Facility.initialize();
        }
        Life.initializeFacility = initializeFacility;
    })(Life = Iz.Life || (Iz.Life = {}));
})(Iz || (Iz = {}));
(function (Iz) {
    var Life;
    (function (Life) {
        var FacilityUtil;
        (function (FacilityUtil) {
            function getLevel(facilityId) {
                const facility = Life.Facility.get(facilityId);
                return facility ? facility.level : 1;
            }
            FacilityUtil.getLevel = getLevel;
            function setLevel(facilityId, level) {
                const facility = Life.Facility.get(facilityId);
                if (!facility)
                    return;
                facility.level = level;
            }
            FacilityUtil.setLevel = setLevel;
            function isOpen(facilityId) {
                return Life.Facility.isOpenFacility(facilityId);
            }
            FacilityUtil.isOpen = isOpen;
            function isOpenShop(facilityId) {
                return Life.Facility.isOpenShop(facilityId);
            }
            FacilityUtil.isOpenShop = isOpenShop;
        })(FacilityUtil = Life.FacilityUtil || (Life.FacilityUtil = {}));
    })(Life = Iz.Life || (Iz.Life = {}));
})(Iz || (Iz = {}));
(() => {
    "use strict";
    const _createGameObjects = DataManager.createGameObjects;
    DataManager.createGameObjects = function () {
        _createGameObjects.call(this);
        Iz.Life.initializeFacility();
    };
    const _DataManager_makeSaveContents = DataManager.makeSaveContents;
    DataManager.makeSaveContents = function () {
        const contents = _DataManager_makeSaveContents.call(this);
        contents.facility = {
            list: Iz.Life.Facility.list.map((info) => info.cloneData()),
        };
        return contents;
    };
    const _DataManager_extractSaveContents = DataManager.extractSaveContents;
    DataManager.extractSaveContents = function (contents) {
        _DataManager_extractSaveContents.call(this, contents);
        contents.facility.list.forEach((data) => {
            Iz.Life.Facility.setData(data);
        });
    };
    const FacilityOpenSwitchId = 50;
    const ShopOpenSwitchId = 51;
    const _Game_Map_update = Game_Map.prototype.update;
    Game_Map.prototype.update = function (sceneActive) {
        if (sceneActive) {
            const facility = Iz.Life.Facility.getByMap(this.mapId());
            if (facility) {
                // 施設
                {
                    const current = $gameSwitches.value(FacilityOpenSwitchId);
                    const next = Iz.Life.Facility.isOpenFacility(facility.id);
                    if (current !== next) {
                        $gameSwitches.setValue(FacilityOpenSwitchId, next);
                    }
                }
                // 店
                {
                    const current = $gameSwitches.value(ShopOpenSwitchId);
                    const next = Iz.Life.Facility.isOpenShop(facility.id);
                    if (current !== next) {
                        $gameSwitches.setValue(ShopOpenSwitchId, next);
                    }
                }
            }
        }
        _Game_Map_update.call(this, sceneActive);
    };
})();
(() => {
    "use strict";
    function getFacilityOpenTimeText(facilityId) {
        const schedule = Iz.Life.Facility.getSchedule(facilityId);
        if (!schedule) {
            return "";
        }
        const begin = Iz.System.toChronusFormat(schedule.begin_hour, schedule.begin_minute);
        const end = Iz.System.toChronusFormat(schedule.end_hour, schedule.end_minute);
        return Iz.System.toChronusDurationFormat(begin, end);
    }
    function getShopOpenTimeText(facilityId) {
        const schedule = Iz.Life.Facility.getShopSchedule(facilityId);
        if (!schedule) {
            return "";
        }
        const begin = Iz.System.toChronusFormat(schedule.begin_hour, schedule.begin_minute);
        const end = Iz.System.toChronusFormat(schedule.end_hour, schedule.end_minute);
        return Iz.System.toChronusDurationFormat(begin, end);
    }
    const _Window_Base_convertEscapeCharacters = Window_Base.prototype.convertEscapeCharacters;
    Window_Base.prototype.convertEscapeCharacters = function (text) {
        let res = _Window_Base_convertEscapeCharacters.call(this, text);
        res = res.replace(/\x1bFACTIME\[(\d+)\]/gi, (_, p1) => getFacilityOpenTimeText(Number.parseInt(p1)));
        res = res.replace(/\x1bSHOPTIME\[(\d+)\]/gi, (_, p1) => getShopOpenTimeText(Number.parseInt(p1)));
        return res;
    };
})();
//# sourceMappingURL=Facility.js.map