/*:
 * @target MZ
 * @plugindesc インジケータ
 *
 * @help
 * イベントのメモ欄にタグを付けてインジケータのラベルを表示
 *
 * ・１つ表示したい場合
 * <indicator:名前>
 * 「名前」部分が表示名になります
 *
 * ・複数表示したい場合
 * <indicator:["名前１","名前２"]>
 * 配列で定義します（※２つまで表示可、配列の先頭側が下に表示）
 *
 *
 * ==============================
 * マスターデータ定義
 * ==============================
 *
 * マスターデータに設定をすることでエイリアスの定義、
 * 先頭にアイコンの表示等を行えます
 *
 * [定義]
 * id: エイリアス名
 * input_key: 入力名
 * display_name: 表示名
 *
 * 例えば以下の設定では エイリアスをpickにして「拾う」と表示します
 *
 * [定義側]
 * id: pick
 * display_name: 拾う
 *
 * [タグ側]
 * <indicator:["pick"]>
 * ※エイリアスを使う場合は配列にしてください
 *
 * ---------
 *
 * input_keyにはラベルがどの入力に対応したものかを設定します
 * 現状用意してあるものは以下
 *
 * action
 * sub
 * ok
 * cancel
 *
 * 各input_keyに対応したアイコンが自動で表示されます
 * （actionならZ、subならX　etc..）
 * アイコンの表示がいらない（テキストだけ）なら空欄にしてください
 *
 *
 *
 */
var Iz;
(function (Iz) {
    var Life;
    (function (Life) {
        Life.IndicatorTag = "indicator";
        Life.IndicatorMax = 2;
        class IndicatorProxy {
            constructor() {
                this._infos = [];
                for (let i = 0; i < Life.IndicatorMax; i++) {
                    const info = { enable: false, key: "", text: "", atlasKey: "", frameKey: "" };
                    this._infos[i] = info;
                }
            }
            getAtlasKey(key) {
                const param = $dataIndicator.get(key);
                if (!param || !param.input_key)
                    return "";
                const imageParam = Life.AtlasInputImageList.get(param.input_key);
                return imageParam ? imageParam.atlasKey : "";
            }
            getFrameKey(key) {
                const param = $dataIndicator.get(key);
                if (!param || !param.input_key)
                    return "";
                const imageParam = Life.AtlasInputImageList.get(param.input_key);
                return imageParam ? imageParam.frameKey : "";
            }
            setInfos(keyList) {
                const len = Math.min(Math.max(keyList.length, 0), this._infos.length);
                for (let i = 0; i < len; i++) {
                    const key = keyList[i];
                    const param = $dataIndicator.get(key);
                    const info = this._infos[i];
                    info.key = key;
                    info.text = param ? param.display_name : key;
                    info.atlasKey = this.getAtlasKey(key);
                    info.frameKey = this.getFrameKey(key);
                    info.enable = true;
                }
            }
            clear() {
                this._infos.forEach((info) => {
                    info.enable = false;
                });
            }
            getInfo(index) {
                return this._infos[index];
            }
            has(key) {
                return !!this._infos.find((info) => info.enable && info.key === key);
            }
        }
        Life.Indicator = new IndicatorProxy();
    })(Life = Iz.Life || (Iz.Life = {}));
})(Iz || (Iz = {}));
(function (Iz) {
    var Life;
    (function (Life) {
        const MaxWidth = 320;
        const MinWidth = 160;
        const Height = 28;
        class Sprite_Indicator extends Sprite {
            constructor() {
                const bitmap = new Bitmap(MaxWidth, Height);
                // 領域確認用
                // bitmap.fillAll("yellow");
                super(bitmap);
                this.z = 11;
                this.setFrame(0, 0, MaxWidth, Height);
                this.createTextRoot();
                this.createText();
                this.createInputImage();
            }
            get textSprite() {
                return this._text;
            }
            calcRootWidth() {
                const offset = 64;
                const w = Math.floor(Math.min(Math.max(this._text.measureTextWidth() + offset, MinWidth), MaxWidth));
                return w;
            }
            setText(text) {
                this._text.setText(text);
                const width = this.calcRootWidth();
                this.setRootWidth(width);
            }
            setRootWidth(w) {
                const offset = Math.floor((MaxWidth - w) / 2);
                this._root.bitmap?.clear();
                this._root.bitmap?.fillRect(offset, 0, w, Height, "black");
                this._input.x = offset;
            }
            setImage(atlasKey, frameKey) {
                if (this._atlasKey === atlasKey && this._frameKey === frameKey) {
                    return;
                }
                this._atlasKey = atlasKey;
                this._frameKey = frameKey;
                if (!atlasKey || !frameKey) {
                    this._input.bitmap = Sprite_Indicator.NullBitmap;
                    return;
                }
                this._input.setAtlas(atlasKey, frameKey);
            }
            createTextRoot() {
                const bitmap = new Bitmap(MaxWidth, Height);
                bitmap.paintOpacity = 128;
                bitmap.fillAll("black");
                const sprite = new Sprite(bitmap);
                this.addChild(sprite);
                this._root = sprite;
            }
            createText() {
                const sprite = new Sprite_Text(MaxWidth, Height);
                sprite.setAlign("center");
                sprite.setText("");
                sprite.setFontSize(18);
                this._root.addChild(sprite);
                this._text = sprite;
            }
            createInputImage() {
                const bitmap = new Bitmap(24, 24);
                const sprite = new Sprite(bitmap);
                // sprite.y = 2;
                // sprite.x = 5;
                this._root.addChild(sprite);
                this._input = sprite;
            }
        }
        Sprite_Indicator.NullBitmap = new Bitmap(24, 24);
        Life.Sprite_Indicator = Sprite_Indicator;
        const COMMAND_CODE_COMMENT = 108;
        function extractIndicatorComment(list) {
            return list
                .filter((data) => data.code === COMMAND_CODE_COMMENT)
                .map((data) => {
                const regex = /<indicator:(\S+)>/;
                const result = regex.exec(data.parameters[0]);
                const info = { tags: [] };
                if (result) {
                    if (result[1].startsWith("[")) {
                        info.tags = JSON.parse(result[1]);
                    }
                    else {
                        info.tags = [result[1]];
                    }
                }
                return info;
            })
                .filter((data) => data.tags.length > 0);
        }
        Life.extractIndicatorComment = extractIndicatorComment;
        function getIndicatorLabel(list) {
            return extractIndicatorComment(list).find((data) => data.tags.length > 0);
        }
        Life.getIndicatorLabel = getIndicatorLabel;
        function existNoIndicator(list) {
            return list.some((command) => {
                if (command.code !== COMMAND_CODE_COMMENT)
                    return false;
                if (command.parameters[0] !== "<NoIndicator:>")
                    return false;
                return true;
            });
        }
        Life.existNoIndicator = existNoIndicator;
    })(Life = Iz.Life || (Iz.Life = {}));
})(Iz || (Iz = {}));
(() => {
    const _Game_Event_setupPage = Game_Event.prototype.setupPage;
    Game_Event.prototype.setupPage = function () {
        _Game_Event_setupPage.call(this);
        const page = this.page();
        if (page) {
            this._noIndicator = Iz.Life.existNoIndicator(page.list);
            this._indicatorLabel = Iz.Life.getIndicatorLabel(page.list);
        }
    };
    const _Spriteset_Map_CreateLowerLayer = Spriteset_Map.prototype.createLowerLayer;
    Spriteset_Map.prototype.createLowerLayer = function () {
        _Spriteset_Map_CreateLowerLayer.call(this);
        this._indicators = [];
        for (let i = 0; i < Iz.Life.IndicatorMax; i++) {
            const sprite = new Iz.Life.Sprite_Indicator();
            this._tilemap.addChild(sprite);
            this._indicators.push(sprite);
        }
    };
    const _Spriteset_Map_update = Spriteset_Map.prototype.update;
    Spriteset_Map.prototype.update = function () {
        _Spriteset_Map_update.call(this);
        const isEventRunning = $gameMap.isEventRunning();
        let changed = false;
        for (let i = 0; i < this._indicators.length; i++) {
            const sprite = this._indicators[i];
            const info = Iz.Life.Indicator.getInfo(i);
            if (!info.enable || isEventRunning) {
                if (sprite.visible) {
                    changed = true;
                }
                sprite.visible = false;
                continue;
            }
            if (!sprite.visible) {
                changed = true;
            }
            sprite.visible = true;
            const tileWidth = $gameMap.tileWidth();
            const tileHeight = $gameMap.tileHeight();
            const baseX = $gameMap.adjustX($gamePlayer._realX) * tileWidth - 134;
            const baseY = $gameMap.adjustY($gamePlayer._realY) * tileHeight - 80 - i * 32;
            sprite.position.set(baseX, baseY);
            if (sprite.textSprite.text !== info.text) {
                sprite.setText(info.text);
                changed = true;
            }
            sprite.setImage(info.atlasKey, info.frameKey);
        }
        if (changed) {
            let maxWidth = 0;
            const list = this._indicators.filter((s) => s.visible);
            list.forEach((s) => {
                const width = s.calcRootWidth();
                if (width > maxWidth) {
                    maxWidth = width;
                }
            });
            list.forEach((s) => {
                s.setRootWidth(maxWidth);
            });
        }
    };
})();
// 個別対応
(function (Iz) {
    var Life;
    (function (Life) {
        Life.IndicatorKey_Talk = "talk";
        Life.IndicatorKey_Gift = "gift";
        Life.IndicatorKey_Fishing = "fishing";
        Life.IndicatorKey_ConsumeUse = "consume_use";
        Life.IndicatorKey_Plant = "plant";
        Life.IndicatorKey_Brush = "brushing";
        Life.IndicatorKey_Pregnancy = "pregnancy";
        Life.IndicatorKey_Milking = "milking";
        Life.IndicatorKey_Sprinkler_Put = "sprinkler_put";
        Life.IndicatorKey_Sprinkler_Pick = "sprinkler_pick";
        Life.IndicatorKey_Sprinkler_Recover = "sprinkler_recover";
        Life.IndicatorKey_FarmLeveling = "farm_leveling";
        Life.IndicatorKey_Hoe = "hoe";
        Life.IndicatorKey_WaterPot = "waterpot";
        Life.IndicatorKey_TakeWater = "takewater";
        Life.IndicatorKey_Ax = "ax";
        Life.IndicatorKey_Incubator_Put = "incubator_put";
        Life.IndicatorKey_Incubator_Remove = "incubator_remove";
        Life.IndicatorKey_Incubator_PutEgg = "incubator_put_egg";
        function getEventTags(event) {
            if (!drawableIndicator(event))
                return undefined;
            const label = event._indicatorLabel;
            if (label) {
                return [...label.tags];
            }
            if (Object.hasOwn(event.event().meta, Iz.Life.IndicatorTag)) {
                const keyList = PluginManagerEx.findMetaObject(event.event(), [Iz.Life.IndicatorTag]);
                if (Array.isArray(keyList._parameter)) {
                    // ex.) <indicator:[pick,mining]>
                    return [...keyList._parameter];
                }
                // ex.) <indicator:寝る>
                return new Array(keyList._parameter);
            }
            return undefined;
        }
        Life.getEventTags = getEventTags;
        function drawableIndicator(event) {
            // インジケータ非表示設定
            if (event._noIndicator)
                return false;
            // セルフスイッチがDのイベントはインジケータを表示しない
            const key = [event._mapId, event.eventId(), "D"];
            return !$gameSelfSwitches.value(key);
        }
    })(Life = Iz.Life || (Iz.Life = {}));
})(Iz || (Iz = {}));
//# sourceMappingURL=Indicator.js.map