/*:
 * @target MZ
 * @plugindesc クエスト
 * @author INAZUMA GAMES
 *
 * @help
 *
 *
 *
 */
var Iz;
(function (Iz) {
    var Life;
    (function (Life) {
        const WakeupSwitchId = 1007;
        Life.QuestState = {
            Init: 0,
            Unlocked: 1,
            InProgress: 2,
            Complete: 3,
        };
        const NotifyType = {
            Unlock: 0,
            Start: 1,
            Forward: 2,
            Complete: 3,
        };
        class ConditionCheckTime {
            isValidTime(hour, minute) {
                if (!Number.isInteger(hour))
                    return false;
                if (hour < 0 || hour >= 25)
                    return false;
                if (!Number.isInteger(minute))
                    return false;
                if (minute < 0 || minute >= 60)
                    return false;
                return true;
            }
            convertConditionTime(s) {
                const time = { hour: -1, minute: 0 };
                const splits = s.split(":");
                if (splits.length === 2) {
                    const hour = Number.parseInt(splits[0]);
                    const minute = Number.parseInt(splits[1]);
                    if (this.isValidTime(hour, minute)) {
                        time.hour = hour;
                        time.minute = minute;
                    }
                    else {
                        console.warn("不正な時間を検出");
                    }
                }
                else {
                    console.warn("不正な時間設定を検出");
                }
                return time;
            }
            check(id) {
                const data = this.getData(id);
                if (!data)
                    return true;
                const timeBegin = this.convertConditionTime(data.time_begin);
                const timeEnd = this.convertConditionTime(data.time_end);
                if (!this.isValidTime(timeBegin.hour, timeBegin.minute))
                    return false;
                if (!this.isValidTime(timeEnd.hour, timeEnd.minute))
                    return false;
                const hour = Iz.GlobalMap.getChronus().getHour();
                const minute = Iz.GlobalMap.getChronus().getMinute();
                const b1 = hour > timeBegin.hour || (hour === timeBegin.hour && minute >= timeBegin.minute);
                const b2 = hour < timeEnd.hour || (hour === timeEnd.hour && minute <= timeBegin.minute);
                return b1 && b2;
            }
            text(id) {
                const data = this.getData(id);
                if (!data)
                    return "";
                return Iz.String.format(Iz.Text.get("QUEST_CONDITION_FORMAT_1"), data.time_begin, data.time_end);
            }
            getData(id) {
                return $dataQuestConditionTime.get(id);
            }
        }
        class ConditionCheckDayOfWeek {
            check(id) {
                const data = this.getData(id);
                if (!data)
                    return true;
                return Iz.GlobalMap.getChronus().getWeekIndex() === data.day_of_week;
            }
            text(id) {
                const data = this.getData(id);
                if (!data)
                    return "";
                const dayOfWeekName = Iz.System.getDayOfWeekName(data.day_of_week);
                return Iz.String.format(Iz.Text.get("QUEST_CONDITION_FORMAT_16"), dayOfWeekName);
            }
            getData(id) {
                return $dataQuestConditionDayOfWeek.get(id);
            }
        }
        class ConditionCheckItem {
            check(id) {
                const data = this.getData(id);
                if (!data)
                    return true;
                return $gameParty.numItems($dataItems[data.item_id]) >= data.num;
            }
            text(id) {
                const data = this.getData(id);
                if (!data)
                    return "";
                return Iz.String.format(Iz.Text.get("QUEST_CONDITION_FORMAT_2"), $dataItems[data.item_id].name, data.num);
            }
            onStart(id) {
                this.checkConsumeItem(id);
            }
            onComplete(id) {
                this.checkConsumeItem(id);
            }
            checkConsumeItem(id) {
                const data = this.getData(id);
                if (!data)
                    return;
                if (data.consume_on_update) {
                    $gameParty.loseItem($dataItems[data.item_id], data.num, false);
                }
            }
            getData(id) {
                return $dataQuestConditionItem.get(id);
            }
        }
        class ConditionCheckGroup {
            check(id) {
                const data = this.getData(id);
                if (!data)
                    return true;
                return Life.getItemNumByGroup(data.group_id) >= data.num;
            }
            text(id) {
                const data = this.getData(id);
                if (!data)
                    return "";
                return Iz.String.format(Iz.Text.get("QUEST_CONDITION_FORMAT_3"), Life.getItemGroupName(data.group_id), data.num);
            }
            onComplete(id) {
                const data = this.getData(id);
                if (!data)
                    return;
                if (data.consume_on_complete) {
                    Life.consumeItemByGroup(data.group_id, data.num);
                }
            }
            getData(id) {
                return $dataQuestConditionGroup.get(id);
            }
        }
        class ConditionCheckMap {
            check(id) {
                const data = this.getData(id);
                if (!data)
                    return true;
                return $gameMap.mapId() === data.map;
            }
            text(id) {
                const data = this.getData(id);
                if (!data)
                    return "";
                return Iz.System.getMapName(data.map);
            }
            getData(id) {
                return $dataQuestConditionMap.get(id);
            }
        }
        class ConditionCheckLocation {
            check(id) {
                const data = this.getData(id);
                if (!data)
                    return true;
                const location = $globalMap.getLocation($gameMap.mapId(), $gamePlayer.x, $gamePlayer.y);
                return !!location && data.location === location.id;
            }
            text(id) {
                const data = this.getData(id);
                if (!data)
                    return "";
                const location = $dataLocation[data.location];
                return location ? location.name : "";
            }
            getData(id) {
                return $dataQuestConditionLocation.get(id);
            }
        }
        class ConditionCheckCharacterMap {
            check(id) {
                const data = this.getData(id);
                if (!data)
                    return true;
                return ($gameMap.mapId() > 0 &&
                    $gameMap.mapId() === $globalMap.nonPlayer(data.character).mapId());
            }
            text(id) {
                const data = this.getData(id);
                if (!data)
                    return "";
                return Iz.System.getMapName($globalMap.nonPlayer(data.character).mapId());
            }
            getData(id) {
                return $dataQuestConditionCharacterMap.get(id);
            }
        }
        class ConditionCheckCharacterLocation {
            check(id) {
                const data = this.getData(id);
                if (!data)
                    return true;
                const character = $globalMap.nonPlayer(data.character);
                return $gamePlayer.isSameLocation(character);
            }
            text(id) {
                const data = this.getData(id);
                if (!data)
                    return "";
                const location = $globalMap.nonPlayer(data.character).location();
                if (!location)
                    return "";
                return location.name;
            }
            getData(id) {
                return $dataQuestConditionCharacterLocation.get(id);
            }
        }
        class ConditionCheckWeather {
            check(id) {
                const data = this.getData(id);
                if (!data)
                    return true;
                if (data.weather < 0)
                    return true;
                return Iz.Life.getWeatherType() === data.weather;
            }
            text(id) {
                const data = this.getData(id);
                if (!data)
                    return "";
                const weatherName = Iz.Text.get(`WEATHER_${data.weather}`);
                return Iz.String.format(Iz.Text.get("QUEST_CONDITION_FORMAT_6"), weatherName);
            }
            getData(id) {
                return $dataQuestConditionWeather.get(id);
            }
        }
        class ConditionCheckTalkCharacter {
            check(id) {
                const data = this.getData(id);
                if (!data)
                    return true;
                if (!$gameMap.isEventRunning())
                    return false;
                const character = $gameMap.interpreter().character(0);
                if (!Iz.GlobalMap.IsNPC(character))
                    return false;
                return character.characterId === data.talk_character;
            }
            text(id) {
                const data = this.getData(id);
                if (!data)
                    return "";
                const actor = $gameActors.actor(data.talk_character);
                return Iz.String.format(Iz.Text.get("QUEST_CONDITION_FORMAT_7"), actor ? actor.name() : "");
            }
            getData(id) {
                return $dataQuestConditionTalkCharacter.get(id);
            }
        }
        class ConditionCheckLikability {
            check(id) {
                const data = this.getData(id);
                if (!data)
                    return true;
                return Life.Status.getLikability(data.character) >= data.value;
            }
            text(id) {
                const data = this.getData(id);
                if (!data)
                    return "";
                const actor = $gameActors.actor(data.character);
                return Iz.String.format(Iz.Text.get("QUEST_CONDITION_FORMAT_8"), actor ? actor.name() : "", data.value);
            }
            getData(id) {
                return $dataQuestConditionLikability.get(id);
            }
        }
        class ConditionCheckLikabilityLevel {
            check(id) {
                const data = this.getData(id);
                if (!data)
                    return true;
                return Life.Status.getLikabilityLevel(data.character) >= data.value;
            }
            text(id) {
                const data = this.getData(id);
                if (!data)
                    return "";
                const actor = $gameActors.actor(data.character);
                return Iz.String.format(Iz.Text.get("QUEST_CONDITION_FORMAT_18"), actor ? actor.name() : "", 
                // data.value,
                `\\IL[likability${data.value}]`);
            }
            getData(id) {
                return $dataQuestConditionLikabilityLevel.get(id);
            }
        }
        class ConditionCheckQuest {
            check(id) {
                const data = this.getData(id);
                if (!data)
                    return true;
                return Life.Quest.getState(data.quest_id) === Life.QuestState.Complete;
            }
            text(id) {
                const data = this.getData(id);
                if (!data)
                    return "";
                const questContent = Life.Quest.getQuestContent(data.quest_id);
                return Iz.String.format(Iz.Text.get("QUEST_CONDITION_FORMAT_9"), questContent ? questContent.title : "");
            }
            getData(id) {
                return $dataQuestConditionQuest.get(id);
            }
        }
        class ConditionCheckTarget {
            check(id) {
                const data = this.getData(id);
                if (!data)
                    return true;
                if (!$gameMap.isEventRunning())
                    return false;
                if ($gameMap.mapId() !== data.map)
                    return false;
                const character = $gameMap.interpreter().character(0);
                if (!character)
                    return false;
                const integer = Number.parseInt(data.event);
                let event;
                if (Number.isInteger(integer) && integer > 0) {
                    return character.eventId() === integer;
                }
                return character.event().name === data.event;
            }
            text(id) {
                const data = this.getData(id);
                if (!data)
                    return "";
                return Iz.String.format(Iz.Text.get("QUEST_CONDITION_FORMAT_10"), data.display_name);
            }
            getData(id) {
                return $dataQuestConditionTarget.get(id);
            }
        }
        class ConditionCheckLivestock {
            check(id) {
                const data = this.getData(id);
                if (!data)
                    return true;
                switch (data.range) {
                    case Life.Data.QuestLivestockRange.Shed:
                        return Life.DairyUtil.getHappinessShed(data.shed_id) >= data.value;
                    case Life.Data.QuestLivestockRange.All:
                        return Life.DairyUtil.getHappinessAll() >= data.value;
                    default:
                        return false;
                }
            }
            text(id) {
                const data = this.getData(id);
                if (!data)
                    return "";
                const shed = $dataDairyShed[data.shed_id];
                return Iz.String.format(Iz.Text.get("QUEST_CONDITION_FORMAT_11"), `${shed ? shed.name : ""}`, data.value);
            }
            getData(id) {
                return $dataQuestConditionLivestock.get(id);
            }
        }
        class ConditionCheckWakeup {
            check(id) {
                // ※現状QuestConditionWakeupの情報はいらないので判定しない
                // const data = this.getData(id);
                // if(!data)
                //     return true;
                // 起床時判定
                return isWakeup();
            }
            text(id) {
                const data = this.getData(id);
                if (!data)
                    return "";
                return Iz.Text.get("QUEST_CONDITION_FORMAT_12");
            }
            getData(id) {
                return $dataQuestConditionWakeup.get(id);
            }
        }
        class ConditionCheck15 {
            check(id) {
                return true;
            }
            text(id) {
                return "未使用";
            }
        }
        class ConditionCheckBuilding {
            check(id) {
                const data = this.getData(id);
                if (!data)
                    return true;
                const BuildingSwitchId = 41;
                return $gameSwitches.value(BuildingSwitchId) === data.value;
            }
            text(id) {
                const data = this.getData(id);
                if (!data)
                    return "";
                return Iz.String.format("{0}", Iz.Text.get(`QUEST_CONDITION_FORMAT_17_${data.value ? 1 : 2}`));
            }
            getData(id) {
                return $dataQuestConditionBuilding.get(id);
            }
        }
        class ConditionCheckNpcMap {
            check(id) {
                const data = this.getData(id);
                if (!data)
                    return true;
                return data.map === $globalMap.nonPlayer(data.character).mapId();
            }
            text(id) {
                const data = this.getData(id);
                if (!data)
                    return "";
                const actor = $gameActors.actor(data.character);
                const mapName = Iz.System.getMapName(data.map);
                return Iz.String.format(Iz.Text.get("QUEST_CONDITION_FORMAT_19"), actor ? actor.name() : "", mapName);
            }
            getData(id) {
                return $dataQuestConditionNpcMap.get(id);
            }
        }
        class ConditionCheckWorkLevel {
            check(id) {
                const data = this.getData(id);
                if (!data)
                    return true;
                return Iz.Life.Status.getWorkLevel(data.type) >= data.level;
            }
            text(id) {
                const data = this.getData(id);
                if (!data)
                    return "";
                return Iz.String.format(Iz.Text.get(`QUEST_CONDITION_FORMAT_20_${data.type}`), data.level);
            }
            getData(id) {
                return $dataQuestConditionWorkLevel.get(id);
            }
        }
        class ConditionCheckRequest {
            check(id) {
                const data = this.getData(id);
                if (!data)
                    return true;
                return $gameParty.numItems($dataItems[data.item_id]) >= 1;
            }
            text(id) {
                const data = this.getData(id);
                if (!data)
                    return "";
                const item = $dataItems[data.item_id];
                if (!item)
                    return "";
                return Iz.String.format(Iz.Text.get("QUEST_CONDITION_FORMAT_21"), item.name);
            }
            onComplete(id) {
                const data = this.getData(id);
                if (!data)
                    return;
                // 依頼のアイテムは自動消費する
                $gameParty.loseItem($dataItems[data.item_id], 1, false);
            }
            getData(id) {
                return $dataQuestConditionRequest.get(id);
            }
        }
        class ConditionCheckElapsedDays {
            check(id) {
                const data = this.getData(id);
                if (!data)
                    return true;
                return Life.Quest.getLeftDays(data.quest_id) <= 0;
            }
            text(id) {
                const data = this.getData(id);
                if (!data)
                    return "";
                return Iz.String.format(Iz.Text.get("QUEST_CONDITION_FORMAT_22"), Life.Quest.getLeftDays(data.quest_id));
            }
            onBegin(id) {
                const data = this.getData(id);
                if (!data)
                    return;
                Life.Quest.setLeftDays(data.quest_id, data.days);
            }
            getData(id) {
                return $dataQuestConditionElapsedDays.get(id);
            }
        }
        class ConditionCheckPlaceGroup {
            check(id) {
                const data = this.getData(id);
                if (!data)
                    return true;
                return Life.includePlaceGroup({ placeGroupId: data.place_group, map: $gameMap.mapId() });
            }
            text(id) {
                const data = this.getData(id);
                if (!data)
                    return "";
                const placeData = $dataPlaceGroup.get(data.place_group);
                return placeData?.name ?? "";
            }
            getData(id) {
                return $dataQuestConditionPlaceGroup.get(id);
            }
        }
        class ConditionCheckDairyHatched {
            check(id) {
                const data = this.getData(id);
                if (!data)
                    return false;
                return !!Life.Dairy.hatchedLivestock === data.hatched;
            }
            text(id) {
                const data = this.getData(id);
                if (!data)
                    return "";
                return Iz.Text.get(data.hatched ? "QUEST_CONDITION_FORMAT_24_1" : "QUEST_CONDITION_FORMAT_24_2");
            }
            getData(id) {
                return $dataQuestConditionDairyHatched.get(id);
            }
        }
        class ConditionCheckDairyLivestocks {
            check(id) {
                const data = this.getData(id);
                if (!data)
                    return false;
                const targetLivestocks = [];
                if (data.livestock_id > 0) {
                    targetLivestocks.push(data.livestock_id);
                }
                else if (data.livestock_group > 0) {
                    targetLivestocks.push(...getLivestocksByGroup(data.livestock_group));
                }
                else {
                    return false;
                }
                const livestocksNum = Life.Dairy.livestocks
                    .filter((s) => !Life.isEventShed(s.shedId))
                    .filter((s) => targetLivestocks.includes(s.id)).length;
                switch (data.comparison_operator) {
                    case Life.Data.ComparisonOperator.NotEqual:
                        return livestocksNum !== data.num;
                    case Life.Data.ComparisonOperator.Equal:
                        return livestocksNum === data.num;
                    case Life.Data.ComparisonOperator.GreaterThanOrEqual:
                        return livestocksNum >= data.num;
                    case Life.Data.ComparisonOperator.GreaterThan:
                        return livestocksNum > data.num;
                    case Life.Data.ComparisonOperator.LessThanOrEqual:
                        return livestocksNum <= data.num;
                    case Life.Data.ComparisonOperator.LessThan:
                        return livestocksNum < data.num;
                    default:
                        return false;
                }
            }
            text(id) {
                const data = this.getData(id);
                if (!data)
                    return "";
                const format = Iz.Text.get(`QUEST_CONDITION_FORMAT_25_${data.comparison_operator}`);
                let name = "";
                if (data.livestock_id > 0) {
                    const dataLivestock = $dataDairyLivestock[data.livestock_id];
                    name = dataLivestock.name;
                }
                else if (data.livestock_group > 0) {
                    const dataGroup = $dataDairyLivestockGroup.get(data.livestock_group);
                    if (dataGroup) {
                        name = dataGroup.name;
                    }
                }
                return Iz.String.format(format, name, data.num);
            }
            getData(id) {
                return $dataQuestConditionDairyLivestocks.get(id);
            }
        }
        class ConditionCheckSeason {
            check(id) {
                const data = this.getData(id);
                if (!data)
                    return false;
                const month = $gameSystem.chronus().getMonth();
                return month === data.season;
            }
            text(id) {
                const data = this.getData(id);
                if (!data)
                    return "";
                const seasonName = Iz.System.getMonthName(data.season);
                const format = Iz.Text.get("QUEST_CONDITION_FORMAT_26");
                return Iz.String.format(format, seasonName);
            }
            getData(id) {
                return $dataQuestConditionSeason.get(id);
            }
        }
        class ConditionCheckBuy {
            check(id) {
                const data = this.getData(id);
                if (!data)
                    return false;
                return $gameParty.hasItem($dataItems[data.item_id]);
            }
            text(id) {
                const data = this.getData(id);
                if (!data)
                    return "";
                const item = $dataItems[data.item_id]?.name ?? "";
                const format = Iz.Text.get("QUEST_CONDITION_FORMAT_27");
                return Iz.String.format(format, item);
            }
            getData(id) {
                return $dataQuestConditionBuy.get(id);
            }
        }
        class ConditionCheckDate {
            check(id) {
                const data = this.getData(id);
                if (!data)
                    return false;
                const dateCharacterVariableId = 801;
                if (!Life.isInDate())
                    return false;
                if ($gameVariables.value(dateCharacterVariableId) !== data.character)
                    return false;
                return true;
            }
            text(id) {
                const data = this.getData(id);
                if (!data)
                    return "";
                return Iz.Text.get("QUEST_CONDITION_FORMAT_28");
            }
            getData(id) {
                return $dataQuestConditionDate.get(id);
            }
        }
        class ConditionCheckRelationship {
            check(id) {
                const data = this.getData(id);
                if (!data)
                    return false;
                return Life.Status.getRelationShip(data.character) === data.relationship;
            }
            text(id) {
                const data = this.getData(id);
                if (!data)
                    return "";
                const actor = $gameActors.actor(data.character);
                if (!actor)
                    return "";
                const format = Iz.Text.get("QUEST_CONDITION_FORMAT_29");
                return Iz.String.format(format, actor.name(), Life.Status.getRelationshipName(data.relationship));
            }
            getData(id) {
                return $dataQuestConditionRelationship.get(id);
            }
        }
        class ConditionCheckDairyBirthed {
            check(id) {
                const data = this.getData(id);
                if (!data)
                    return false;
                return !!Life.Dairy.birthedLivestock === data.birthed;
            }
            text(id) {
                const data = this.getData(id);
                if (!data)
                    return "";
                return Iz.Text.get(data.birthed ? "QUEST_CONDITION_FORMAT_30_1" : "QUEST_CONDITION_FORMAT_30_2");
            }
            getData(id) {
                return $dataQuestConditionDairyBirthed.get(id);
            }
        }
        class ConditionCheckFalseOrTrue {
            check(id) {
                return !!id;
            }
            text(id) {
                return id ? "true" : "false";
            }
        }
        class RewardCheckItem {
            give(id) {
                const data = this.getData(id);
                if (!data)
                    return;
                $gameParty.gainItem($dataItems[data.item_id], data.num);
            }
            text(id) {
                const data = this.getData(id);
                if (!data)
                    return "";
                return Iz.String.format(Iz.Text.get("QUEST_REWARD_FORMAT_1"), $dataItems[data.item_id].name, data.num);
            }
            getData(id) {
                const data = $dataQuestRewardItem.get(id);
                return data;
            }
        }
        class RewardCheckMoney {
            give(id) {
                const data = this.getData(id);
                if (!data)
                    return;
                $gameParty.gainGold(data.money);
            }
            text(id) {
                const data = this.getData(id);
                if (!data)
                    return "";
                return Iz.String.format(Iz.Text.get("QUEST_REWARD_FORMAT_2"), data.money, TextManager.currencyUnit);
            }
            getData(id) {
                const data = $dataQuestRewardMoney.get(id);
                return data;
            }
        }
        class RewardCheckLikability {
            give(id) {
                const data = this.getData(id);
                if (!data)
                    return;
                Life.Status.addLikability(data.character, data.likability);
            }
            text(id) {
                const data = this.getData(id);
                if (!data)
                    return "";
                const actor = $gameActors.actor(data.character);
                return Iz.String.format(Iz.Text.get("QUEST_REWARD_FORMAT_3"), actor ? actor.name() : "", data.likability);
            }
            getData(id) {
                const data = $dataQuestRewardLikability.get(id);
                return data;
            }
        }
        class RewardCheckLivestock {
            give(id) {
                const data = this.getData(id);
                if (!data)
                    return;
                for (let i = 0; i < data.num; i++) {
                    const option = {
                        id: data.livestock,
                        shedId: data.shed_id,
                    };
                    Life.Dairy.addLivestock(option, false);
                }
            }
            text(id) {
                const data = this.getData(id);
                if (!data)
                    return "";
                const dataLivestock = $dataDairyLivestock[data.livestock];
                return Iz.String.format(Iz.Text.get("QUEST_REWARD_FORMAT_4"), dataLivestock ? dataLivestock.name : "", data.num);
            }
            getData(id) {
                const data = $dataQuestRewardLivestock.get(id);
                return data;
            }
        }
        class RewardCheckFacility {
            give(id) {
                const data = this.getData(id);
                if (!data)
                    return;
                Life.FacilityUtil.setLevel(data.facility_id, data.level);
            }
            text(id) {
                return Iz.Text.get("QUEST_REWARD_FORMAT_5");
            }
            getData(id) {
                const data = $dataQuestRewardFacility.get(id);
                return data;
            }
        }
        class RewardCheckGP {
            give(id) {
                const data = this.getData(id);
                if (!data)
                    return;
                Life.Status.addGoddessPoint(data.value);
            }
            text(id) {
                return "";
            }
            getData(id) {
                const data = $dataQuestRewardGP.get(id);
                return data;
            }
        }
        class RewardCheckRelationship {
            give(id) {
                const data = this.getData(id);
                if (!data)
                    return;
                Life.Status.setRelationShip(data.character, data.relationship);
            }
            text(id) {
                return "";
            }
            getData(id) {
                const data = $dataQuestRewardRelationship.get(id);
                return data;
            }
        }
        class RewardCheckNpc {
            give(id) {
                const data = this.getData(id);
                if (!data)
                    return;
                if (data.command === Life.Data.QuestNpcCommand.Appear) {
                    const hour = Iz.GlobalMap.getChronus().getHour();
                    $globalMap.appearNonPlayer(data.character, hour);
                }
            }
            text(id) {
                return "";
            }
            getData(id) {
                const data = $dataQuestRewardNpc.get(id);
                return data;
            }
        }
        class RewardCheckVariable {
            give(id) {
                const data = this.getData(id);
                if (!data)
                    return;
                const value = Number(data.value);
                if (Number.isFinite(value)) {
                    if (Number.isInteger(value)) {
                        $gameVariables.setValue(data.variable_id, value);
                    }
                    else {
                        $gameVariables.setValueFloat(data.variable_id, value);
                    }
                }
                else {
                    $gameVariables.setValue(data.variable_id, data.value);
                }
            }
            text(id) {
                return "";
            }
            getData(id) {
                const data = $dataQuestRewardVariable.get(id);
                return data;
            }
        }
        class QuestProxy {
            initialize() {
                this._list = new Map();
                for (const key of $dataQuest.keys()) {
                    this._list.set(key, {
                        id: key,
                        state: Life.QuestState.Init,
                        progress: 0,
                        param: {
                            leftDays: 0,
                            reorderLeftDays: 0,
                        },
                    });
                }
                this._conditionChecks = [];
                this._conditionChecks[Life.Data.QuestConditionType.Time] = new ConditionCheckTime();
                this._conditionChecks[Life.Data.QuestConditionType.Item] = new ConditionCheckItem();
                this._conditionChecks[Life.Data.QuestConditionType.Group] = new ConditionCheckGroup();
                this._conditionChecks[Life.Data.QuestConditionType.Map] = new ConditionCheckMap();
                this._conditionChecks[Life.Data.QuestConditionType.Location] = new ConditionCheckLocation();
                this._conditionChecks[Life.Data.QuestConditionType.Weather] = new ConditionCheckWeather();
                this._conditionChecks[Life.Data.QuestConditionType.TalkCharacter] =
                    new ConditionCheckTalkCharacter();
                this._conditionChecks[Life.Data.QuestConditionType.Likability] =
                    new ConditionCheckLikability();
                this._conditionChecks[Life.Data.QuestConditionType.Quest] = new ConditionCheckQuest();
                this._conditionChecks[Life.Data.QuestConditionType.Target] = new ConditionCheckTarget();
                this._conditionChecks[Life.Data.QuestConditionType.Livestock] =
                    new ConditionCheckLivestock();
                this._conditionChecks[Life.Data.QuestConditionType.Wakeup] = new ConditionCheckWakeup();
                this._conditionChecks[Life.Data.QuestConditionType.CharacterMap] =
                    new ConditionCheckCharacterMap();
                this._conditionChecks[Life.Data.QuestConditionType.CharacterLocation] =
                    new ConditionCheckCharacterLocation();
                this._conditionChecks[Life.Data.QuestConditionType.ProgressMax] = new ConditionCheck15();
                this._conditionChecks[Life.Data.QuestConditionType.DayOfWeek] =
                    new ConditionCheckDayOfWeek();
                this._conditionChecks[Life.Data.QuestConditionType.Building] = new ConditionCheckBuilding();
                this._conditionChecks[Life.Data.QuestConditionType.LikabilityLevel] =
                    new ConditionCheckLikabilityLevel();
                this._conditionChecks[Life.Data.QuestConditionType.NpcMap] = new ConditionCheckNpcMap();
                this._conditionChecks[Life.Data.QuestConditionType.WorkLevel] =
                    new ConditionCheckWorkLevel();
                this._conditionChecks[Life.Data.QuestConditionType.Request] = new ConditionCheckRequest();
                this._conditionChecks[Life.Data.QuestConditionType.ElapsedDays] =
                    new ConditionCheckElapsedDays();
                this._conditionChecks[Life.Data.QuestConditionType.PlaceGroup] =
                    new ConditionCheckPlaceGroup();
                this._conditionChecks[Life.Data.QuestConditionType.DairyHatched] =
                    new ConditionCheckDairyHatched();
                this._conditionChecks[Life.Data.QuestConditionType.DairyLivestocks] =
                    new ConditionCheckDairyLivestocks();
                this._conditionChecks[Life.Data.QuestConditionType.Season] = new ConditionCheckSeason();
                this._conditionChecks[Life.Data.QuestConditionType.Buy] = new ConditionCheckBuy();
                this._conditionChecks[Life.Data.QuestConditionType.Date] = new ConditionCheckDate();
                this._conditionChecks[Life.Data.QuestConditionType.Relationship] =
                    new ConditionCheckRelationship();
                this._conditionChecks[Life.Data.QuestConditionType.DairyBirthed] =
                    new ConditionCheckDairyBirthed();
                this._conditionChecks[Life.Data.QuestConditionType.FalseOrTrue] =
                    new ConditionCheckFalseOrTrue();
                this._rewardChecks = [];
                this._rewardChecks[Life.Data.QuestRewardType.Item] = new RewardCheckItem();
                this._rewardChecks[Life.Data.QuestRewardType.Money] = new RewardCheckMoney();
                this._rewardChecks[Life.Data.QuestRewardType.Likability] = new RewardCheckLikability();
                this._rewardChecks[Life.Data.QuestRewardType.Livestock] = new RewardCheckLivestock();
                this._rewardChecks[Life.Data.QuestRewardType.Facility] = new RewardCheckFacility();
                this._rewardChecks[Life.Data.QuestRewardType.Relationship] = new RewardCheckRelationship();
                this._rewardChecks[Life.Data.QuestRewardType.GP] = new RewardCheckGP();
                this._rewardChecks[Life.Data.QuestRewardType.Npc] = new RewardCheckNpc();
                this._rewardChecks[Life.Data.QuestRewardType.Variable] = new RewardCheckVariable();
                this._conditionResults = {
                    start: new Map(),
                    complete: new Map(),
                };
                this._checkIfNeededOnMapEnter = false;
                this._readGroupFlgs = new Set();
            }
            applyQuestData(saveData) {
                saveData.list.forEach((data) => {
                    const existData = this.getQuestData(data.id);
                    if (existData) {
                        Object.assign(existData, data);
                    }
                    else {
                        // マスターデータにないデータは取り込まない
                    }
                });
                saveData.readGroupFlgs.forEach((data) => {
                    this.setReadGroupFlg(data, true);
                });
            }
            getQuestData(questId) {
                return this._list.get(questId);
            }
            getQuestDataList() {
                return Array.from(this._list.values());
            }
            getReadGroupFlgs() {
                return [...this._readGroupFlgs.values()];
            }
            getProgress(questId) {
                const quesstData = this.getQuestData(questId);
                return quesstData?.progress ?? 0;
            }
            getProgressMax(questId) {
                const questData = this.getQuestData(questId);
                if (!questData)
                    return 0;
                const limit = Math.max(0, $dataQuestContent.filter((c) => !!c && c.id === questId).length - 1);
                return limit;
            }
            setProgress(questId, progress) {
                const questData = this.getQuestData(questId);
                if (!questData)
                    return;
                const limit = this.getProgressMax(questId);
                questData.progress = Math.min(Math.max(progress, 0), limit);
            }
            getQuestContent(questId) {
                // 進行度に応じた内容を取得
                return $dataQuestContent
                    .filter((c) => !!c)
                    .filter((c) => c.id === questId)
                    .sort((a, b) => a.index - b.index)
                    .find((c) => c.index === this.getProgress(questId));
            }
            checkUnlock() {
                const notifyList = [];
                [...this._list.values()]
                    .filter((data) => data.state === Life.QuestState.Init)
                    .forEach((data) => {
                    if (this.checkUnlockCondition(data.id)) {
                        if (this.unlock(data.id)) {
                            notifyList.push({ questId: data.id, type: NotifyType.Unlock });
                        }
                    }
                });
                return notifyList;
            }
            checkStart(autoCheck) {
                const notifyList = [];
                [...this._list.values()]
                    .filter((data) => data.state === Life.QuestState.Unlocked)
                    .filter((data) => {
                    // complate()とcheck()時はauto_startがtrueの時のみ自動的に開始する
                    if (autoCheck) {
                        const dataQuest = $dataQuest.get(data.id);
                        return dataQuest?.auto_start;
                    }
                    return true;
                })
                    .forEach((data) => {
                    if (this.checkStartCondition(data.id)) {
                        if (this.start(data.id)) {
                            notifyList.push({ questId: data.id, type: NotifyType.Start });
                        }
                    }
                });
                return notifyList;
            }
            checkForward() {
                const notifyList = [];
                [...this._list.values()]
                    .filter((data) => data.state === Life.QuestState.InProgress)
                    .forEach((data) => {
                    // TODO: 最後の進行度（content）かどうか判断する
                    let forwared = false;
                    while (this.getProgress(data.id) < this.getProgressMax(data.id)) {
                        if (this.checkCompleteCondition(data.id)) {
                            if (this.forward(data.id)) {
                                if (!forwared) {
                                    forwared = true;
                                    notifyList.push({ questId: data.id, type: NotifyType.Forward });
                                }
                            }
                        }
                        else {
                            break;
                        }
                    }
                });
                return notifyList;
            }
            checkComplete() {
                const notifyList = [];
                [...this._list.values()]
                    .filter((data) => data.state === Life.QuestState.InProgress)
                    .forEach((data) => {
                    if (this.checkCompleteCondition(data.id)) {
                        const success = this.complete(data.id);
                        if (success) {
                            notifyList.push({ questId: data.id, type: NotifyType.Complete });
                        }
                    }
                });
                return notifyList;
            }
            // 条件を満たす全てのクエストをチェックする
            check() {
                const notifyList = [];
                while (true) {
                    const list = [];
                    const unlocks = this.checkUnlock();
                    const starts = this.checkStart(true);
                    const forwards = this.checkForward();
                    const completes = this.checkComplete();
                    list.push(...unlocks);
                    list.push(...starts);
                    list.push(...forwards);
                    list.push(...completes);
                    if (list.length === 0) {
                        break;
                    }
                    notifyList.push(...list);
                }
                this.notify(notifyList);
            }
            requestCheck() {
                this._checkIfNeededOnMapEnter = true;
            }
            checkIfNeeded() {
                if (this._checkIfNeededOnMapEnter) {
                    this.check();
                    this._checkIfNeededOnMapEnter = false;
                }
            }
            checkOnStart(questId) {
                const notifyList = [];
                if (this.start(questId)) {
                    notifyList.push({ questId: questId, type: NotifyType.Start });
                }
                this.notify(notifyList);
            }
            checkOnRestart(questId) {
                const notifyList = [];
                if (this.restart(questId)) {
                    notifyList.push({ questId: questId, type: NotifyType.Start });
                }
                this.notify(notifyList);
            }
            checkOnForward(questId) {
                const notifyList = [];
                if (this.forward(questId)) {
                    notifyList.push({ questId: questId, type: NotifyType.Forward });
                }
                this.notify(notifyList);
            }
            // 完了後のアンロック・開始のみをチェック
            checkOnComplete(questId) {
                const notifyList = [];
                const success = this.complete(questId);
                if (success) {
                    notifyList.push({ questId: questId, type: NotifyType.Complete });
                }
                while (true) {
                    const unlocks = this.checkUnlock();
                    const starts = this.checkStart(true);
                    notifyList.push(...unlocks);
                    notifyList.push(...starts);
                    if (unlocks.length === 0 && starts.length === 0) {
                        break;
                    }
                }
                this.notify(notifyList);
            }
            checkOnForceComplete(questId) {
                const notifyList = [];
                const success = this.forceComplete(questId);
                if (success) {
                    notifyList.push({ questId: questId, type: NotifyType.Complete });
                }
                while (true) {
                    const unlocks = this.checkUnlock();
                    const starts = this.checkStart(true);
                    notifyList.push(...unlocks);
                    notifyList.push(...starts);
                    if (unlocks.length === 0 && starts.length === 0) {
                        break;
                    }
                }
                this.notify(notifyList);
            }
            notify(list) {
                // NOTE:
                // 通知が多い場合にどうするか
                // 通知タイプが最後に近いもののみ表示する
                const notifyMap = new Map();
                list.forEach((info) => {
                    const data = notifyMap.get(info.questId);
                    if (data) {
                        if (info.type > data.type) {
                            data.type = info.type;
                        }
                    }
                    else {
                        notifyMap.set(info.questId, info);
                    }
                });
                let notified = false;
                notifyMap.forEach((info, i) => {
                    const questContent = this.getQuestContent(info.questId);
                    if (questContent) {
                        if (info.type === NotifyType.Unlock) {
                            Iz.System.notify(Iz.String.format(Iz.Text.get("QUEST_NOTIFY_UNLOCK"), questContent.title), i === 0);
                            notified = true;
                        }
                        else if (info.type === NotifyType.Start) {
                            Iz.System.notify(Iz.String.format(Iz.Text.get("QUEST_NOTIFY_START"), questContent.title), i === 0);
                            notified = true;
                        }
                        else if (info.type === NotifyType.Forward) {
                            Iz.System.notify(Iz.String.format(Iz.Text.get("QUEST_NOTIFY_FORWARD")), i === 0);
                            notified = true;
                        }
                        else if (info.type === NotifyType.Complete) {
                            Iz.System.notify(Iz.String.format(Iz.Text.get("QUEST_NOTIFY_COMPLETE"), questContent.title), i === 0);
                            notified = true;
                        }
                        if (info.type === NotifyType.Complete) {
                            const rewards = this.getRewards(info.questId);
                            rewards
                                .map((r) => this.getRewardName(r))
                                .filter((text) => text)
                                .forEach((text) => {
                                // TODO: 報酬種別ごとの文言
                                Iz.System.notify(Iz.String.format(Iz.Text.get("QUEST_NOTIFY_REWARD"), text), false);
                                notified = true;
                            });
                        }
                    }
                });
                if (notified) {
                    Iz.System.playSe("system/Quest");
                }
            }
            unlock(questId) {
                const data = this.getQuestData(questId);
                if (!data || data.state !== Life.QuestState.Init)
                    return false;
                data.state = Life.QuestState.Unlocked;
                this.execOnBegin(questId);
                return true;
            }
            start(questId) {
                const data = this.getQuestData(questId);
                if (!data || data.state !== Life.QuestState.Unlocked)
                    return false;
                data.state = Life.QuestState.InProgress;
                this.execOnStart(questId);
                this.execOnBegin(questId);
                return true;
            }
            restart(questId) {
                const data = this.getQuestData(questId);
                if (!data || data.state !== Life.QuestState.Complete)
                    return false;
                data.state = Life.QuestState.InProgress;
                this.execOnStart(questId);
                this.execOnBegin(questId);
                return true;
            }
            forward(questId) {
                if (this.getState(questId) !== Life.QuestState.InProgress)
                    return false;
                const content = this.getQuestContent(questId);
                if (!content)
                    return false;
                const progress = this.getProgress(questId);
                this.setProgress(questId, progress + 1);
                const newProgress = this.getProgress(questId);
                if (newProgress <= progress) {
                    return false;
                }
                this.setReadGroupFlgByQuest(questId, true);
                this.execOnComplete(questId); // checkCompleteConditionで確認した１つ前のcontentの完了後の処理
                this.execOnBegin(questId); // 現在のcontentの開始時の処理
                return true;
            }
            complete(questId) {
                const data = this.getQuestData(questId);
                if (!data || data.state !== Life.QuestState.InProgress)
                    return false;
                const quest = $dataQuest.get(questId);
                if (!quest)
                    return false;
                data.state = Life.QuestState.Complete;
                data.param.reorderLeftDays = quest.reorder_days;
                this.setReadGroupFlgByQuest(questId, true);
                this.execOnComplete(questId);
                this.giveReward(questId);
                return true;
            }
            forceComplete(questId) {
                const data = this.getQuestData(questId);
                if (!data || data.state === Life.QuestState.Complete)
                    return false;
                const quest = $dataQuest.get(questId);
                if (!quest)
                    return false;
                data.state = Life.QuestState.Complete;
                data.param.leftDays = 0;
                data.param.reorderLeftDays = quest.reorder_days;
                this.setProgress(questId, this.getProgressMax(questId));
                // NOTE:
                // エディタ側管理の進行度を完了済みの数値にする(1000)
                const COMPLETED_VALUE = 1000;
                $gameVariables.setValue(questId, COMPLETED_VALUE);
                // NOTE:
                // onComplete()でのアイテム消費や依頼アイテムの消費がされなくなるが、
                // デバッグ用途での影響はないため許容する
                // this.execOnComplete();
                this.giveReward(questId);
                return true;
            }
            isReadable(questId) {
                const data = $dataQuest.get(questId);
                if (!data) {
                    return false;
                }
                return !this._readGroupFlgs.has(data.read_group);
            }
            setReadGroupFlg(readGroup, flg) {
                if (readGroup <= 0) {
                    return;
                }
                if (flg) {
                    this._readGroupFlgs.add(readGroup);
                }
                else {
                    this._readGroupFlgs.delete(readGroup);
                }
            }
            setReadGroupFlgByQuest(questId, flg) {
                const quest = $dataQuest.get(questId);
                if (!quest) {
                    return false;
                }
                return this.setReadGroupFlg(quest.read_group, flg);
            }
            giveReward(questId) {
                const quest = $dataQuest.get(questId);
                if (!quest)
                    return;
                const rewards = $dataQuestReward.filter((data) => data && data.id === quest.reward);
                rewards.forEach((reward) => {
                    const checker = this._rewardChecks[reward.type];
                    checker.give(reward.type_id);
                });
            }
            getRewards(questId) {
                const quest = $dataQuest.get(questId);
                if (!quest)
                    return [];
                const rewards = $dataQuestReward.filter((data) => data && data.id === quest.reward);
                return rewards;
            }
            getRewardName(reward) {
                return this._rewardChecks[reward.type].text(reward.type_id);
            }
            getConditions(conditionId) {
                const conditions = $dataQuestCondition.filter((data) => data && data.id === conditionId);
                return conditions;
            }
            // getConditionName(condition: Data.QuestCondition) {
            getConditionName(type, type_id) {
                return this._conditionChecks[type].text(type_id);
            }
            hasStartConditionType(questId, type) {
                const quest = $dataQuest.get(questId);
                if (!quest)
                    return false;
                const content = this.getQuestContent(questId);
                if (!content)
                    return false;
                const list = $dataQuestCondition.filter((data) => data && data.id === quest.condition_start);
                return list.some((data) => data.type === type);
            }
            hasCompleteConditionType(questId, type) {
                const quest = $dataQuest.get(questId);
                if (!quest)
                    return false;
                const content = this.getQuestContent(questId);
                if (!content)
                    return false;
                const list = $dataQuestCondition.filter((data) => data && data.id === content.condition_complete);
                return list.some((data) => data.type === type);
            }
            checkActiveCondition(questId) {
                const quest = $dataQuest.get(questId);
                if (!quest)
                    return false;
                return this.checkCondition(quest.condition_active).result;
            }
            checkUnlockCondition(questId) {
                if (!this.checkActiveCondition(questId)) {
                    return false;
                }
                const quest = $dataQuest.get(questId);
                return this.checkCondition(quest ? quest.condition_unlock : 0).result;
            }
            checkStartCondition(questId) {
                if (!this.checkActiveCondition(questId)) {
                    return false;
                }
                const quest = $dataQuest.get(questId);
                if (!quest)
                    return false;
                if (!this.isReadable(questId))
                    return false;
                const placeCondition = this.checkPlaceCondition(quest.condition_start_place);
                const startCondition = this.checkCondition(quest.condition_start);
                const success = placeCondition && startCondition.result;
                if (success) {
                    this._conditionResults.start.set(questId, startCondition);
                }
                else {
                    this._conditionResults.start.delete(questId);
                }
                return success;
            }
            checkRestartCondition(questId) {
                const data = this.getQuestData(questId);
                if (!data)
                    return false;
                if (data.param.reorderLeftDays > 0)
                    return false;
                // NOTE:
                // 現状、残り日数以外は開始条件と同じにしている
                return this.checkStartCondition(questId);
            }
            checkCompleteCondition(questId) {
                if (!this.checkActiveCondition(questId)) {
                    return false;
                }
                const content = this.getQuestContent(questId);
                if (!content)
                    return false;
                if (!this.isReadable(questId))
                    return false;
                const placeCondition = this.checkPlaceCondition(content.place);
                const completeCondition = this.checkCondition(content.condition_complete);
                const success = placeCondition && completeCondition.result;
                if (success) {
                    this._conditionResults.complete.set(questId, completeCondition);
                }
                else {
                    this._conditionResults.complete.delete(questId);
                }
                return success;
            }
            getConditionStatus(conditionId) {
                const res = {
                    or: [],
                    and: [],
                };
                // or
                {
                    const list = $dataQuestCondition
                        .filter((data) => data && data.id === conditionId)
                        .filter((data) => data.operator === Life.Data.QuestConditionOperator.Or)
                        .map((data) => {
                        return {
                            data: data,
                            meet: this.checkConditionType(data.type, data.type_id),
                        };
                    });
                    res.or = list;
                }
                // and
                {
                    const list = $dataQuestCondition
                        .filter((data) => data && data.id === conditionId)
                        .filter((data) => data.operator === Life.Data.QuestConditionOperator.And)
                        .map((data) => {
                        return {
                            data: data,
                            meet: this.checkConditionType(data.type, data.type_id),
                        };
                    });
                    res.and = list;
                }
                return res;
            }
            checkCondition(conditionId) {
                const list = this.getConditionStatus(conditionId);
                const orTarget = list.or.find((status) => status.meet);
                if (list.or.length > 0 && !orTarget) {
                    return { result: false, list: [] };
                }
                const res = list.and.every((status) => status.meet);
                const resList = [];
                if (orTarget) {
                    resList.push({ type: orTarget.data.type, type_id: orTarget.data.type_id });
                }
                resList.push(...list.and
                    .filter((status) => status.meet)
                    .map((status) => {
                    return { type: status.data.type, type_id: status.data.type_id };
                }));
                return { result: res, list: resList };
            }
            checkPlaceCondition(placeId) {
                const place = $dataQuestPlace.get(placeId);
                if (!place)
                    return true;
                return this.checkConditionType(place.type, place.type_id);
            }
            checkConditionType(type, type_id) {
                const obj = this._conditionChecks[type];
                return obj ? obj.check(type_id) : true;
            }
            getState(questId) {
                const data = this.getQuestData(questId);
                if (!data)
                    return Life.QuestState.Init;
                return data.state;
            }
            getLeftDays(questId) {
                const data = this.getQuestData(questId);
                return data?.param.leftDays ?? 0;
            }
            setLeftDays(questId, days) {
                const data = this.getQuestData(questId);
                if (!data)
                    return;
                data.param.leftDays = days;
            }
            newday() {
                const notifyList = this.checkUnlock();
                this.notify(notifyList);
                this._list.forEach((data) => {
                    data.param.leftDays = Math.max(0, data.param.leftDays - 1);
                    data.param.reorderLeftDays = Math.max(0, data.param.reorderLeftDays - 1);
                });
                this._readGroupFlgs.clear();
            }
            isPermanent(questId) {
                return !!$dataQuest.get(questId)?.permanent;
            }
            execOnBegin(questId) {
                let condition = 0;
                if (this.getState(questId) === Life.QuestState.Unlocked) {
                    condition = $dataQuest.get(questId)?.condition_start ?? 0;
                }
                else if (this.getState(questId) === Life.QuestState.InProgress) {
                    condition = this.getQuestContent(questId)?.condition_complete ?? 0;
                }
                this.getConditions(condition).forEach((c) => {
                    this._conditionChecks[c.type].onBegin?.(c.type_id);
                });
            }
            execOnStart(questId) {
                // 開始時処理
                const result = this._conditionResults.start.get(questId);
                result?.list.forEach((c) => {
                    this._conditionChecks[c.type].onStart?.(c.type_id);
                });
            }
            execOnComplete(questId) {
                // 完了後処理
                const result = this._conditionResults.complete.get(questId);
                result?.list.forEach((c) => {
                    this._conditionChecks[c.type].onComplete?.(c.type_id);
                });
            }
            convertEscapeCharacterQuest(placeId, text_) {
                const place = $dataQuestPlace.get(placeId);
                const placeText = place ? Life.Quest.getConditionName(place.type, place.type_id) : "";
                let text = text_;
                text = text.replace(/\x1bQplace/gi, placeText);
                text = text.replace(/\\Qplace/gi, placeText);
                text = Iz.GlobalMap.convertEscapeCharacterGM(text);
                return text;
            }
        }
        Life.Quest = new QuestProxy();
        function initializeQuest() {
            Life.Quest.initialize();
        }
        Life.initializeQuest = initializeQuest;
        function isWakeup() {
            return $gameSwitches.value(WakeupSwitchId);
        }
        Life.isWakeup = isWakeup;
        function getLivestocksByGroup(livestockGroup) {
            const sets = new Set($dataDairyLivestockGroupContent
                .filter((data) => data && data.id === livestockGroup)
                .map((data) => data.livestock_id));
            return [...sets];
        }
        Life.getLivestocksByGroup = getLivestocksByGroup;
    })(Life = Iz.Life || (Iz.Life = {}));
})(Iz || (Iz = {}));
(function (Iz) {
    var Life;
    (function (Life) {
        var QuestUtil;
        (function (QuestUtil) {
            function check() {
                Life.Quest.check();
            }
            QuestUtil.check = check;
            function meetSpecificStartCondition(questId) {
                if (Life.MemoryUtil.isMemoryScene())
                    return false;
                // デート中はデート中条件のみ対象
                if (Life.isInDate()) {
                    if (!Life.Quest.hasStartConditionType(questId, Life.Data.QuestConditionType.Date)) {
                        return false;
                    }
                }
                return true;
            }
            function meetStartCondition(questId) {
                if (!meetSpecificStartCondition(questId))
                    return false;
                return (Life.Quest.getState(questId) === Life.QuestState.Unlocked && Life.Quest.checkStartCondition(questId));
            }
            QuestUtil.meetStartCondition = meetStartCondition;
            function meetRestartCondition(questId) {
                if (!meetSpecificStartCondition(questId))
                    return false;
                return (Life.Quest.getState(questId) === Life.QuestState.Complete && Life.Quest.checkRestartCondition(questId));
            }
            QuestUtil.meetRestartCondition = meetRestartCondition;
            function meetSpecificCompleteCondition(questId) {
                if (Life.MemoryUtil.isMemoryScene())
                    return false;
                // デート中はデート中条件のみ対象
                if (Life.isInDate()) {
                    if (!Life.Quest.hasCompleteConditionType(questId, Life.Data.QuestConditionType.Date)) {
                        return false;
                    }
                }
                return true;
            }
            function meetForwardCondition(questId) {
                if (!meetSpecificCompleteCondition(questId))
                    return false;
                if (Life.Quest.getState(questId) !== Life.QuestState.InProgress)
                    return false;
                if (Life.Quest.getProgress(questId) >= Life.Quest.getProgressMax(questId))
                    return false;
                return Life.Quest.checkCompleteCondition(questId);
            }
            QuestUtil.meetForwardCondition = meetForwardCondition;
            function meetCompleteCondition(questId) {
                if (!meetSpecificCompleteCondition(questId))
                    return false;
                return (Life.Quest.getState(questId) === Life.QuestState.InProgress &&
                    Life.Quest.checkCompleteCondition(questId));
            }
            QuestUtil.meetCompleteCondition = meetCompleteCondition;
            function start(questId) {
                if (Life.MemoryUtil.isMemoryScene())
                    return;
                Life.Quest.checkOnStart(questId);
            }
            QuestUtil.start = start;
            function restart(questId) {
                if (Life.MemoryUtil.isMemoryScene())
                    return;
                Life.Quest.checkOnRestart(questId);
            }
            QuestUtil.restart = restart;
            function forward(questId) {
                if (Life.MemoryUtil.isMemoryScene())
                    return;
                Life.Quest.checkOnForward(questId);
            }
            QuestUtil.forward = forward;
            function saveMemory(questId) {
                if (isHeart(questId)) {
                    Life.MemoryUtil.setUnlockQuest(questId, true);
                    Life.saveSystemData();
                }
            }
            function complete(questId) {
                if (Life.MemoryUtil.isMemoryScene())
                    return;
                Life.Quest.checkOnComplete(questId);
                saveMemory(questId);
            }
            QuestUtil.complete = complete;
            function forceComplete(questId) {
                if (Life.MemoryUtil.isMemoryScene())
                    return;
                Life.Quest.checkOnForceComplete(questId);
                saveMemory(questId);
            }
            QuestUtil.forceComplete = forceComplete;
            function newday() {
                Life.Quest.newday();
            }
            QuestUtil.newday = newday;
            function isUnlocked(questId) {
                return Life.Quest.getState(questId) === Life.QuestState.Unlocked;
            }
            QuestUtil.isUnlocked = isUnlocked;
            function isInProgress(questId) {
                return Life.Quest.getState(questId) === Life.QuestState.InProgress;
            }
            QuestUtil.isInProgress = isInProgress;
            function isCompleted(questId) {
                return Life.Quest.getState(questId) === Life.QuestState.Complete;
            }
            QuestUtil.isCompleted = isCompleted;
            function getProgress(questId) {
                return Life.Quest.getProgress(questId);
            }
            QuestUtil.getProgress = getProgress;
            function setReadGroupFlg(questId, flg) {
                return Life.Quest.setReadGroupFlgByQuest(questId, flg);
            }
            QuestUtil.setReadGroupFlg = setReadGroupFlg;
            // 常設クエストの受注数
            function getOrderedPermanents() {
                return Life.Quest.getQuestDataList()
                    .filter((data) => isInProgress(data.id))
                    .filter((data) => Life.Quest.isPermanent(data.id)).length;
            }
            QuestUtil.getOrderedPermanents = getOrderedPermanents;
            // 受注可能か
            function canOrder(questId) {
                return meetRestartCondition(questId) || meetStartCondition(questId);
            }
            QuestUtil.canOrder = canOrder;
            function isHeart(questId) {
                return !!$dataQuest.get(questId)?.heart;
            }
            QuestUtil.isHeart = isHeart;
        })(QuestUtil = Life.QuestUtil || (Life.QuestUtil = {}));
    })(Life = Iz.Life || (Iz.Life = {}));
})(Iz || (Iz = {}));
(() => {
    const _createGameObjects = DataManager.createGameObjects;
    DataManager.createGameObjects = function () {
        _createGameObjects.call(this);
        Iz.Life.initializeQuest();
    };
    const _DataManager_makeSaveContents = DataManager.makeSaveContents;
    DataManager.makeSaveContents = function () {
        const contents = _DataManager_makeSaveContents.call(this);
        contents.quest = {
            list: Iz.Life.Quest.getQuestDataList(),
            readGroupFlgs: Iz.Life.Quest.getReadGroupFlgs(),
        };
        return contents;
    };
    const _DataManager_extractSaveContents = DataManager.extractSaveContents;
    DataManager.extractSaveContents = function (contents) {
        _DataManager_extractSaveContents.call(this, contents);
        Iz.Life.Quest.applyQuestData(contents.quest);
    };
    const _Scene_Boot_onDatabaseLoaded = Scene_Boot.prototype.onDatabaseLoaded;
    Scene_Boot.prototype.onDatabaseLoaded = function () {
        _Scene_Boot_onDatabaseLoaded.call(this);
        if (!Utils.isOptionValid("test_quest")) {
            const questList = [...$dataQuest.values()].filter((data) => !data.test);
            $dataQuest.clear();
            for (const data of questList) {
                $dataQuest.set(data.id, data);
            }
        }
        if (Iz.Life.isTrial()) {
            const fullversionOnlyQuests = [...$dataQuest.values()]
                .filter((data) => !!data && data.fullversion_only)
                .map((data) => data.id);
            const contents = $dataQuestContent.filter((data) => !!data && !fullversionOnlyQuests.includes(data.id));
            $dataQuestContent.splice(0);
            // NOTE: dbConvererでnullを0番目に挿入するのを廃止した方がいいかも？
            // @ts-ignore
            $dataQuestContent.push(null);
            $dataQuestContent.push(...contents);
            const questList = [...$dataQuest.values()].filter((data) => !data.fullversion_only);
            $dataQuest.clear();
            for (const data of questList) {
                $dataQuest.set(data.id, data);
            }
        }
    };
    const _Scene_Load_terminate = Scene_Load.prototype.terminate;
    Scene_Load.prototype.terminate = function () {
        _Scene_Load_terminate.call(this);
        if (this._loadSuccess) {
            Iz.Life.Quest.requestCheck();
        }
    };
    const _Scene_Map_start = Scene_Map.prototype.start;
    Scene_Map.prototype.start = function () {
        _Scene_Map_start.call(this);
        Iz.Life.Quest.checkIfNeeded();
    };
})();
//# sourceMappingURL=Quest.js.map