/*:
 * @target MZ
 * @plugindesc
 *
 * @help
 * ショップの処理を商品の追加と表示に分けることで
 * 細かな在庫制御を可能にしたシンプルなプラグインです
 *
 *
 * @command initialize
 * @text initialize
 * @desc ショップの初期化
 *
 * @command addGoods
 * @text addGoods
 * @desc 商品のリストを追加
 *
 * @arg items
 * @text 商品に追加するアイテム
 * @desc 商品に追加するアイテム
 * @type item[]
 * @default []
 *
 * @arg priceRate
 * @text 価格倍率
 * @desc 商品価格を倍率で手動設定します
 *
 * @arg free
 * @text 商品価格を0に設定します
 * @desc 商品価格を0に設定します
 * @type boolean
 *
 * @command addShopItem
 * @text addShopItem
 * @desc 商品を追加
 *
 * @arg item
 * @text 商品に追加するアイテム
 * @desc 商品に追加するアイテム
 * @type item
 *
 * @arg price
 * @text 価格
 * @desc 商品価格を手動設定します 設定が無い場合はデフォルトの価格になります
 * @type number
 *
 * @arg priceRate
 * @text 価格倍率
 * @desc 商品価格を倍率で手動設定します priceの方が優先されます
 *
 * @arg limit
 * @text 1度の購入数制限
 * @desc 1以上の数値で1度の購入数を制限します それ以外は無制限です
 * @type number
 *
 *
 * @command open
 * @text open
 * @desc ショップを開く
 *
 * @arg closeSceneAfterPurchase
 * @text 購入後にショップシーンを閉じる
 * @desc アイテムを一つ購入するとショップシーンを閉じるようになります
 * @type boolean
 * @default false
 *
 * @command closeSceneAfterPurchase
 * @text closeSceneAfterPurchase
 * @desc アイテムを一つ購入するとショップシーンを閉じるようになります
 *
 * @command addShopTags
 * @text addShopTags
 * @desc ショップ識別用タグ付け
 *
 * @arg tags
 * @text タグ
 * @desc タグ
 * @type string[]
 * @default []
 *
 */
/*~struct~ShopItem:
 * @param item
 * @text アイテム
 * @desc アイテム
 * @type item
 *
 * @param price
 * @text 価格
 * @desc 商品価格を手動設定します 設定が無い場合はデフォルトの価格になります
 *
 * @param limit
 * @text 1度の購入数制限
 * @desc 1以上の数値で1度の購入数を制限します それ以外は無制限です
 * @type number
 *
 */
(function () {
    // 種別
    // 0: アイテム 1: 武器 2: 防具
    // 価格の上書き判定
    // 0: デフォルト価格 0以外: 上書き価格を使用
    let closeSceneAfterPurchase = false;
    const shopItems = [];
    const shopTags = [];
    function initializeShop() {
        shopItems.splice(0);
        shopTags.splice(0);
    }
    const ItemKind = 0;
    function createGood(itemId, overridePrice) {
        const good = [
            ItemKind,
            itemId,
            overridePrice !== undefined ? 1 : 0,
            overridePrice !== undefined ? overridePrice : 0,
        ];
        return good;
    }
    PluginManagerEx.registerCommand(document.currentScript, "initialize", (args) => {
        initializeShop();
    });
    PluginManagerEx.registerCommand(document.currentScript, "addGoods", (args) => {
        const items = args.items;
        items.forEach((item) => {
            let price = 0;
            if (args.priceRate) {
                const dataItem = $dataItems[item];
                if (dataItem) {
                    price = Math.floor(dataItem.price * args.priceRate);
                }
            }
            if (args.free === true) {
                price = 0;
            }
            shopItems.push({
                item: item,
                overridePrice: args.free === true || price > 0,
                price: price,
                limit: 0,
            });
        });
    });
    PluginManagerEx.registerCommand(document.currentScript, "addShopItem", (args) => {
        const itemId = args.item || 0;
        if (!itemId)
            return;
        const free = args.price === 0 || args.priceRate === 0;
        let price = 0;
        if (typeof args.price === "number" && Number.isInteger(args.price)) {
            price = args.price;
        }
        else if (args.priceRate) {
            const dataItem = $dataItems[itemId];
            if (dataItem) {
                price = Math.floor(dataItem.price * args.priceRate);
            }
        }
        const shopItem = {
            item: itemId,
            overridePrice: free || price > 0,
            price: price,
            limit: Number(args.limit) || 0,
        };
        shopItems.push(shopItem);
    });
    PluginManagerEx.registerCommand(document.currentScript, "open", (args) => {
        SceneManager.push(Scene_Shop);
        const goods = [];
        shopItems.forEach((item) => {
            const good = createGood(item.item, item.overridePrice ? item.price : undefined);
            goods.push(good);
        });
        SceneManager.prepareNextScene(goods, true);
        closeSceneAfterPurchase = args.closeSceneAfterPurchase === true;
    });
    PluginManagerEx.registerCommand(document.currentScript, "closeSceneAfterPurchase", (args) => {
        closeSceneAfterPurchase = true;
    });
    PluginManagerEx.registerCommand(document.currentScript, "addShopTags", (args) => {
        const tags = args.tags;
        shopTags.push(...tags);
    });
    const _dobuy = Scene_Shop.prototype.doBuy;
    Scene_Shop.prototype.doBuy = function (number) {
        _dobuy.call(this, number);
        if (closeSceneAfterPurchase) {
            this.popScene();
            closeSceneAfterPurchase = false;
        }
    };
    const _Scene_Shop_maxBuy = Scene_Shop.prototype.maxBuy;
    Scene_Shop.prototype.maxBuy = function () {
        const max = _Scene_Shop_maxBuy.call(this);
        const item = this._item;
        if (DataManager.isItem(item)) {
            const target = shopItems.find((shopItem) => {
                return shopItem.item === item.id;
            });
            if (target && target.limit > 0) {
                return Math.min(max, target.limit);
            }
        }
        return max;
    };
    const _Scene_Shop_terminate = Scene_Shop.prototype.terminate;
    Scene_Shop.prototype.terminate = function () {
        _Scene_Shop_terminate.call(this);
        initializeShop();
        closeSceneAfterPurchase = false;
    };
    Scene_Shop.prototype.getShopTags = function () {
        return shopTags;
    };
})();
//# sourceMappingURL=Shop.js.map